/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.network;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.net.ssl.SSLSocket;

public class HttpInputStream {

	public static final String SIMPLE_RESPONSE_HEADER = "HTTP/1.0 200 OK SIMPLE RESPONSE\r\n\r\n";
	
	/**	Note the BUFFER SIZE cannot be made larger.  There is a problem using
		larger size.  The process will consume lots of CPU time. */
	private static final int	BUFFER_SIZE = 4096;
	private static final String CRLF = "\r\n";
	private static final String CRLF2 = CRLF + CRLF;
	private static final String LF = "\n";
	private static final String LF2 = LF + LF;
	
	private static final Pattern patternChunked
		= Pattern.compile("(\\w+);?\\S*\\s*", Pattern.CASE_INSENSITIVE | Pattern.MULTILINE);	//;?\\S*\\s*", Pattern.CASE_INSENSITIVE);
	
	private HttpHeader mHeader = null;
	private int readBodyLength = 0;						// body length read so far
	private byte[] buf = new byte[BUFFER_SIZE];			// read buffer for body
	//private InputStream in = null;
	private BufferedInputStream in = null;
	private String simpleResponseBuffer = null;			// buffer for simple response during header processing
	private StringBuffer msb = new StringBuffer(1024);
	private long mLastActiveTime = System.currentTimeMillis();
	private LastActiveAdapter mLastActiveAdapter = null;
	private boolean mCloseConnection = true;
	private Socket mSocket = null;
	
	public HttpInputStream(InputStream in) {
//		this.in = in;
//		there is a bug using BufferedInputStream eating 99% CPU time so go back to use general InputStream.
		this.in = new BufferedInputStream(in);
	}
	
	public HttpInputStream(HttpConnection conn, LastActiveAdapter lastActiveAdapter) throws IOException {
		this(conn.mSocket.getInputStream());
		mSocket = conn.mSocket;
		mLastActiveAdapter = lastActiveAdapter;
	}

	public void setSocket(Socket socket) {
		mSocket = socket;
	}

	public int available() throws IOException {
		int avail = 0;
		int oneByte = -1;
		int timeout = 0;
		avail = in.available();
		if (avail == 0 && mSocket != null && mSocket instanceof SSLSocket) {
			try {
				timeout = mSocket.getSoTimeout();
				mSocket.setSoTimeout(1);
				in.mark(256);
				oneByte = in.read();
				in.reset();
				avail = in.available();
			} catch (SocketTimeoutException e) {
				avail = 0;
			} finally {
				mSocket.setSoTimeout(timeout);
			}
		}
		
		return avail;
	}

	/**
	Read a header at the beginning.
	*/
	public synchronized HttpHeader readHeader() throws HttpMalformedHeaderException, IOException {
		String msg = "";
        //byte onebyte[] = new byte[1];
        int		oneByte = -1;
        boolean eoh = false;
        boolean neverReadOnce = true;
        int len = 0;
        boolean headerCompared = false;

		processBeforeReadHeader();
		
		in.mark(256);		// allow restoring buffer if HTTP/0.9
        do {
        	oneByte = in.read();
        	
        	if (oneByte == -1) {
        		eoh = true;
        		if (neverReadOnce) {
        			sleep(20);
        			continue;
        		}
				break;
        	} else {
        		neverReadOnce = false;
        	}
            msb.append((char) oneByte);

			if (!headerCompared && msb.length() > 7) {
				msg = msb.toString();
				headerCompared = true;
				if (!HttpRequestHeader.isRequestLine(msg) && !HttpResponseHeader.isStatusLine(msg)) {
					eoh = true;
					in.reset();
					msg = SIMPLE_RESPONSE_HEADER;
				}
			}

            if (((char) oneByte) == '\n' && isHeaderEnd(msb)) {
                eoh = true;
                msg = msb.toString();
            }
		} while (!eoh || neverReadOnce);

		setLastActive();
		mHeader = HttpHeader.getInstance(msg);
		processAfterReadHeader();
		return mHeader;

	}
	
	private boolean isHeaderEnd(StringBuffer sb) {
		int len = sb.length();
		if (len > 2) {
			if (LF2.equals(sb.substring(len-2))) {
				return true;
			}
		}
		
		if (len > 4) {
			if (CRLF2.equals(sb.substring(len-4))) {
				return true;
			}
		}
		
		return false;
	}

	private int readBody(byte[] data) throws IOException {
		int len = 0;
		int remainingLen = 0;

		if (mHeader.getContentLength() == -1) {
			len = in.read(data);
		} else {
			remainingLen = mHeader.getContentLength() - readBodyLength;
			if (remainingLen < data.length && remainingLen > 0) {
				len = in.read(data,0,remainingLen);
			} else if (remainingLen > data.length) {
				len = in.read(data);
			}
		}

		if (len > 0) {
			setLastActive();
			readBodyLength += len;
		}

		return len;
	}

	public HttpBody readBody() throws IOException {
		return readBody(mHeader);
	}

	public HttpBody readBody(HttpHeader hdr) {

		HttpBody body = null;
		
		if (hdr.isTransferEncodingChunked()) {
			body = readBodyChunked(hdr);
		} else {
			body = new HttpBody();
			readEntity(hdr.getContentLength(), body, null);
		}
		body.setLength(body.length());
		return body;
		
	}

	private void processBeforeReadHeader() {
		msb.setLength(0);
		mCloseConnection = true;
        readBodyLength = 0;
	}

	private void processAfterReadHeader() {
	}
			
	private int readEntity(int expectedLength, HttpBody entityBody, HttpOutputStream out) {

		int len = 0;
		int remainingLength = 0;
		int readEntityLength = 0;
		
		/*
		// code obsolete after use of buffered stream
		if (simpleResponseBuffer != null) {
			byte[] temp = simpleResponseBuffer.getBytes("8859_1");
			for (int i=0; i<temp.length; i++) {
				data[i] = temp[i];
			}
			len = temp.length;
			simpleResponseBuffer = null;
		} else
		*/

		if (expectedLength == 0) {
			return 0;
		}

		remainingLength = expectedLength - readEntityLength;
				
		try {			
			do {
	
				len = 0;
						
				if (expectedLength < 0 || remainingLength > buf.length) {
					len = in.read(buf);
				} else {
					//(remainingLength < buf.length && remainingLength > 0) {
					len = in.read(buf, 0, remainingLength);
				}

				if (len > 0) {
					setLastActive();
					readEntityLength += len;	
					remainingLength = expectedLength - readEntityLength;

					// add to entity body if provided
					if (entityBody != null) {
						entityBody.append(buf, len);
					}
	
					// pipe to output 
					if (out != null) {
						out.write(buf,len);
						out.flush();
					}
				} else {
					sleep(10);
				}
				
			} while (len != -1 && (expectedLength < 0 || expectedLength > 0 && remainingLength > 0));
		} catch (IOException e) {
			close();
		}
		
		return readEntityLength;
	}
	
	public HttpBody readBodyChunked(HttpHeader hdr) {
		int len = 0;
		int readLen = 0;

		// Refer RFC 2616
		// section 3.6.1 on chunked format
		// section 19.4.6 on chunked transfer coding pseudo code
		
		String line = null;
		int chunkSize = 0;
		int readChunkSize = 0;
		HttpBody body = new HttpBody();
		
		/*
      	length := 0
       	read chunk-size, chunk-extension (if any) and CRLF
       	while (chunk-size > 0) {
        	read chunk-data and CRLF
          	append chunk-data to entity-body
          	length := length + chunk-size
          	read chunk-size and CRLF
       	}
       	read entity-header
       	while (entity-header not empty) {
        	append entity-header to existing header fields
          	read entity-header
       	}
       
		*/

		Matcher matcher = null;
		try {
			//line = readLine().replaceAll("(\\r\\n|\\n| )","");
			line = readLine();
			matcher = patternChunked.matcher(line);
			matcher.find();
			String result = matcher.group(1);

			chunkSize = Integer.parseInt(result, 16);	//matcher.group(1), 16);
			while (chunkSize > 0) {
				readChunkSize = 0;
				while (readChunkSize < chunkSize) {
					len = readEntity(chunkSize - readChunkSize, body, null);
					readChunkSize += len;
				}
				readLen += chunkSize;
				line = readLine();	// read CRLF after entity
				line = readLine();	// read chunk header
				//line = readLine().replaceAll("(\\r\\n|\\n| )","");
				matcher = patternChunked.matcher(line);
				matcher.find();
				chunkSize = Integer.parseInt(matcher.group(1), 16);	//matcher.group(1), 16);
			}
			
			String entityHeaderLine = null;
			do {
				entityHeaderLine = readLine();
			} while (!entityHeaderLine.equals(hdr.getDelimiter()));
		} catch (IOException e) {
		}
						
		// Content-Length := length
       	// Remove "chunked" from Transfer-Encoding


		hdr.setHeader(HttpHeader.TRANSFER_ENCODING, null);
		hdr.setContentLength(readLen);
		body.setLength(body.length());

		return body;
	}
	
	private String readLine() throws IOException {
        int		oneByte = -1;
        boolean eol = false;
        int len = 0;
        String msg = null;
		msb.setLength(0);

		do {
			oneByte = in.read();
	        if (oneByte == -1) {
	        	eol = true;
	        } else {
	        	msb.append((char) oneByte);
	    	}

			msg = msb.toString();
	        if (msg.endsWith(CRLF) || msg.endsWith(LF)) {
		        eol = true;
	        }
	    } while (!eol);
	    
		setLastActive();
		return msb.toString();
	}
	
	
	/*
	public synchronized boolean isEndOfBody(HttpHeader hdr) throws IOException {
		if (hdr.getContentLength() == -1) {
			return false;
		} else if (hdr.getContentLength() > readBodyLength) {
			return false;
		
		//} else if (simpleResponseBuffer != null) {
		//	return false;
		//
		} else {
			return true;
		}
	}
	*/

	public void close() {
		try {
			in.close();
		} catch (Exception e) {
		}
		HttpUtil.close(mSocket);
	}

	public synchronized void pipeBody(HttpOutputStream out) throws IOException {

/*
		int len = 0;
		while (header.getContentLength() != 0 && !isEndOfBody() && len >= 0) {
			len = readBody(buf);
			if (len > 0) {
				out.write(buf,len);
				out.flush();
			} else {
				break;
			}
			
		}
*/
		pipeBody(out, false);
	}
	
	public synchronized HttpBody pipeBody(HttpOutputStream out, boolean isReturnBody) throws IOException {

		HttpBody body = null;
		
		if (isReturnBody) {
			body = new HttpBody();
		}

		readEntity(mHeader.getContentLength(), body, out);
		return body;
	}

	private void sleep(int ms) {
		try {
			Thread.sleep(ms);
		} catch (InterruptedException e) {
		}
	}

	private void setLastActive() {
		mLastActiveTime = System.currentTimeMillis();
		if (mLastActiveAdapter != null) {
			mLastActiveAdapter.setLastActive();
		}
	}
	
	long lastActiveTimeMillis() {
		return mLastActiveTime;
	}
	
	public int read() throws IOException {
		return in.read();
	}
	public int read(byte[] b) throws IOException {
		return in.read(b);
	}
	
	public int read(byte[] b, int off, int len) throws IOException {
		return in.read(b, off, len);
	}
		



}
