/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.net.URI;
import java.util.Hashtable;
import java.util.Random;
import java.util.regex.Pattern;

import com.proofsecure.paros.network.HttpBody;
import com.proofsecure.paros.network.HttpHeader;
import com.proofsecure.paros.network.HttpRequestHeader;
import com.proofsecure.paros.network.HttpResponseHeader;
import com.proofsecure.paros.network.HttpStatusCode;
import com.proofsecure.paros.Global;

class Analyser extends AbstractTest {

	private Hashtable mAnalysedEntityTable = null;
	private static Random	staticRandomGenerator = 	new Random();
	
	/** remove HTML HEAD as this may contain expiry time which dynamic changes */
	private static final String p_REMOVE_HEADER = "(?m)(?i)(?s)<HEAD>.*?</HEAD>";
	private static final String[] staticSuffixList = { ".cfm", ".jsp", ".php", ".asp", ".aspx", ".dll", ".exe", ".pl"};
	private final static Pattern patternNotFound = Pattern.compile("(\\bnot\\b(found|exist))|(\\b404\\berror\\b)|(\\berror\\b404\\b)", PATTERN_PARAM);

	public Analyser() {
		mAnalysedEntityTable = new Hashtable();
	}

	public String getTestName() {
		return "Analyser";
	}

	protected void scan() throws Exception {
	}


	void startTest(ParsedEntity startEntity) {
		mAnalysedEntityTable.clear();

		if (startEntity.isRoot() && startEntity.getChildCount() == 0) {
			return;
		}

		writeOutput("Running analyser ...");
		
		analyseServer(startEntity);

		// analyse host so if startEntity does not include a host it will be checked


		if (!startEntity.isRoot()) {
			try {
				preTestAnalyse(getHostEntity(startEntity));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

	}
	/**
	Check if given response header and body indicated the file exist.
	*/	
	protected boolean isFileExist(ParsedEntity entity, HttpRequestHeader reqh, HttpResponseHeader resh, HttpBody resBody) {

		// RFC
		if (resh.getStatusCode() == HttpStatusCode.NOT_FOUND) {
			return false;
		}
	
		ParsedEntity refEntity = entity;
		if (refEntity.isLeaf() && !((ParsedEntity) refEntity.getParent()).isRoot()) {
			refEntity = (ParsedEntity) refEntity.getParent();
		}

		// get sample with same relative path position when possible.
		// if not exist, use the rotot path.	
		SampleResponse sample = (SampleResponse) mAnalysedEntityTable.get(refEntity.getURIHostPath().toString());
		if (sample == null) {
			sample = (SampleResponse) mAnalysedEntityTable.get(getHostEntity(refEntity).getURIHostPath().toString());
		}
		
		// check if any analysed result.

		if (sample == null) {
			System.out.println("sample not found. RefEntity = " + refEntity.getURIHostPath().toString());
			if (resh.getStatusCode() == HttpStatusCode.OK) {
				// no anlaysed result to confirm, assume file exist and return
				return true;
			} else {
				return false;
			}
		}
		
		// check for redirect response.  If redirect to same location, then file does not exist
		if (HttpStatusCode.isRedirection(resh.getStatusCode())) {
			try {
				if (sample.resHeader.getStatusCode() == resh.getStatusCode()) {
					String location = resh.getHeader(HttpHeader.LOCATION);
					if (location != null && location.equals(sample.resHeader.getHeader(HttpHeader.LOCATION))) {
						return false;
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
			return true;
		}
		
		// Not success code
		if (resh.getStatusCode() != HttpStatusCode.OK) {
			return false;
		}

		// remain only OK response here

		String body = resBody.toString().replaceAll(p_REMOVE_HEADER, "");
		if (sample.errorPageType == SampleResponse.ERROR_PAGE_STATIC) {
			if (sample.resBody.equals(body)) {
				return false;
			}
			return true;
		}

		try {
			URI uri = new URI(reqh.getURIHostPathQuery());
			if (sample.errorPageType == SampleResponse.ERROR_PAGE_DYNAMIC_BUT_DETERMINISTIC) {
				body = resBody.toString().replaceAll(getPathRegex(uri), "");
				if (sample.resBody.equals(body)) {
					return false;
				}
				return true;
			}
		} catch (Exception e) {
			e.printStackTrace();

		}

		// nothing more to determine.  Check for possible not found page pattern.
		if (matchBodyPattern(patternNotFound, null)) {
			return false;
		}

		return true;		
	}		

	/**
	Analyse entity (should be a folder unless it is host level) in-order.
	*/
	private void analyseServer(ParsedEntity entity) {

		ParsedEntity tmp = null;
		
		if (entity == null) {
			return;
		}
		// analyse entity if not root and not leaf.
		// Leaf is not analysed because only folder entity is used to determine if path exist.
		try {
			if (!entity.isRoot()) {
				if (!entity.isLeaf() || entity.isLeaf() && ((ParsedEntity) entity.getParent()).isRoot()) {
					preTestAnalyse(entity);
				}
			}
		} catch (Exception e) {
			e.printStackTrace();

		}
				
		for (int i=0; i<entity.getChildCount(); i++) {
			try {
				tmp = (ParsedEntity) entity.getChildAt(i);
				//if (!tmp.isLeaf()) {
					analyseServer(tmp);
				//}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	Analyse a single folder entity.  Results are stored into mAnalysedEntityTable.
	*/
	private void preTestAnalyse(ParsedEntity entity) throws Exception {

		HttpResponseHeader	resHeader	= null;
		HttpBody			resBody		= null;

		// if analysed already, return;
		// move to host part
		ParsedEntity folderEntity = entity;
		
		// try 2 random file which does not exist
		init(folderEntity);

		//System.out.println("analysing: " + getRequestHeader().getURIHostPath());

		// already exist one.  no need to test
		synchronized (mAnalysedEntityTable) {
			if (mAnalysedEntityTable.get(getRequestHeader().getURIHostPath()) != null) {
				return;
			}
		}

		URI tmpUri = new URI(getRequestHeader().getURIHostPath());
		String path = getRandomPathSuffix(folderEntity, tmpUri);
		URI uri1 = new URI(tmpUri.getScheme(), null, tmpUri.getHost(), tmpUri.getPort(), path, null, null);
		//System.out.println("comparing url: " + uri1.toString());
		getRequestHeader().setURI(uri1.toString());

		//System.out.println("analysing2 " + getRequestHeader().getURIHostPath());		

		sendAndReceive();

		resHeader = new HttpResponseHeader(getResponseHeader().toString());


		// standard RFC response, no further check is needed
		
		
		if (resHeader.getStatusCode() == HttpStatusCode.NOT_FOUND) {
			addAnalysedHost(folderEntity, resHeader, "", SampleResponse.ERROR_PAGE_RFC);
			return;
		}

		if (HttpStatusCode.isRedirection(resHeader.getStatusCode())) {
			addAnalysedHost(folderEntity, resHeader, "", SampleResponse.ERROR_PAGE_REDIRECT);
			return;
		}
		if (resHeader.getStatusCode() != HttpStatusCode.OK) {
			addAnalysedHost(folderEntity, resHeader, "", SampleResponse.ERROR_PAGE_NON_RFC);
			return;
		}
	

		resHeader = new HttpResponseHeader(getResponseHeader().toString());
		resBody		= new HttpBody(getResponseBody().toString());

		init(folderEntity);
		tmpUri = new URI(getRequestHeader().getURIHostPath());
		path = getRandomPathSuffix(folderEntity, tmpUri);
		URI uri2 = new URI(tmpUri.getScheme(), null, tmpUri.getHost(), tmpUri.getPort(), path, null, null);
		//System.out.println("comparing url: " + uri2.toString());

		getRequestHeader().setURI(uri2.toString());
		sendAndReceive();

		// remove HTML HEAD as this may contain expiry time which dynamic changes		
		String resBody1 = resBody.toString().replaceAll(p_REMOVE_HEADER, "");
		String resBody2 = getResponseBody().toString().replaceAll(p_REMOVE_HEADER, "");

		// check if page is static.  If so, remember this static page
		if (resBody1.equals(resBody2)) {
			addAnalysedHost(folderEntity, resHeader, resBody1, SampleResponse.ERROR_PAGE_STATIC);
			return;
		}

		// else check if page is dynamic but deterministic
		resBody1 = resBody1.toString().replaceAll(getPathRegex(uri1),"");
		resBody2 = resBody2.toString().replaceAll(getPathRegex(uri2),"");
		if (resBody1.equals(resBody2)) {
			addAnalysedHost(folderEntity, resHeader, resBody1, SampleResponse.ERROR_PAGE_DYNAMIC_BUT_DETERMINISTIC);
			return;
		}

		// else mark app "undeterministic".
		addAnalysedHost(folderEntity, resHeader, "", SampleResponse.ERROR_PAGE_UNDETERMINISTIC);
	
		
	}

	/**
	Get a random path relative to the current entity.  Whenever possible, use a suffix exist in the children
	according to a priority of staticSuffixList.
	@param	entity	The current entity.
	@param	uri		The uri of the current entity.
	@return	A random path (eg /folder1/folder2/1234567.chm) relative the entity.
	*/	
	private String getRandomPathSuffix(ParsedEntity entity, URI uri) {
		String resultSuffix = getChildSuffix(entity, true);
		
		String path = "";
		path = (uri.getPath() == null) ? "" : uri.getPath();
		path = path + (path.endsWith("/") ? "" : "/") + Long.toString(Math.abs(staticRandomGenerator.nextLong())) + Global.getEyeCatcher();
		path = path + resultSuffix;

		return path;

	}
	
	/**
	Get a suffix from the children which exists in staticSuffixList.
	An option is provided to check recursively.    Note that the immediate
	children are always checked first before further recursive check is done.
	@param	entity	The current entity.
	@param	performRecursiveCheck	True = get recursively the suffix from all the children.
	@return	The suffix ".xxx" is returned.  If there is no suffix found, an empty string is returned.
	*/
	private String getChildSuffix(ParsedEntity entity, boolean performRecursiveCheck) {

		String resultSuffix = "";
		String suffix = null;
		ParsedEntity childEntity = null;
		try {

			for (int i=0; i<staticSuffixList.length; i++) {
				suffix = staticSuffixList[i];
				for (int j=0; j<entity.getChildCount(); j++) {
					childEntity = (ParsedEntity) entity.getChildAt(j);
					if (childEntity.getURIHostPath().getPath().endsWith(suffix)) {
						return suffix;
					}
				}
			}
			
			if (performRecursiveCheck) {
				for (int j=0; j<entity.getChildCount(); j++) {
					resultSuffix = getChildSuffix((ParsedEntity) entity.getChildAt(j), performRecursiveCheck);
					if (!resultSuffix.equals("")) {
						return resultSuffix;
					}
				}
			}
														
		} catch (Exception e) {
		}
		
		return resultSuffix;
	}

	
	private void addAnalysedHost(ParsedEntity entity, HttpResponseHeader resHeader, String body, int errorPageType) {


		String host = null;
		host = entity.getURIHostPath().toString();

		SampleResponse sample = new SampleResponse();
		sample.resHeader = resHeader;
		sample.resBody			= body;
		sample.errorPageType = errorPageType;

		synchronized (mAnalysedEntityTable) {
			mAnalysedEntityTable.put(host, sample);
		}
	}

	/**
	Get entity up to the root indicating the host path of this entity.
	*/
	private ParsedEntity getHostEntity(ParsedEntity entity) {
		ParsedEntity hostEntity = entity;
		if (hostEntity.isRoot()) {
			return hostEntity;
		}
		
		while (!((ParsedEntity) hostEntity.getParent()).isRoot()) {
			hostEntity = (ParsedEntity) hostEntity.getParent();
		}
		return hostEntity;
	}

	private String getPathRegex(URI uri) {
		StringBuffer sb = new StringBuffer(100);
		
		// case should be sensitive
		//sb.append("(?i)");
		
		String host = uri.getScheme() + "://" + uri.getHost();
		if (uri.getPort() != -1) {
			host = host + ":" + uri.getPort();
		}
		sb.append(getPathPattern(host));
		
		String path = "";
		if (uri.getPath() != null) {
			path = uri.getPath();
			if (uri.getQuery() != null) {
				path = path + "(\\?" + uri.getQuery() + ")?";
			}
			sb.append(path);			
		}
		
		return sb.toString();
	}
	
	private String getPathPattern(String path) {
		return "(" + path + ")?";
	}
	
}

class SampleResponse {

	static final int	ERROR_PAGE_RFC							= 0;
	static final int	ERROR_PAGE_NON_RFC						= 1;
	static final int	ERROR_PAGE_REDIRECT						= 2;
	static final int	ERROR_PAGE_STATIC						= 3;
	static final int	ERROR_PAGE_DYNAMIC_BUT_DETERMINISTIC	= 4;
	static final int	ERROR_PAGE_UNDETERMINISTIC				= 5;

	HttpResponseHeader	resHeader = null;
	String				resBody	= "";
	int					errorPageType = ERROR_PAGE_RFC;
}