/*
Paros and its related class files.
Paros is an HTTP/HTTPS proxy for assessing web application security.
Copyright (C) 2003-2004 www.proofsecure.com

This program is free software; you can redistribute it and/or
modify it under the terms of the Clarified Artistic License
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
Clarified Artistic License for more details.

You should have received a copy of the Clarified Artistic License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package com.proofsecure.paros.scan;

import java.io.IOException;
import java.net.SocketException;
import java.io.EOFException;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import com.proofsecure.paros.network.HttpStatusCode;

class TestBufferOverflow extends TestAbstractParam {

	private static final int TEST_LENGTH = 16384;
	private static final String TEST_CHAR = "A";

	private static String 	staticLongStringA = null;
	private static String	staticLongFormatString = null;
	
	static {
		StringBuffer sb = new StringBuffer(TEST_LENGTH);
		for (int i=0; sb.length() < TEST_LENGTH; i++) {
			sb.append(TEST_CHAR);
		}
		staticLongStringA = sb.toString();
		
		sb.setLength(0);
		for (int i=0; sb.length() < TEST_LENGTH / 2; i++) {
			// append "%x%n" - format string
			sb.append("%25x%25n");
		}
		staticLongFormatString = sb.toString();
	}
	
	private static String[] PARAM_LIST = {staticLongStringA, staticLongFormatString};
	private static Pattern patternOverflow = Pattern.compile(TEST_CHAR + "{" + TEST_LENGTH/2 + "," + TEST_LENGTH + "}", PATTERN_PARAM);
	
	TestBufferOverflow() {
	}

    public String toString() {
        return "TestBufferOverflow";
    }
    
	public String getTestName() {
		return "Long input and buffer overflow";
	}	
	
	protected void check(boolean isBody, String paramKey, String paramValue, String query, int insertPos) throws IOException {

		String bingoQuery = null;
		String displayURI = null;
		String newQuery = null;
		boolean isError = false;				// check if abrupt IO error occured
		String resBodyNormal = null;
		
		int pos = 0;
		long defaultTimeUsed = 0;
		long timeUsed = 0;
		long lastTime = 0;
		
		// always try normal query first
		newQuery = insertQuery(query, insertPos, paramKey + "=" + paramValue);
		createMessage(isBody, newQuery);
		lastTime = System.currentTimeMillis();
		sendAndReceive();
		defaultTimeUsed = System.currentTimeMillis() - lastTime;
		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK) {
			return;
		}
		resBodyNormal = getResponseBody().toString();
		
		// loop parameters
		for (int i=0; i<PARAM_LIST.length; i++) {
			if (i == 0) {
				// try without the parameter
				bingoQuery = insertQuery(query, insertPos, "");
			} else {
				bingoQuery = insertQuery(query, insertPos, paramKey + "=" + PARAM_LIST[i]);
			}
			createMessage(isBody, bingoQuery);
			lastTime = System.currentTimeMillis();

			try {
				sendAndReceive();
			} catch (SocketException e) {
				// abrupt IO error occured - eg sudden close of connection
				isError = true;
			} catch (EOFException e) {
				// abrupt IO error occured - eg sudden close of connection
				isError = true;
			} catch (IOException e) {
				// other type of error, exit
				throw e;
			}

			defaultTimeUsed = System.currentTimeMillis() - lastTime;
			displayURI = getRequestHeader().getURIHostPathQuery();

			if (isError) {
				bingo(30001, AlertItem.RISK_HIGH, AlertItem.SUSPICIOUS, "", paramKey, "");
				return;
			}
		
			if (checkResult(bingoQuery, resBodyNormal)) {
				return;
			}
		}

		
	}

	private boolean checkResult(String paramKey, String normalHTTPResponse) {

		if (getResponseHeader().getStatusCode() != HttpStatusCode.OK
			&& !HttpStatusCode.isServerError(getResponseHeader().getStatusCode())) {
			return false;
		}
		
		// remove false positive if parameter have no effect on output
		if (getResponseBody().toString().equals(normalHTTPResponse)) {
			return false;
		}
		
		Matcher matcher = patternOverflow.matcher(getResponseHeader().toString());
		if (matcher.find()) {
			bingo(30000, AlertItem.RISK_MEDIUM, AlertItem.WARNING, "", paramKey, "");
			return true;
		}
		
		return false;
		
	}

	

	
	protected void scan() throws Exception {
		boolean skip = (getEntity().mState == ParsedEntity.DO_NOT_SCAN);
			
		writeStatus(getTestName() + ": " + (skip? "(skipped) " : "") + getRequestHeader().getURIHostPath());
		if (skip) {
			return;
		}
		init();
		checkUrlOrBody(false, myQuery);
		checkUrlOrBody(true, myReqBody.toString());
	}


}