/*
 *
 * Paros and its related class files.
 * 
 * Paros is an HTTP/HTTPS proxy for assessing web application security.
 * Copyright (C) 2003-2004 Chinotec Technologies Company
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the Clarified Artistic License
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Clarified Artistic License for more details.
 * 
 * You should have received a copy of the Clarified Artistic License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.parosproxy.paros.control;
 
import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.parosproxy.paros.Constant;
import org.parosproxy.paros.Paros;
import org.parosproxy.paros.model.Model;
import org.parosproxy.paros.model.Session;
import org.parosproxy.paros.view.SessionDialog;
import org.parosproxy.paros.view.View;


/**
 *
 * To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Generation - Code and Comments
 */
public class MenuFileControl {

    private static Log log = LogFactory.getLog(MenuFileControl.class);

    private View view = null;
    private Model model = null;
    private Control control = null;
    
    public MenuFileControl(Model model, View view, Control control) {
        this.view = view;
        this.model = model;
        this.control = control;
    }
    
	public void exit() {
	    if (model.getSession().isNewState()) {
			if (view.showConfirmDialog("The current session is not saved.  Discard?") != JOptionPane.OK_OPTION) {
				return;
			}
			model.getSession().discard();
	    }

	    control.shutdown();
	    log.info(Constant.PROGRAM_TITLE + " terminated.");
		System.exit(0);
	}
	
	public void newSession(boolean isPromptNewSession) throws ClassNotFoundException, Exception {
		if (isPromptNewSession) {
		    if (model.getSession().isNewState()) {
				if (view.showConfirmDialog("The current session is not saved.  Discard and create new session?") != JOptionPane.OK_OPTION) {
					return;
				}
				model.getSession().discard();
		    } else if (view.showConfirmDialog("The current session will be closed.  Create new session?") != JOptionPane.OK_OPTION) {
				return;
			}
			model.createAndOpenUntitledDb();
		}
		
		Session session = new Session(model);
	    log.info("new session file created");
	    model.setSession(session);

		view.getSiteTreePanel().getTreeSite().setModel(session.getSiteTree());
		
		if (isPromptNewSession) {
		    SessionDialog dialog = view.getSessionDialog("New Session");
		    dialog.initParam(session);
		    dialog.showDialog(false);
		    saveAsSession();
		}

		control.getExtensionLoader().sessionChangedAllPlugin(session);
		// refresh display
		
		view.getMainFrame().setTitle(Constant.PROGRAM_NAME + " " + Constant.PROGRAM_VERSION + " - " + session.getSessionName());
		
	}
	
	public void openSession() {
		JFileChooser chooser = new JFileChooser();
		File file = null;
	    chooser.setFileFilter(new FileFilter() {
	           public boolean accept(File file) {
	                if (file.getName().endsWith(".session")) {
	                    return true;
	                }
	                return false;
	            }
	           public String getDescription() {
	               return "Paros session";
	           }
	    });
	    int rc = chooser.showOpenDialog(view.getMainFrame());
	    if(rc == JFileChooser.APPROVE_OPTION) {
			try {
	    		file = chooser.getSelectedFile();
	    		if (file == null) {
	    			return;
	    		}
	    		Session session = model.getSession();
	    		session.open(file.getAbsolutePath());
	    	    log.info("opening session file " + file.getAbsolutePath());
	    	    control.getExtensionLoader().sessionChangedAllPlugin(session);
	    		view.getMainFrame().setTitle(Constant.PROGRAM_NAME + " - " + model.getSession().getSessionName());
			} catch (Exception e) {
			    view.showWarningDialog("Error opening session file");
	    	    log.error("error opening session file " + file.getAbsolutePath());
			    log.error(e.getMessage());
			}
	    }
	}
	public void saveSession() {
	    Session session = model.getSession();

	    if (session.isNewState()) {
		    view.showWarningDialog("Please use Save As...");
		    return;
	    }
	    
		try {
		    
    		session.save(session.getFileName());
    	    log.info("saving session file " + session.getFileName());
    	    
		} catch (Exception e) {
		    view.showWarningDialog("Error saving session file.");
    	    log.error("error saving session file " + session.getFileName());
    	    log.error(e.getMessage());
    	    
		}
	    
	}
	
	public boolean saveAsSession() throws Exception {
	    Session session = model.getSession();

	    JFileChooser chooser = new JFileChooser();
	    chooser.setFileFilter(new FileFilter() {
	           public boolean accept(File file) {
	                if (file.getName().endsWith(".session")) {
	                    return true;
	                }
	                return false;
	            }
	           public String getDescription() {
	               return "Paros session";
	           }
	    });
		File file = null;
	    int rc = chooser.showSaveDialog(view.getMainFrame());
	    if(rc == JFileChooser.APPROVE_OPTION) {
    		file = chooser.getSelectedFile();
    		if (file == null) {
    			return false;
    		}
    		String fileName = file.getAbsolutePath();
    		if (!fileName.endsWith(".session")) {
    		    fileName += ".session";
    		}
    		session.save(fileName);
	    } else {
	        return false;
	    }
	    return true;
	}
	
	public void properties() {
	    SessionDialog dialog = view.getSessionDialog("Session Properties");
	    dialog.initParam(model.getSession());
	    dialog.showDialog(false);
	}
}
