/*
*
* Paros and its related class files.
* 
* Paros is an HTTP/HTTPS proxy for assessing web application security.
* Copyright (C) 2003-2004 Chinotec Technologies Company
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the Clarified Artistic License
* as published by the Free Software Foundation.
* 
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* Clarified Artistic License for more details.
* 
* You should have received a copy of the Clarified Artistic License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package org.parosproxy.paros.core.spider;

import java.util.HashSet;
import java.util.Vector;

import org.apache.commons.httpclient.URI;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.parosproxy.paros.network.ConnectionParam;
import org.parosproxy.paros.network.HttpMessage;
import org.parosproxy.paros.network.HttpSender;




/**
 *
 * To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Generation - Code and Comments
 */
public class Spider {
    private static Log log = LogFactory.getLog(Spider.class);

	private HttpSender httpSender = null;
	private boolean isSubmitForm = true;
	private Vector listenerList = new Vector();
	private ConnectionParam connectionParam = null;
	private Vector queue = new Vector();
	private SpiderThread[] spiderThread = null;
	private HttpMessageQueue visitedLink = new HttpMessageQueue();
	private SpiderParam spiderParam = null;
	private HashSet seedHostNameSet = new HashSet();
	private int[] depthQueueCount = null;
	
	
	public Spider(SpiderParam spiderParam, ConnectionParam param) {
	    this.connectionParam = param;
	    this.spiderParam = spiderParam;
	    spiderThread = new SpiderThread[spiderParam.getThread()];
	    httpSender = new HttpSender(param);
	    httpSender.setFollowRedirect(true);
	    depthQueueCount = new int[spiderParam.getMaxDepth()+1];
	}

	public void addSpiderListener(SpiderListener listener) {
		listenerList.add(listener);		
	}
	
	public synchronized void addSeed(HttpMessage msg) {
	    URI uri = msg.getRequestHeader().getURI();
	    String hostName = null;
	    int port = 80;
	    
        try {
		    log.info("seeding " + msg.getRequestHeader().getURI().toString());

            hostName = uri.getHost();
            port = uri.getPort();
    	    if (port > 0) {
    	        hostName = hostName + ":" + Integer.toString(port);
    	    }
    	    
    	    seedHostNameSet.add(hostName);
    	    addQueue(msg, 0);

        } catch (URIException e) {
            e.printStackTrace();
        }
	    
	}
	
	public synchronized void addQueue(HttpMessage msg, int depth) {
	    
	    if (depth > spiderParam.getMaxDepth() || getVisitedLink().contains(msg)) {
	        return;
	    }
	    
	    // no need to add if in queue already
	    for (int i=0; i<queue.size(); i++) {
	        QueueItem item = (QueueItem) (queue.get(i));
	        if (item.equals(msg)) {
	            return;
	        }
	    }
	    
	    QueueItem item = new QueueItem(msg);
	    item.setDepth(depth);
	    queue.add(item);
	    if (depth < depthQueueCount.length) {
	        depthQueueCount[depth]++;
	    }
	}
    
    void checkIfAllThreadCompleted() {
        for (int i=0; i<spiderThread.length; i++) {
            if (!spiderThread[i].isCompleted()) {
                // cannot check thread alive here because child issued this check.  They
                // will not stop yet.
                return;
            }
        }
        
        // all thread finished running
        notifyListenerSpiderComplete();
        
    }



    /**
     * @return Returns the httpSender.
     */
    public HttpSender getHttpSender() {
        return httpSender;
    }

    /**
     * @return Returns the queue.
     */
    public Vector getQueue() {
        return queue;
    }


	
    /**
     * @return Returns the visited.
     */
    public HttpMessageQueue getVisitedLink() {
        return visitedLink;
    }
	
	private void notifyListenerFoundURI(HttpMessage msg) {
	    SpiderListener listener = null;
	    for (int i=0;i<listenerList.size();i++) {
	        listener = (SpiderListener) listenerList.get(i);
	        listener.foundURI(msg);
	    }

	}
	
	private void notifyListenerSpiderComplete() {
	    SpiderListener listener = null;
	    
	    notifyListenerSpiderProgress(null, 100);
	    
	    for (int i=0;i<listenerList.size();i++) {
	        listener = (SpiderListener) listenerList.get(i);
	        listener.spiderComplete();	        
	    }
	    log.info("Spider completed");
	    
	}

	synchronized void SpiderProgress(QueueItem item) {
	    int scale = 100/(spiderParam.getMaxDepth()+1);
	    int percentage = scale * item.getDepth();

	    int outstanding= 0;
	    for (int i=0;i<queue.size();i++) {
	        QueueItem poll= (QueueItem) queue.get(i);
	        if (poll != null) {
		        if (poll.getDepth() <= item.getDepth()) {
		            outstanding++;
		        }
	        }
	    }
	    
	    percentage += scale *(depthQueueCount[item.getDepth()]-outstanding)/depthQueueCount[item.getDepth()];

	    try {
	        notifyListenerSpiderProgress(item.getMessage().getRequestHeader().getURI(), percentage);
	        Thread.sleep(100);
	    } catch (Exception e) {
	     
	    }
	}
	
	private void notifyListenerSpiderProgress(URI uri, int percentageComplete) {
	    SpiderListener listener = null;

	    for (int i=0;i<listenerList.size();i++) {
	        listener = (SpiderListener) listenerList.get(i);
	        listener.spiderProgress(uri, percentageComplete);	        
	    }
	}

	private void notifyListenerReadURI(HttpMessage msg) {
	    SpiderListener listener = null;

	    log.info("crawled " + msg.getRequestHeader().getURI().toString());

	    for (int i=0;i<listenerList.size();i++) {
	        listener = (SpiderListener) listenerList.get(i);
	        listener.readURI(msg);
	    }

	}
	
	public void removeSpiderListener(SpiderListener listener) {
		listenerList.remove(listener);
	}
	
    /**
     * @return Returns the spiderParam.
     */
    public SpiderParam getSpiderParam() {
        return spiderParam;
    }
    /**
     * @param spiderParam The spiderParam to set.
     */
    public void setSpiderParam(SpiderParam spiderParam) {
        this.spiderParam = spiderParam;
    }
    /**
     * @param visited The visited to set.
     */
    public void setVisitedLink(HttpMessageQueue visitedLink) {
        this.visitedLink = visitedLink;
    }
    
    public void start() {
	    log.info("spider started.");

        for (int i=0; i<spiderThread.length; i++) {
            if (spiderThread[i] != null && spiderThread[i].isAlive()) {
                spiderThread[i].setStop(true);
            }

            spiderThread[i] = new SpiderThread(this);        
            spiderThread[i].start();
            
        }
    }
    
    public void stop() {
        for (int i=0; i<spiderThread.length; i++) {
            spiderThread[i].setStop(true);
            try {
                spiderThread[i].join(4000);
            } catch (InterruptedException e) {
            }
        }
	    log.info("spider stopped.");
	                
    }
    
    public boolean isSeedScope(URI uri) {
        boolean result = false;
        
        String hostName = null;
        try {
            hostName = uri.getHost();
            if (uri.getPort() > 0) {
                hostName = hostName + ":" + uri.getPort();
            }

            String[] hostList = (String[]) seedHostNameSet.toArray(new String[0]);
            for (int i=0; i<hostList.length; i++) {
                if (hostList[i] == null) continue;
                if (hostName.endsWith(hostList[i])) {
                    return true;
                }
            }

        } catch (URIException e) {
            e.printStackTrace();
        }
        return false;
    }

    void foundURI(HttpMessage msg) {
        notifyListenerFoundURI(msg);
        
    }
    
    void readURI(HttpMessage msg) {
        notifyListenerReadURI(msg);
    }

}



    
