/*
*   Copyright (C) 2000 PROTOS Project Consortium
*   [http://www.ee.oulu.fi/research/ouspg/protos]
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 2
*   as published by  the Free Software Foundation
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package FI.protos.ouspg.wrapper;

import FI.protos.wrapper.BugCat2;

import java.io.*;
import java.net.*;
import java.util.jar.*;
import java.lang.reflect.*;

/**
 * An udp Bug cat.
 */
public class UDPBugCat extends BugCat2 {

  /**
   * Default constructor. Pretty empty.
   */
  public UDPBugCat() {

  }
    
  /**
   * Command-line help.
   */
  public void help() {
    /* Overriden help method from superclas */
    System.out.println("Usage java -jar <jarfile>.jar [ [OPTIONS] | -host <hostngame> ]\n");
    System.out.println("  -delay <milliseconds>   Delay between cases" +
		       " send to host.\n                         " + 
		       " Defaults to 1000ms.");
    System.out.println("  -file <file>            Send file <file> instead" +
		       " of testcase(s)");
    System.out.println("  -help                   Display this help");
    System.out.println("  -host <hostname>        hostname to send" +
		       " packets");
    System.out.println("  -jarfile <file>         Get data from alternate" +
		       " bugcat JAR-file <file>");
    System.out.println("  -port <index>           Portnumber to send" +
		       " packets on host.\n                         " +
		       " Defaults to 9200.");
    System.out.println("  -single <index>         Inject single test " +
		       "case <index>");
    System.out.println("  -start <index>          Inject test cases " +
		       "starting from <index>");
    System.out.println("  -stop <index>           Stop test case " +
		       "injection to <index>");

    System.out.print ("\n");
    System.exit(0);
  }
  
  /**
   * Stub for Command-line help. Activated by -h command line option
   */
  public void h() {
    this.help ();
    System.exit(0);
  }

  /**
   * Hostname command line option.
   */
  private String hostname = null;
  
  /**
   * Set hostname by command line option '-host'.
   * @param s The hostname.
   */
  public void host (String s) {
    hostname = s;
  }

  /**
   * Port command line option. Defaults to 9200.
   */
  private int port = 9200;
  
  /**
   * Set port number by command line option '-port'.
   * @param s The port number.
   */
  public void port (String s) {
    try {
      port = new Integer (s).intValue();
    } catch (Exception e) {
      port = -1;
    }

    if (port < 1 || port > 0xffff) {
      System.out.println ("Invalid port number. Port number must be " +
			  "between [1..65536].");
      System.exit (0);
    }
  }

  /**
   * Injection delay command line option. Defaults to 1000 ms
   */
  private int injection_delay = 1000;
  
  /**
   * Set injection delay by command line option '-injection_delay'.
   */
  public void delay (String s) {
    try {
      injection_delay = new Integer (s).intValue();
    } catch (Exception e) {
      injection_delay = -1;
    }
    if (injection_delay < 0 || injection_delay > 360000) {
      System.out.println ("Invalid injection delay. Injection delay " + 
			  "must be between [0..360000] ms.");
      System.exit (0);
    }
  }
    
  /**
   * Main routine, parses command-line arguments and starts injection.
   * @param args Command-line arguments.
   */
  public static void main(String args[]) {
    UDPBugCat cat =new UDPBugCat();
    cat.run(args);
  }

  //
  // Method redefined from super class
  //
 
  public void prepare() throws IOException {
      /* Check that some hostname is given. Otherwise give help */
      if (hostname == null) {
	  this.help ();
      }
  }

  public void inject(int index, byte[] metaData, byte[] data)
    throws IOException {
      
    /* Display statistics about case to be injected */
    if (data.length > 65150) System.out.print ("Payload too large error" +
					       " [not injecting]: ");
      
    System.out.println("test case #" +index
		       +": meta-data " +metaData.length +" bytes, "
		       +"data " +data.length +" bytes");
    
    /* Check that we have a valid host before trying to inject anything */
    if (hostname != null) {
      /* check for valid host here */
      
      /* Apply injection delay */
      try {
	Thread.currentThread().sleep (injection_delay);
      } catch (Exception e) {
	System.out.println ("Excepted while waiting (injection delay) " +
			    "before next injection");
      }
      try {
	DatagramSocket socket =new DatagramSocket();
	InetAddress addr = InetAddress.getByName(hostname);
	DatagramPacket my_data = new DatagramPacket(data,data.length,
						    addr,port);
	socket.send(my_data);
	socket.close();
	socket = null;
      } catch (SocketException s) {
	System.out.println ("Cannot create datagram socket. Aborting.");
	System.exit (-1);
      } catch (UnknownHostException uh) {
	System.out.println ("Cannot resolve host. Aborting.");
	System.exit (-1);
      } catch (Exception e) {
	System.out.println ("Error building datagram. Aborting.");
	System.exit (-1);
      }
    } else {
      throw new IOException("Injecting testcase, but socket not open");
    } /* end if hostname != null */
  }
  
}







