package FI.protos.wrapper;

import java.io.*;
import java.util.jar.*;
import java.lang.reflect.*;

/**
 * Base class for test case injection utility for distribution.
 * Version for test cases injected into separate files.
 */
abstract public class BugCat2 {

  //
  // Methods to be redefined in real cat class.
  //

  /**
   * Command-line help.
   */
  public void help() {
    System.out.println(
       "  -single <index>         Inject single test case <index>\n"
      +"  -start <index>          Inject test cases starting from <index>\n"
      +"  -stop <index>           Stop test case injection to <index>\n"
      +"  -jarfile <file>         Get data from bugcat jar-file <file>\n"
      +"  -file <file>            Get single case from file <file>\n"
      +"  -help                   Give this help\n");
    System.exit(0);
  }

  /**
   * Prepare for injection.
   * Redefine in derived classes, if preparation needed.
   * @throws IOException If preparation fails.
   */
  public void prepare() throws IOException {
  }

  /**
   * Inject a test case.
   * The meta-data is reserved for future expansions, ignore now.
   * Redefine in derived classes.
   * @param index Index of injected test case.
   * @param metaData Meta-data of the test case.
   * @param data Data of the test case.
   * @throws IOException If the injection fails.
   */
  public void inject(int index, byte[] metaData, byte[] data)
    throws IOException {
  }

  /**
   * Finish after injection. Not called if injection fails.
   * Redefine in derived classes, if finish needed.
   * @throws IOException If finish fails.
   */
  public void finish() throws IOException {
  }

  //
  // Methods not needing redefinition
  //

  /**
   * Test case file prefix.
   */
  public static String CASE_FILE_PREFIX ="testcases/";

  /**
   * Magic 32-bit number in beginning of test case file.
   */
  public final static int MAGIC_NUMBER =0x00009e17;

  /**
   * Run me.
   * @param args Command-line arguments.
   */
  public void run(String args[]) {
    try {
      if (args.length ==0) { //give help
        System.out.println("(help by command line argument: -help)");
      }
      //parse command-line arguments
      int index =0;
      while (index <args.length) {
        String name =args[index];
        if (!name.startsWith("-")) {
          throw new IllegalArgumentException(
            "command line argument: " +name +" should start with '-'");
        }
        name =name.substring(1); //name of command-line argument
        index++;
        Method[] all_method =getClass().getMethods();
        Method method =null;
        for(int a =0; a <all_method.length; a++) { //find method to call
          if (all_method[a].getName().equals(name)) {
            method =all_method[a];
            break;
          }
        }
        if (method ==null) {
          throw new IllegalArgumentException(
            "unknown command line argument: -" +name);
        }
        //get parameters of the argument
        Object parameter[] =new Object[method.getParameterTypes().length];
        for(int a =0; a <parameter.length; a++) {
          if (index ==args.length) {
            throw new IllegalArgumentException(
              "missing " +(a +1) +". parameter for argument: -" +name);
          }
          parameter[a] =args[index];
          index++;
        }
        method.invoke(this, parameter); //call the method
      }
      parseTestCases();
    } catch (IOException e) {
      System.err.println("ERROR: " +e.getMessage() +"\n");
    } catch (IllegalArgumentException e) {
      System.err.println("ERROR: " +e.getMessage() +"\n");
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Current test-case file for debugging.
   */
  protected String filename;

  /**
   * Read buffer size.
   */
  protected final static int READ_BUFFER_SIZE =2048;

  /**
   * Parse test cases.
   * @throws IOException If cannot access or inject test cases.
   */
  public void parseTestCases() throws IOException {
    if (singleFile !=null) {
      parseSingleFile(); //parse a test case from single file
    } else {
      parseJarFile(); //parse test case(s) from jar file
    }
  }

  /**
   * Parse test cases from single file.
   * @throws IOException If cannot access or inject test cases.
   */
  public void parseSingleFile() throws IOException {
    System.out.println("reading data from file: " +singleFile);
    filename =singleFile;

    //do the injection
    prepare();

    int index =0; //test case index, always 0
    byte[] meta_data =new byte[0]; //do not have meta-data
    byte[] data; //actual test case data
    byte[] buffer =new byte[READ_BUFFER_SIZE];

    try {
      FileInputStream file_in =new FileInputStream(singleFile);

      //read whole test case data into memory
      ByteArrayOutputStream data_buf =new ByteArrayOutputStream();
      int total =0;
      int read =file_in.read(buffer);
      while (read >-1) {
	data_buf.write(buffer, 0, read);
	total +=read;
	read =file_in.read(buffer);
      }
      data =data_buf.toByteArray();
      inject(index, meta_data, data); //inject and out!
    } catch (EOFException e) { //end of file, not a big deal(TM)
    }
    finish();
  }

  /**
   * Parse test cases from JAR file.
   * @throws IOException If cannot access or inject test cases.
   */
  public void parseJarFile() throws IOException {
    if (stopIndex <startIndex) {
      throw new IllegalArgumentException(
        "Stop index is smaller than start index, nothing done");
    }

    //resolve JAR file name
    String jar_file =null;

    if (jarFileOption ==null) {
      //java.class.path should always exist, but give default to be on safe side
      String class_path =System.getProperty("java.class.path", ":");
      if ((class_path.indexOf(File.pathSeparatorChar) <0)) {
        System.out.println( //no ':' (UNIX) or ';' (Win) in path
          "single-valued 'java.class.path', using it's value for jar file name");
        jar_file =class_path;
      } else {
        System.out.println(
	  "multi-valued 'java.class.path',"
          +" must use command line option '-jarfile' or '-file'");
        help();
      }
    } else {
      jar_file =jarFileOption;
    }

    //open jar file
    System.out.println("reading data from jar file: " +jar_file);
    JarInputStream jar_in;
    try {
      jar_in =new JarInputStream(new FileInputStream(jar_file), true);
    } catch (IOException e) {
      throw new IOException("Cannot find file '" +jar_file +"'");
    }

    //do the injection
    prepare();

    int prefix_length =CASE_FILE_PREFIX.length(); //needed in index parsing
    int index; //test case index
    byte[] meta_data =new byte[0]; //do not have meta-data
    byte[] data; //actual test case data
    byte[] buffer =new byte[READ_BUFFER_SIZE];
    JarEntry entry;

    try {
      for(;;) {
        //find file for test case by looking for correct file prefix
        index =-1;
        do {
          entry =jar_in.getNextJarEntry();
          if (entry ==null) { //all files found
            break;
          }
          filename =entry.getName();
          if (filename.startsWith(CASE_FILE_PREFIX)) { //found test case file
            index =new Integer(filename.substring(prefix_length)).intValue();
          }
        }
        while (index <0);

        if (entry ==null) { //stop here
          break;
        }
        if ((index <startIndex) ||(index >stopIndex)) {
          continue; //do not trust any particular order of test cases in jar file
        }

        //read whole test case data into memory
        ByteArrayOutputStream data_buf =new ByteArrayOutputStream();
        int total =0;
        int read =jar_in.read(buffer);
        while (read >-1) {
          data_buf.write(buffer, 0, read);
          total +=read;
          read =jar_in.read(buffer);
        }
        data =data_buf.toByteArray();
        inject(index, meta_data, data); //inject!
      }
    } catch (EOFException e) { //end of file, not a big deal(TM)
    } catch (NumberFormatException e) {
      throw new RuntimeException("Internal error, invalid test case file '"
        +filename +"'");
    }
    finish();
  }

  /**
   * Set start test case.
   * @param index Index of start test case.
   */
  public void start(String index) {
    try {
      startIndex =Integer.parseInt(index);
    } catch (NumberFormatException e) {
      throw new IllegalArgumentException("Invalid index '" +index +"'");
    }
  }

  /**
   * Index of start test case.
   */
  protected int startIndex =0;

  /**
   * Set stop test case.
   * @param index Index of stop test case.
   */
  public void stop(String index) {
    try {
      stopIndex =Integer.parseInt(index);
    } catch (NumberFormatException e) {
      throw new IllegalArgumentException("Invalid index '" +index +"'");
    }
  }

  /**
   * Index of start test case.
   */
  protected int stopIndex =Integer.MAX_VALUE;

  /**
   * Inject a single test case.
   * @param index Index of the test case.
   */
  public void single(String index) {
    try {
      startIndex =stopIndex =Integer.parseInt(index);
    } catch (NumberFormatException e) {
      throw new IllegalArgumentException("Invalid index '" +index +"'");
    }
  }

  /**
   * The JAR file.
   * @param file The JAR file.
   */
  public void jarfile(String file) {
    jarFileOption =file;
  }

  /**
   * The JAR file, specified by command-line.
   */
  protected String jarFileOption;

  /**
   * Single case file.
   * @param file The file name.
   */
  public void file(String file) {
    singleFile =file;
  } 

  /**
   * A single file with single test case.
   */
  protected String singleFile;
}
