/*
*   Copyright (C) 2000, 2001 PROTOS Project Consortium
*   [http://www.ee.oulu.fi/research/ouspg/protos]
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 2
*   as published by  the Free Software Foundation
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package FI.protos.wrapper;

import java.io.*;
import java.net.*;
import java.util.jar.*;
import java.lang.reflect.*;

/**
 * TCP/IP client socket injector for LDAPv3 Test Suite.
 */
public class LDAPCat extends BugCat {

  /**
   * Command-line help.
   */
  public void help() {
    System.out.println(
       "Usage java -jar <jarfile> [OPTIONS]\n"
       +"\n"
       +" -delay <ms>         Wait <ms> milliseconds after each request.\n"
       +"                     Defaults to 1000 ms.\n"
       +" -file <file>        Send file <file> instead of test cases.\n"
       +" -help               Display this help.\n"
       +" -host <hostname>    LDAP server hostname, must be always set!\n"    
       +" -jarfile <file>     Use test case jar-file <file>.\n"
       +" -port <index>       LDAP server port. Defaults to 389.\n" 
       +" -showreply          Show replies from server.\n"
       +" -single <index>     Inject single test case <index>.\n"
       +" -start <index>      Inject test cases starting from <index>.\n"
       +" -stop <index>       Stop test case injection to <index>.\n"
    );
    System.exit(0);
  }
  
  /**
   * Stub for Command-line help. Activated by -h command line option
   */
  public void h() {
    this.help ();
    System.exit(0);
  }

  /**
   * Server hostname.
   */
  protected String hostname =null;
  
  /**
   * Set hostname by command line option '-host'.
   * @param option The hostname.
   */
  public void host (String option) {
    hostname =option;
  }

  /**
   * Server port number.
   */
  protected int port =389;
  
  /**
   * Set port number by command line option '-port'.
   * @param option The port number.
   */
  public void port (String option) {
    try {
      port =new Integer(option).intValue();
    } catch (NumberFormatException e) {
      port =-1;
    }

    if ((port <1) ||(port >0xffff)) {
      System.err.println("Invalid port number. Value must be "
	+"between [1..65536].");
      System.exit(0);
    }
  }

  /**
   * Waiting time after test case.
   */
  protected int waitTime =1000;
  
  /**
   * Set test case waiting time by command line option '-delay'.
   * @param option The time (ms).
   */
  public void delay(String option) {
    try {
      waitTime =new Integer(option).intValue();
    } catch (Exception e) {
      waitTime =-1;
    }
    if ((waitTime <0) ||(waitTime >360000)) {
      System.err.println ("Invalid waiting time. Value " + 
	"must be between [0..360000] ms.");
      System.exit(0);
    }
  }
    
  /**
   * True if should show replies from server.
   */
  protected boolean showReply =false;

  /**
   * Enable show replies by command line option '-showreply'.
   */
  public void showreply() {
    showReply =true;
  }
  

  /**
   * Main routine, parses command-line arguments and starts injection.
   * @param args Command-line arguments.
   */
  public static void main(String args[]) {
    LDAPCat cat =new LDAPCat();
    cat.run(args);
  }

  /**
   * Used socket.
   */
  public Socket socket;

  //
  // Method redefined from super class
  //
 
  public void prepare() throws IOException {
    /* Check that some hostname is given. Otherwise give help */
    if (hostname ==null) {
      System.err.println("Server hostname not specified!");
      this.help();
    }
  }

  public void inject(int index, byte[] metaData, byte[] data)
    throws IOException {
      
    try {
      socket =null;

      // open socket connection

      while (socket ==null) {
	try {
	  socket =new Socket(hostname, port);
	} catch (IOException e) {
	  System.out.println("Could not connect to " +hostname
	    +" port " +port +", retrying after a while...");
	  Thread.sleep(2000);
	}
      }
      socket.setSoTimeout(waitTime);

      // write test case

      System.out.println("Writing test case #" +index
	+", " +data.length +" bytes, to " +hostname
	+" port " +port);
    
      armTimeout(waitTime); //arm timeout for writing

      OutputStream out =socket.getOutputStream();
      out.write(data);
      out.flush();

      disarmTimeout(); //disarm the timeout

      // read reply

      InputStream in =socket.getInputStream();
      ByteArrayOutputStream buf =new ByteArrayOutputStream();
      try {
	System.out.println("Wrote, waiting " +waitTime +" ms for reply");

	// read every byte available (terminating on exception in all cases)
	for(;;) {
	  int abyte =in.read();
	  if (abyte ==-1) { //end-of-stream, server has disconnected
	    break;
	  }
	  buf.write(abyte);
	}
      } catch (InterruptedIOException e) { } //wait time elapsed
	
      socket.close();

      // show reply or at least its length

      byte read[] =buf.toByteArray();
      System.out.println(read.length +" bytes received");
      if (showReply) {
	System.out.print(dump(read, 0, read.length));
      }
    } catch (IOException e) {
      System.out.println ("Test case interrupted by error: " +e.getMessage());
    } catch (InterruptedException e) {
      System.exit(0);
    } finally {
      disarmTimeout(); //must be done always
    }
  }
  
  protected void fireTimeout() {
    System.out.println("Writing takes too long... timeout!");
    try {
      socket.close();
    } catch (IOException e) { }
  }
}







