/*
*   Copyright (C) 2001 PROTOS Project Consortium
*   [http://www.ee.oulu.fi/research/ouspg/protos]
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 2
*   as published by  the Free Software Foundation
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package FI.protos.ouspg.wrapper;

import FI.protos.ouspg.wrapper.BugCatZero;

import java.io.*;
import java.net.*;

/**
 * An udp zerocase Bug cat.
 */
public class UDPTrapBugCat extends BugCatZero
{

  /*
   * Internal variables
   */
  /* Remote address set by host command line option. */
  private InetAddress addr = null;
  /* port to connect to */
  private int port = 162;
  /* injection delay in milliseconds */
  private int idelay = 100;
  /* extra time to wait before closing socket in milliseconds*/
  private int cdelay = 0;
  /* time to wait for reply in milliseconds */
  private int wdelay = 0;
  /* dump reply ? */
  private boolean showreply = false;
  /* send zerocase between cases ? */
  private boolean zerocase = false;
  /* guess;) */
  private DatagramSocket socket = null;
  

  /**
   * Default constructor. 
   */
  public UDPTrapBugCat() {}
  
  /**
   * Stub for Command-line help. Activated by -h command line option
   */
  public void h() {
    this.help ();
  }

  /*
    *Command-line help. Activated by -help command line option
   */
  public void help () {
    System.out.println("Usage java -jar <jarfile>.jar [ [OPTIONS] |"+
		       " -host <hostname> ]\n");
    System.out.println
      ("  -closedelay <ms>      Extra delay before closing socket.\n" + 
       "                        Defaults to 0 ms (milliseconds).");
    System.out.println
      ("  -delay <ms>           Time to wait before sending new test-case.\n"+ 
       "                        Defaults to 100 ms (milliseconds).");
    System.out.println
      ("  -file <file>          Send file <file> instead of test-case(s)");
    System.out.println
      ("  -help                 Display this help");
    System.out.println
      ("  -host <hostname>      Hostname to send (inject) test-cases");
    System.out.println
      ("  -jarfile <file>       Get data from an alternate bugcat\n"+
       "                        JAR-file <file>");
    System.out.println
      ("  -port <index>         Portnumber to send packets on host.\n" +
       "                        Defaults to " + port + ".");
    System.out.println
      ("  -single <index>       Inject a single test-case <index>");
    System.out.println
      ("  -start <index>        Inject test-cases starting from <index>");
    System.out.println
      ("  -stop <index>         Stop test-case injection to <index>");
    System.out.print ("\n");
    System.exit(0);
  }
  
  /**
   * Set hostname by command line option '-host'.
   * @param s The hostname.
   */
  public void host (String s) {
    try {
      addr = InetAddress.getByName(s);
    } catch (UnknownHostException e) {
      System.out.println ("ERROR: Unknown host " + e.getMessage());
      System.exit (1);
    }
  }
  
  /**
   * Set port number by command line option '-port'.
   * @param s The port number.
   */
  public void port (String s) {
    try {
      port = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid port number " + e.getMessage());
      System.exit (0);
    }
  }
  
  /**
   * Set closedelay by command line option '-closedelay'.
   */
  public void closedelay (String s) {
    try {
      cdelay = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid closedelay " + e.getMessage());
      System.exit (0);
    }
  }

  /**
   * Set injection delay by command line option '-injectiondelay'.
   */
  public void delay (String s) {
    try {
      idelay = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid injection delay " + e.getMessage());
      System.exit (0);
    }
  }

  /**
   * Set replywait delay by command line option '-replywait'.
   */
  public void replywait (String s) {
    try {
      wdelay = new Integer (s).intValue();
    } catch (NumberFormatException e) {
      System.out.println("ERROR: Invalid replywait " + e.getMessage());
      System.exit (0);
    }
  }

  /**
   * Dump replies. Enabled by command line option '-showreply'.
   */
  public void showreply () {
    showreply = true;
  }

  /**
   * Send valid case between test-cases. 
   * Enabled by command line option '-zerocase'.
   */
  public void zerocase () {
    zerocase = true;
  }

  /**
   * Main routine, parses command-line arguments and starts injection.
   * @param args Command-line arguments.
   */
  public static void main(String args[]) {
    UDPTrapBugCat cat =new UDPTrapBugCat();
    cat.run(args);
  }
    
  /**
   * Method redefined from super class
   */
  public void prepare() throws IOException {
    /* Check that some hostname is given. Otherwise give help */
    if (addr ==null) {
      this.help ();
      System.exit (0);
    }
  }
  
  public void finish() throws IOException {
    /* Nothing done here at this time */
  }
  
  public void inject(int index, byte[] metaData, byte[] data)
    throws IOException {
    
    System.out.println("test-case #" +index
		       +": injecting meta-data " +metaData.length +" bytes, "
		       +"data " +data.length +" bytes");

    /* main injection loop here */
    magic(data, wdelay);
      
    if (!zerocase) return;
   
    /* verify with zerocase */
    int iwdelay = (wdelay != 0) ? wdelay : 1;
    System.out.println ("test-case #" +index+": injecting valid case...");
    while (magic(this.getZeroCase(), iwdelay) != 0) {
      System.out.println ("test-case #" +index + ": No reply to" +
			  " valid case within "+iwdelay+" ms."+
			  " Retrying...");
      iwdelay = (iwdelay < 60000) ? (iwdelay * 2) : iwdelay;
    }

  }

  /* do the actual injection */
  private int magic (byte[] data, int wto) {
    DatagramPacket dps = new DatagramPacket(data, data.length,
					    addr, port);

    /* prepare for empty data */
    if (data == null) {
      System.out.println ("WARNING: null data received. Unable to send/receive");
      return 0;
    }

    try {
      /* Init socket */
      socket =new DatagramSocket();
      socket.connect(addr, port);

      /* apply injection delay */
      delay (idelay);
      
      /* send case */
      socket.send(dps);
      if (wto == 0) return 0;
      
      /* read reply (one datagram, one datagram only..) */
      socket.setSoTimeout(wto);
      DatagramPacket dp = new DatagramPacket (new byte[0xffff], 0xffff);
      System.out.print("waiting " + wto +" ms for reply...");
      socket.receive (dp);
      System.out.println(dp.getLength() +" bytes received");
      if (showreply) {
	System.out.print(dump(dp.getData(), 0, dp.getLength()));
      }
    } catch (InterruptedIOException e) { 
      System.out.println ("0 bytes received");
      return 1;
    } catch (SocketException se) {
      System.out.println ("ERROR: " + se.getMessage());
      System.exit (-1);
    } catch (Exception e) {
      System.out.println ("ERROR: " + e.getMessage());
      System.exit (-1);
    } finally {       
      /* do closedelay and close socket */
      delay (cdelay);
      if (socket != null) {
	socket.close();
	socket = null;
      }
    } /* end try */
    return 0;
  } 

  /* 
   * private methods
   */ 
  private void delay (int t) {
    /* Apply closedelay */
    try {
      Thread.currentThread().sleep(t);
    } catch (Exception e) { 
      System.out.println ("ERROR: " + e.getMessage());
      System.exit (-1);
    }
  }

  
} /* end of class */

// Local variables:
// c-basic-offset: 2
// End:


