/*
 *   Copyright (C) 2000, 2001 PROTOS Project Consortium
 *   [http://www.ee.oulu.fi/research/ouspg/protos]
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2
 *   as published by  the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package FI.protos.ouspg.wrapper;

import java.io.*;
import java.util.jar.*;
import java.lang.reflect.*;
import java.text.*;

/**
 * Base class for test case injection utility for distribution.
 */
abstract public class BugCatZero implements Runnable {

  //
  // methdos used to access zerocase (valid) data
  //
  private byte[] zcbuff = null;
  public byte[] getZeroCase () {
    return zcbuff;
  }

  public void setZeroCase (byte[] b) {
    zcbuff = b;
  }
  

  //
  // Methods to be redefined in real cat class
  //

  /**
   * Command-line help.
   */
  public void help() {
    System.out.println(
       "  -single <index>         Inject single test case <index>\n"
      +"  -start <index>          Inject test cases starting from <index>\n"
      +"  -stop <index>           Stop test case injection to <index>\n"
      +"  -jarfile <file>         Get data from bugcat jar-file <file>\n"
      +"  -file <file>            Get single case from file <file>\n"
      +"  -help                   Give this help\n");
    System.exit(0);
  }

  /**
   * Prepare for injection.
   * Redefine in derived classes, if preparation needed.
   * @throws IOException If preparation fails.
   */
  public void prepare() throws IOException {
  }

  /**
   * Inject a test case.
   * The meta-data is reserved for future expansions, ignore now.
   * Redefine in derived classes.
   * @param index Index of injected test case.
   * @param metaData Meta-data of the test case.
   * @param data Data of the test case.
   * @throws IOException If the injection fails.
   */
  public void inject(int index, byte[] metaData, byte[] data)
    throws IOException {
  }

  /**
   * Finish after injection. Not called if injection fails.
   * Redefine in derived classes, if finish needed.
   * @throws IOException If finish fails.
   */
  public void finish() throws IOException {
  }

  //
  // Timeout function for subclasses which need it
  //

  /**
   * Timeout fires, redefine in subclass when needed.
   */
  protected void fireTimeout() {
    System.err.println("Timeout internal error!");
  }

  /**
   * Arm the timeout.
   * @param period Timeout period (ms).
   */
  protected void armTimeout(int period) {
    disarmTimeout();

    fireTime =System.currentTimeMillis() +(long)period;
    timeoutArm =true;
    timeoutDisarm =false;
    new Thread(this).start();
  }

  /**
   * Disarm the armed timeout.
   * The call blocks until timeout is really armed or timeout fires.
   */
  protected void disarmTimeout() {
    synchronized (timeoutMutex) {
      timeoutDisarm =true; //request disarmament
      try {
        while (timeoutArm) { //wait until timeout no longer running
          timeoutMutex.notify();
          timeoutMutex.wait();
        }
      } catch (InterruptedException e) { }
    }
  }

  /**
   * Synchronization mutex for timeout attributes.
   */
  private Object timeoutMutex =new Object();

  /**
   * True, if timeout armed.
   */
  private boolean timeoutArm =false;

  /**
   * True, if timeout should be disarmed.
   */
  private boolean timeoutDisarm =false;

  /**
   * Timeout fire time.
   */
  private long fireTime;

  /**
   * Main loop for timeout thread.
   * Not enclosed in inner class to avoid multiple class files from this
   * .java file. Multiple files would create problems in wrapping.
   */
  public void run() {
    synchronized (timeoutMutex) {
      try {
        for(;;) {
          if (timeoutDisarm) { //timeout disarmed
            break;
          }
          long time_left =fireTime -System.currentTimeMillis();
          if (time_left <=0) { //timeout fires, just once
            fireTimeout();
            break;
          }
          timeoutMutex.wait(time_left);
        }
      } catch (InterruptedException e) {
        fireTimeout(); //fire timeout on interrupt also
      }
      timeoutArm =false;
      timeoutMutex.notify();
    }
  }

  //
  // Dump method
  //

  /**
   * Dump binary data using classic notation, 16 bytes per row,
   * ASCII values shown in extreme left.
   * @param data The data.
   * @param offset Start offset from data.
   * @param length Length of clip dumped.
   * @return The dump.
   */
  public static String dump(byte[] data, int offset, int length) {
    StringBuffer b =new StringBuffer();

    DecimalFormat addressFormat =new DecimalFormat("00000000");

    int location =offset; //current dump location
    int stop =offset +length; //stop location

    while (location <stop) {
      int width; //width of this line (bytes)
      if ((stop -location) >=16) { //full line
	width =16;
      } else { //partial line
	width =stop -location; 
      }

      //dump address
      String addr =addressFormat.format(offset +location);
      b.append(addr);
      b.append(" ");

      //dump hex values, unsigned
      for(int a =0; a <width; a++) {
        int value =(data[location +a] +256) %256;
        if (value <=0xf) { //always two digits
          b.append(" 0");
        } else {
          b.append(" ");
        }
        b.append(Integer.toHexString(value));
      }

      //fill with spaces if last line
      for(int a =width; a <16; a++) {
        b.append("   ");
      }

      b.append("  ");
      //dump characters, if in range 32..126
      for(int a =0; a <width; a++) {
        byte value =data[location +a];
        if ((value >=32) &&(value <=127)) {
          b.append((char)value);
        } else {
          b.append('.'); //place dot '.' where not printable 7-bit ASCII
        }
      }

      //continue to next line or exit
      b.append("\n");
      location +=width;
    }
    return b.toString();
  }

  //
  // Internal methods not needing redefinition
  //

  /**
   * Test case file prefix.
   */
  public static String CASE_FILE_PREFIX ="testcases/";

  /**
   * Run me, thrill me, kiss me, kill me.
   * @param args Command-line arguments.
   */
  public void run(String args[]) {
    try {
      if (args.length ==0) { //give help
        System.out.println("(help by command line argument: -help)");
      }
      //parse command-line arguments
      int index =0;
      while (index <args.length) {
        String name =args[index];
        if (!name.startsWith("-")) {
          throw new IllegalArgumentException(
            "command line argument: " +name +" should start with '-'");
        }
        name =name.substring(1); //name of command-line argument
        index++;
        Method[] all_method =getClass().getMethods();
        Method method =null;
        for(int a =0; a <all_method.length; a++) { //find method to call
          if (all_method[a].getName().equals(name)) {
            method =all_method[a];
            break;
          }
        }
        if (method ==null) {
          throw new IllegalArgumentException(
            "unknown command line argument: -" +name);
        }
        //get parameters of the argument
        Object parameter[] =new Object[method.getParameterTypes().length];
        for(int a =0; a <parameter.length; a++) {
          if (index ==args.length) {
            throw new IllegalArgumentException(
              "missing " +(a +1) +". parameter for argument: -" +name);
          }
          parameter[a] =args[index];
          index++;
        }
        method.invoke(this, parameter); //call the method
      }
      parseTestCases();
    } catch (IOException e) {
      System.err.println("ERROR: " +e.getMessage() +"\n");
    } catch (IllegalArgumentException e) {
      System.err.println("ERROR: " +e.getMessage() +"\n");
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Current test-case file for debugging.
   */
  protected String filename;

  /**
   * Read buffer size.
   */
  protected final static int READ_BUFFER_SIZE =2048;

  /**
   * Parse test cases.
   * @throws IOException If cannot access or inject test cases.
   */
  public void parseTestCases() throws IOException {
    if (singleFile !=null) {
      parseSingleFile(); //parse a test case from single file
    } else {
      parseJarFile(); //parse test case(s) from jar file
    }
  }

  /**
   * Parse test cases from single file.
   * @throws IOException If cannot access or inject test cases.
   */
  public void parseSingleFile() throws IOException {
    System.out.println("reading data from file: " +singleFile);
    filename =singleFile;

    // do the injection
    prepare();

    int index =0; //test case index, always 0
    byte[] meta_data =new byte[0]; //do not have meta-data
    byte[] data; //actual test case data
    byte[] buffer =new byte[READ_BUFFER_SIZE];

    try {
      FileInputStream file_in =new FileInputStream(singleFile);

      //read whole test case data into memory
      ByteArrayOutputStream data_buf =new ByteArrayOutputStream();
      int total =0;
      int read =file_in.read(buffer);
      while (read >-1) {
       	data_buf.write(buffer, 0, read);
      	total +=read;
      	read =file_in.read(buffer);
      }
      data =data_buf.toByteArray();
      inject(index, meta_data, data); //inject and out!
    } catch (EOFException e) { //end of file, not a big deal(TM)
    }
    finish();
  }

  /**
   * Parse test cases from JAR file.
   * @throws IOException If cannot access or inject test cases.
   */
  public void parseJarFile() throws IOException {
    
    if (stopIndex <startIndex) {
      throw new IllegalArgumentException
	("Stop index is smaller than start index, nothing done");
    }
      
    // resolve JAR file name

    String jar_file =null;

    if (jarFileOption ==null) {
      //java.class.path should always exist, but give default anyway
      String class_path =System.getProperty("java.class.path", ":");
      if ((class_path.indexOf(File.pathSeparatorChar) <0)) {
        System.out.println( //no ':' (UNIX) or ';' (Win) in path
          "single-valued 'java.class.path'"
          +", using it's value for jar file name");
        jar_file =class_path;
      } else {
        System.out.println(
           "Note, multi-valued 'java.class.path',"
          +" must use command line option '-jarfile'\nor '-file'");
        help();
      }
    } else {
      jar_file =jarFileOption;
    }

    // open jar file

    System.out.println("reading data from jar file: " +jar_file);
    JarInputStream jar_in;
    try {
      jar_in =new JarInputStream(new FileInputStream(jar_file), true);
    } catch (IOException e) {
      throw new IOException("Cannot find file '" +jar_file +"'");
    }

    // do the injection

    prepare();

    int prefix_length =CASE_FILE_PREFIX.length(); //needed in index parsing
    int index; //test case index
    byte[] meta_data =new byte[0]; //do not have meta-data
    byte[] data; //actual test case data
    byte[] buffer =new byte[READ_BUFFER_SIZE];
    JarEntry entry;

    try {
      for(;;) {
        //find file for test case by looking for correct file prefix
        index =-1;
        do {
          entry =jar_in.getNextJarEntry();
          if (entry ==null) { //all files found
            break;
          }
          filename =entry.getName();
          if (filename.startsWith(CASE_FILE_PREFIX)) { //found test case file
            //get test case index from file name (= fist sequence of digits)
            int filename_l =filename.length();
            int i =prefix_length;
            while ((i <filename_l) &&
                   (!Character.isDigit(filename.charAt(i)))) {
              i++;
            }
            int s =i;
            while ((s <filename_l) &&Character.isDigit(filename.charAt(s))) {
              s++;
            }
            if (i ==s) { //no digits, cannot be test case file
              continue;
            }
            index =new Integer(filename.substring(i, s)).intValue();
          }
        }
        while (index <0);

        if (entry ==null) { //stop here
          break;
        }
	
        if (((index <startIndex) ||(index >stopIndex)) && index != 0) {
          continue; //do not trust any particular order of test cases
        }

        //read whole test case data into memory
        ByteArrayOutputStream data_buf =new ByteArrayOutputStream();
        int total =0;
        int read =jar_in.read(buffer);
        while (read >-1) {
          data_buf.write(buffer, 0, read);
          total +=read;
          read =jar_in.read(buffer);
        }
	if (index ==0) zcbuff =data_buf.toByteArray(); // store zerocase
        data =data_buf.toByteArray();
        if ((index >= startIndex) && (index <= stopIndex)) 
	  inject(index, meta_data, data); //inject!
      }
    } catch (EOFException e) { //end of file, not a big deal(TM)
    } catch (NumberFormatException e) {
      throw new RuntimeException("Internal error, invalid test case file '"
        +filename +"'");
    }
    finish();
  }

  //
  // Default command-line options
  //

  /**
   * Set start test case.
   * @param index Index of start test case.
   */
  public void start(String index) {
    try {
      startIndex =Integer.parseInt(index);
    } catch (NumberFormatException e) {
      //throw new IllegalArgumentException("Invalid index '" +index +"'");
      System.out.println ("ERROR: Invalid index " + e.getMessage());
      System.exit (1);
    }
  }

  /**
   * Index of start test case.
   */
  protected int startIndex =0;

  /**
   * Set stop test case.
   * @param index Index of stop test case.
   */
  public void stop(String index) {
    try {
      stopIndex =Integer.parseInt(index);
    } catch (NumberFormatException e) {
      //throw new IllegalArgumentException("Invalid index '" +index +"'");
      System.out.println ("ERROR: Invalid index " + e.getMessage());
      System.exit (1);
    }
  }

  /**
   * Index of start test case.
   */
  protected int stopIndex =Integer.MAX_VALUE;

  /**
   * Inject a single test case.
   * @param index Index of the test case.
   */
  public void single(String index) throws IllegalArgumentException {
    try {
      startIndex =stopIndex =Integer.parseInt(index);
    } catch (NumberFormatException e) {
      //throw new IllegalArgumentException("Invalid index '" +index +"'");
      System.out.println ("ERROR: Invalid index " + e.getMessage());
      System.exit (1);
    }
  }

  /**
   * The JAR file.
   * @param file The JAR file.
   */
  public void jarfile(String file) {
    jarFileOption =file;
  }

  /**
   * The JAR file, specified by command-line.
   */
  protected String jarFileOption;

  /**
   * Single case file.
   * @param file The file name.
   */
  public void file(String file) {
    singleFile =file;
  } 

  /**
   * A single file with single test case.
   */
  protected String singleFile;
}

// Local variables:
// c-basic-offset: 2
// End:

