/*
*   Copyright (C) 2001 PROTOS Project Consortium, 2002 OUSPG
*   [http://www.ee.oulu.fi/research/ouspg/protos]
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License version 2
*   as published by  the Free Software Foundation
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package FI.protos.ouspg.wrapper;

import FI.protos.ouspg.wrapper.InjectorUtilities;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * TCP Injector
 */
public class TCPInjector extends Injector {
    protected int port = 1720;
    protected int replyWait = 100;
    protected int sendWait = 2000;
    protected int closeDelay = 50;
    protected int retryDelay = 100;
    protected int retryDelayLimit = 5000;
    protected int retryLimit = 0;
    protected int repeatLimit = 0;
    protected int validCaseIndex = 0;
    protected boolean showreply = false;
    protected boolean showsent = false;
    protected boolean useValidCase = false;
    protected boolean oldtcp = false;
    protected InetAddress host = null;
    protected Socket socket = null;

    public TCPInjector () {
    }

    /**
     * Stub for Command-line help. Activated by -h command line option
     */
    public void h() {
	this.help ();
    }

    /*
     *Command-line help. Activated by -help command line option
     */
    public void help () {
	System.out.println("Usage: java -jar <jar>"+
			   " --host <hostname> [OPTIONS]");
	System.out.println
	    (" --help                 Display this help.");
	System.out.println
	    (" --host <hostname>      Hostname to send test cases.");
	System.out.println
	    (" --port <port>          Port number on host to send cases.\n"+
	     "                        Default: "+port+".");
	System.out.println
	    (" --sendwait <ms>        Maximum time to wait for test case\n" +
	     "                        sending to complete.\n"+
	     "                        Value 0 means infinite time.\n"+
	     "                        Default: "+sendWait+" milliseconds.");
	System.out.println
	    (" --replywait <ms>       Maximum time to wait for a reply.\n" +
	     "                        Value 0 disables reading replies.\n"+
	     "                        Default: "+replyWait+" milliseconds.");
	System.out.println
	    (" --closedelay <ms>      Additional sleeping time before,\n"+ 
	     "                        closing socket\n"+
	     "                        Default: "+closeDelay+" milliseconds.");
	System.out.println
	    (" --retrylimit <count>   Maximum amount of connection/valid\n"+
             "                        case attempt retries until aborting\n"+
	     "                        the test run. 0 means MAX_INT.\n"+
	     "                        Default: "+retryLimit+".");
	System.out.println
	    (" --retrydelay <ms>      Initial time to sleep before next\n"+ 
	     "                        connection/valid case attempt.\n"+
	     "                        Doubled after each unsuccessful one\n"+
	     "                        until retrydelaylimit is reached\n"+
	     "                        Default: "+retryDelay+" milliseconds.");
	System.out.println
	    (" --retrydelaylimit <ms> Maximum time to sleep before next\n"+ 
	     "                        connection/valid case attempt.\n"+
	     "                        Default: "+retryDelayLimit+
	     " milliseconds.");
	System.out.println 
	    (" --showreply            Show hex dump of received packets.");
	System.out.println 
	    (" --showsent             Show hex dump of sent packets.");
	System.out.println
	    (" --single <index>       Inject a single test case <index>.");
	System.out.println
	    (" --start <index>        Start injecting from case <index>.");
	System.out.println
	    (" --stop <index>         Stop injecting after <index>.\n"+
	     "                        If smaller that start index, cases\n"+
	     "                        are injected in reverse order.");
	System.out.println
	    (" --repeat <count>       Repeat test run <count> times.\n"+
	     "                        Default: "+repeatLimit+".");
	System.out.println
	    (" --validcase            After each test case inject a valid\n"+
	     "                        case (test case #0) until a reply\n"+
	     "                        is received. May be used to check if\n"+
	     "                        the target is still responding.\n"+
	     "                        Default: off.");
	System.out.println
	    (" --oldtcp               Don't drain TCP socket of incoming\n"+
             "                        data before closing it. Default: off.");
	System.out.print ("\n");
	System.exit(0);
    }

    /**
     * Set destination port number by command line option '-port'.
     * @param s The port number.
     */
    public void port (String s) {
	try {
	    port = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid port number "
			       + e.getMessage());
	    System.exit (0);
	}
    }

    /**
     * Set destination host by command line option '-host'.
     * @param s The host.
     */
    public void host (String s) {
	try {
	    host = InetAddress.getByName(s);
	} catch (UnknownHostException e) {
	    System.out.println ("ERROR: invalid host " + e.getMessage());
	    System.exit (1);
	}
    }

    /**
     * Set close delay by command line option '-closedelay'.
     */
    public void closedelay(String s) {
	try {
	    closeDelay = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid closedelay value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }

    /**
     * Set retry delay by command line option '-retrydelay'.
     */
    public void retrydelay(String s) {
	try {
	    retryDelay = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid retrydelay value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }
    
    /**
     * Set retry delay maximum by command line option '-retrydelaylimit'.
     */
    public void retrydelaylimit(String s) {
	try {
	    retryDelayLimit = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid retrydelaylimit value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }

    /**
     * Set retrylimit by command line option '-retrylimit'.
     */
    public void retrylimit(String s) {
	try {
	    retryLimit = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid retrylimit value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }

    /**
     * Set sendwait value by command line option '-sendwait'.
     */
    public void sendwait(String s) {
	try {
	    sendWait = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid sendwait value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }
 
    /**
     * Set replywait value by command line option '-replywait'.
     */
    public void replywait(String s) {
	try {
	    replyWait = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid replywait value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }
 
    /**
     * Set repeat value by command line option '-repeat'.
     */
    public void repeat(String s) {
	try {
	    repeatLimit = new Integer (s).intValue();
	} catch (NumberFormatException e) {
	    System.out.println("ERROR: Invalid repeat value " 
			       + e.getMessage());
	    System.exit (0);
	}
    }
    
    /**
     * Dump replies. Enabled by command line option '-showreply'.
     */
    public void showreply() {
	showreply = true;
    }
    
    /**
     * Dump sent packets. Enabled by command line option '-showsent'.
     */
    public void showsent() {
	showsent = true;
    }

    /**
     * Valid case without setting valid case index.
     */
    public void validcase() {
	useValidCase = true;
    }

    /**
     * Don't drain socket before closing it.
     */
    public void oldtcp() {
	oldtcp = true;
    }

    /**
     * Main routine, parses command-line arguments and starts injection.
     * @param args Command-line arguments.
     */
    public static void main(String args[]) {
	TCPInjector inj = new TCPInjector();
	inj.run(args);
    }

    /**
     * Run me, thrill me, kiss me, kill me.
     * @param args Command-line arguments.
     */
    public void run(String args[]) {
	if (args.length ==0) { //give help
	    help();
	} else {
	    try {
		InjectorUtilities.parseCommandLine(args, this);
	    } catch(Throwable t) {
		System.err.println("Error: " +t.getMessage() +"\n");
		help();
	    }
	}

	if (host == null) {
	    System.out.println("Hostname must be specified.");
	    help();
	}

	//InjectorUtilities.setTestCaseFormat("testcases/0000000");

	byte[] validCase = InjectorUtilities.getTestCase(validCaseIndex);

	// repeat test run for desired times
	for (int repeatIndex = 0; repeatIndex <= repeatLimit; repeatIndex++) {

	    System.out.println("Starting test run "+repeatIndex+"/"
			       +repeatLimit);

	    int caseIndex = startIndex;

	    // test run: inject test cases from startindex to stopindex
	    for(;;) {

		byte[] testCase = InjectorUtilities.getTestCase(caseIndex);

		if (testCase == null) {
		    //no more test cases, start next run
		    break;
		}

		//inject test case
		inject(caseIndex, testCase, false);
	    
		if (useValidCase) {
		    int retries = 0;
		    int sleepTime = retryDelay;

		    //inject valid case (case #0)
		    while (!inject(caseIndex, validCase, true)) {
			
			retries++;
			
			if (retries == retryLimit) {
			    System.out.println("#"+caseIndex+
					       ": Valid case retry limit "+
					       retryLimit+" reached.");
			    System.exit(0);
			}
			System.out.println("#"+caseIndex+
					   ": No response to valid case."+
					   " Retry "+retries+"/"
					   +retryLimit+" after "
					   +sleepTime+" ms...");
			delay(sleepTime);
			
			if ((sleepTime * 2) < retryDelayLimit) {
			    sleepTime *= 2;
			} else {
			    sleepTime = retryDelayLimit;
			}
		    }
		} // if validcase
		
		if (caseIndex == stopIndex) {
		    //stopindex reached, start next run
		    break;
		}
		if (startIndex < stopIndex) {
		    caseIndex++;
		} else {
		    caseIndex--;
		}
	    } // for(;;)
	} // for (repeat)
	System.out.println("Done.");
    }

    private boolean inject(int index, byte[] data,  boolean isValidcase) {
	InputStream in = null;
	OutputStream out = null;
	byte[] replyData;
	boolean validcaseSuccess = false;

	try {
	    //connect
	    connect(index);

	    if (isValidcase) {
		System.out.println("#"+index+": Injecting valid case.");
	    } else {
		System.out.println("#"+index+": Injecting test case, "
				   +data.length+" bytes.");
	    }

	    socket.setSoTimeout(sendWait);
	    armTimeout(sendWait); //arm timeout for writing

	    out = socket.getOutputStream();
	    in = socket.getInputStream();

	    //send test case
	    out.write(data);
	    out.flush();

	    disarmTimeout(); //disarm timeout

	    if (showsent) {
		System.out.print(dump(data, 0, data.length));
	    }
	    if (replyWait != 0) {
		socket.setSoTimeout(replyWait);
		//get reply
		System.out.print("#"+index+": Waiting " +replyWait+
				 " ms for reply...");
		replyData = getReply(in);
		System.out.println(replyData.length +" bytes received");

		if (replyData.length > 0)
		    validcaseSuccess = true;
		
		if (showreply) {
		    System.out.print(dump(replyData, 0, replyData.length));
		}
	    }
	    //disconnect
	    if (closeDelay > 0) {
		System.out.println("#"+index+": Waiting "+closeDelay+
				   " ms before closing connection.");
		delay(closeDelay);
	    }
	    
	    out.close();
	    if (!oldtcp)
		 // work around RST problem with windows
		in.skip(in.available());
	    in.close();
	    socket.close();

	} catch (IOException e) {
	    System.out.println("#"+index+": ERROR: "+e.getMessage());
	    socket = null;
	}

	return validcaseSuccess;

    }

    /**
     * Connects to target host. Retries if connection fails.
     */
    private void connect(int index) {

	int sleepTime = retryDelay;
	int retries = 0;
	socket = null;

	do {
	    try {
		socket =new Socket(host, port);
		System.out.println("#"+index+": Connect success.");
 		return;
	    } catch (IOException e) {
		retries++;
		System.out.println("#"+index+": Connect failed: " 
				   +e.getMessage()+". Retry "+retries+"/"
				   +retryLimit+" after "+sleepTime+" ms...");
		delay(sleepTime);
		if ((sleepTime * 2) < retryDelayLimit) {
		    sleepTime *= 2;
		} else {
		    sleepTime = retryDelayLimit;
		}
	    }
	} while (retries != retryLimit);

	System.out.println("#"+index+": Connection retry limit "
			   +retryLimit+" reached.");
	System.exit(0);
    }

    /**
     * Reads response to a test case from target host. 
     * Returns a zero size byte array if no response is received.
     */
    private byte[] getReply(InputStream in) {

	ByteArrayOutputStream baos = new ByteArrayOutputStream();

	try {
	    //read every byte available 
	    //(terminating on exception in all cases)
	    for(;;) {
		int abyte =in.read();
		if (abyte ==-1) { //end-of-stream, server has disconnected
		    break;
		}
		baos.write(abyte);
	    }
	} catch (InterruptedIOException e) { 
	    //reply wait time elapsed
	} catch (IOException ei) {
	    System.out.print("failed: "+ei.getMessage()+". ");
	}
	return (baos.toByteArray());
    }

    /**
     * Sleeps for given time.
     */
    private void delay (int ms) {
	try {
	    Thread.currentThread().sleep(ms);
	} catch (InterruptedException ie) { 
	    System.exit (0);
	} catch (IllegalArgumentException iae) { 
	    System.out.println ("ERROR: " + iae.getMessage());
	    System.exit (1);
	}
    }

    protected void fireTimeout() {
	System.out.println("Writing takes too long... timeout!");
	try {
	    socket.close();
	} catch (IOException e) { }
    }

}
