/*
 *   Copyright (C) 2000, 2001 PROTOS Project Consortium, 2002 OUSPG
 *   [http://www.ee.oulu.fi/research/ouspg/protos]
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2
 *   as published by  the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package FI.protos.ouspg.wrapper;

import FI.protos.ouspg.wrapper.InjectorUtilities;
import java.io.*;
import java.util.jar.*;
import java.lang.reflect.*;
import java.text.*;

abstract public class Injector implements Runnable {

    /**
     * Synchronization mutex for timeout attributes.
     */
    private Object timeoutMutex =new Object();
    
    /**
     * True, if timeout armed.
     */
    private boolean timeoutArm =false;
    
    /**
     * True, if timeout should be disarmed.
     */
    private boolean timeoutDisarm =false;
    
    /**
     * Timeout fire time.
     */
    private long fireTime;

    /**
     * Test run start index
     */
    protected int startIndex =0;

    /**
     * Test run stop index
     */
    protected int stopIndex =Integer.MAX_VALUE;

    /**
     * Main loop for timeout thread.
     * Not enclosed in inner class to avoid multiple class files from this
     * .java file. Multiple files would create problems in wrapping.
     */
    public void run() {
	synchronized (timeoutMutex) {
	    try {
		for(;;) {
		    if (timeoutDisarm) { //timeout disarmed
			break;
		    }
		    long time_left =fireTime -System.currentTimeMillis();
		    if (time_left <=0) { //timeout fires, just once
			fireTimeout();
			break;
		    }
		    timeoutMutex.wait(time_left);
		}
	    } catch (InterruptedException e) {
		fireTimeout(); //fire timeout on interrupt also
	    }
	    timeoutArm =false;
	    timeoutMutex.notify();
	}
    }
    
    /**
     * Arm the timeout.
     * @param period Timeout period (ms).
     */
    protected void armTimeout(int period) {
	disarmTimeout();
	
	fireTime =System.currentTimeMillis() +(long)period;
	timeoutArm =true;
	timeoutDisarm =false;
	new Thread(this).start();
    }
    
    /**
     * Disarm the armed timeout.
     * The call blocks until timeout is really armed or timeout fires.
     */
    protected void disarmTimeout() {
	synchronized (timeoutMutex) {
	    timeoutDisarm =true; //request disarmament
	    try {
		while (timeoutArm) { //wait until timeout no longer running
		    timeoutMutex.notify();
		    timeoutMutex.wait();
		}
	    } catch (InterruptedException e) { }
	}
    }

    /**
     * Timeout fires, redefine in subclass when needed.
     */
    protected void fireTimeout() {
	System.err.println("Timeout internal error!");
    }

    /**
     * Command-line help.
     */
    public void help() {
	System.out.println(
         " --single <index>         Inject single test case <index>\n"
        +" --start <index>          Inject test cases starting from <index>\n"
        +" --stop <index>           Stop test case injection to <index>\n"
        +" --help                   Give this help\n");
	System.exit(0);
    }

    /**
     * Set start test case.
     * @param index Index of start test case.
     */
    public void start(String index) {
	try {
	    startIndex =Integer.parseInt(index);
	} catch (NumberFormatException e) {
	    System.out.println ("ERROR: Invalid index " + e.getMessage());
	    System.exit (1);
	}
    }
    
    /**
     * Set stop test case.
     * @param index Index of stop test case.
     */
    public void stop(String index) {
	try {
	    stopIndex =Integer.parseInt(index);
	} catch (NumberFormatException e) {
	    //throw new IllegalArgumentException("Invalid index '" +index +"'");
	    System.out.println ("ERROR: Invalid index " + e.getMessage());
	    System.exit (1);
	}
    }
    
    /**
     * Inject a single test case.
     * @param index Index of the test case.
     */
    public void single(String index) throws IllegalArgumentException {
	try {
	    startIndex =stopIndex =Integer.parseInt(index);
	} catch (NumberFormatException e) {
	    System.out.println ("ERROR: Invalid index " + e.getMessage());
	    System.exit (1);
	}
    }


    /**
     * Dump binary data using classic notation, 16 bytes per row,
     * ASCII values shown in extreme left.
     * @param data The data.
     * @param offset Start offset from data.
     * @param length Length of clip dumped.
     * @return The dump.
     */
    public static String dump(byte[] data, int offset, int length) {
	StringBuffer b =new StringBuffer();
	
	DecimalFormat addressFormat =new DecimalFormat("00000000");
	
	int location =offset; //current dump location
	int stop =offset +length; //stop location
	
	while (location <stop) {
	    int width; //width of this line (bytes)
	    if ((stop -location) >=16) { //full line
		width =16;
	    } else { //partial line
		width =stop -location; 
	    }
	    
	    //dump address
	    String addr =addressFormat.format(offset +location);
	    b.append(addr);
	    b.append(" ");
	    
	    //dump hex values, unsigned
	    for(int a =0; a <width; a++) {
		int value =(data[location +a] +256) %256;
		if (value <=0xf) { //always two digits
		    b.append(" 0");
		} else {
		    b.append(" ");
		}
		b.append(Integer.toHexString(value));
	    }
	    
	    //fill with spaces if last line
	    for(int a =width; a <16; a++) {
		b.append("   ");
	    }
	    
	    b.append("  ");
	    //dump characters, if in range 32..126
	    for(int a =0; a <width; a++) {
		byte value =data[location +a];
		if ((value >=32) &&(value <=127)) {
		    b.append((char)value);
		} else {
		    //place dot '.' where not printable 7-bit ASCII
		    b.append('.');
		}
	    }
	    
	    //continue to next line or exit
	    b.append("\n");
	    location +=width;
	}
	return b.toString();
    }
    /**
     * Run me, thrill me, kiss me, kill me.
     * @param args Command-line arguments.
     */
    public void run(String args[]) {
	if(args != null) {
	    try {
		InjectorUtilities.parseCommandLine(args, this);
	    } catch(Throwable t) {
		System.err.println("Error: " +t.getMessage() +"\n");
		help();
	    }
	} else {
	    System.out.println("(help by command line argument: --help)");
	}
    }
}
