/**
 * Data preview subsystem
 *
 * This is the main entry point for all client-side datapreview functionality
 *
 */


// namespace in which to save all sourcetypes generated by the preview app
// value is either a string value for the namespace, i.e., 'splunk_datapreview'
// or 'system' for the local system global store
var SOURCETYPE_DESTINATION_NS = 'system';


Splunk.namespace("Splunk.preview");

Splunk.preview.Controller = function(filePath, continue_to, return_to, reset_to, dynLibs) {
    this.mysettings = new Splunk.preview.Settings();
    this.path = filePath;
    this.continue_to = continue_to;
    this.return_to = return_to;
    this.reset_to = reset_to;
    this.st_name = null;
    this.st_type = null;
    this.error_panel = $('#main_error_panel');
    this.userHasApplied = false;
    this.chartingLibLoaded = false;
    this.pendingMetadata = null;
    this.dynLibs = dynLibs;
};

Splunk.preview.Controller.prototype = {

    main: function() {
        // load dynamic libraries
        $script(this.dynLibs, $.proxy(function() {
            this.chartingLibLoaded = true;
            if (this.pendingMetadata) {
                metaview.render(this.pendingMetadata);
                this.pendingMetadata = null;
            }
        },this));
    
        // Event bindings
        $('.ts_apply').live('click', $.proxy(this.handleApply, this));
        $('.resetSettings').live('click', $.proxy(this.handleReset, this));
        
        $('a#continue').bind('click', $.proxy(this.handleContinue, this));
        $('#bt_continue').bind('click', $.proxy(this.handleContinue, this));
        $('#bt_cancel').bind('click', $.proxy(this.goFilePicker, this));

        confview.updateSettings();
        confview.render();
        
        stview.bind('submit', $.proxy(function() {
            this.st_type = stview.getSelectionValue();
            
            if (this.st_type == 'auto') {
                this.st_name = stview.autoSourcetype;
            } else if (this.st_type == 'new') {
                this.mysettings.set('sourcetype', '^^idontexistyet^^');
                this.renderViews();
            } else if (this.st_type == 'existing') {
                this.st_name = stview.getPresetValue();
                this.mysettings.set('sourcetype', this.st_name);
                this.renderViews();
            }
        }, this));
        
        // Begin loading data
        $.when(
            Splunk.preview.Sourcetype.fetchNames()
                .pipe(function(values) {
                    stview.presetValues = values;
                })
            ,
            this.renderViews()
            )
        .then(function(f1, f2) {
            stview.autoSourcetype = samp.settings.get('PREFERRED_SOURCETYPE');
            stview.render();
        });
        
    },
    
    /**
     * Send user to preview file selection page
     */
    goFilePicker: function() {
        if (this.reset_to.length == 0) {
            history.go(-1);
        }
        var path_query = this.reset_to.split('?');
        var props = Splunk.util.queryStringToProp(path_query[1]);
        props.source = this.path;
        props.breadcrumbs = props.breadcrumbs.replace('+',' ');
        var redirect_url = path_query[0] + '?' + $.param(props);
        window.location.href = redirect_url;
    },

    /**
     * Starts the various requests to fetch data
     */
    renderViews: function() {
        eventview.clear();
        metaview.clear();
        
        eventview.showThrobber();

        // get sample from URI
        return Splunk.preview.createSample(this.path, this.mysettings)
            .pipe(
                $.proxy(function(sample) {
                    samp = sample;
                    confview.loadProps(sample.settings);
                    confview.updateSettings();
                    var job = sample.initSearchJob();
                    $(job).bind('stateChange', $.proxy(this.handleJobStateChange, this));
                    $(job).bind('propertiesUpdate', $.proxy(this.handleJobPropertiesUpdate, this));
                    job.startPoll();
                }, this)
            ,
                $.proxy(function(error_info) {
                    // error_info.messages
                    // error_info.statusCode
                    // error_info.errorThrown
                    var strings = [];
                    $.each(error_info.messages, function(ix,msg) { 
                        if (msg.type != 'HTTP') {
                            strings.push(msg.text); 
                        }
                    });
                    this.promptFatalError(strings);
                }, this)
            );


    },


    showErrors: function(messages) {
        if (messages && messages.length) {
            // TODO: this truncation is temporary; need to refactor messaging
            this.error_panel.text(messages.slice(0,2).join('; '));
        }
    },


    clearErrors: function() {
        this.error_panel.html('');
    },
        

    promptFatalError: function(messages) {

        eventview.clearThrobber();

        var errorList = $('#fatal_popup ul').html('');
        var i,L;
        for (i=0,L=messages.length; i<L; i++) {
            errorList.append($('<li/>').text(messages[i]));
        }

        // determine the right buttons to show on error
        var buttons;
        if (this.userHasApplied) {
            buttons = [
                {
                    label: _('OK'),
                    type: 'primary'
                }
            ];

        } else {
            buttons = [
                {
                    label: _('Select a different file'),
                    type: 'primary',
                    callback: $.proxy(this.goFilePicker, this)
                }
            ];
        }


        var fatalPopup = new Splunk.Popup($('#fatal_popup'), {
            title: 'Error processing file',
            buttons: buttons
        });
    },

        
    handleJobPropertiesUpdate: function(evt) {
        if (samp.searchJob.get('doneProgress') != null) {
            eventview.setProgress(samp.searchJob.get('doneProgress'));
        }
    },

    /**
     * Fills in rest of data when job finishes
     */
    handleJobStateChange: function(evt, state) {

        if (state >= splunk.service.JobStates.DONE) {

            // deal with job messaging
            var jobMessages = samp.searchJob.get('messages');
            var desiredLevels = ['error', 'warn', 'info'];
            var outputText = [];
            var i,L;
            for (var i=0,L=desiredLevels.length; i<L; i++) {
                if (jobMessages[desiredLevels[i]]) {
                    outputText = outputText.concat(jobMessages[desiredLevels[i]]);
                }
            }
            this.showErrors(outputText);

            // assume that errors on job mean something is busted
            if (samp.searchJob.get('eventCount') == 0 
                || samp.searchJob.get('isFailed')
                || samp.searchJob.get('isZombie')
                || jobMessages.error 
            ) {
                
                eventview.clearThrobber();
                metaview.clearThrobber();
                stview.close();
                if (!outputText.length) {
                    outputText = [_('Could not generate a preview of this file.')];
                }
                this.promptFatalError(outputText);

            } else {

                samp.fetchEvents()
                    .pipe(function(events) {
                        eventview.clearThrobber();
                        eventview.render(events);
                    })
                    .pipe(function() {
                        metaview.showThrobber();
                        return samp.fetchMetadata();
                    })
                    .pipe($.proxy(function(results) {
                        metaview.clearThrobber();
                        if (this.chartingLibLoaded) {
                            metaview.render(results);
                        } else {
                            this.pendingMetadata = results;
                        }
                    },this));
            
            }
        }
    },
        
    handleApply: function (evt) {
        if (evt) {
            evt.preventDefault();
        }
        
        // first reset
        this.mysettings.clear();
        this.clearErrors();
        this.userHasApplied = true;

        this.processConfig();
        this.renderViews();
        return true;
    },
    
    handleReset: function (evt) {
        if (evt) {
            evt.preventDefault();
        }
        
        this.clearErrors();
        this.mysettings.clear();
       
        confview.resetSettings(); 
        this.handleApply();   

        return true;
    },
    
    handleContinue: function () {
        if (confview.settingsChanged()) {
            var shouldApply = confirm(_('You have unapplied changes. Would you like to apply them now?'));
            if (shouldApply) {
                this.handleApply();
            }
            return false;
        }
    
        if ((this.st_type == 'auto' || this.st_type == 'existing') && !this.userHasApplied) {
            var redirect_url = this.makeContinueUrl(this.st_name);
            window.location.href = redirect_url;
            
        } else {
            stconfirm.render(this.mysettings, this.makeContinueUrl(), this.return_to);    
        }
    },

    handleSubmit: function() {
        stconfirm.clearErrors();
        var st_name = $('#st_name').val();
        if (!st_name) {
            stconfirm.showError(_('Provide a sourcetype name'));
            return;
        }

        // set flag so that this sourcetype will show up in future lists
        // of "available" sourcetypes
        var settingsToSave = this.mysettings.toSourcetypeSettings();
        settingsToSave['pulldown_type'] = 1;

        var src = new Splunk.preview.Sourcetype(
            SOURCETYPE_DESTINATION_NS, 
            Splunk.util.getConfigValue('USERNAME'), 
            settingsToSave);
        
        src.save(st_name)
            .done($.proxy(function(saved_object) {
                stconfirm.close();
                stsuccess.render(st_name, this.makeContinueUrl(st_name), this.return_to);
            }, this))

            .fail(function(err) {
                var err_msg;
                if (err.statusCode == 409) {
                    err_msg = _('This sourcetype already exists. Please enter a different name.');
                }
                else {
                    err_msg = _('Save failed: ') + $.map(err.messages, function(x){return x.text;}).join('; ');
                }
                stconfirm.showError(err_msg);
            });
    },
    

    processConfig: function() {
        // make sure to first set the simple, then override with any user-set
        // keys from the advanced box
        this._processSimpleConfig();
        this._processAdvancedConfig();
        confview.updateSettings();
    },
        
    _processSimpleConfig: function()  {
        
        $.each(confview.getSimpleSettings(), $.proxy(function(key, val){
            if ( val.value != '' ) {
                if ( val.name == 'lb_type' ) {
                    
                    switch (val.value) {
                        case 'auto':
                            break;

                        case 'single_line':
                            this.mysettings.set('SHOULD_LINEMERGE', 'false');
                            break;

                        case 'regex':
                            this.mysettings.set('SHOULD_LINEMERGE', true);
                            break;                            
                            
                        default:
                            break;
                    }
                } else {
                    if (val.name in Splunk.preview.Settings.properties) {
                        // filter out non-props form entities
                        this.mysettings.set(val.name, val.value);
                    }
                }
            }
        }, this));
    },

    _processAdvancedConfig: function() {
        var propsData = confview.getAdvancedSettings();
        var rows = propsData.split('\n');
        $.each(rows, $.proxy(function(ix,row) {
            var kv = row.split('=');
            if (kv.length == 2) {
                this.mysettings.set(kv[0], kv[1]);
            }
        }, this));
    },
    

    /**
     * Returns a relative URI that points to a data input creation page with:
     * - the "more settings" tab open
     * - the sourcetype mode set to "manual"
     * - the sourcetype name set to the sourcetype parameter
     * - (file input) the input path set to the sample filepath
     *
     */ 
    makeContinueUrl: function(sourcetype) {

        // set defaults
        var sourcetype = sourcetype || '';
        var path = Splunk.util.make_url('/manager/search/data/inputs/monitor/_new');
        var query_string = {
            'action': 'edit',
            'redirect_override_cancel': Splunk.util.getPath(this.return_to),
            'def.sourcetype': sourcetype,
            'def.spl-ctrl_sourcetypeSelect': 'sourcetype',
            'def.spl-ctrl_EnableAdvanced': '1'
        };
                              
        // assume that page will generally have continue links
        if (this.continue_to && this.continue_to.indexOf('?') > -1) {
            var continue_parts = this.continue_to.split('?');
            path = continue_parts[0];
            query_string = $.extend(query_string, Splunk.util.queryStringToProp(continue_parts[1]));
        }

        // only file monitor can get the name prepopulated; all other input
        // types have different needs
        if (path.indexOf('inputs/monitor') > -1) {
            query_string['def.name'] = this.path;
        }

        return path + '?' + Splunk.util.propToQueryString(query_string);

    }

    

    
};


window.eventview = new Splunk.preview.EventView();
window.metaview = new Splunk.preview.MetadataView();
window.confview = new Splunk.preview.ConfigurationView();
window.stview = new Splunk.preview.SourcetypeModeView();
window.stconfirm = new Splunk.preview.SourcetypeConfirmView();
window.stsuccess = new Splunk.preview.SourcetypeSuccessView();

