import logging, re

import cherrypy
from splunk.appserver.mrsparkle import *
from splunk.appserver.mrsparkle.lib import i18n
import splunk.searchhelp.searchhelper as sh
import splunk.searchhelp.utils as shu
import splunk.search as se

import xml.sax.saxutils as su
import splunk.bundle as bundle
import splunk.mining.FieldLearning as ifl

import splunk.util,datetime

logger = logging.getLogger('splunk.appserver.controllers.etb')

_STRING_DELIMITER = "\n"
MAX_SAMPLES = 100
MIN_SAMPLES = 20
MAX_LINES = 15

requiredFields = ["sid", "offset"]

CERROR = "error"
CWARN  = "warn"
CMSG   = "info"
CSUCCESS = "success"

CVALIDLIST = [CERROR, CSUCCESS, CWARN, CMSG]

def errorCMP(x,y):
    x = re.search('class="(.*?)"', x).group(1)    
    y = re.search('class="(.*?)"', y).group(1)    
    return cmp(CVALIDLIST.index(y),CVALIDLIST.index(x))

LISTTYPE = type([])
DICTTYPE = type({})
def addMessage(parent, msg, msgtype):
    if msgtype not in CVALIDLIST:
        raise Exception("Invalid message type '%s' for '%s'" % (msgtype, msg))
    if type(parent) == DICTTYPE:
        parent = parent['messages']
    parent.append('<p class="%s">%s</p>' % (su.escape(msgtype), su.escape(msg)))


"""
1. given an event, we generate a series of checkboxes for attribute=values and _raw phrases.
2. given series of checkboxes or customizes eventtypestring, we search and get example results
3. allow saving and testing
"""


class ETBController(BaseController):
    """/etb"""

    @route('/')
    @expose_page(must_login=True, methods=['GET', 'POST'])
    def index(self, **kwargs):

        cherrypy.response.headers['content-type'] = MIME_HTML
        args = getArgs(kwargs)
        args['testingurl'] = None
        args['etstyles'] = []
        
        try:
            # IF WE HAVE AN EVENTTYPE AND THE USER IS SAVING IT, RETURN SUCCESS
            if len(args['eventtype'])>0 and save(args):
                args['events'] = []
                args['checkedValues'] = {}
                return self.render_template('etb/index.html', args)
        except Exception, e:
            addMessage(args, "Unable to save eventtype: %s" % e, CERROR)
            #import traceback #addMessage(args, "<pre>Stacktrace: %s</pre>" % traceback.format_exc(), CWARN)

        try:
            getFieldInfo(args)
        except Exception, e:
            # better error generated by getevent
            pass
            
        # if user modified eventtype and didn't hit preview button, use his; otherwise generate it
        if args['edited']:
            eventtype = args['eventtype']
        else:
            eventtype = args['generatedSearch']

        args['eventtype'] = eventtype
        args['events'] = []
        try:
            args['events'] = getSampleEvents(eventtype, args)
            if not eventTypeExists(args) and len(eventtype)>0:
                args['testingurl'] = self.make_url(['app', args.get('namespace'), 'flashtimeline'],  _qs=[('q', 'search ' + eventtype)])
            args['etstyles'] = getStyles(args)        
            
        except Exception, e:
            addMessage(args, "Unable to get sample events: %s" % e, CERROR)



        # sort warnings before messages. weak.
        args['messages'].sort(errorCMP)
        args['messages'].reverse()

        return self.render_template('etb/index.html', args)

def getStyles(args):
    names = []
    try:
        ns = args['namespace']
        cssfile = os.path.join(os.environ['SPLUNK_HOME'], 'etc', 'apps', ns, 'appserver', 'static', 'application.css')
        f = open(cssfile, "r")
        lines = f.readlines()
        for line in lines:
            thesenames = re.findall(".splEvent-([^ {]*)", line)
            names.extend(thesenames)
        names.sort()
    except Exception, e:
        pass
    finally:
        f.close()
                          
    return names
    


def getSampleEvents(eventtype, args):
    results = []

    if eventtype != '':

        if eventtype.strip().startswith("|") or len(shu.getJustCommands(eventtype, None)) > 1:
            raise Exception("Eventtypes cannot contain search commands")
        
        eventtype = eventtype.replace('\\', '\\\\')
        query = "search %s | head %s | fields | abstract maxlines=%s " % (eventtype, MAX_SAMPLES, MAX_LINES)
        maxtime = args.get('maxtime', None)
        if maxtime != None:
            # try to use maxtime to get selecteed event at top
            epochmaxtime = splunk.util.dt2epoch(splunk.util.parseISO(maxtime))
            results = se.searchAll(query, latest_time=epochmaxtime, status_buckets=1)

        # if not enough events, research without time constraint
        if len(results) < MIN_SAMPLES:
            results = se.searchAll(query, status_buckets=1)

        results =  [ r.raw.getRaw() for r in results ]
    return results

def ignoredField(fieldname):
    return fieldname == '' or fieldname == None or fieldname.startswith('_') or fieldname.startswith('date_') or fieldname == 'punct' or fieldname == 'timestartpos' or fieldname == 'timeendpos'

def getFieldInfo(args):
    event = args['event']
    job = se.getJob(args['sid'])
    summary = job.summary
    if event != None:
        eventValues = {}
        fieldValues = {}
        fieldOrder = []
        for attr in event:
            if ignoredField(attr):
                continue
            fieldOrder.append(attr)            
            eventValues[attr] = [str(v) for v in event[attr]] # list
            fieldValues[attr] = summary.fields.get(attr,{'distinctCount':0, 'modes':[]})
            
            #fieldcount = fieldinfo['count']
            #fielddistinctcount = fieldinfo['distinctCount']
            #fieldValues[attr] = fieldinfo['modes']  # (vd['value'],vd['count'])
        args['eventValues'] = eventValues
        args['fieldValues'] = fieldValues

        fieldOrder.sort(lambda x, y: fieldValues[y]['distinctCount'] - fieldValues[x]['distinctCount'])
        args['fieldOrder'] =  fieldOrder

def quoteAsNeeded(val):
    if not val.isalnum():
        val = val.replace('"', '\\"')
        val = '"%s"' % val
    return val

def getCondition(attr, val):
    val = quoteAsNeeded(val)    
    if attr.startswith('_raw'):
        return val
    else:
        return '%s=%s' % (attr, val)
    
def getArgs(requestArgs):

    for field in requiredFields:
        if field not in requestArgs:
            raise AttributeError, 'Required field "%s" not provided' % field

    checkedValues = {}
    # make a map of attrs to set of values checked
    for k, v in requestArgs.items():
        if k.startswith("check"):
            k = k[7:] # check+2digits
            if not k in checkedValues:
                checkedValues[k] = set()
            checkedValues[k].add(v)

    # now generate search from checked features
    searchCondition = []
    for attr, vals in checkedValues.items():
        conds = [getCondition(attr, val) for val in vals]
        if len(conds) == 1:
            searchCondition.append(getCondition(attr,list(vals)[0]))
        else:
            searchCondition.append('(%s)' % (' OR '.join(conds)))
    generatedSearch = ' '.join(searchCondition)
    
    maxtime = None
    messages = []
    eventSearch, event = getJobInfo(requestArgs, messages)
    rootSearch = ''
    if event != None:
        searchArgs = eventSearch.split('|')[0]
        if searchArgs.startswith("search "):
            searchArgs = searchArgs[7:]
        searchArgs = searchArgs.strip()
        rootSearch = searchArgs
        if searchArgs != "*":
            generatedSearch = ("%s %s" % (searchArgs, generatedSearch)).strip()

        eventValues = {}
        for attr in event:
            eventValues[attr] = event[attr][0] # just use first value
        #eventValues['raw'] = event.raw.getRaw()
        if '_time' in event:
            maxtime = splunk.util.getISOTime(event.toDateTime() + datetime.timedelta(seconds=1))


    return {
    'username':cherrypy.session['user']['name'],
    'namespace': requestArgs.get('namespace','search'),
    'sid': requestArgs['sid'],
    'offset': requestArgs['offset'],
    'messages' :messages,
    'eventtype': requestArgs.get('eventtype',''),
    'rootSearch': rootSearch,
    'generatedSearch': generatedSearch,
    'saveresults': 'save' in requestArgs,
    'edited': requestArgs.get('edited','') == "True" and not 'preview' in requestArgs,
    'maxtime': maxtime,
    'successmessage': '',
    'checkedValues': checkedValues,
    'event': event,
    'eventTypeName': requestArgs.get('eventTypeName', ""),
    'eventTypeStyle': requestArgs.get('eventTypeStyle', ""),
    'eventTypePriority': requestArgs.get('eventTypePriority', "5")
    
    }

def getJobInfo(requestArgs, messages):
    try:
        sid = requestArgs['sid']
        offset = requestArgs['offset']        
        if sid != '' and offset != '':
            job = se.getJob(sid)
            return job.eventSearch, job.events[int(offset)]
    except Exception, e:
        addMessage(messages, 'Unable to get sample event.  The search job has probably expired: "%s"' % e, CERROR)
        return None, None

def eventTypeExists(args, eventTypeName=None):

    if len(args['eventtype']) == 0:
        return True

    eventtype = args['eventtype']
    props = bundle.getConf('eventtypes', None, args['namespace'], args['username'])
    # for each prop stanza
    for stanzaname in props.keys():
        if eventTypeName == stanzaname:
            addMessage(args, "'%s' eventtype already exists." % eventTypeName, CWARN)
            return True
        stanza = props[stanzaname]
        search = stanza.get("search", "")
        if search.strip().lower() == eventtype.strip().lower():

            # just be nice and don't warn the poor guy if he didn't modify the default search
            if args['rootSearch'] == args['generatedSearch']:
                addMessage(args, "Click on event Event Type Feature checkboxes below to make a new unique eventtype.", CMSG)
            else:
                addMessage(args, "The proposed eventtype is already defined as eventtype '%s'." % stanzaname, CWARN)
            return True
    return False

def illegalStanzaCharacters(name):
    illegalChars = "[]\n\r"
    for ch in illegalChars:
        if ch in name:
            return True
    return False

def invalidEventTypeName(args):
    badname = False
    name = args['eventTypeName']
    if name == '' or len(name) > 50 or illegalStanzaCharacters(name):
        addMessage(args, "Illegal eventtype name.", CERROR)
        return True
    if eventTypeExists(args, eventTypeName=name):
        return True
    return False

def save(args):
    shouldSave = args['saveresults'] == True
    if not shouldSave or invalidEventTypeName(args):
        return False

    etname = args['eventTypeName']
    ns = args['namespace']
    user = args['username']
    etconf = bundle.getConf('eventtypes', None, ns, user)
    stanzaname = etname
    etconf.createStanza(stanzaname)
    # write out each regex to props.conf
    etconf[stanzaname]['search'] = args['eventtype'].replace('\n', ' ').replace('\r','')


    style = args['eventTypeStyle']
    # if the eventtype has a style, set up an custom renderer
    if style != "" and style !="None":
        priority = args['eventTypePriority']    
        
        render_conf = bundle.getConf('event_renderers', None, ns, user)
        stanzaname = etname
        render_conf.createStanza(stanzaname)
        render_conf[stanzaname]['eventtype'] = etname
        render_conf[stanzaname]['css_class'] = style
        render_conf[stanzaname]['priority'] = priority
    

    # great success!
    successmsg = "'%s' is now saved as an eventtype." % etname
    addMessage(args, successmsg, CSUCCESS)
    args['successmessage'] = successmsg 
    return True


 #.ugettext(message)
def unit_test():
    class FakeSession(dict):
        id = 5
    sessionKey = splunk.auth.getSessionKey('admin', 'changeme')
    try:
        cherrypy.session['sessionKey'] = sessionKey
    except AttributeError:
        setattr(cherrypy, 'session', FakeSession())
        cherrypy.session['sessionKey'] = sessionKey
    cherrypy.session['user'] = { 'name': 'admin' }
    cherrypy.session['id'] = 12345
    cherrypy.config['module_dir'] = '/'
    cherrypy.config['build_number'] = '123'
    cherrypy.request.lang = 'en-US'
    # roflcon
    class elvis:
        def ugettext(self, msg):
            return msg
    cherrypy.request.t = elvis()
    # END roflcon

    etber = ETBController()

    argc = len(sys.argv)
    if argc == 3:
        search = sys.argv[1]
        example = sys.argv[2]
        print "search: '%s' example: '%s'" % (search, example)
        out = etber.index(search="index=main %s" % search, fieldname='pid', examples=example, save='true', sid=1255116240.31, offset=0)
        #out = etber.index(search="index=main %s" % search, fieldname='pid', examples=example, save='true')
        print out
    else:
        print 'Usage: %s "restriction" "example"' % sys.argv[0]

if __name__ == '__main__':
    unit_test()
