import cherrypy, logging, datetime, hashlib
from splunk.appserver.mrsparkle import *
import splunk.util, json
from splunk.appserver.mrsparkle.lib import times, i18n

logger = logging.getLogger('splunk.appserver.controllers.util')


class UtilityController(BaseController):
    '''
    General utilities
    /util
    '''


    @route('/:m=md5')
    @expose_page(must_login=False)
    def md5(self, thing, salt=None, **args):
        '''
        Provides md5 hashing for strings.
        
        Usage:
        
            /util/md5?thing=helloworld ==> fc5e038d38a57032085441e7fe7010b0
            /util/md5?thing=helloworld&salt=23 ==> b6bc495e785755a8385a2f8b4ef12ec7
        '''
        hash = hashlib.md5()
        if salt:
            hash.update('%s%s' % (salt, thing))
        else:
            hash.update(str(thing))
        return hash.hexdigest()

    @route('/:log/:js')
    @expose_page(must_login=False, methods='POST')
    def log_js(self, log, js, data='[]', **args):
        '''
        Provides javascript logging service.
        
        Args:
            
            data: A JSON encoded collection of JavaScript log messages to process.  Default is an empty collection.

        '''
        out = {'events_logged': "0"}
        # non-logged in users cannot write to disk
        if cherrypy.session.get('sessionKey', None) is None:
            return self.render_json(out)
        
        try:
            logs = json.loads(data)
        except Exception, e:
            logger.error("Could not parse json from javascript logger. Data set was:\n %s" % data)
            return self.render_json(out)
        successCount = 0;
        for item in logs:
            event = []
            event.append('name=javascript')
            event.append('class=%s' % item.get('class','undefined'))
            event.append('%s' % item.get('message',''))
            eventFormat = ", ".join(event);
            try:
                getattr(logger, item.get('level', 'info'))(eventFormat)
                successCount = successCount + 1
            except Exception, e:
                logger.error("Could not log javascript event. Event was:\n%s" % eventFormat)
        out['events_logged'] = str(successCount)
        return self.render_json(out)


    #
    # time-related utilities
    # all hosted under /util/time
    #


    @route('/:m=time/:n=zoneinfo/:tz')
    @set_cache_level('always')
    @expose_page(must_login=False)
    def get_tztable(self, tz, **unused):
        '''
        Returns the Olsen database entries for the current server timezone.
        
        The 'tz' parameter is required for caching purposes.  It is not
        actually used.
        
        /util/time/zoneinfo/<SOME_TZ_LABEL>
        '''
        
        cherrypy.response.headers['content-type'] = MIME_TEXT
        
        try:
            return times.getServerZoneinfo()
        except Exception, e:
            logger.exception(e)
            return ''



    @route('/:m=time/:n=parser')
    @set_cache_level('never')
    @expose_page(must_login=False)
    def parse_time(self, ts, date_format='medium', time_format='medium', **unused):
        '''
        Returns a JSON structure of unix timestamps translated into both
        ISO-8601 format and a localized string.
        
        ts:
            one or more unix timestamps or relative time identifiers
            
        date_format:
            short - 10/17/09
            medium - Oct 17, 2009
            long - October 17, 2009
            
        time_format:
            short - 3:49 PM
            medium - 3:49:33.000 PM
            long - 3:49:40.000 PM -0700
        '''

        try:
             parsed = times.splunktime2Iso(ts)
        except Exception, e:
            logger.exception(e)
            return '{}'
            
        output = {}
        
        for key in parsed:
            
            localizedargs = {
                'dt': splunk.util.parseISO(parsed[key]),
                'date_base_format': date_format,
                'time_base_format': time_format
            }
                
            output[key] = {
                'iso': parsed[key],
                'localized': i18n.format_datetime_microseconds(**localizedargs)
            }
            
        return self.render_json(output)


