import os, copy, logging, time
import re
import urllib
import json
import lxml.etree as et
import splunk.bundle
import splunk.entity as en
import splunk.saved
import splunk.util
import splunk.search.Parser
import splunk.search.Transformer as xformer

from splunk.appserver.mrsparkle import *
from splunk.search.TransformerUtil import *

from splunk.appserver.mrsparkle.lib import util, viewconf, viewstate, jsonresponse, i18n, cached
from splunk.appserver.mrsparkle.lib.module import moduleMapper 
from splunk.appserver.mrsparkle.lib.memoizedviews import memoizedViews 
from splunk.appserver.mrsparkle import controllers
import controllers.report
from splunk import search
from lib.apps import local_apps

logger = logging.getLogger('splunk.appserver.controllers.view')

# define the splunkd entity path where views are stored
VIEW_ENTITY_CLASS = 'data/ui/views'

# define path to get current app view organization data
NAV_ENTITY_CLASS = 'data/ui/nav'
NAV_ENTITY_NAME = 'default'
NAV_ALT_ENTITY_NAME_S = 'default-%s'
NAV_CLASS_FREE = 'free'

# define the default view for saved searches.
DEFAULT_DISPLAYVIEW = 'flashtimeline'

class InvalidViewException(Exception):
    pass

class ViewController(BaseController):
    
    # /////////////////////////////////////////////////////////////////////////
    #  Supporting methods
    # /////////////////////////////////////////////////////////////////////////
    
    def __init__(self):
        '''
        Boot up the view controller
        
        The view system renders the views for the current user and namespace.
        Module definitions, including those registered inside apps, are cached
        inside this controller object (the long-lived instance held by
        cherrypy)
        '''
        
        super(ViewController,self).__init__()

        
    def getAppManifest(self):
        '''
        Returns a dict of all available apps to current user
        '''
        
        output = cached.getEntities('apps/local', search=['disabled=false','visible=true'], count=-1)
                
        return output
        
        
    def getViewManifest(self, namespace, currentViewName=None, viewstate_id=None, refresh=0):
        '''
        Returns a flat dict of all available views to current user and namespace.
        Iterate over this output to inspect the properties of the various views.
        '''
        
        output = {}
        
        # get available views to user/app
        memoizedViews.getAvailableViews(namespace, refresh, output)

        #
        # handle sticky state params for current view
        # the load sequence (from lowest the highest priority) is:
        # 1) default defined in <module>.conf
        # 2) view configuration XML
        # 3) specified viewstate param set; _current if not specified
        #

        if currentViewName and currentViewName in output:

            currentUser = cherrypy.session['user'].get('name')
            
            try:
                persisted = viewstate.get(currentViewName, viewstate_id=viewstate_id, namespace=namespace, owner=currentUser)

            except splunk.ResourceNotFound:
                # just stop trying to load non-existent viewstates
                logger.warn('getViewManifest - unable to load requested viewstate id=%s' % viewstate_id)
                return output
            
            logger.debug('getViewManifest - loading overlay viewstate id=%s' % viewstate_id)

            # Make a deep copy to avoid modifying entries in the memoizedViews cache.
            output[currentViewName] = copy.deepcopy(output[currentViewName]) 

            rosters = output[currentViewName]['layoutRoster']
            
            keyedParams = output[currentViewName].get('keyedParamMap')
            
            for panelName in rosters:
                for currentModule in rosters[panelName]:
                    
                    moduleId = currentModule['id']
                    if moduleId in persisted.modules:
                        for paramName in persisted.modules[moduleId]:
                            logger.debug('PERSISTENCE - override module=%s paramName=%s: %s ==> %s' % (
                                moduleId,
                                paramName,
                                currentModule['params'].get(paramName, 'UNDEFINED'),
                                persisted.modules[moduleId][paramName]
                            ))
                            currentModule['params'][paramName] = persisted.modules[moduleId][paramName]

        return output
        

    
    def getAppNav(self, app, viewManifest, searches):
        '''
        Retrieves the app nav hierarchy, and generates a tree of links
        
            1.  read in XML definition
            2.  replace run-time tokens, i.e. all unclassified views, recent 
                items, etc.
            3.  return hierarchical ordered list of labels and links to 
                views/searches/links
            
        '''

        navDefinition = None
        navAltClass = None

        # set alternate class
        if cherrypy.config.get('is_free_license'):
            navAltClass = NAV_CLASS_FREE

        # try alternate nav
        if navAltClass:
            try:
                navDefinition = en.getEntity(NAV_ENTITY_CLASS, NAV_ALT_ENTITY_NAME_S % NAV_CLASS_FREE, namespace=app)
            except splunk.ResourceNotFound:
                pass
            
        # if no alt, then proceed with default
        if not navDefinition:
            try:
                navDefinition = en.getEntity(NAV_ENTITY_CLASS, NAV_ENTITY_NAME, namespace=app)
            except splunk.ResourceNotFound:
                logger.warn('"%s" app does not have a navigation configuration file defined.' % app) # TK mgn 06/19/09
            except Exception, e:
                logger.exception(e)
                raise

        # parse the XML
        try:
            parser = et.XMLParser(remove_blank_text=True)
            navDefinition = et.XML(navDefinition['eai:data'], parser)
        except et.XMLSyntaxError, e:
            logger.error('Invalid app nav XML encountered: %s' % e)
        except Exception, e:
            logger.error('Unable to parse nav XML for app=%s; %s' % (app, e))

        output = []

        # if application has nav defined
        if navDefinition:

            # empty nav means don't do anything; omitted nav is treated down below
            if len(navDefinition) == 0:
                return output

            self._replaceNavTokens(navDefinition, viewManifest, app, searches)
            output = self._decorateNavItems(navDefinition, viewManifest, app)

            # check for the default view; if no default set, pick the first
            # view listed in nav; if none, try to get first in manifest            
            defaultNodes = navDefinition.xpath('//view[@default]')
            for node in defaultNodes:
                if splunk.util.normalizeBoolean(node.get('default')):
                    defaultView = node.get('name')
                    break
            else:
                fallbackNodes = navDefinition.xpath('//view[@name]')
                for node in fallbackNodes:
                    defaultView = node.get('name')
                    break
                else:
                    defaultView = sorted(viewManifest.keys())[0]

            
        # otherwise dump all views into a generic menu
        else:
            logger.warn('Unable to process navigation configuration for app "%s"; using defaults.' % app) # TK mgn 06/19/09
            DEFAULT_VIEW_COLLECTION = 'Default Views'
            output.append({
                'label': DEFAULT_VIEW_COLLECTION, 
                'submenu': [
                    {'label': _(viewManifest[name]['label']), 'uri': name} 
                    for name 
                    in sorted(viewManifest.keys())
                ]
            })
            
            defaultView = DEFAULT_DISPLAYVIEW

        return output, defaultView
        

    def _replaceNavTokens(self, navDefinition, viewManifest, app, searches):
        '''
        Inserts the proper view and saved search items as required by the XML
        nodes placed into the nav XML data.  Modified the 'navDefinition' lxml
        node in-place.

        The XML nodes currently recognized are:
            <view source="unclassified" />
            <view source="all" />
        '''
        
        # get a list of explicitly marked views and saved searches
        markedViews = []
        for node in navDefinition.xpath('//view[@name]'):
            markedViews.append(node.get('name'))
        
        markedSaved = []
        for node in navDefinition.xpath('//saved[@name]'):
            if searches.has_key(node.get('name')):
                markedSaved.append(node.get('name'))
            else:
                node.getparent().remove(node)
        #
        # handle views
        # identify the <view source="" /> nodes and fill in with views
        #
        for node in navDefinition.xpath('//view[@source]'):
            source = node.get('source')
            match  = node.get('match', '').lower()

            if source == 'all':
                for viewName in sorted(viewManifest.keys()):
                    if match and viewName.lower().find(match) == -1:
                        continue
                    linkNode = et.Element('view')
                    linkNode.set('name', viewName)
                    node.addprevious(linkNode)
            
            elif source == 'unclassified':
                for viewName in sorted(viewManifest.keys()):
                    if (viewName in markedViews) or (match and viewName.lower().find(match) == -1):
                        continue
                    linkNode = et.Element('view')
                    linkNode.set('name', viewName)
                    node.addprevious(linkNode)
 		    if match:
                       markedViews.append(viewName) 
            
            else:
                logger.warn('Unable to process view item; unknown source: %s' %  source)
                
            node.getparent().remove(node)
            
        #    
        # handle saved searches
        # identify the <saved source="" /> nodes and fill in with the proper
        # saved search items; allow matching on name substring
        #
        for node in navDefinition.xpath('//saved[@source]'):
            source = node.get('source', '').lower()
            match = node.get('match', '').lower()

            if source == 'all':
                keys = splunk.util.objUnicode(searches.keys())
                for savedName in sorted(keys, key=unicode.lower):
                    if match and savedName.lower().find(match) == -1:
                        continue
                    savedNode = et.Element('saved')
                    savedNode.set('name', savedName)
                    savedNode.set('sharing', searches[savedName].get('eai:acl',{}).get('sharing'))
                    if node.get('view'):
                        savedNode.set('view', node.get('view'))
                    node.addprevious(savedNode)

            elif source == 'unclassified':
                keys = splunk.util.objUnicode(searches.keys())
                for savedName in sorted(keys, key=unicode.lower):
                    if savedName not in markedSaved:
                        if match and savedName.lower().find(match) == -1:
                            continue
                        savedNode = et.Element('saved')
                        savedNode.set('name', savedName)
                        savedNode.set('sharing', searches[savedName].get('eai:acl',{}).get('sharing'))
                        if node.get('view'):
                            savedNode.set('view', node.get('view'))
                        node.addprevious(savedNode)
                        if match:
                            markedSaved.append(savedName)

            else:
                logger.warn('Unable to process saved search item; unknown source: %s' % source)
                
            node.getparent().remove(node)
        
        
        
    def _decorateNavItems(self, branch, viewManifest, app):
        '''
        Rewrites the incoming nav definition by decorating view names with
        proper links, and saved searches as views with search name specified.
        This recursive method is used by getAppNav().
        
        Input Example:
            <nav>
                <collection label="Dashboards">
                    <a href="http://google.com">Google</a>
                </collection>
                <collection label="Views">
                    <view source="all" />
                </collection>
                <collection label="Saved Searches" sort="alpha">
                    <collection label="Recent Searches">
                        <saved source="recent" />
                    </collection>
                    <saved name="All firewall errors" />
                    <divider />
                </collection>
            </nav>
        
        Output Example:
        
        
        '''
        
        output = []
        for node in branch:
            
            # update the view nodes with the proper links and labels
            if node.tag == 'view':
                viewData = viewManifest.get(node.get('name'))
                if viewData:
                    if viewData['isVisible']:
                        output.append({
                            'label': _(viewData.get('label')), 
                            'uri': self.make_url(['app', app, node.get('name', '')])
                        })
                else:
                    logger.warn(_('An unknown view name \"%(view)s\" is referenced in the navigation definition for \"%(app)s\".') % {'view': node.get('name'), 'app': app})
                    
            # update saved searches and point them to the saved search redirector
            elif node.tag == 'saved':
                if node.get('view'):
                    uri = self.make_url(
                        ['app', app, node.get('view')],
                        {'s': node.get('name')}
                    )
                else:
                    uri = self.make_url(
                        ['app', app, '@go'],
                        {'s': node.get('name')}
                    )
                output.append({
                    'label': node.get('name'), 
                    'uri': uri,
                    'sharing': node.get('sharing', None)
                })
                
            elif node.tag == 'a':
                uri = node.get('href')
                output.append({
                    'label': _(node.text), 
                    'uri': self.make_url(uri) if uri.startswith('/') else uri
                })

            elif node.tag == 'divider':
                output.append({
                    'label': '------', 
                    'uri': '#', 
                    'divider': 'actionsMenuDivider'
                })
                
            elif node.tag == 'collection':
                subcollection = {
                    'label': _(node.get('label')), 
                    'submenu': self._decorateNavItems(node, viewManifest, app)
                }
                
                # only show submenu if it contains something
                if len(subcollection['submenu']) > 0:
                    output.append(subcollection)

        return output
        
        
    def getAppConfig(self, appName, appList, permalinkInfo, currentViewName=None, build_nav=True):
        '''
        Returns a dict of properties for appName.
        '''

        # determine viewstate set
        viewstate_id = None
        if 'vs' in cherrypy.request.params:
            viewstate_id = cherrypy.request.params['vs']
        elif 'DATA' in permalinkInfo:
            viewstate_id = permalinkInfo['DATA']['vsid']
            
        # set default output
        output = {
            'is_visible': False,
            'label': appName,
            'nav': {},
            'can_alert': False,
            'available_views': self.getViewManifest(appName, currentViewName, viewstate_id=viewstate_id)
        }

        if appName != 'system':
            if appName not in appList:
                logger.warn(_('Splunk cannot load app "%s" because it could not find a related app.conf file.') % appName)
                return output

            appConfig = appList[appName]

            output['is_visible'] = splunk.util.normalizeBoolean(appConfig['visible'])
            output['label']      = splunk.util.normalizeBoolean(appConfig['label']) 
            output['version']    = appConfig.get('version')

        try:
            searches = en.getEntities('saved/searches', namespace=appName, search='is_visible=1 AND disabled=0', count=500, _with_new='1')
            if '_new' in searches:
                output['can_alert'] = 'alert.severity' in searches['_new'].get('eai:attributes', {}).get('optionalFields', [])
                del searches['_new']
        except splunk.ResourceNotFound:
            logger.warn('Unable to retrieve current saved searches')
            searches = {}            

        # get app nav
        if build_nav:
            output['nav'], tmp_dv = self.getAppNav(appName, output['available_views'], searches)
        else:
            output['nav'] = {}
        
        return output


    def filePathToUrl(self, filePath, pivotPath='modules'):
        '''
        Converts a filesystem path to a relative URI path, on the assumption that
        both paths contain a common pivot path segment
        '''

        parts = filePath.split(os.sep)
        try: pivot = parts.index(pivotPath)
        except: return False
        return '/' + os.path.join(*parts[pivot:]).replace('\\', '/')
    
    def _getTermsForSavedSearch(self, savedSearchName, namespace) :
        savedSearch = en.getEntity("saved/searches", savedSearchName, namespace=namespace)
        return savedSearch.get('qualifiedSearch', False)

        
    # /////////////////////////////////////////////////////////////////////////
    #  Main route handlers
    # /////////////////////////////////////////////////////////////////////////
        
    @route('/:app')
    @expose_page()
    def appDispatcher(self, app, setup=None):
        '''
        Redirect user to the default view, as specified in the nav XML
        
        Include a check against the setup properties to determine if we need
        to redirect to the setup page instead.
        
        The 'setup' param indicates if this handler should redirect to
        the app setup page, if requested
        '''
        
        apps = self.getAppManifest()

        if not apps.has_key(app):
            cherrypy.response.status = 404
            return self.render_template('view/404_app.html', {'app':app, 'apps':apps})
            
        # locate default view
        views = self.getViewManifest(namespace=app, refresh=1)
        nav, defaultView = self.getAppNav(app, views, {})
        defaultViewUri = ['app', app, defaultView]

        # check the app's setup status
        if apps[app].getLink('setup') and apps[app]['configured'] == '0':
            
            # show interstitial first
            if not setup:
                return self.render_template('view/app_setup.html', {'bypass_link': self.make_url(defaultViewUri), 'app_label': apps[app]['label'], 'appList': apps})
                
            logger.info('requested app is unconfigured, redirecting; app=%s configured=%s setup=%s' % (app, apps[app]['configured'], apps[app].getLink('setup')))
            return self.redirect_to_url(['manager', app, 'apps', 'local', app, 'setup'], _qs={
                'action':'edit', 
                'redirect_override': self.make_url(['app', app, defaultView], translate=False)
                })

        # otherwise, continue to default view
        raise self.redirect_to_url(defaultViewUri)

    
    @route('/:app/:p=@go')
    @expose_page()
    def redirect(self, app, **kwargs):
        '''
        Dispatches generic object requests to the appropriate view.  Currently
        used for redirecting saved searches and sids

        TODO: this needs to be app configurable
        '''
        view = None

        def getViewForSavedSearch(savedSearchName, app, owner=None):
            try:
                saved = en.getEntity('saved/searches', savedSearchName, namespace=app, owner=owner)
                app = saved.get('request.ui_dispatch_app') or app
                view = saved.get('request.ui_dispatch_view') or saved.get('displayview') or saved.get('view') or DEFAULT_DISPLAYVIEW
                return app, view
            except splunk.ResourceNotFound, e:
                raise cherrypy.HTTPError(404, _('The following requested saved search is unknown: "%s". As a result, Splunk is unable to redirect to a view.') % savedSearchName)

        def redirect(app, view, options):
            return self.redirect_to_url(['app', app, view], _qs=options)

        if 's' in kwargs:
            app, view = getViewForSavedSearch(kwargs['s'], app)
            logger.info('loading saved search "%s" into view "%s"' % (kwargs['s'], view))

            # removes the sid explicitly if one is set, this allows @go to follow the regular view.py behavior
            if 'sid' in kwargs:
                del kwargs['sid']

        elif 'sid' in kwargs:
            logger.info('loading SID "%s"' % kwargs['sid'])
            
            try:
                job = splunk.search.getJob(kwargs['sid'])
            except splunk.ResourceNotFound, e:
                logger.warn('unable to load search job, SID "%s" not found' % kwargs['sid'])
                output = {
                    'not_found_str': _("The search you requested could not be found."),
                    'explanation_str': _("The search has probably expired or been deleted."),
                    'rerun_str': _("Clicking \"Rerun search\" will run a new search based on the expired search's search string in the expired search's original time period.  Alternatively, you can return back to Splunk."),
                    'url': None
                }

                if e.resourceInfo:
                    ss_namespace = e.resourceInfo.get('app')
                    ss_owner = e.resourceInfo.get('owner')
                    ss_name = e.resourceInfo.get('name')
                    ss_now = e.resourceInfo.get('dispatch.now')
                    ss_loc = e.resourceInfo.get('location')
                    
                    if ss_namespace and ss_name:
                        url = ['app', ss_namespace, '@go']
                        qs = {'s': ss_name}
                        if ss_now:
                            qs['now'] = ss_now
                        output['url'] = self.make_url(url, _qs=qs)


                not_found_str = _("The search you requested could not be found.")
                explanation_str = _("The search has probably expired or been deleted.")
                rerun_str = _("Clicking \"Rerun search\" will run a new search based on the expired search's search string in the expired search's original time period.  Alternatively, you can return back to Splunk.")

                cherrypy.response.status = 404
                return self.render_template('/errors/missing_job.html', output)
                           
            if job.isSavedSearch and job.label:
                app, view = getViewForSavedSearch(job.label, app)
                logger.info('search job was run by saved search scheduler, predefined view: %s' % view)

            else:
                try:
                    request = job.request
                    view = request.get('ui_dispatch_view', False)
                    if view:
                        logger.info('search job specified view: %s' % view)
                except AttributeError:
                    pass

                if not view:
                    view = DEFAULT_DISPLAYVIEW
                    
        if view:
            if 'p' in kwargs:
                del kwargs['p']

            # optimization to render stock reports in a printer friendly view
            # that removes interactive chrome
            if kwargs.get('media') == 'print' and view == 'report_builder_display':
                view = 'report_builder_print'

            redirect(app, view, kwargs)

        raise cherrypy.HTTPError(400, _('The requested object is unknown. You must provide an sid or saved search as a query param. e.g. ?sid=<sid> or ?s=<saved search name>.'))

    def processPermalink(self, app, now, earliest, latest, remote_server_list, q, sid, s, view_id):
        """
        Take the q, s, sid parameters and generate a uniform data structure for handling it.
        """
        permalinkInfo = {}

        # PROCESSING ARGUMENTS FOR PERMALINK.
        
        # 0. Verify that between s, q, and sid,  only one is actually specified. 
        # NOTE - when we have the ability for messaging here to send warnings down to the user, we should just 
        # pick the first one specified, between s, q, sid.   And if a second or third is specified,  
        # then maybe check whether it's all still consistent,   but write some stuff that tells the user either 
        # that the args were inconsistent,  or that some of the args were totally ignored. 
        if s and (q or sid):
            raise Exception, _("If you specify s (saved search name), you cannot also specify q or sid (job id).")
        if sid and (q or s):
            raise Exception, _("If you specify sid (job id), you cannot also specify q or s (saved search name).")
        if q and (s or sid):
            raise Exception, _("If you specify q, you cannot also specify s (saved search name) or sid (job id).")
        
        # 1. Saved search name, aka 's'.
        if (s) :
            try:
                savedSearchObject = search.getSavedSearch(s, namespace=app, owner=cherrypy.session['user'].get('name'))
            except splunk.ResourceNotFound:
                msg = _('The saved search "%(savedSearchName)s" could not be found.') % {'savedSearchName': s}
                lib.message.send_client_message('error', msg)
            else:
                if savedSearchObject.get('disabled') == '1':
                    msg = _('The saved search "%(savedSearchName)s" is disabled.') % {'savedSearchName': s}
                    lib.message.send_client_message('error', msg)
                else:
                    # ensure that the saved search has a viewstate object to save stuff to
                    # -- if saved search is owned by another user and is publically read-only,
                    #    the auto viewstate will fail
                    # TODO #1 it seems odd to do this here rather than in resurrectFromSavedSearch
                    #      for instance HiddenSavedSearch will now not get the same treatment. 
                    #      it's probably OK, but seems inconsistent.
                    if savedSearchObject.get('vsid') == None:
                        newVsid = viewstate.generateViewstateId(make_universal=True)
                        self.setViewstate(app, view_id, newVsid, _is_shared=True, _is_autogen=True)
                        savedSearchObject['vsid'] = newVsid
                        logger.info('Saved search "%s" has no viewstate; auto generating vsid=%s' % (s, newVsid))
                        try:
                            en.setEntity(savedSearchObject)
                        except splunk.AuthorizationFailed:
                            logger.warn('loaded a saved search without a viewstate; current user not authorized to generate viewstate')
                        except splunk.ResourceNotFound, e:
                            logger.warn('There was an error generating the view state object. The view will likely continue to work, but will not be able to persist its state.')
                        except Exception, e:
                            logger.exception(e)

                    try:
                        context = util.resurrectFromSavedSearch(
                            savedSearchObject=savedSearchObject,
                            hostPath=self.splunkd_urlhost,
                            namespace=app,
                            owner=cherrypy.session['user'].get('name'),
                            now=now
                            )
                    except splunk.SearchException:
                        msg = _('The saved search "%(savedSearchName)s" has a syntax error.') % {'savedSearchName': s}
                        lib.message.send_client_message('error', msg)

                        # scaffold the fallback structure
                        context = {
                            'fullSearch': savedSearchObject.get('search'),
                            'baseSearch': savedSearchObject.get('search'),
                            'decompositionFailed': True,
                            'intentions': [],
                            'earliest': savedSearchObject.get('dispatch.earliest_time'),
                            'latest': savedSearchObject.get('dispatch.latest_time')
                        }
                    
                    # TODO #2.  And obviously by pulling it up we would delete this line
                    context["vsid"] = savedSearchObject.get('vsid')

                    permalinkInfo['DATA'] = {'mode': 'saved', 'name': s, 'vsid': context["vsid"]}
                    permalinkInfo['toBeResurrected'] = context

        # 2. search Id.        
        elif (sid) :
            try :
                job = splunk.search.getJob(sid=sid,sessionKey=cherrypy.session['sessionKey'])
                jsonableJob = job.toJsonable(timeFormat='unix')
                
                # See -- SPL-24020  Saving a report using relative time ranges such as "Previous business week" gives me epoch earliest and lastest times in the save dialog    
                # to fix SPL-24020 it now only uses the earliestTime, latestTime from the job when the job was dispatched by the scheduler. 
                #    Note: earliest/latest from the job are always absolute epochTime.
                #    in all other cases it will use the earliest, latest args from the client, which may be relative time terms. 
                #    see SPL-24020 for further comments.
                earliestTime = None
                latestTime   = None
                
                request = jsonableJob.get("request")
                delegate = jsonableJob.get("delegate")

                if delegate:
                    earliestTime = jsonableJob["earliestTime"]
                    latestTime   = jsonableJob["latestTime"]

                    # handle the cases where the search was all time, or partially all time
                    if request:
                        # if there was no latest time set, treat the lastest time as the time the search was run
                        # if there was no earliest time set, make sure we clear the earliest time

                        if not request.get("latest_time"):
                            latestTime = jsonableJob["createTime"]

                        if not request.get("earliest_time"):
                            earliestTime = None
                            
                elif request:
                    earliestTime = request.get("earliest_time")
                    latestTime   = request.get("latest_time")
                else :
                    logger.error("Found job with no delegate that also had no request parameter. Unable to resurrect earliest and latest times.")
                    
                context = util.resurrectSearch(
                    hostPath = self.splunkd_urlhost,
                    q = job.search, 
                    earliest = earliestTime, 
                    latest   = latestTime, 
                    remote_server_list = None, # job.searchProviders,
                    namespace = app,
                    owner=cherrypy.session['user'].get('name')
                )

                
                # try to get the relevant viewstate; first check the URI, if not
                # passed, then check if job was dispatched from a saved search;
                # if so, try to pull the persisted viewstate id
                context["job"] = jsonableJob
                savedSearchName = None
                if 'vs' in cherrypy.request.params:
                    context["vsid"] = cherrypy.request.params['vs']
                else:
                    savedSearchObject = splunk.saved.getSavedSearchFromSID(sid)
                    if savedSearchObject:
                        savedSearchName = savedSearchObject.name
                        
                        vsid = savedSearchObject.get('vsid')
                        if vsid and len(vsid) > 0:
                            context['vsid'] = savedSearchObject['vsid']
                
                permalinkInfo['DATA'] = {'mode': 'sid', 'name': savedSearchName, 'sid': sid, 'vsid': context.get('vsid')} 
                permalinkInfo['toBeResurrected'] = context

            except splunk.AuthorizationFailed:
                lib.message.send_client_message('error', _('Permission to access job with sid = %(sid)s was denied.') % {'sid': sid})

            except splunk.ResourceNotFound : 
                lib.message.send_client_message('error', _('Splunk cannot find a job with an sid = %(sid)s. It may have expired or been deleted.') % {'sid': sid})
            
            
        # 3) straight up search language string + any manually passed in getargs.
        elif (q):
            # scaffold the fallback structure
            permalinkInfo['toBeResurrected'] = {
                    'fullSearch': q,
                    'baseSearch': q,
                    'decompositionFailed': True,
                    'intentions': [],
                    'earliest': earliest,
                    'latest': latest
                    }

            try:
                logger.debug("DECOMPOSITION > %s" % q)
                context = util.resurrectSearch(
                    hostPath = self.splunkd_urlhost,
                    q = q, 
                    earliest = earliest, 
                    latest = latest, 
                    remote_server_list = remote_server_list,
                    namespace = app,
                    owner = cherrypy.session['user'].get('name')
                    )
                if 'vs' in cherrypy.request.params:
                    context["vsid"] = cherrypy.request.params['vs']

                permalinkInfo['toBeResurrected'] = context

            except splunk.SearchException, e:
                # TODO: Make sure this message coming from splunkd is eventually translated.
                # It cannot just be wrapped in _() right now.
                lib.message.send_client_message('error', e.message)

        return permalinkInfo

    def buildViewTemplate(self, app, view_id, action=None, q=None, sid=None, s=None, earliest=None, latest=None, remote_server_list=None, render_invisible=False, build_nav=True, now=None, include_app_css_assets=True):
        """
        Build the template args required to render a view
        The render() handler below calls this, as do other controllers that need to embed a view 
        into their own templates (see the AdminController for example)
        """

        # assert on view name; may only be alphanumeric, dot, dash, or underscore
        view_id_checker = re.compile('^[\w\.\-\_]+$')
        if not view_id_checker.match(view_id):
            raise cherrypy.HTTPError(400, _('Invalid view name requested: "%s". View names may only contain alphanumeric characters.') % view_id)
        
        # get list of all UI apps
        appList = self.getAppManifest()

        permalinkInfo = self.processPermalink(app, now, earliest, latest, remote_server_list, q, sid, s, view_id)
        
        # get current app configuration
        appConfig = self.getAppConfig(app, appList, permalinkInfo, view_id, build_nav)

        if not appConfig['is_visible'] and not render_invisible:
            raise cherrypy.HTTPError(404, _('App "%s" does not support UI access.  See its app.conf for more information.') % appConfig['label'])
        
        # get all views for current user in app context
        availableViews = appConfig['available_views']
        if len(availableViews) == 0:
            raise cherrypy.HTTPError(404, _('App "%s" does not have any available views.') % app)
        
        # get template layout for current config
        currentViewConfig = availableViews.get(view_id)
        if not currentViewConfig:
            raise cherrypy.HTTPError(404, _('Splunk cannot find the  "%s" view.') % view_id)
        
        if currentViewConfig.get('objectMode') == 'XMLError' and currentViewConfig.get('message'):
            raise cherrypy.HTTPError(400, _('XML Syntax Error: %s' % currentViewConfig.get('message')))
        
        # get asset rosters
        moduleMap = currentViewConfig.get('layoutRoster', {})
        cssList = []
        jsList = []
        for moduleName in currentViewConfig.get('activeModules', []):
            moddef = moduleMapper.getInstalledModules()[moduleName]
            if 'css' in moddef:
                cssList.append(self.filePathToUrl(moddef['css']))
            if 'js' in moddef:
                jsList.append(self.filePathToUrl(moddef['js']))

        # compile link list for all available views
        viewList = [{'label': availableViews[k].get('label', k), 'uri': self.make_url(['app', app, k])} for k in availableViews]
        viewList.sort(lambda x,y: cmp(x['label'], y['label']))
        
        # gather application specific static content
        # allowed css is 'application.css', then any css declared in viewConfig
        customCssList = []
        printCssList = []
        customJsList = []
        
        if not app in local_apps:
            local_apps.refresh(True)
            if not app in local_apps:
                raise cherrypy.HTTPError(404, _('App  "%s" does not exist.') % app)
            
        #js_assests
        allowedJs = ['application.js']
        if local_apps[app].has_key('static'):
            appstatics = local_apps[app]['static'].get('js', [])
            for filename in appstatics:
                if filename in allowedJs:
                    customJsList.append("/static/app/%s/%s" % (app,filename))        
        
        #css assets
        if include_app_css_assets:
            allowedCss = [currentViewConfig.get('stylesheet')]
            allowedCss.append('application.css')
                
            allowedPrintCss = ['print.css']
            if local_apps[app].has_key('static'):
                appstatics = local_apps[app]['static'].get('css', [])
                for filename in appstatics:
                    if filename in allowedCss:
                        customCssList.append("/static/app/%s/%s" % (app,filename))
                    elif filename in allowedPrintCss:
                        printCssList.append("/static/app/%s/%s" % (app,filename))
                appPatches = local_apps[app]['patch'].get('css', [])
                for patch in appPatches:
                    customCssList.append(patch)
        
        # translate the view label
        label = currentViewConfig.get('label')
        if label:
            label = _(label)
        else:
            label = '(%s)' % view_id
        
        # Safely get the displayView. Somehow this gets set to None elsewhere
        # which causes some problems further up in the stack
        displayView = currentViewConfig.get('displayView')
        if displayView == None:
            displayView = view_id
        
        # assemble the template vars
        templateArgs = {
        
            # define standard params
            'APP': {'id': app, 'label': appConfig['label'], 'is_visible': appConfig['is_visible'], 'can_alert': appConfig['can_alert']},
            'VIEW': {'id': view_id,
                     'label': label,
                     'displayView': displayView,
                     'refresh': currentViewConfig.get('refresh', 10),
                     'onunloadCancelJobs': currentViewConfig.get('onunloadCancelJobs'),
                     'autoCancelInterval': currentViewConfig.get('autoCancelInterval'),
                     'template': currentViewConfig.get('template', []),
                     'objectMode': currentViewConfig.get('objectMode'),
                     'nativeObjectMode': currentViewConfig.get('nativeObjectMode'),
                     'hasAutoRun': currentViewConfig.get('hasAutoRun'),
                     'editUrlPath': currentViewConfig.get('editUrlPath'),
                     'canWrite': currentViewConfig.get('canWrite'),
                     'hasRowGrouping': currentViewConfig.get('hasRowGrouping'),
                    },
            'DATA': {'mode': None},
            'toBeResurrected': None,
            
            # define HTML asset params
            'navConfig': appConfig['nav'],
            'appList': appList,
            'viewList': viewList,
            'modules':  moduleMap,
            "cssFiles" : cssList,
            "customCssFiles" : customCssList,
            "printCssFiles" : printCssList,
            "jsFiles"  : jsList,
            "customJsFiles": customJsList,
            'splunkReleaseVersionParts': self._normalizedVersions(splunk.getReleaseVersion()),
            "make_static_app_url": self.make_static_app_url_closure(app)                  
        }
        
        if 'DATA' in permalinkInfo:
            templateArgs['DATA'] = permalinkInfo['DATA']

        if 'toBeResurrected' in permalinkInfo:
            templateArgs['toBeResurrected'] = permalinkInfo['toBeResurrected']
        
        return templateArgs   

    def _normalizedVersions(self, version):
        """
        Takes a loosely defined version number, replaces all word characters to underscores and returns in order matching variants.
        
        Ex:
        version: 4.2.2
        matches:
        ['4', '4_2', '4_2_2']
        """
        
        version_delim = '_'
        version = re.sub("([^\w]+)", version_delim, version) #"elvis@#$#@$4.4" -> "elvis_4_4"
        version_parts = version.split(version_delim)
        version_name = None
        versions = []

        for version_part in version_parts:
            if version_name is None:
                version_name = version_part
            else:
                version_name = version_name + version_delim + version_part
            versions.append(version_name)
        return versions
        
    def make_static_app_url_closure(self, app):
        def make_static_app_url(path, *a, **kw):
            return util.make_url(['static', 'app', app, path], *a, **kw)
        return make_static_app_url

    @route('/:app/:view_id')
    @expose_page(handle_api=True)
    @set_cache_level('never')
    def render(self, app, view_id, action=None, q=None, sid=None, s=None, earliest=None, latest=None, remote_server_list=None, now=None, output='html', **kw):
        '''
        Handle main view requests
        '''

        #
        # DEBUG
        #

        if output == 'pdf':
            # make a sub-request to the pdf report controller
            ctrl = cherrypy.request.app.root.report
            prefix = util.generateBaseLink()
            try:
                session_key = cherrypy.session['sessionKey']
                owner = cherrypy.session['user']['name']
                cherrypy.session.release_lock()
                return ctrl.requestPDF(session_key=session_key,
                    owner = owner,
                    request_path = prefix + util.current_url_path(False),
                    override_disposition = 'inline; filename="dashboard.pdf"'
                    )
            except controllers.report.SimpleError, e:
                # render an HTML version of the dashboard instead, including the PDF error
                lib.message.send_client_message('error', _("An error occurred while rendering the PDF: ")+e._message)
                output = 'html'
                # fall through to HTML renderer
                

        if kw.get('showsource'):
            templateArgs = {
                'viewConfig': viewconf.loads(en.getEntity(VIEW_ENTITY_CLASS, view_id, namespace=app).get('eai:data'), view_id, isStorm=splunk.util.normalizeBoolean(cherrypy.config.get('storm_enabled'))),
                'view_id': view_id
            }
            try:
                templateArgs['viewXml'] = viewconf.dumps(templateArgs['viewConfig'])
            except Exception, e:
                logger.exception(e)
                templateArgs['viewXml'] = '(unable to generate XML: %s)' % e
                
            return self.render_template('/view/pyramid.html', templateArgs)
            
        if kw.get('showtree'):
            viewConfig = viewconf.loads(en.getEntity(VIEW_ENTITY_CLASS, view_id, namespace=app).get('eai:data'), view_id, isStorm=splunk.util.normalizeBoolean(cherrypy.config.get('storm_enabled')))
            
            def convertModuleToJit(module):
                newModule = {
                    'id': '',
                    'name': '',
                    'children': []
                }
                
                if 'className' in module:
                    newModule['name'] = module['className']
                    if module['className'] in seenModules:
                        seenModules[module['className']] += 1
                    else:
                        seenModules[module['className']] = 1
                    newModule['id'] = '_'.join([module['className'], str(seenModules[module['className']])])

                if 'children' in module:
                    for childModule in module['children']:
                        newModule['children'].append(convertModuleToJit(childModule))
                return newModule

            
            seenModules = {}
            moduleTree = [{'id': 'root', 'name': 'root', 'children': []}]
            if 'modules' in viewConfig.keys():
                for module in viewConfig['modules']:
                    moduleTree[0]['children'].append(convertModuleToJit(module))
                    
            templateArgs = {'moduleTree': moduleTree}
            return self.render_template('/view/tree2.html', templateArgs)

        time1 = time.time()
        templateArgs = self.buildViewTemplate(app, view_id, action, q, sid, s, earliest, latest, remote_server_list, now=now)
                
        #try:
        #    templateArgs = self.buildViewTemplate(app, view_id, action, q, sid, s, earliest, latest, remote_server_list, now=now)
        #except cherrypy.HTTPError as error:
        #    cherrypy.response.status = error.status
        #    return self.render_template('view/404_view.html', {"status":error.status, "message":error[1], "app":app})            
        time2 = time.time()
        try:
            output = self.render_template('/view/' + templateArgs['VIEW']['template'], templateArgs)
        # If we couldn't render the template from the global view templates then
        # it is likely an app specified template.
        # TODO: Leverage mako lookups to handle to order of directories so we 
        # don't have to do this.
        except:
            output = self.render_template(templateArgs['VIEW']['template'], templateArgs)
        time3 = time.time()
        logger.info('PERF - viewTime=%ss templateTime=%ss' % (round(time2-time1, 4), round(time3-time2, 4)))
        if cherrypy.request.is_api:
            # set non json serializable types to None revisit if required.
            for i in templateArgs['appList']:
                templateArgs['appList'] = None
            templateArgs['make_url'] = None
            templateArgs['make_route'] = None
            templateArgs['attributes'] = None
            templateArgs['controller'] = None
            templateArgs['make_static_app_url'] = None
            templateArgs['h'] = None
            return self.render_json(templateArgs)
        else:
            cherrypy.response.headers['content-type'] = MIME_HTML
            return output
    

    @route('/:app/:view_id/:viewstate_id', methods='GET')
    @expose_page(handle_api=True, methods='GET')
    def getViewstate(self, app, view_id, viewstate_id):
        '''
        Returns a JSON structure representing all of the module params for
        view_id.
        '''

        try:
            viewManifest = self.getViewManifest(namespace=app, currentViewName=view_id, viewstate_id=viewstate_id)
        except splunk.ResourceNotFound:
            raise cherrypy.HTTPError(status=404, message=(_('Viewstate not found; view=%s viewstate=%s') % (view_id, viewstate_id)))

        
        currentView = viewManifest[view_id]

        output = {}

        for panelName in currentView['layoutRoster']:
            for currentModule in currentView['layoutRoster'][panelName]:
                output[currentModule['id']] = currentModule.get('params', {})

        return self.render_json(output)
        
        
        
    @route('/:app/:view_id/:viewstate_id')
    @expose_page(handle_api=True, methods=['GET','POST'])
    def setViewstate(self, app, view_id, viewstate_id, _is_shared=False, _is_autogen=False, **form_args):
        '''
        Persists module params to the specific viewstate object.  Writes are
        done in an overlay fashion; unspecified params will not be overwritten.
        Parameters are accepted in the following format:

            <module_DOM_id>.<param_name>=<param_value>

        Ex:

            SearchBar_0_0_0.useTypeahead=true

        Only string values are accepted; complex data types are not persistable.
        '''

        output = jsonresponse.JsonResponse()

        # determine the desired viewstate
        altView, vsid = viewstate.parseViewstateHash(viewstate_id)
        if altView == None:
            altView = view_id

        # scaffold object mapper
        vs = viewstate.Viewstate()
        vs.namespace = app
        vs.view = altView
        vs.id = vsid

        if splunk.util.normalizeBoolean(_is_shared):
            vs.owner = 'nobody'
        else:
            vs.owner = cherrypy.session['user'].get('name')

        # add in stub property
        if _is_autogen and len(form_args) == 0:
            form_args['is.autogen'] = 1

        if len(form_args) == 0:
            logger.warn('setViewstate - no parameters received; nothing persisted')
            output.success = False
            output.addError(_('No parameters received; aborting'))
            return self.render_json(output)
            
        # insert all passed params
        for key in form_args:
            parts = key.split('.', 1)
            if len(parts) < 2:
                logger.warn('setViewstate - invalid viewstate param name: %s; aborting' % key)
                output.success = False
                output.addError(_('Invalid viewstate param name: %s; aborting') % key)
                return self.render_json(output)

            vs.modules.setdefault(parts[0], {})
            vs.modules[parts[0]][parts[1]] = form_args[key]
            logger.debug('setViewstate - setting module=%s param=%s value=%s' % (parts[0], parts[1], form_args[key]))

        # commit
        try:
            viewstate.commit(vs)
        except Exception, e:
            logger.exception(e)
            output.success = False
            output.addError(str(e))
            return self.render_json(output)


        # set sharing bit, if requested; should succeed even for normal users
        if splunk.util.normalizeBoolean(_is_shared):
            try:
                viewstate.setSharing(vs, 'global')
            except Exception, e:
                logger.exception(e)
                output.success = False
                output.addError(str(e))
                return self.render_json(output)

        return self.render_json(output)
