import cherrypy, json, time, logging
import splunk
import splunk.appserver.mrsparkle
from splunk.appserver.mrsparkle.lib import i18n, times, cached, util

logger = logging.getLogger('splunk.appserver.lib.config')
    
def _get_active_config():
    return {
        'SEARCH_RESULTS_TIME_FORMAT': i18n.ISO8609_MICROTIME,
        'DISPATCH_TIME_FORMAT': cherrypy.config.get('DISPATCH_TIME_FORMAT'),
        'MRSPARKLE_ROOT_PATH': cherrypy.request.script_name,
        'MRSPARKLE_PORT_NUMBER': cherrypy.config.get('server.socket_port'),
        'VERSION_LABEL': cherrypy.config.get('version_label', 'UNKNOWN_VERSION'),
        'BUILD_NUMBER': cherrypy.config.get('build_number', 0),
        'BUILD_PUSH_NUMBER': cherrypy.config.get('_push_version', 0),
        'LOCALE': i18n.current_lang_url_component(),
        'FLASH_MAJOR_VERSION': cherrypy.config.get('flash_major_version', 0),
        'FLASH_MINOR_VERSION': cherrypy.config.get('flash_minor_version', 0),
        'FLASH_REVISION_VERSION': cherrypy.config.get('flash_revision_version', 0),
        "JS_LOGGER_MODE": cherrypy.config.get('js_logger_mode', 0),
        "JS_LOGGER_MODE_SERVER_END_POINT": cherrypy.config.get('js_logger_mode_server_end_point', '/'),
        "JS_LOGGER_MODE_SERVER_POLL_BUFFER": cherrypy.config.get('js_logger_mode_server_poll_buffer', 100000),
        "JS_LOGGER_MODE_SERVER_MAX_BUFFER": cherrypy.config.get('js_logger_mode_server_max_buffer', 1000),
        'NOCACHE' : cherrypy.config.get('js_no_cache', False),
        'UI_UNIX_START_TIME': int(cherrypy.config.get('start_time', 0)),
        'DEFAULT_NAMESPACE': splunk.getDefault('namespace'),
        'SYSTEM_NAMESPACE': splunk.appserver.mrsparkle.SYSTEM_NAMESPACE,
        'UI_INACTIVITY_TIMEOUT': getCherrypyConfigIntSafe('ui_inactivity_timeout', 60),
        'SERVER_TIMEZONE_OFFSET': getServerTimezoneOffset(),
        'SERVER_ZONEINFO': getServerZoneInfo(),
        'SPLUNKD_FREE_LICENSE': cherrypy.config.get('is_free_license'),
        'FORM_KEY': util.getFormKey(),
        'USERNAME': cherrypy.session.get('user', {}).get('name'),
        'JSCHART_TEST_MODE': cherrypy.config.get('jschart_test_mode', False)
    }

def _get_active_unauthorized_config():
    return {
        'MRSPARKLE_PORT_NUMBER': cherrypy.config.get('server.socket_port'),
        'UI_INACTIVITY_TIMEOUT': getCherrypyConfigIntSafe('ui_inactivity_timeout', 60),
        'FORM_KEY': util.getFormKey(),
        'SERVER_ZONEINFO': '',
    }

def _get_app_config(app):
    try:
        rawConfig = splunk.bundle.getConf('app', namespace=app)
    except splunk.ResourceNotFound:
        return {}
    return {
        'APP_BUILD': rawConfig['install'].get('build', 0)
    }

def getCherrypyConfigIntSafe(key, default):
    """returns int value under the key in cherrypy.config; if it's not an integer, returns default value"""
    try:
        i = int(cherrypy.config.get(key, default))
        return (i<0)*0 + (i>0)*i
    except ValueError:
        logger.warn('%s key is not integer, assuming default value %s', key, default)
        return default
    
def getServerTimezoneOffset():
    """  returns the offset from GMT in seconds  """
    # Somewhat shockingly, this clunky if/else is the official way to get the actual timezone offset, 
    # ie a offset int that is accurate in both DST and non-DST times.
    if (time.localtime()[-1] == 1): 
        return time.altzone
    else:
        return time.timezone

@cached.memoized(cache_age=30)
def getServerZoneInfo():
    '''
    Returns server's zoneinfo table.  Memoized.
    '''
    try:
        return times.getServerZoneinfo()
    except Exception, e:
        logger.exception(e)
        return ''

def getConfig(sessionKey=None, namespace=None):
    '''
    Returns the configuration information for the main Splunk frontend.
    The values returned from the endpoint are subject to the following:
    
    1) values are idempotent
    2) any time values are in ISO-8601 format
    3) values are typed appropriately
    
    These values should be treated as read-only.
    
    '''
    # unauthed calls get the bare minimum
    if not sessionKey:
        args = _get_active_unauthorized_config()
    else:
        args = _get_active_config()

    if namespace:
        args.update(_get_app_config(namespace))

    logger.debug('config values: %s' % args)
    return args
