import urllib
import cherrypy
import splunk
import logging
import os
import json

from splunk.appserver.mrsparkle.lib import util, startup, message
from routes import RequestRefused
from decorator import decorator
from splunk.appserver.mrsparkle.list_helpers.formattermapper import FormatterMapper
from cherrypy.lib import http

logger = logging.getLogger('splunk.appserver.mrsparkle.lib.decorators')

ONLY_API=1
SPLUNKD_SESSION_KEY = 'sessionKey'

# SSO related constants
# These are referenced in account.py and debug.py
DEFAULT_REMOTE_USER_HEADER = 'REMOTE_USER'
REMOTE_USER_SESSION_KEY = 'REMOTE_USER'
SPLUNKWEB_REMOTE_USER_CFG = 'remoteUser'
SPLUNKWEB_TRUSTED_IP_CFG = 'trustedIP'
SPLUNKWEB_SSO_MODE_CFG = 'SSOMode'

def chain_decorators(fn, *declist):
    """
    Called from a decorator to chain other decorators together
    eg. chain_decorators(fn, require_login(), cherrypy.expose)
    """

    for dec in declist[::-1]: # wrap them in the same order as they'd be wrapped if they were used normally
        fn = dec(fn)

    @decorator
    def rundecs(_fn, *a, **kw):
        return fn(*a, **kw)

    return rundecs

def expose_page(must_login=True, handle_api=False, methods=None, verify_session=True, verify_sso=True, trim_spaces=False):
    """
    Use this instead of cherrypy.expose
    Ensures that user's are logged in to view the page by default
    
    set handle_api=True to have requests beginning with /api sent to the handler 
    as well as non-api requests
    set handle_api=ONLY_API to have it only accept api requests
    (check cherrypy.request.is_api to see whether this is an api request if set to True)

    set methods to a list of method names to accept for this handler (default=any)
    """
    @decorator
    def check(fn, self, *a, **kw):
        is_api = util.is_api()
        request = cherrypy.request
        if not handle_api and is_api:
            raise RequestRefused(404)
        if handle_api is ONLY_API and not is_api:
            raise RequestRefused(404)
        _methods = methods
        if _methods:
            if isinstance(_methods, basestring):
                _methods = [ _methods ]
            if request.method not in _methods:
                raise RequestRefused(405)
        
        # verify that version info is good; do it here so that any URI access
        # will trigger the check
        startup.initVersionInfo()

        # add a convenience property to all request objects to get at the
        # current relative URI
        request.relative_uri = request.path_info + (('?' + request.query_string) if request.query_string else '')
        if cherrypy.config.get('root_endpoint') not in ['/', None, '']:
            request.relative_uri = cherrypy.config.get('root_endpoint') + request.relative_uri

        # CSRF protection
        # Disable in tests by setting cherrypy.config.update({'environment': 'test_suite'})
        if verify_session and request.method == 'POST' and not cherrypy.config.get('environment') == 'test_suite':
            is_xhr = util.is_xhr()
            form_key = request.headers.get('X-Splunk-Form-Key') if is_xhr else request.params.get('splunk_form_key')
            # verify that the incoming form key matches server's version
            if not util.isValidFormKey(form_key):
                if is_xhr:
                    logger.warn('CSRF: validation failed because client XHR did not include proper header')
                else:
                    logger.warn('CSRF: validation failed because HTTP POST did not include expected parameter')
                if must_login:
                    if is_xhr:
                        raise cherrypy.HTTPError(401, _('Splunk cannot authenticate the request. CSRF validation failed.'))
                    else:
                        return self.redirect_to_url('/account/login', _qs=[ ('return_to', util.current_url_path()) ] )
                logger.warn('CSRF: skipping 401 redirect response because endpoint did not request protection')

        # basic input cleansing
        if trim_spaces:
            for key, value in kw.iteritems():
                if isinstance(value, basestring):
                    kw[key] = value.strip()
                    if kw[key] != value:
                        logger.debug('Leading/trailing whitespaces were trimmed in "%s" argument' % key)
                
        return fn(self, *a, **kw)


    def dec(fn):
        if must_login:
            return chain_decorators(fn, check, sso_ip_validation(verify_sso), sso_check(), require_login(), ExceptionHandler(), cherrypy.expose)(fn)
        else:
            return chain_decorators(fn, check, sso_ip_validation(verify_sso), ExceptionHandler(), cherrypy.expose)(fn)
            
    return dec

    
def clean_session():
    '''Safely clean the session. This is used primarily by the SSO mechanism.'''
    # Secure the file
    cherrypy.session.escalate_lock()

    # Clears the data from the in memory session
    cherrypy.session.clear()

    # Abandons the session. Strangely we still need to call clear
    # even though a read of regenerate() seems to imply it works
    # on the in memory session (though never calls clear()).
    cherrypy.session.regenerate()


def sso_ip_validation(verify_sso=True):
    '''
    SSO strict mode lockdown.                      
    Screen the incoming requests and ensure they are originating from a valid IP address.
    If we're in SSO strict mode we lock down all endpoints, except those that specify verify_sso=False
    via the expose_page decorator.
    '''
    @decorator
    def validate_ip(fn, self, *a, **kw):
        if util.in_sso_mode() and verify_sso:
            incoming_request_ip = cherrypy.request.remote.ip
            splunkweb_trusted_ip = splunk.util.stringToFieldList(cherrypy.config.get(SPLUNKWEB_TRUSTED_IP_CFG))
            sso_mode = cherrypy.request.config.get(SPLUNKWEB_SSO_MODE_CFG, 'strict')
            current_remote_user = cherrypy.session.get(REMOTE_USER_SESSION_KEY)
            
            if incoming_request_ip not in splunkweb_trusted_ip:
                if current_remote_user:
                    logger.warn('There was a user logged by SSO and somehow the splunkweb trustedIP is no longer valid. Removing the logged in user.')
                    clean_session()
                
                if sso_mode and sso_mode.lower() == 'strict':
                    raise cherrypy.HTTPError(403, _("Forbidden: Strict SSO Mode"))
                    
        return fn(self, *a, **kw)
        
    return validate_ip

def sso_check():
    '''
    Preforms the SSO validation and authentication.
    '''
        
    def login(handler_inst, user):
        '''
        Attempts to login the user via splunkd's trusted endpoint.
        This will only ever work if splunkd is in trusted auth mode.
        '''
        # Clean the sessionKey, something has gone wrong, but we're going to make it right.
        clean_session()

        # Get the session key from the trusted endpoint, this should not raise an exception
        sessionKey = splunk.auth.getSessionKeyForTrustedUser(user)

        if sessionKey != None:
            # Escalate the lock again just in case.
            cherrypy.session.escalate_lock()

            # Store the splunkd session key
            cherrypy.session[SPLUNKD_SESSION_KEY] = sessionKey
            
            # Store the incoming user in the Remote-User header. This is critical!
            cherrypy.session[REMOTE_USER_SESSION_KEY] = user

            # now get the user's full name
            en = splunk.entity.getEntity('authentication/users', user, sessionKey=sessionKey)
            fullName = user
            if en and 'realname' in en and en['realname']:
                fullName = en['realname']

            # This was stolen from account.py
            cherrypy.session['user'] = {
                'name': user,
                'fullName': fullName,
                'id': -1
            }
        else:
            logger.warn('Could not authenticate user %s via SSO. Does %s have a matching splunk account with the same username?' % (user, user))
            handler_inst.redirect_to_url('/account/sso_error')

    @decorator
    def preform_sso_check(fn, self, *a, **kw):
        '''
        The core of SSO.  This validates SSO users, ensures that users are logged out of
        Splunk when logged out of SSO, etc.
        '''

        if util.in_sso_mode():
            logger.debug('In SSO mode.')
            
            # Retrieve the name of the remote user header passed from the proxy
            remote_user_header = cherrypy.request.config.get(SPLUNKWEB_REMOTE_USER_CFG) or DEFAULT_REMOTE_USER_HEADER
            
            # Retrieve the incoming remote user
            incoming_remote_user = cherrypy.request.headers.get(remote_user_header)
            
            # Retrieve the last user who visited via SSO and was stored in the session
            current_remote_user = cherrypy.session.get(REMOTE_USER_SESSION_KEY)
            
            # Retrieve the splunkd session key
            splunkd_session_key = cherrypy.session.get(SPLUNKD_SESSION_KEY)
            
            # The IP address of the request, used to validate the incoming req.
            incoming_request_ip = cherrypy.request.remote.ip
            
            # The list of IP addresses that are trusted by splunkweb
            splunkweb_trusted_ip = splunk.util.stringToFieldList(cherrypy.config.get(SPLUNKWEB_TRUSTED_IP_CFG))
            
            # We should only ever evaluate this condition if SSOMode is not strict but the IPs don't match.
            # In this case we allow the user to login via regular splunk auth.
            # Not happy that we do this check twice per request, but without this line users would be
            # authed into SSO even if originating from an invalid IP.
            if incoming_request_ip not in splunkweb_trusted_ip:
                logger.info('The incoming request did not originate from a trusted IP address, thus SSO will not be performed. However, the user will be able to login to Splunk manually. Splunkweb must have SSOMode set to permissive.')
                return fn(self, *a, **kw)
            
            logger.debug('Request comes from a trusted source, will validate %s header' % remote_user_header)
            
            # There should be 8 cases here (2^3 possible choices between the three options).
            # Some of these are redundant, but I've left them verbose so it's easier to understand
            # why certain cases are handled in specific ways.
            if incoming_remote_user:
       
                # Case 1: They don't have a session, or have an invalid session 
                if not splunkd_session_key or not splunk.auth.ping(sessionKey=splunkd_session_key):
                    logger.debug('SSO CASE 1: %s header is set but no valid splunkd session key could be found.' % remote_user_header)
                    login(self, incoming_remote_user)

                # Case 2: They have a session, but they don't seem to match the previously logged in user.
                # Maybe they logged out of SSO, but not splunk?
                elif current_remote_user and incoming_remote_user != current_remote_user:
                    logger.debug('SSO CASE 2: %s header is set, splunkd session key is available and valid, but there is a mismatch between the user stored in the session and the incoming user. %s != %s' % (remote_user_header, current_remote_user, incoming_remote_user))
                    login(self, incoming_remote_user)
        
                # Case 3: They have a session but there is no current logged in user. The only way this
                # could knowingly happen is if the user's session is destroyed somewhere and the current
                # SSO user is not placed back in the session before coming back here. Maybe this could happen
                # if the user access splunkweb on its direct port, not the proxy. First time login
                # should have been handled by case 1, but this is put here for redundancy. Also note,
                # this requires we stash the Remote-User in the session after a user logins
                # in account/login. Perhaps less than ideal but it's not obvious how else to guard
                # against a user logging into SSO, logging out of splunk, logging in to splunk as admin
                # then logging out of SSO, then another user comes along, logs into SSO, hits splunk and
                # because we never checked for current_remote_user == None (assuming this wasn't here)
                # they suddenly login as admin!
                elif current_remote_user == None:
                    logger.warn('SSO CASE 3: %s header is set and a valid session key exists, but the visitor was strangely not previously stored in the session.' % remote_user_header)
                    login(self, incoming_remote_user)

                # Case 4: The user has a valid session and matches the current_remote_user. They are authed
                # which is confirmed by the require_login decorator.
                logger.debug('SSO CASE 4: %s header is set, user in the session matches the user in the %s header and the user is authed.' % (remote_user_header, remote_user_header))

            else:
                # Case 5: There was a user here and now they are gone. This is sketchy, maybe someone
                # got into the actual splunkweb server when they shouldn't have after logging into SSO.
                # Logout them out to be sure.
                if current_remote_user:
                    logger.error('SSO CASE 5: %s header is NOT set, but a remote user was stored in the session before. Logged in user will be logged out.' % remote_user_header)
                    clean_session()
       
                logger.debug('SSO CASE 5,6,7,8: %s header is NOT set, the user\'s fate will be determined by the validity of their session key.' % remote_user_header)

                # Case 6: The user has an invalid sessionKey, and no current_remote_user. In this case
                # we let the require_login handler take care of the invalid auth.

                # Case 7: The user has a valid session but did not provide an incoming_remote_user or
                # a current_remote_user.  They are allowed through if the require_login decorator allows
                # them access.  Unclear if this should be handled differently.  Presumably someone could
                # want splunkweb access for super special users who can access the splunkweb server
                # directly even while in SSO mode.

                # Case 8: There is no incoming_remote_user and no current_remote_user and no session.
                # They will be punted by require_login.
                
        return fn(self, *a, **kw)
        
    return preform_sso_check

def lock_session(fn):
    """
    Use this if your handler will make changes to cherrypy.session
    It causes CherryPy to acquire an exclusive lock on the session for the 
    duration of the request ensuring there aren't any race conditions with
    other requests that are also accessing session data.
    """
    fn.lock_session = True
    return fn

def require_login():
    """
    If for some reason you're not using the expose_page decorator
    you can use this to require a user to be logged in instead.
    use expose_page though. really.
    """
    @decorator
    def check_login(fn, self, *a, **kw):
        session_key = cherrypy.session.get('sessionKey', None)
        is_api = util.is_api()

        if not session_key:
            logger.info('require_login - no splunkd sessionKey variable set; cherrypy_session=%s request_path=%s' % (cherrypy.session.id, cherrypy.request.path_info))
            logger.debug('require_login - cookie request header: %s' % unicode(cherrypy.request.cookie))
            logger.debug('require_login - cookie response header: %s' % unicode(cherrypy.response.cookie))
            if is_api or util.is_xhr():
                logger.info('require_login - is api/XHR request, raising 401 status')
                raise cherrypy.HTTPError(401)
            else:
                logger.info('require_login - redirecting to login')
                self.redirect_to_url('/account/login', _qs=[ ('return_to', util.current_url_path()) ] )
            
        try:
            return fn(self, *a, **kw)
        except splunk.AuthenticationFailed:
            logger.info('sessionKey rejected by splunkd')
            cherrypy.session.delete()
            if is_api or util.is_xhr():
                raise cherrypy.HTTPError(401)
            else:
                self.redirect_to_url('/account/login', _qs=[ ('return_to', util.current_url_path()) ] )

    return check_login


def ExceptionHandler():
    """
    Handles exceptions returned by simpleRequest
    """
    @decorator
    def handle_exceptions(fn, self, *a, **kw):
        from controllers.admin import AdminController

        try:
            return fn(self, *a, **kw)
            
        except splunk.AuthenticationFailed:
            # redirect to the login page if auth fails
            cherrypy.session['sessionKey'] = None
            self.redirect_to_url('/account/login', _qs=[ ('return_to', util.current_url_path()) ] )
            
        except splunk.AuthorizationFailed, e:
            if isinstance(self, AdminController):
                return self.render_admin_template('admin/error.html', {'namespace' : 'search', 'excp_msg': e, 'excp_details' : 'None'})   
            else:
                raise
                
        except splunk.SplunkdConnectionException, e:
            logger.exception(e)
            raise cherrypy.HTTPError(503, _('The splunkd daemon cannot be reached by splunkweb.  Check that there are no blocked network ports or that splunkd is still running.'))
            
        except splunk.BadRequest, e:
            logger.exception(e)
            if e.msg == "Couldn't parse xml reply":
                raise cherrypy.HTTPError(500, _('The splunkd python dispatcher was unable to properly process script output.'))
            else:
                raise
                
    return handle_exceptions


def conditional_etag():
    '''
    DEPRECATED.  Use @set_cache_level('etag') instead.
    
    Similar to the util.apply_etag(content) method this wraps the 
    entire response with predefined 304 behavior.
    '''
    @decorator
    def apply_etag(fn, self, *a, **kw):
        response = fn(self, *a, **kw)
        if (util.apply_etag(response)):
            return None
        else:
            return response
        
    return apply_etag

def set_cache_level(cache_level):
    '''
    This is a convience wrapper for util.set_cache_level, providing
    backwards compatibility with the original set_cache_level decorator.
    
    The body of this method was moved to the util module so that individual
    modules could dictate whether or not they need to cache their getResults
    responses.
    '''
    @decorator
    def apply_cache_headers(fn, self, *a, **kw):
        response = fn(self, *a, **kw)
        return util.set_cache_level(cache_level, response)
    return apply_cache_headers

def normalize_list_params():
    '''
    Requires the underlying class implements the default list params:
        COUNT
        OFFSET
        SORT_KEY
        SORT_DIR
    '''
    
    @decorator
    def apply_normalized_list_params(fn, self, *a, **kw):
        params = {
            'count': kw.get('count', self.COUNT),
            'offset': kw.get('offset', self.OFFSET),
            'sort_key': kw.get('sort_key', self.SORT_KEY),
            'sort_dir': kw.get('sort_dir', self.SORT_DIR)
        }
        
        try:
            params['count'] = int(params['count'])
        except ValueError:
            params['count'] = int(self.COUNT)
            
        try:
            params['offset'] = int(params['offset'])
        except ValueError:
            params['offset'] = int(self.OFFSET)
        
        kw.update(params)

        return fn(self, *a, **kw)
    
    return apply_normalized_list_params

def format_list_template():

    def decode_fields(fields):
        if type(fields) != list:
            fields = [fields]
    
        decoded_fields = []
        for field in fields:
            try:
                decoded_fields.append(json.loads(field))
            except ValueError, e:
                decoded_fields.append(field)
        return decoded_fields
    
    @decorator
    def response_template(fn, self, *a, **kw):

        # Retrieve the list of dictionaries
        kw['list_data'] = fn(self, *a, **kw)

        # protect against dir traversal 
        # We might want to consider more strict filtering, but shouldn't
        # have any issues with double encoding or unicode filtering based
        # on quick testing.
        output_mode = os.path.join('lists', '%s.html' % os.path.basename(kw.get('output_mode', 'li')))
        
        fields = kw.get('fields')
        staticFields = kw.get('staticFields')
       
        # TODO: make this more sane using a cherrypy tool that translates
        # field[0][label]=foo,field[0][value]=bar to:
        # field = {'0': {'label':'foo', 'value':'bar'}}
        if fields:
            kw['fields'] = decode_fields(fields)

        if staticFields:
            kw['staticFields'] = decode_fields(staticFields)

        # Some list templates require delimiters.
        # This may not be the ideal place to put this, but where else?
        if not kw.get('delimiter'):
            kw['delimiter'] = ''

        return self.render_template(output_mode, kw)

    return response_template

