import os, sys, platform, shutil, tarfile
import stat
import logging as logger
import splunk.clilib.cli_common
import datetime
import optparse
import fnmatch, re
import glob
import socket
import traceback
# work on 3.2 (zzz but...)
try:
    import splunk.clilib.bundle_paths
except:
    pass

SPLUNK_HOME   = os.environ['SPLUNK_HOME']
RESULTS_LOC   = os.path.join(SPLUNK_HOME, 'var', 'run', 'splunk', 'diag-temp')
MSINFO_FILE   = 'msinfo-sum.txt'
SYSINFO_FILE  = 'systeminfo.txt'
COMPOSITE_XML = os.path.join(SPLUNK_HOME, 'var', 'run', 'splunk', 'composite.xml')

system_info = None # systeminfo.txt file obj; if you want to use reporting
                   # functions externally, set this as desired.

##################
# various cruft around naming the diag file.
#
def get_diag_date_str():
    return datetime.date.today().isoformat()

def get_splunkinstance_name():
    # octavio says the hostname is preferred to machine-user, and that
    # multiple-splunks-per-host is now rare, so just use hostname
    return socket.gethostname()

def get_results_dir():
    results_path = os.path.join(RESULTS_LOC, get_diag_name())
    if os.name == "nt":
        # SPL-34231, horrid (microsoft) hack to support longer filenames --
        # normally windows rejects 260+ char paths, unless unicode and
        # containing this magic cookie
        return u"\\\\?\\" + results_path
    else:
        return results_path

#def get_diag_name():  # impelmented via generator
def _diag_name_generator():
    # hack to create 'static' value
    date_str = get_diag_date_str()
    servername = get_splunkinstance_name()
    diag_name = "diag-%s-%s" % (servername, date_str)
    logger.info('Selected diag name of: ' + diag_name)
    while True:
        yield diag_name
get_diag_name = _diag_name_generator().next

def get_tar_name():
     return(os.path.join(SPLUNK_HOME, get_diag_name() + ".tar.gz"))

##################
# main utility functions to gather data
#
def sytemUsername():
    if os.name == 'posix':
        import pwd
        # get the name for the UID for the current process
        username = pwd.getpwuid(os.getuid())[0]
    elif os.name == 'nt':
        # thanks internets -- http://timgolden.me.uk/python/win32_how_do_i/get-the-owner-of-a-file.html`
        import win32api
        import win32con
        username = win32api.GetUserNameEx(win32con.NameSamCompatible)
    else:
        username = 'unknown for platform:' + os.name
    system_info.write('diag launched by: ' + username)

def splunkVersion():
    """ Use splunk version to figure the version"""

    system_info.write('\n\n********** Splunk Version **********\n\n')
    sver = os.popen('splunk version')
    system_info.write(sver.read())


# this uses python's uname function to get info in a cross-platform way.
def systemUname():
    """ Python uname output """ 

    system_info.write('\n\n********** Uname **********\n\n')
    suname = platform.uname()
    system_info.write(str(suname))
    #SPL-18413
    system_info.write("\n")
    system_info.write('\n\n********** splunkd binary format **********\n\n')
    splunkdpath = os.path.join(SPLUNK_HOME, 'bin', 'splunkd') 
    if suname[0] == 'Windows':
        splunkdpath += ".exe"
    arch = str(platform.architecture(splunkdpath)) 
    system_info.write(arch)
    if suname[0] == 'Linux':
        system_info.write('\n\n********** Linux distribution info **********\n\n')
        system_info.write(os.popen('lsb_release -a').read())


def networkConfig():
    """ Network configuration  """ 

    system_info.write('\n\n********** Network Config  **********\n\n')
    # we call different utilities for windows and "unix".
    if os.name == "posix":
        # if running as a non-root user, you may not have ifconfig in your path.
        # we'll attempt to guess where it is, and if we can't find it, just
        # assume that it is somewhere in your path.
        ifconfigCmd = '/sbin/ifconfig'
        if not os.path.exists(ifconfigCmd):
            ifconfigCmd = 'ifconfig'
        system_info.write(os.popen('%s -a' % ifconfigCmd).read())
    else:
        system_info.write(os.popen('ipconfig /all').read())


def networkStat():
    """ Network Status """ 

    system_info.write('\n\n********** Network Status **********\n\n')
    # just like with ifconfig, we attempt to guess the path of netstat.
    # if we can't find it, we leave it up to it being in your path.
    # also, if we're on windows, just go with the path-less command.
    netstatCmd = '/bin/netstat'
    if not os.name == "posix" or not os.path.exists(netstatCmd):
        netstatCmd = 'netstat'
    system_info.write(os.popen("%s -a -n" % netstatCmd).read())


def systemResources():
    """ System Memory """ 

    # on windows, we use msinfo to get all the relevant output.
    if os.name == "posix":
        system_info.write('\n\n********** System Ulimit **********\n\n')
        system_info.write(os.popen('ulimit -a').read())
        system_info.write('\n\n********** System Memory **********\n\n')
        suname = platform.uname()
        #SPL-17593
        if suname[0] == 'SunOS':
            system_info.write(os.popen('vmstat').read())
        elif suname[0] == 'Darwin':
            system_info.write(os.popen('vm_stat').read())
        elif suname[0] == 'Linux':
            system_info.write(os.popen('free').read())
        else:
            # try vmstat for hpux, aix, etc
            system_info.write(os.popen('vmstat').read())
        system_info.write('\n\n********** DF output **********\n\n')
        system_info.write(os.popen('df').read())
        system_info.write('\n\n********** mount output **********\n\n')
        system_info.write(os.popen('mount').read())
        system_info.write('\n\n********** cpu info **********\n\n')
        if suname[0] == 'SunOS':
            system_info.write(os.popen('/usr/sbin/psrinfo -v').read())
        elif suname[0] == 'Darwin':
            system_info.write(os.popen('system_profiler SPHardwareDataType').read())
        elif suname[0] == 'Linux':
            if os.path.exists('/proc/cpuinfo'):
                system_info.write(open('/proc/cpuinfo').read())
            else:
                system_info.write("/proc/cpuinfo unavailable. no /proc mounted?\n")
        elif suname[0] == 'AIX':
            aix_horror = """ for processor in `lsdev -c processor | awk '{ print $1; }'` ; do 
                                echo $processor;  
                                lsattr -E -l $processor; 
                             done """
            system_info.write(os.popen(aix_horror).read())
        elif suname[0] == 'FreeBSD':
            system_info.write(os.popen("sysctl -a | egrep -i 'hw.machine|hw.model|hw.ncpu'").read())
            system_info.write(os.popen("isainfo -v").read())
        elif suname[0] == 'HP-UX':
            hpux_horror = "echo 'selclass qualifier cpu;info;wait;infolog' | cstm"
            system_info.write(os.popen(hpux_horror).read())
        else:
            system_info.write("access to cpu data not known for platform.")
    else:
        os.popen('start /wait msinfo32.exe /report "%s" /categories +SystemSummary' % MSINFO_FILE)
        try:
            shutil.move(MSINFO_FILE, get_results_dir())
        except IOError, e: 
            # user probably clicked cancel on the msinfo gui
            err_msg = "Couldn't copy windows system info file to diag. %s"
            logger.warn(err_msg % (e.strerror,))

def splunkDBPaths():
    # first get all the index path config strings
    index_paths = []

    index_confs  = splunk.clilib.cli_common.getMergedConf('indexes')

    volumes = {}
    pathKeys = ['homePath', 'coldPath', 'thawedPath']
    for stanza_name in index_confs.keys():
        if stanza_name == 'default':
            continue
        stanza = index_confs[stanza_name]
        # ignore disabled index stanzas
        if stanza.get('disabled') == "true":
            continue
        if stanza_name.startswith('volume:'):
            if not stanza.has_key('path'):
                logger.warn("The indexing volume %s does not have a path defined, this is an error." % (stanza_name))
                continue
            volumes[stanza_name] = stanza['path']
        else:
            # it's an index definition, get the paths
            for pathKey in pathKeys:
                if not stanza.has_key(pathKey):
                    logger.warn("The index %s does not have a value set for %s, this is unusual." % (stanza_name, pathKey))
                else:
                    index_paths.append(stanza.get(pathKey))

    def normalize_path(btool_path):
        # SPL-25568 first try to hack around use of '/' on Windows.. soo ugly
        # 'safe' because / is not legal path character on win32.  
        # bundle layer should normalize this to universal separator or
        # always platform specific.
        if os.name == "nt":
            if (not '\\' in btool_path) and ('/' in btool_path):
                btool_path = btool_path.replace('/','\\')

        # if it's absolute, there's no processing to do here.
        # (also fixes SPL-26129)
        if os.path.isabs(btool_path):
            return btool_path

        # btool normally gives us $VARS in the path (how it is in file)
        # This is clumsy but workable on unix, but nonfunctional on windows
        # so substitute the full path in there manually.  
        # Start by splitting the very first section of the path, and checking to
        # see if it's a unix variable.
        try:
            pathBegin, pathEnd = btool_path.split(os.path.sep, 1)
        except ValueError:
            err_mesg = "No directory separator found in index path: %s\n" % btool_path
            sys.stderr.write(err_mesg)
            if not system_info.closed:
                system_info.write(err_mesg)
            return None

        if pathBegin.startswith('$'):
            # if it does start with $, remove the $ and lookup the remainder in our env vars,
            # replacing the variable with the result.
            try:
                pathBegin = os.environ[pathBegin[1:]]
                # rebuild the path
                return os.path.join(pathBegin, pathEnd)
            except KeyError, key_e:
                # only log error, let dir or ls show the problem as well
                err_mesg = "No env var available from index path: %s\n" % key_e.args[0]
                sys.stderr.write(err_mesg)
                # hack fix for SPL-33455, instead should make errors file.
                if not system_info.closed:
                    system_info.write(err_mesg)

        return btool_path

    def expand_vol_path(orig_path, volumes=volumes):
        if not orig_path.startswith('volume:'):
            return orig_path
        tmp_path = orig_path
        if os.name == "nt" and  (not '\\' in tmp_path) and ('/' in tmp_path):
            tmp_path = orig_path.replace('/','\\')
        if not os.path.sep in tmp_path:
            logger.warn("Volume based path '%s' contains no directory seperator." % orig_path)
            return None
        volume_id, tail = tmp_path.split(os.path.sep, 1)
        if not volume_id in volumes:
            logger.warn("Volume based path '%s' refers to undefined volume '%s'." % (orig_path, volume_id))
            return None
        return os.path.join(volumes[volume_id], tail)

    # detect and expand volume paths


    paths = map(expand_vol_path, index_paths)
    paths = filter(None, paths) # remove chaff from expand_vol_path
    paths = map(normalize_path, paths)
    paths = filter(None, paths) # remove chaff from normalize_paths
    return paths

def splunkDBListing():
    """ Index Listing Output""" 
    dbPaths = splunkDBPaths()

    system_info.write('\n\n********** find **********\n\n')
    for dbPath in dbPaths:
        if os.name == "posix":
            # get a recursive listing of that path.
            # SPL-19438 -- show if dir is symlink
            system_info.write("\nls -ld \"%s\" \n" % dbPath)
            system_info.write(os.popen("ls -ld \"%s\"\n" % dbPath).read())
            system_info.write("\nls -lR \"%s\"\n" % dbPath)
            system_info.write(os.popen("ls -lR \"%s\"" % dbPath).read())
        else:
            system_info.write(os.popen("dir /s/a \"%s\"" % dbPath).read())

def get_recursive_dir_info(dir_path, dest_file):
    """ do a recursive listing of the given dir and write the results on the given file """
    try:
        if not os.path.exists(os.path.dirname(dest_file)):
            os.makedirs(os.path.dirname(dest_file)) 
        f = open(dest_file, 'w') 
    except IOError, e: 
        # report fail, but continue along
        err_msg = "Error creating dirlisting file for '%s' at '%s', %s, continuing..."
        logger.warn(err_msg % (dir_path, dest_file, e.strerror))
        return
    try:
       if os.name == "posix":
          f.write("ls -lR \"%s\"\n" % dir_path)
          f.write(os.popen("ls -lR \"%s\"" % dir_path).read())
       else:
          f.write("dir /s/a \"%s\"\n" % dir_path)
          f.write(os.popen("dir /s/a \"%s\"" % dir_path).read())
    finally: 
       f.close()

def get_bucketmanifest_filenames(path):
    """return relative paths for all .bucketManifest files under a given path (hopefully a SPLUNK_DB or index)
     reqested by vishal, SPL-31499"""
    filenames = []

    for dir, subdirs, files in os.walk(path):
        data_files = filter(lambda s: s == (".bucketManifest"), files)
        relative_dir = dir[len(path)+1:]
        # prepend the directory
        paths = map( lambda fname: os.path.join(relative_dir, fname), data_files)
        filenames.extend(paths)
    return filenames

def get_worddata_filenames(path):
    "return relative paths for Host/Source/SourceTypes .data files under a given path (hopefully a SPLUNK_DB or index)"
    filenames = []
    wanted_filenames = ("Hosts.data", "Sources.data", "SourceTypes.data")

    for dir, subdirs, files in os.walk(path):
        data_files = filter(lambda f: f in wanted_filenames, files)
        relative_dir = dir[len(path)+1:]
        # prepend the directory
        paths = map( lambda fname: os.path.join(relative_dir, fname), data_files)
        filenames.extend(paths)
    return filenames

###################
# internal utility

def copytree_special(src, dst, symlinks=False, ignore=None):
    """shutil.copytree safe for named pipes and other shenanigans"""
    if not sys.version_info >= (2,6):
        # 3.x, dont implement ignore, rare case anyway
        return shutil.copytree(src, dst, symlinks)
    # 4.x version
    special_files = []
    def special_file_filter(directory, dir_contents, prior_ignore=ignore,
                            special_files=special_files):
        # first we give any prior ignorer a chance to prune the listing
        logger.debug('dir_contents at start of copytree:' + str(dir_contents))
        if prior_ignore:
            ignore_files = prior_ignore(directory, dir_contents)
            logger.debug('prior ignore eliminated:' + str(ignore_files))
        else:
            ignore_files = []
        for filename in dir_contents:
            if filename in ignore_files:
                continue
            logger.debug('checking file: ' + filename)
            f_path = os.path.join(directory, filename)
            f_stat = os.lstat(f_path)
            if stat.S_ISLNK(f_stat.st_mode):
                continue # KEEP.  symlink flag on copytree dictates policy
            elif stat.S_ISSOCK(f_stat.st_mode):
                special_files.append(('socket', f_path))
            elif stat.S_ISCHR(f_stat.st_mode) or stat.S_ISBLK(f_stat.st_mode):
                special_files.append(('device', f_path))
            elif stat.S_ISFIFO(f_stat.st_mode):
                special_files.append(('fifo', f_path))
            elif stat.S_ISDIR(f_stat.st_mode) or stat.S_ISREG(f_stat.st_mode):
                continue # KEEP
            else:
                raise AssertionError("%s stat was not any known kind of file" % f_path)
            # reached end, so goes on ignore list
            ignore_files.append(filename)
        logger.debug('returning:' + str(ignore_files))
        return ignore_files
    def make_fake_special_file(f_type, f_path, src_dir, dst_dir):
        tgt_path = os.path.join(dst_dir, f_path + '.' + f_type)
        tgt_dir = os.path.dirname(tgt_path)
        if not os.path.exists(tgt_dir):
            os.makedirs(tgt_dir)
        open(tgt_path, 'w') # make zero byte file, immediately closed
        shutil.copystat(os.path.join(src_dir, f_path), tgt_path)

    copy_errors = None
    try:
        shutil.copytree(src, dst, symlinks, special_file_filter)
    except shutil.Error, copy_errors:
        pass
    for f_type, f_path in special_files:
        try:
            make_fake_special_file(f_type, f_path, src, dst)
        except IOError, e:
            err_msg = "Error creating fake special file for '%s', %s, continuing..."
            logger.warn(err_msg % (f_path, e.strerror))
    if copy_errors:
        raise copy_errors # pylint: disable-msg=E0702
        # apparently pylint cannot understand 'if'
    return

def copy_file_with_parent(src_file, dest_file):
    "copy a file while creating any parent directories for the target"
    # ensure directories exist
    dest_dir = os.path.dirname(dest_file)
    try:
        if not os.path.isdir(dest_dir):
            os.makedirs(os.path.dirname(dest_file))
        shutil.copy(src_file, dest_file)
    except IOError, e:
        # windows sucks
        err_msg = "Error duping file for '%s' to '%s', %s, continuing..."
        logger.warn(err_msg % (src_file, dest_file, e.strerror))

def copy_indexfiles(target_path):
    "copy index files such as .data and .bucketmanifest in all indices to the results dir"

    indexdata_filenames = []

    for dbPath in splunkDBPaths():
        rel_paths = get_worddata_filenames(dbPath)
        rel_paths.extend(get_bucketmanifest_filenames(dbPath))
        # make absolute paths
        abs_paths = map(lambda f: os.path.join(dbPath, f), rel_paths)
        indexdata_filenames.extend(abs_paths)

    for src_file in indexdata_filenames:
        dest_file = src_file

        # trim SPLUNK_HOME from target directory
        if dest_file.startswith(SPLUNK_HOME):
                dest_file = dest_file[len(SPLUNK_HOME):]

        # windows tomfoolery....
        # make an informative path to store the indexdata file
        if os.name != 'posix':
            unc, dest_file = os.path.splitunc(dest_file)
            # yank initial backslashes off UNC expression for constructing
            # relative path later
            unc_dirnames = unc.lstrip('\\')
        else:
            unc_dirnames = ''
        # capture drive letter if exists
        drive , dest_file = os.path.splitdrive(dest_file)
        if drive:
            # letter only, since windows is unahppy with colons in dirnames
            drive = drive[0]

        # make relative, for join to work
        if dest_file[0] == os.sep:
            dest_file = dest_file[1:]

        dest_file = os.path.join(target_path, unc_dirnames, drive, dest_file)

        copy_file_with_parent(src_file, dest_file)

def copy_dispatch_dir(target_path):
     dispatch_dir = os.path.join(SPLUNK_HOME, "var", "run", "splunk", "dispatch") 
    
     # collect info iff dispatch dir is present
     if not os.path.exists(dispatch_dir):    
         return
 
     get_recursive_dir_info(dispatch_dir,  os.path.join(target_path, "dir_info.txt")) 
     for job in os.listdir(dispatch_dir):
         try:
             job_dir = os.path.join(dispatch_dir, job)
             if not os.path.isdir(job_dir):
                continue
             
             get_recursive_dir_info(job_dir, os.path.join(target_path, job, "dir_info.txt"))
             # copy only files in the job's dir, skip all the .gz stuff
             for f in os.listdir(job_dir):
                src_file = os.path.join(job_dir, f)
                # don't block forever reading from named pipes and so on.
                if not os.path.isfile(src_file):
                    continue
                # don't capture the customer's data, or zero byte status indicator files.
                if f.endswith('.gz') or f.startswith('results') or f.endswith(".token"):
                   continue
                       
                copy_file_with_parent(src_file, os.path.join(target_path, job, f)) 
         except OSError, e: 
             # report fail, but continue along
             err_msg = "Error capturing data for dispatch job dir '%s', %s, continuing..."
             logger.warn(err_msg % (job_dir, e.strerror))
             pass

def copy_pooled_data():
    # maybe get 'pooled' or searchhead pooling config data
    pooled_path = None
    try:
        pooled_path = splunk.clilib.bundle_paths.get_shared_storage()
    except AttributeError:
        pass

    if not pooled_path:
        return

    logger.info("Copying Splunk search-head pooled data...")

    try:
        # use copytree(..., ignore=func ) pattern here, but not elsewhere since
        # pooling feature only exists with newer splunk + newer python
        shutil.copytree(os.path.join(pooled_path), os.path.join(get_results_dir(), "search_pool"), symlinks = True, ignore=shutil.ignore_patterns('results.csv*', '*.token', 'events', 'buckets'))
    except shutil.Error, copy_errors:
        # If files get moved or whatever during copy (like someone is moving
        # manager) this can happen
        msg = "Some problems were encountered copying the pooled config dir (likely thesese are not a problem):\n" + str(copy_errors)
        logger.warn(msg)

def copy_win_checkpoints():
    "get the checkpoint files used by windows inputs"
    p_storage = os.path.join("var", "lib", "splunk", "persistentStorage")
    checkpoint_files = ["wmi_checkpoint", "regmon-checkpoint"]
    checkpoint_dirs = ["WinEventLog", "ADmon"]

    for c_file in checkpoint_files:
        c_path = os.path.join(p_storage, c_file)
        src = os.path.join(SPLUNK_HOME, c_path)
        tgt = os.path.join(get_results_dir(), c_path)
        if os.path.exists(src):
            copy_file_with_parent(src, tgt)

    for c_dir in checkpoint_dirs:
        c_dir_path = os.path.join(p_storage, c_dir)
        src = os.path.join(SPLUNK_HOME, c_dir_path)
        tgt = os.path.join(get_results_dir(), c_dir_path)
        if os.path.exists(src):
            copytree_special(src, tgt)

def deleteTree(path):
    """ Delete a directory, if it exists. """
    def handle_readonly_filedir_errors(func_called, path, exc):
        "try to make this more like rm -rf"
        error_type = exc[1]
        full_perms = stat.S_IRWXU | stat.S_IRWXG | stat.S_IRWXO

        parent = os.path.dirname(path)
        os.chmod(parent, full_perms)
        os.chmod(path, full_perms)
        if func_called == os.rmdir:
            os.rmdir(path)
        elif func_called == os.remove:
            os.remove(path)
        else:
            raise
    if os.path.isdir(get_results_dir()):
        shutil.rmtree(get_results_dir(), onerror=handle_readonly_filedir_errors)

def clean_temp_files():
    if os.path.isdir(RESULTS_LOC):
        for f in os.listdir(RESULTS_LOC):
            specific_tempdir = os.path.join(RESULTS_LOC, f)
            if os.path.isdir(specific_tempdir):
                deleteTree(specific_tempdir)

def local_getopt():
    # handle arguments
    parser = optparse.OptionParser()
    parser.add_option("--exclude", action="append", dest="exclude_list", 
                      help="glob-style file pattern to exclude (repeatable)", metavar="pattern")
    parser.add_option("--all-dumps", type="string", dest="all_dumps", 
                      help="get every crash .dmp file, opposed to default of a more useful subset", metavar="bool")
    options, args =  parser.parse_args()
    if not options.exclude_list:
        options.exclude_list = []
    return options, args

def read_config(options):
    server_conf = splunk.clilib.cli_common.getMergedConf('server')
    diag_stanza = server_conf.get('diag', {})
    exclude_terms = [v for (k, v) in diag_stanza.items() if k.startswith("EXCLUDE")]
    options.exclude_list.extend(exclude_terms)
    return options

def build_filename_filters(globs):
    if not globs:
        return []
    glob_to_re = lambda s: re.compile(fnmatch.translate(s))
    return map(glob_to_re, globs)

def filter_dumps(dump_filenames):
    class Dump:
        def __init__(self, filename):
            self.filename=filename
            self.stat = os.stat(filename)
    dumps = map(Dump, dump_filenames)
    total_dumpspace = sum(map(lambda d: d.stat.st_size, dumps))
    dump_megs = total_dumpspace / 1024 / 1024 
    if dump_megs >= 2000:
        logger.warn("""Note, you have %sMB in memory dump files in your $SPLUNK_HOME/var/log/splunk directory.  
You may want to prune, clean out, or move this data.""" % dump_megs)
    now = datetime.datetime.now()
    def age_filter(dump):
        age = now - datetime.datetime.fromtimestamp(dump.stat.st_mtime)
        if age.days >= 30:
            logger.error("Not including crashdump %s, over 30 days old" % dump.filename)
            return False
        return True
    dumps = filter(age_filter, dumps)
    # sort dumps by time, newest first.
    dumps.sort(cmp=lambda x, y: cmp(x.stat.st_mtime, y.stat.st_mtime))
    dumps.reverse()
    # try to get the useful crashes, starting with the two most recent.
    useful_crashes = dumps[:2] 
    # now get a third, going backwards if they are consective, to try to find
    # the start of whatever the recent problem is
    i = 2
    while i < len(dumps):
        cur_crash = dumps[i]
        if i+1 == len(dumps):
            # if we get to the end, use that
            useful_crashes.append(cur_crash)
            break
        cur_crashtime = datetime.datetime.fromtimestamp(cur_crash.stat.st_mtime)
        older_crashtime = datetime.datetime.fromtimestamp(dumps[i+1].stat.st_mtime)
        tdelta = cur_crashtime - older_crashtime
        # 2 days we assume is large enough to indicate this is the start
        if tdelta.days >= 2:
            useful_crashes.append(cur_crash)
            break
        i+=1
    return map(lambda x: x.filename, useful_crashes)

def create_diag(options):
    try:
        logger.info("Ensuring clean temp dir...")
        # delete any old results tarball, and create a fresh results directory
        clean_temp_files()
    except Exception, e:
        #logger.error(e)
        logger.error("""Couldn't initally clean diag temp directory.  
To work around move %s aside, or delete it.
A listing of the entire contents of this directory will help us identify the problem.""" % get_results_dir())
        raise

    os.makedirs(get_results_dir())
    if os.path.exists(get_tar_name()):
        os.unlink(get_tar_name())
    
    # make sure it's a supported os.
    if not os.name in ("posix", "nt"):
        logger.error("Unsupported OS (%s)." % os.name)
        sys.exit(1)

    logger.info("Starting splunk diag...")

    try:
        try:
            global system_info
            system_info = open(os.path.join(get_results_dir(), SYSINFO_FILE), 'w')
        except IOError:
            logger.error("Exiting: Cannot create system info file.  Permissions may be wrong.")
            sys.exit(1)

        # who is running me?
        sytemUsername()

        #get the splunk version
        splunkVersion()
        
        #uname
        systemUname()
        
        #ifconfig
        networkConfig()
        
        #netstat
        networkStat()
        
        #ulimit
        systemResources()

        #ls
        splunkDBListing()
    finally:
        system_info.close()
    
    #copy etc and log into results too
    logger.info("Copying Splunk configuration files...")
    try:
        copytree_special(os.path.join(SPLUNK_HOME, "etc"), os.path.join(get_results_dir(), "etc"), symlinks = True)
    except shutil.Error, copy_errors:
        # If files get moved or whatever during copy (like someone is moving
        # manager) this can happen
        msg = "Some problems were encountered copying etc/... (likely thesese are not a problem):\n" + str(copy_errors)
        logger.warn(msg)
        #system_info.write(msg + "\n")

    logger.info("Copying Splunk log files...")
    try:
        logdir = os.path.join(SPLUNK_HOME, "var", "log", "splunk")
        targetdir = os.path.join(get_results_dir(), "log")
        # current splunk uses python 2.6, so use this to skip dump files.
        if sys.version_info >= (2,6) and not options.all_dumps:
            skip_dumps = shutil.ignore_patterns("*.dmp")
            copytree_special(logdir, targetdir, symlinks = True, ignore = skip_dumps)
            # now get a list of the dmp files, if any exist
            dumps = glob.glob(os.path.join(logdir, "*.dmp"))
            useful_dumps = filter_dumps(dumps)
            for dump in useful_dumps:
                shutil.copy(dump, targetdir)
        else:
            copytree_special(logdir, targetdir, symlinks = True)
    except shutil.Error, copy_errors:
        # log files might be rotated, I suppose?
        msg = "Some problems were encountered copying log files:\n" + str(copy_errors)
        logger.warn(msg)
        #system_info.write(msg + "\n")

    copy_pooled_data()

    # get worddata "metadata" files and ".bucketManifest" files
    logger.info("Copying index worddata, and bucket info files...")
    copy_indexfiles(get_results_dir())

    if not os.path.exists(COMPOSITE_XML):
        logger.warn("Unable to find composite.xml file, product has likely not been started.")
    else:
        try:
            shutil.copy(COMPOSITE_XML, get_results_dir())
        except IOError, e:
            # windows sucks
            err_msg = "Error copying in composite.xml: '%s' continuing..."
            logger.warn(err_msg % e.strerror)

    logger.info("Copying Splunk log files in the dispatch dir...")
    copy_dispatch_dir(os.path.join(get_results_dir(), "dispatch"))

    if os.name == "nt":
        logger.info("Copying windows input checkpoint files...")
        copy_win_checkpoints()

    filter_list = build_filename_filters(options.exclude_list)
    def tar_exclude(filename):
        for regex in filter_list:
            if regex.match(filename):
                return True # don't tar
        return False

    #create a tar.gz file out of the results dir
    logger.info("Creating archive file...")
    destFile = tarfile.open(get_tar_name(), 'w:gz')
    destFile.add(get_results_dir(), os.path.basename(get_results_dir()), exclude=tar_exclude)
    destFile.close()

def main():
    # handle options
    cmdline_options, args = local_getopt()
    options = read_config(cmdline_options)

    # Do the work, trying to ensure the temp dir gets cleaned even on failure
    try:
        create_diag(options)
    except Exception, e:
        logger.error("Exception occurred while generating diag, we are deeply sorry.")
        logger.error(traceback.format_exc())
        logger.info("We will now try to clean out the temp directory...")
        clean_temp_files()
        return

    # and for normal conclusion..
    try:
        #clean up the results dir
        logger.info("Cleaning up...")
        clean_temp_files()
    finally:
        logger.info("Splunk diagnosis file created: %s" % get_tar_name())


def pclMain(args, fromCLI):
    return main()

if __name__ == "__main__":
    # get logging behavior similar to normal splunk invocation
    # logging apis suck, renaming them sucks more
    sh = logger.StreamHandler()
    sh.setFormatter(logger.Formatter("%(message)s"))
    l = logger.getLogger()
    l.setLevel(logger.INFO)
    l.addHandler(sh)
    main()
