#!/usr/bin/perl

# Code to parse a file of snort alerts and produce HTML output intended for
# diagnostic inspection and tracking down problems.  The model is that one is
# using a cron job or similar to produce a daily/hourly/whatever file of snort
# alerts.  This script can be run on each such file to produce a convenient
# HTML breakout of all the alerts.

# Currently it does not handle portscan preprocessor alerts - it ignores them.

# Usage:

# snortsnarf.pl <alertfile>

# if alertfile is absent, /var/log/snort.alert is assumed

# The script will produce a directory snfout.alertfile full of a very
# large number of html files.  This is placed under the current directory.  
# The program will not run if the directory already exists.  Start at 
# index.html with browser.  

# It is believed to work with Perl 5 on the following OS's
# OpenBSD 2.6
# RedHat Linux 6.1
# Windows NT 4.0 (NTFS only) (tweak $dirsep and $root below).

# It probably works with most other Unixen/Linuxen, and is certain not to 
# work on the FAT filesystem.

# Known to work with alert files generated with this snort command line:
# snort -D -c oursnort.lib -o -d 

# HTML looks marginally ok with Netscape 4.7.  Other browsers not tested.

# Tested on alert files up to about 500KB.  
# HTML takes up 5 times the space of the original alert file.  
# .tar.gz of the dir is about half the size of the alert file.
# YMMV!

##############################################################################

# Author: Stuart Staniford-Chen, Silicon Defense (stuart@silicondefense.com)
# This code is in the public domain.  Please send complaints, kudos, and 
# especially improvements and bugfixes to me.  This is a one day hack and 
# may only be worth what you paid for it.

# Alert parsing code borrowed from Joey McAlerney, Silicon Defense.

$script = "<a href=\"http://www.silicondefense.com/snortsnarf/\">".
			"Snortsnarf</a>";
$version = "v031000.1";
$author_email = "stuart\@SiliconDefense.com";
$author = "<a href=\"mailto:$author_email\">Stuart Staniford-Chen</a>";

##############################################################################

# portability stuff - toggle for Unix/Windows.

$#dirsep = "\\"; 	# Windows
$dirsep = "\/"; 	# Unix

#$root = "d:\\";	# Windows
#$root = "e:\\";		# Windows
$root = "\/";		# Unix

$html = 'html'; 	# usually html or htm

# Various global variables

$alert_file = $root."var".$dirsep."log".$dirsep."snort.alert";
$sig_page = "index.html";

# temp development hack
# $alert_file = "snort.alert.030800";

##############################################################################

# data structure comments

# alert structure

# $alert = {src => $src,
#		    dest => $dest,
#			sig => $sig,
#			date => $date,
#			time => $time,
#			text => $text}
			
# where $src and $dest are the IPs in strings in dotted decimal format
# $sig is the signature string
# $date is the date as an ASCII string
# $time is the time of day as an ASCII string
# $text is the whole alert in ASCII (used for printing out).

# sig_count is a ref to a hash with signatures as keys and counts as values.

##############################################################################

# Main program

&initialize();

while($alert = &get_alert())
{
  &process_alert($alert)
}

&construct_sig_indexes();
&output_main_sig_page();
&output_per_sig();
&output_per_source();
&output_per_src_dest();

##############################################################################

sub initialize
{
  $alert_file = $ARGV[0] if $ARGV[0];
  my $last = rindex($alert_file,$dirsep);
  my $substr = substr($alert_file,$last+1);
  $output_dir = "snfout.$substr";
  open(ALERT,$alert_file) || die("Couldn't open alert input file $alert_file\n");
  die("$output_dir already exists\n") if -e $output_dir;
  mkdir($output_dir,0);
  chdir($output_dir);
}

##############################################################################

sub get_alert
{
  my($length,$text,$signature,$line1,$line2,$line3,$line4,$line5);
  my($sig_url);
  
    while(<ALERT>) {
        chomp;
        $length = length;

        next if $length == 0;     # who needs a blank line? Not me.

        # ---- Process the first line -----
        #
        # the first line just holds the attack id
        next unless /\[\*\*\]/;                
		$text = $_."\n";
        $signature = substr($_,5,($length-10));

        # Some snort pre-processors  put stuff in the alert file. 
        # As a quick fix, lets just look for a message put out by the
        # spp_portscan pre-processor.
        #
        if($signature =~ m/spp_portscan/g) {
            <ALERT>;                         # line that follows
            next;                          # ok, lets try this again...
        }

        # ---- Process the second line -----        
        #
        $line2 = <ALERT>;
        $text .= $line2;
		chomp($line2);
		
        my $date = substr($line2,0,5);
        my $time = substr($line2,6,15);
        my $remainder =  substr($line2,22,(length $line2));  # grab the rest
                                                          # for regex matching
        $remainder =~ s/ \-\> /-/; 
        my($source,$destination) = split('-',$remainder);
        my($src,$srcport) = split(':',$source);
        my($dst,$dstport) = split(':',$destination);

        # ---- Process the third line -----
        #
        $line3 = <ALERT>;
        $text .= $line3;
		chomp($line3);
		
        my($df,$proto,$ttl,$tos,$id) = undef;
        ($proto,$ttl,$tos,$id,$df) = $line3 =~
/^(\w*)\sTTL\:(\d*)\sTOS\:(\w*)\sID\:(\d*)\s?\s?(DF)?$/;

        # ---- Process the fourth line -----
        #
        $line4 = <ALERT>;
		$text .= $line4;
		chomp($line4);
		
        if($proto eq "TCP") {
            my($flags,$seq,$ack,$win) = $line4 =~
/^([SFRUAP12*]*)\sSeq\:\s(\w*)\s*Ack\:\s(\w*)\s*Win\:\s(\w*)$/;
        }
        elsif($proto eq "UDP") {
            my($UDPlength) = $line4 =~ /^Len\:\s(\d*)$/;
        }
        elsif($proto eq "ICMP") {
            my($ICMPid,$seq,$type) = $line4 =~ /^ID\:(\d*)\s*Seq\:(\d*)\s*(\w*)/;
        }

        # ---- Process the fifth line if there is one -----
        #
        $line5 = <ALERT>;
		chomp($line5);
		
		$text .= $line5."\n" if length $line5 > 0;

		
        my $TCPoptions = ""; 
        if(length $line5 > 0) {
            $TCPoptions = substr($line5,16,(length $line5));
        }
	   my $alert = {
	   				'src'  => $src,
					'dest' => $dst,
					'date' => $date,
					'time' => $time,
					'sig'  => $signature,
					'text' => $text."\n",
				   };
	  return $alert;  
    }
}

##############################################################################

sub process_alert
{
  my($alert) = @_;
  
  $sig_count->{$alert->{'sig'}}++;
  $sig_src_count->{$alert->{'sig'}}{$alert->{'src'}}++;
  $src_count->{$alert->{'src'}}++;
  push @{$src_list->{$alert->{'src'}}}, $alert;
  $earliest_src->{$alert->{'src'}}
  	= &earliest($alert, $earliest_src->{$alert->{'src'}});
  $latest_src->{$alert->{'src'}}
  	= &latest($alert, $latest_src->{$alert->{'src'}});
	
  $sig_src_dest_count->{$alert->{'sig'}}{$alert->{'src'}}{$alert->{'dest'}}++;
  push @{$sig_src_dest_list->{$alert->{'sig'}}{$alert->{'src'}}
  											{$alert->{'dest'}}}, $alert;
  $earliest_sig_src_dest->{$alert->{'sig'}}{$alert->{'src'}}{$alert->{'dest'}}
  	= &earliest($alert, $earliest_sig_src_dest->{$alert->{'sig'}}
										{$alert->{'src'}}{$alert->{'dest'}});
  $latest_sig_src_dest->{$alert->{'sig'}}{$alert->{'src'}}{$alert->{'dest'}}
  	= &latest($alert, $latest_sig_src_dest->{$alert->{'sig'}}
										{$alert->{'src'}}{$alert->{'dest'}});
  $src_dest_count->{$alert->{'src'}}{$alert->{'dest'}}++;  
  push @{$src_dest_sig_list->{$alert->{'src'}}{$alert->{'dest'}}
  											{$alert->{'sig'}}}, $alert;
  $earliest_src_dest->{$alert->{'src'}}{$alert->{'dest'}}
  	= &earliest($alert, $earliest_src_dest->{$alert->{'src'}}{$alert->{'dest'}});
  $latest_src_dest->{$alert->{'src'}}{$alert->{'dest'}}
  	= &latest($alert, $latest_src_dest->{$alert->{'src'}}{$alert->{'dest'}});
}

##############################################################################

sub construct_sig_indexes
{
  my($sig,$count);
  
  foreach $sig (keys %$sig_count)
   {
    $sig_index->{$sig} = $count++;
	if($sig =~ /^IDS(\d+)/)
	 {
	  # ARACHNIDS signature - can make a nice URL for these.
	  $sig_url->{$sig} = "http:\/\/whitehats.com\/IDS\/$1";
	  $sig_entry->{$sig} = "<a href=\"$sig_url->{$sig}\">$sig</a>";
	 }
	else
	 { 
	  $sig_entry->{$sig} = $sig;
	 }
   }
}

##############################################################################

sub output_main_sig_page
{
  my($sig);
  my $page_title = "Snortsnarf: Snort signatures in $alert_file";
  
  open(PAGE,">$sig_page") || die("Couldn't open sig page $sig_page\n");
  select(PAGE);
  &print_page_head($page_title);
  print "<TABLE BORDER CELLPADDING = 5>\n";
  print "<TR><TD>Signature</TD><TD>Count</TD><TD>Link to summary</TD></TR>\n";

  foreach $sig (sort sort_by_sig_count keys %$sig_count)
   {

    print "<TR><TD>$sig_entry->{$sig}</TD><TD>$sig_count->{$sig}</TD>".
			"<TD><a href=\"sig$sig_index->{$sig}.$html\">Summary</a></TD></TR>\n";
   }
  print "</TABLE>\n\n";
  &print_page_foot();
  close(PAGE);
}

sub sort_by_sig_count { $sig_count->{$a} <=> $sig_count->{$b};}

##############################################################################

sub output_per_source
{
   my($src,$src_file,$alert,$early,$late);
   my $page_title;
   
   foreach $src (keys %$src_count)
    {  
     $src_file = "src$src.$html";
     $page_title = "All $src_count->{$src} alerts from $src in $alert_file";
     open(PAGE,">$src_file") || die("Couldn't open source output file $src_file\n");
     select(PAGE);
     &print_page_head($page_title);
	 $early = $earliest_src->{$src};
	 $late = $latest_src->{$src};
	 print "<p>Earliest: ".&pretty_time($early->{'time'},$early->{'date'})
	   						."</p>\n";
	 print "<p>Latest: ".&pretty_time($late->{'time'},$late->{'date'})
	   						."</p>\n";
	 
	 foreach $alert (@{$src_list->{$src}})
	  {
	   print "$sig_entry->{$alert->{'sig'}} ".
	   		"(<a href=\"sig$sig_index->{$alert->{'sig'}}.$html\">other alerts on this signature</a>)\n";
	   print "<pre>\n";
	   print $alert->{'text'};
	   print "</pre>\n";
	  }	 
     &print_page_foot();
     close(PAGE);
    }
}

##############################################################################

sub output_per_sig
{
  my($sig,$sig_file,$src,$dest,$early,$late);
  my $page_title;
  
  foreach $sig (keys %$sig_count)
   {  
    $global_sig = $sig;  # ouch - need to communicate with sort_by_sig_src_count
    $sig_file = "sig$sig_index->{$sig}.$html";
    $page_title = "Summary of alerts in $alert_file for signature:";
    open(PAGE,">$sig_file") || die("Couldn't open sig output file $sig_file\n");
    select(PAGE);
    &print_page_head($page_title);
	print "<h3>$sig_entry->{$sig}</h3>";
	
  	foreach $src (sort sort_by_sig_src_count keys 	
									%{$sig_src_count->{$sig}})
	 {
	  $global_src = $src; # ouch ouch
	  print "<hr>\n";
	  print "<h3><a href=\"src$src.$html\">Source IP: $src</a> ".
	  	"($sig_src_count->{$sig}{$src} alerts of ".
			"$src_count->{$src} from this IP)</h3>\n";
	  print "<TABLE BORDER CELLPADDING = 5>\n";
      print "<TR><TD>Destination</TD><TD>Count</TD>".
	  					"<TD>Earliest</TD><TD>Latest</TD></TR>\n";

  	  foreach $dest (sort sort_by_sig_src_dest_count keys 	
									%{$sig_src_dest_count->{$sig}{$src}})
	   {
	    $early = $earliest_sig_src_dest->{$sig}{$src}{$dest};
		$late = $latest_sig_src_dest->{$sig}{$src}{$dest};
        print "<TR><TD><a href=\"sd-$src-$dest.$html\#$sig_index->{$sig}\">".
						"$dest</a></TD>".
						"<TD>$sig_src_dest_count->{$sig}{$src}{$dest}</TD><TD>".
	  					&pretty_time($early->{'time'},$early->{'date'}).
						"</TD><TD>".
						&pretty_time($late->{'time'},$late->{'date'})."</TD>";
	   }
	  print "</TABLE>\n";
	 }
    &print_page_foot();
    close(PAGE);
   }
}

sub sort_by_sig_src_count 
{ 
 $sig_src_count->{$global_sig}{$b} <=> 
 					$sig_src_count->{$global_sig}{$a};
}

sub sort_by_sig_src_dest_count 
{ 
 $sig_src_dest_count->{$global_sig}{$global_src}{$b} <=> 
 					$sig_src_count->{$global_sig}{$global_src}{$a};
}

##############################################################################

sub output_per_src_dest
{
   my($src,$dest,$src_dest_file,$alert,$sig);
   my($page_title,$early,$late);
   
   foreach $src (keys %$src_count)
    {
	 foreach $dest (keys %{$src_dest_sig_list->{$src}})
	  {
       $src_dest_file = "sd-$src-$dest.$html";
       $page_title = "Alerts from $src to $dest in $alert_file";
       open(PAGE,">$src_dest_file") || die("Couldn't open src_dest output file $src_dest_file\n");
       select(PAGE);
       &print_page_head($page_title);
	   print "<h3>Total of $src_dest_count->{$src}{$dest}".
	   								" alerts on this page.</h3>\n";
	   $early = $earliest_src_dest->{$src}{$dest};
	   $late = $latest_src_dest->{$src}{$dest};
	   print "<p>Earliest: ".&pretty_time($early->{'time'},$early->{'date'})
	   						."</p>\n";
	   print "<p>Latest: ".&pretty_time($late->{'time'},$late->{'date'})
	   						."</p>\n";
	   
	   foreach $sig (sort sort_by_sig_src_dest_count keys 				
	   										%{$src_dest_sig_list->{$src}{$dest}})
		{
		 print "<hr><h3><a name=\"$sig_index->{$sig}\">$sig_entry->{$sig}</a></h3>\n";
	     print("<pre>\n");
	     foreach $alert (@{$src_dest_sig_list->{$src}{$dest}{$sig}})
	      {
	       print $alert->{'text'};
	      }
	     print "</pre>\n";
		}
       &print_page_foot();
       close(PAGE);
	  }
    }
}

##############################################################################

sub print_page_head
{
  my($page_title) = @_;
  
  print "<html>\n<head>\n<title>$page_title</title>\n";
  print "</head>\n<body>\n";
  print "<h2>$page_title</h2>\n\n";
}

##############################################################################

sub print_page_foot
{
 print "<hr>\n";
 print "<i>Generated by $script $version ($author)</i><p>\n";
 print "See also the <a href=\"http://www.clark.net/~roesch/security.html\">".
 			"Snort Page</a> by Marty Roesch";
 print "</body></html>\n";
}

##############################################################################

sub earliest
{
 my($alert1,$alert2) = @_;
 
 return $alert1 unless defined $alert2;
 return $alert2 unless defined $alert1;
 
 my(@pieces1) = (split('/',$alert1->{'date'}),split(':',$alert1->{'time'}));
 my(@pieces2) = (split('/',$alert2->{'date'}),split(':',$alert2->{'time'}));
 
 foreach (0..$#pieces1)
  {
   return $alert1 if $pieces1[$_] < $pieces2[$_];
   return $alert2 if $pieces1[$_] > $pieces2[$_];
  }
 return $alert1;
}

##############################################################################

sub latest
{
 my($alert1,$alert2) = @_;
 
 return $alert1 unless defined $alert2;
 return $alert2 unless defined $alert1;
 
 my(@pieces1) = (split('/',$alert1->{'date'}),split(':',$alert1->{'time'}));
 my(@pieces2) = (split('/',$alert2->{'date'}),split(':',$alert2->{'time'}));
 
 foreach (0..$#pieces1)
  {
   return $alert1 if $pieces1[$_] > $pieces2[$_];
   return $alert2 if $pieces1[$_] < $pieces2[$_];  
  }
 return $alert2;
}

##############################################################################

sub pretty_time
{
  my($time,$date) = @_;
  my(@bits) = split(/\./,$time);
  return "<b>$bits[0]</b>".
  			".$bits[1]".
			" <i>on $date</i>";
}

##############################################################################


