package com.zend.ide.highlighter;

import java.io.*;
import java.util.*;
import com.zend.ide.util.*;
import javax.swing.text.Segment;

%%

%class PhpLexer
%public
%implements Scanner
%unicode
%int
%caseless

%state ST_HTML_IN_HTML
%state ST_HTML_TAG_ATTLIST
%state ST_HTML_IN_COMMENT
%state ST_HTML_ATTRVAL
%state ST_HTML_IN_TAG_HEADER
%state ST_HTML_TAG_NAME
%state ST_HTML_DOUBLE_QUOTES_ATTRVAL
%state ST_HTML_SINGLE_QUOTE_ATTRVAL
%state ST_HIGHLIGHTING_ERROR

%state ST_PHP_IN_SCRIPTING
%state ST_PHP_DOUBLE_QUOTES
%state ST_PHP_SINGLE_QUOTE
%state ST_PHP_BACKQUOTE
%state ST_PHP_HEREDOC
%state ST_PHP_LOOKING_FOR_PROPERTY
%state ST_PHP_COMMENT
%state ST_PHP_LINE_COMMENT

%state ST_JSCRIPT_IN_SCRIPTING
%state ST_JSCRIPT_COMMENT
%state ST_JSCRIPT_LINE_COMMENT
%state ST_JSCRIPT_DOUBLE_QUOTES
%state ST_JSCRIPT_SINGLE_QUOTE

%{
    /** TODO, Do not forget to:
     1. Change the prototype of yylex to:
     public int yylex() { ...
     (Do not throw the IOException)

     2. in JFLEX 3.0 change yy_advance to the following code:
     private int yy_advance() {
     if (yy_currentPos < yy_endRead) return yy_buffer[yy_currentPos++];
     return YYEOF;
     }

     3. in JFLEX 3.5 change yy_refill to the following code:
     private boolean yy_refill() {
     return yy_currentPos >= yy_endRead;
     }
     */

    public static final boolean isPHPState(int state) {
        return state >= ST_PHP_IN_SCRIPTING && state <= ST_PHP_LINE_COMMENT;
    }

    public static final boolean isJavaScriptState(int state) {
        return state >= ST_JSCRIPT_IN_SCRIPTING && state <= ST_JSCRIPT_SINGLE_QUOTE;
    }


    private int defaultReturnValue = -1;

    private int firstPos = -1;       // the first position in the array
    private int heredoc_len = 0;
    private String heredoc = null;
    private boolean asp_tags = true;

    private StateStack phpStack = new StateStack();
    private StateStack htmlStack = new StateStack();
    private StateStack javaScriptStack = new StateStack();

    private StateStack stack = htmlStack;

    private void beginState(int state) {
        // System.out.println("got: |"+yytext()+"| changing to"+state);
        yybegin(state);
    }

    private void pushState(int state) {
        //System.out.println("pushState");
        stack.pushStack(yy_lexical_state);
        yybegin(state);
    }

    private void popState() {
        //System.out.println("popState");
        yybegin(stack.popStack());
    }

    private void enterPHPState() {
        changeStack(phpStack, ST_PHP_IN_SCRIPTING);
    }

    private void exitPHPState() {
        if (!isInsideJScript) {
            changeStack(htmlStack, ST_HTML_IN_HTML);
        } else {
            changeStack(javaScriptStack, ST_JSCRIPT_IN_SCRIPTING);
        }
    }

    boolean isInsideJScript = false;
    private void enterJavaScriptState() {
        isInsideJScript = true;
        changeStack(javaScriptStack, ST_JSCRIPT_IN_SCRIPTING);
    }

    private void exitJavaScriptState() {
        isInsideJScript = false;
        changeStack(htmlStack, ST_HTML_IN_HTML);
    }

    private void changeStack(StateStack newStack, int initializeState) {
        //System.out.println("changeStack");
        stack.pushStack(yy_lexical_state);
        stack = newStack;
        if (stack.isEmpty()) {
            beginState(initializeState);
        } else {
            beginState(stack.popStack());
        }
    }

    StringBuffer stateKeyBuff = new StringBuffer(10);

    /**
     * build a key that represents the current state of the lexer.
     */
    private String buildStateKey() {
        stateKeyBuff.setLength(0);

        stateKeyBuff.append((char) yy_lexical_state);

        StateStack.StackIterator i = htmlStack.createIterator();
        htmlStack.startIterating(i);
        while (htmlStack.hasNext(i)) {
            stateKeyBuff.append((char) htmlStack.next(i));
        }
        i = phpStack.createIterator();
        phpStack.startIterating(i);
        while (phpStack.hasNext(i)) {
            stateKeyBuff.append((char) phpStack.next(i));
        }
        stateKeyBuff.append(isInsideJScript);
        i = javaScriptStack.createIterator();
        javaScriptStack.startIterating(i);
        while (javaScriptStack.hasNext(i)) {
            stateKeyBuff.append((char) javaScriptStack.next(i));
        }

        if (heredoc != null) {
            stateKeyBuff.append(heredoc);
        }
        return stateKeyBuff.toString();
    }

    private LexerStateKey currStateKey = new LexerStateKey();

    protected class BasicLexerState implements LexerState {

        private StateStack phpStack;
        private StateStack htmlStack;
        private boolean isInsideJScript;
        private StateStack javaScriptStack;
        private byte lexicalState = (byte) yy_lexical_state;
        private boolean isPHPState = PhpLexer.this.stack == PhpLexer.this.phpStack;
        private boolean isJavaScriptState = PhpLexer.this.stack == PhpLexer.this.javaScriptStack;

        public BasicLexerState() {
            if (!PhpLexer.this.phpStack.isEmpty()) {
                this.phpStack = PhpLexer.this.phpStack.createClone();
            }
            if (!PhpLexer.this.htmlStack.isEmpty()) {
                this.htmlStack = PhpLexer.this.htmlStack.createClone();
            }
            if (!PhpLexer.this.javaScriptStack.isEmpty()) {
                this.javaScriptStack = PhpLexer.this.javaScriptStack.createClone();
            }
            this.isInsideJScript = PhpLexer.this.isInsideJScript;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o == null) {
                return false;
            }
            if (!(o instanceof BasicLexerState)) {
                return false;
            }
            BasicLexerState tmp = (BasicLexerState) o;
            if (tmp.lexicalState != this.lexicalState) {
                return false;
            }
            if (this.phpStack != null && !this.phpStack.equals(tmp.phpStack)) {
                return false;
            }
            if (this.htmlStack != null && !this.htmlStack.equals(tmp.htmlStack)) {
                return false;
            }
            if (this.javaScriptStack != null && !this.javaScriptStack.equals(tmp.javaScriptStack)) {
                return false;
            }
            if (this.isInsideJScript != tmp.isInsideJScript) {
                return false;
            }
            return this.htmlStack == tmp.htmlStack && this.phpStack == tmp.phpStack && this.javaScriptStack == tmp.javaScriptStack;
        }

        public boolean equalsTop(LexerState obj) {
            return obj != null && obj.getTopState() == lexicalState;
        }

        public int getTopState() {
            return lexicalState;
        }

        protected StateStack getActiveStack() {
            if (isPHPState) {
                return this.phpStack;
            }
            if (isJavaScriptState) {
                return this.javaScriptStack;
            }
            return this.htmlStack;
        }

        public void restoreState(PhpLexer lexer) {
            if (this.htmlStack == null) {
                lexer.htmlStack.clear();
            } else {
                lexer.htmlStack.copyFrom(this.htmlStack);
            }

            if (this.javaScriptStack == null) {
                lexer.javaScriptStack.clear();
            } else {
                lexer.javaScriptStack.copyFrom(this.javaScriptStack);
            }
            lexer.isInsideJScript = this.isInsideJScript;

            if (this.phpStack == null) {
                lexer.phpStack.clear();
            } else {
                lexer.phpStack.copyFrom(this.phpStack);
            }

            if (isPHPState) {
                lexer.stack = lexer.phpStack;
            } else if (isJavaScriptState) {
                lexer.stack = lexer.javaScriptStack;
            } else {
                lexer.stack = lexer.htmlStack;
            }
            lexer.yybegin(this.lexicalState);
        }

        public boolean isSubstateOf(int state) {
            if (lexicalState == state) {
                return true;
            }
            StateStack activeStack = this.getActiveStack();
            if (activeStack == null) {
                return false;
            }
            return activeStack.contains(state);
        }

        public String toString() {
            StateStack stack = getActiveStack();
            String stackStr = (stack == null) ? "null" : stack.toString();
            return "Stack: " + stackStr + ", currState: " + this.lexicalState;
        }

    }

    private class HeredocState implements LexerState {
        private String myHeredoc;
        private LexerState theState;

        public HeredocState(LexerState state) {
            theState = state;
            myHeredoc = PhpLexer.this.heredoc;
        }

        public boolean isSubstateOf(int state) {
            return theState.isSubstateOf(state);
        }

        public void restoreState(PhpLexer lexer) {
            theState.restoreState(lexer);
            lexer.heredoc = myHeredoc;
            lexer.heredoc_len = myHeredoc.length();
        }

        public int getTopState() {
            return theState.getTopState();
        }

        public boolean equalsTop(LexerState obj) {
            return theState.equalsTop(obj);
        }

        public boolean equals(Object obj) {
            return (obj != null &&
                    obj instanceof HeredocState &&
                    ((HeredocState) obj).theState.equals(theState) && ((HeredocState) obj).myHeredoc.equals(myHeredoc));
        }

    }

    // A pool of states. To avoid creation of a new state on each createMemento.
    private static HashMap lexerStates = new HashMap(100);

    public Object createLexicalStateMemento() {
        //System.out.println("lexerStates size:" + lexerStates.size());
        currStateKey.setKey(buildStateKey());
        Object state = lexerStates.get(currStateKey);
        if (state == null) {
            state = new BasicLexerState();
            if (yy_lexical_state == PhpLexer.ST_PHP_HEREDOC) {
                state = new HeredocState((LexerState) state);
            }
            lexerStates.put(currStateKey.clone(), state);
            // System.out.println("createLexicalStateMemento"+state);
        }
        return state;
    }

    // lex to the EOF. and return the ending state.
    public Object getEndingState() {
        lexToEnd();
        return this.createLexicalStateMemento();
    }

    // lex to the end of the stream.
    public int lexToEnd() {
        int curr = this.yylex();
        int last = curr;
        while (curr != YYEOF) {
            last = curr;
            curr = yylex();
        }
        return last;
    }

    public int lexToTokenAt(int offset) {
        if (this.firstPos + offset < this.yy_markedPos) {
            throw new RuntimeException("Bad offset");
        }
        int t = yylex();
        while (this.yy_markedPos < this.firstPos + offset && t != YYEOF) {
            t = yylex();
        }
        return t;
    }

    public int getMarkedPos() {
        return this.yy_markedPos;
    }

    /**
     * return the index where start we started to lex.
     */
    public int getFirstIndex() {
        return firstPos;
    }

    public void getText(int start, int length, Segment s) {
        if (start + length > this.yy_endRead) {
            throw new RuntimeException("bad segment !!");
        }
        s.array = this.yy_buffer;
        s.offset = start;
        s.count = length;
    }

    /**
     * reset to a new segment. this do not change the state of the lexer.
     * This method is used to scan nore than one segment as if the are one segment.
     */
    public void reset(char array[], int offset, int length) {
        this.yy_buffer = array;
        this.yy_currentPos = offset;
        this.yy_markedPos = offset;
        this.yy_pushbackPos = offset;
        this.yychar = offset;
        this.yy_endRead = offset + length;
        this.yy_startRead = offset;
        this.yy_atEOF = yy_currentPos >= yy_endRead;
        this.firstPos = offset;
    }

    /**
     * reset to a new segment. this do not change the state of the lexer.
     * This method is used to scan nore than one segment as if the are one segment.
     */
    public void reset(Segment s) {
        reset(s.array, s.offset, s.count);
    }

    public void setState(Object state) {
        ((LexerState) state).restoreState(this);
    }

/*
  protected void printState(){
    System.out.println("this.yy_startRead:"+yy_startRead);
    System.out.println("this.yy_endRead:"+ this.yy_endRead);
    System.out.println("this.yy_currentPos:" + this.yy_currentPos);
    System.out.println("this.yy_markedPos:"+ this.yy_markedPos);
    System.out.println("this.yychar:"+this.yychar);
    System.out.println("this.yy_atEOF:"+this.yy_atEOF);
    System.out.println("this.yy_reader:"+yy_reader);
  }
*/

    public void setAspTags(boolean b) {
        asp_tags = b;
    }

    public boolean getAspTags() {
        return asp_tags;
    }

    public PhpLexer(int state) {
        beginState(state);
    }

    private static final boolean isLowerCase(String text) {
        if (text == null) {
            return false;
        }
        for (int i = 0; i < text.length(); i++) {
            if (!Character.isLowerCase(text.charAt(i))) {
                return false;
            }
        }
        return true;
    }

 // End user code

%}
LNUM=[0-9]+
DNUM=([0-9]*[\.][0-9]+)|([0-9]+[\.][0-9]*)
EXPONENT_DNUM=(({LNUM}|{DNUM})[eE][+-]?{LNUM})
HNUM="0x"[0-9a-fA-F]+
LABEL=[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*
WHITESPACE=[ \n\r\t]+
TABS_AND_SPACES=[ \t]*
TOKENS=[;:,.\[\]()|\^&+-//*=%!~$<>?@]
ESCAPED_AND_WHITESPACE=[\n\t\r #'.:;,()|\^&+-/*=%!~<>?@]+

PHP_INT_CAST=	    "("{TABS_AND_SPACES}("int"|"integer"){TABS_AND_SPACES}")"
PHP_DOUBLE_CAST=	"("{TABS_AND_SPACES}("real"|"double"|"float"){TABS_AND_SPACES}")"
PHP_STRING_CAST=	"("{TABS_AND_SPACES}"string"{TABS_AND_SPACES}")"
PHP_ARRAY_CAST=	    "("{TABS_AND_SPACES}"array"{TABS_AND_SPACES}")"
PHP_OBJECT_CAST=	"("{TABS_AND_SPACES}"object"{TABS_AND_SPACES}")"
PHP_BOOL_CAST=	    "("{TABS_AND_SPACES}("bool"|"boolean"){TABS_AND_SPACES}")"
PHP_UNSET_CAST=	    "("{TABS_AND_SPACES}("unset"){TABS_AND_SPACES}")"
PHP_CASTING=        {PHP_INT_CAST}|{PHP_DOUBLE_CAST}|{PHP_STRING_CAST}|{PHP_ARRAY_CAST}|{PHP_OBJECT_CAST}|{PHP_BOOL_CAST}|{PHP_UNSET_CAST}
PHP_OPERATOR=       "=>"|"++"|"--"|"==="|"!=="|"=="|"!="|"<>"|"<="|">="|"+="|"-="|"*="|"/="|".="|"%="|"<<="|">>="|"&="|"|="|"^="|"||"|"&&"|"OR"|"AND"|"XOR"|"<<"|">>"

JSCRIPT_OPEN_TAG="<script"{WHITESPACE}*">"|("<script"{WHITESPACE}+"language"{WHITESPACE}*"="{WHITESPACE}*("javascript"|"\"javascript\""|"\'javascript\'"|"jscript"|"\"jscript\""|"\'jscript\'"){WHITESPACE}*">")

TAG_A           =a
TAG_ABBR        =abbr
TAG_ACRONYM     =acronym
TAG_ADDRESS     =address
TAG_APPLET      =applet
TAG_AREA        =area
TAG_B           =b
TAG_BASE        =base
TAG_BASEFONT    =basefont
TAG_BDO         =bdo
TAG_BIG         =big
TAG_BLOCKQUOTE  =blockquote
TAG_BODY        =body
TAG_BR          =br
TAG_BUTTON      =button
TAG_CAPTION     =caption
TAG_CENTER      =center
TAG_CITE        =cite
TAG_CODE        =code
TAG_COL         =col
TAG_COLGRUOP    =colgruop
TAG_DD          =dd
TAG_DEL         =del
TAG_DFN         =dfn
TAG_DIR         =dir
TAG_DIV         =div
TAG_DL          =dl
TAG_DT          =dt
TAG_EM          =em
TAG_FIELDSET    =fieldset
TAG_FONT        =font
TAG_FORM        =form
TAG_FRAME       =frame
TAG_FRAMESET    =frameset
TAG_H1          =h1
TAG_H2          =h2
TAG_H3          =h3
TAG_H4          =h4
TAG_H5          =h5
TAG_H6          =h6
TAG_HEAD        =head
TAG_HR          =hr
TAG_HTML        =html
TAG_I           =i
TAG_IFRAME      =iframe
TAG_IMG         =img
TAG_INPUT       =input
TAG_INS         =ins
TAG_ISINDEX     =isindex
TAG_KBD         =kbd
TAG_LABEL       =label
TAG_LEGEND      =legend
TAG_LI          =li
TAG_LINK        =link
TAG_MAP         =map
TAG_MENU        =menu
TAG_META        =meta
TAG_NOFRAMES    =noframes
TAG_NOSCRIPT    =noscript
TAG_OBJECT      =object
TAG_OL          =ol
TAG_OPTGROUP    =optgroup
TAG_OPTION      =option
TAG_P           =p
TAG_PARAM       =param
TAG_PRE         =pre
TAG_PROMPT      =prompt
TAG_Q           =q
TAG_S           =s
TAG_SAMP        =samp
TAG_SCRIPT      =script
TAG_SELECT      =select
TAG_SMALL       =small
TAG_SPAN        =span
TAG_STRIKE      =strike
TAG_STRONG      =strong
TAG_STYLE       =style
TAG_SUB         =sub
TAG_SUP         =sup
TAG_TABLE       =table
TAG_TBODY       =tbody
TAG_TD          =td
TAG_TEXTAREA    =textarea
TAG_TFOOT       =tfoot
TAG_TH          =th
TAG_THEAD       =thead
TAG_TITLE       =title
TAG_TR          =tr
TAG_TT          =tt
TAG_U           =u
TAG_UL          =ul
TAG_VAR         =var

/* All the HTML TAGS */
ANY_TAG={TAG_A}|{TAG_ABBR}|{TAG_ACRONYM}|{TAG_ADDRESS}|{TAG_APPLET}|{TAG_AREA}|{TAG_B}|{TAG_BASE}|{TAG_BASEFONT}|{TAG_BDO}|{TAG_BIG}|{TAG_BLOCKQUOTE}|{TAG_BODY}|{TAG_BR}|{TAG_BUTTON}|{TAG_CAPTION}|{TAG_CENTER}|{TAG_CITE}|{TAG_CODE}|{TAG_COL}|{TAG_COLGRUOP}|{TAG_DD}|{TAG_DEL}|{TAG_DFN}|{TAG_DIR}|{TAG_DIV}|{TAG_DL}|{TAG_DT}|{TAG_EM}|{TAG_FIELDSET}|{TAG_FONT}|{TAG_FORM}|{TAG_FRAME}|{TAG_FRAMESET}|{TAG_H1}|{TAG_H2}|{TAG_H3}|{TAG_H4}|{TAG_H5}|{TAG_H6}|{TAG_HEAD}|{TAG_HR}|{TAG_HTML}|{TAG_I}|{TAG_IFRAME}|{TAG_IMG}|{TAG_INPUT}|{TAG_INS}|{TAG_ISINDEX}|{TAG_KBD}|{TAG_LABEL}|{TAG_LEGEND}|{TAG_LI}|{TAG_LINK}|{TAG_MAP}|{TAG_MENU}|{TAG_META}|{TAG_META}|{TAG_OL}|{TAG_NOFRAMES}|{TAG_NOSCRIPT}|{TAG_OBJECT}|{TAG_OL}|{TAG_OPTGROUP}|{TAG_OPTION}|{TAG_P}|{TAG_PARAM}|{TAG_PRE}|{TAG_PROMPT}|{TAG_Q}|{TAG_S}|{TAG_SAMP}|{TAG_SCRIPT}|{TAG_SELECT}|{TAG_SMALL}|{TAG_SPAN}|{TAG_STRIKE}|{TAG_STRONG}|{TAG_STYLE}|{TAG_SUB}|{TAG_SUP}|{TAG_TABLE}|{TAG_TBODY}|{TAG_TD}|{TAG_TEXTAREA}|{TAG_TFOOT}|{TAG_TH}|{TAG_THEAD}|{TAG_TITLE}|{TAG_TR}|{TAG_TT}|{TAG_U}|{TAG_UL}|{TAG_VAR}

/* all the table tags */
TABLE_TAG={TAG_TABLE}|{TAG_TBODY}|{TAG_TD}|{TAG_TFOOT}|{TAG_TH}|{TAG_THEAD}|{TAG_TR}

/* all the heading tags */
HEADING_TAG={TAG_H1}|{TAG_H2}|{TAG_H3}|{TAG_H4}|{TAG_H5}|{TAG_H6}

%%

<YYINITIAL>{
	(([^<]|"<"[^?%s<]){1,5})|"<s"|"<"        	{return PhpConstants.T_INLINE_HTML;}
}

/* ============================================
   This rule containes ALL non-php states  !!!
   When we get a php open tag in any of the non-php states we change to ST_PHP_IN_SCRIPTING.
   This must be the last rule.
   ============================================ */
<YYINITIAL,ST_HTML_IN_HTML,ST_HTML_TAG_ATTLIST,ST_HTML_IN_COMMENT,ST_HTML_ATTRVAL,ST_HTML_IN_TAG_HEADER,ST_HTML_TAG_NAME,ST_HTML_DOUBLE_QUOTES_ATTRVAL,ST_HTML_SINGLE_QUOTE_ATTRVAL,ST_JSCRIPT_IN_SCRIPTING,ST_JSCRIPT_COMMENT,ST_JSCRIPT_LINE_COMMENT,ST_JSCRIPT_DOUBLE_QUOTES,ST_JSCRIPT_SINGLE_QUOTE> {
	"<?php"{WHITESPACE}                         {enterPHPState(); return PhpConstants.T_PHP_OPEN_TAG;}

	"<?php_track_vars?>"([\n]|"\r\n")?          {return PhpConstants.T_PHP_RESERVED_WORD;}

	"<?=" {enterPHPState();   return  PhpConstants.T_PHP_OPEN_TAG;}

	"<?"|"<script"{WHITESPACE}+"language"{WHITESPACE}*"="{WHITESPACE}*("php"|"\"php\""|"\'php\'"){WHITESPACE}*">"
	{
		if (yylength()>=2) { /* yyleng>2 means it's not <? but <script> */
            enterPHPState();
			return  PhpConstants.T_PHP_OPEN_TAG;
		} else {
			return  PhpConstants.T_INLINE_HTML;
		}
	}

	"<%" {
		if (asp_tags) {
            enterPHPState();
            return PhpConstants.T_PHP_OPEN_TAG;
		} else {
		    return  PhpConstants.T_INLINE_HTML;
		}
	}
}

<ST_HTML_IN_HTML> {
	"<"{HEADING_TAG}                    {defaultReturnValue = PhpConstants.T_HTML_TAG_HEADING; return PhpConstants.T_HTML_TAG_HEADING;}
	"<"{HEADING_TAG}{WHITESPACE}        {defaultReturnValue = PhpConstants.T_HTML_TAG_HEADING; beginState(ST_HTML_IN_TAG_HEADER); return PhpConstants.T_HTML_TAG_HEADING;}
    "<"{HEADING_TAG}">"                 {return PhpConstants.T_HTML_TAG_HEADING;}
	(<\/)({HEADING_TAG})(>?)            {return PhpConstants.T_HTML_TAG_HEADING;}

	"<"{TABLE_TAG}                      {defaultReturnValue = PhpConstants.T_HTML_TAG_TABLE;   return PhpConstants.T_HTML_TAG_TABLE;}
	"<"{TABLE_TAG}{WHITESPACE}          {defaultReturnValue = PhpConstants.T_HTML_TAG_TABLE;   beginState(ST_HTML_IN_TAG_HEADER); return PhpConstants.T_HTML_TAG_TABLE;}
	"<"{TABLE_TAG}">"                   {return PhpConstants.T_HTML_TAG_TABLE;}
	(<\/)({TABLE_TAG})(>?)              {return PhpConstants.T_HTML_TAG_TABLE;}

    {JSCRIPT_OPEN_TAG}                  {enterJavaScriptState(); return PhpConstants.T_JSCRIPT_OPEN_TAG;}

	"<"{ANY_TAG}                        {defaultReturnValue = PhpConstants.T_HTML_TAG_DEFAULT; return PhpConstants.T_HTML_TAG_DEFAULT;}
	"<"{ANY_TAG}{WHITESPACE}            {defaultReturnValue = PhpConstants.T_HTML_TAG_DEFAULT; beginState(ST_HTML_IN_TAG_HEADER); return PhpConstants.T_HTML_TAG_DEFAULT;}
	"<"{ANY_TAG}">"                     {return PhpConstants.T_HTML_TAG_DEFAULT;}
	(<\/){ANY_TAG}(>?)                  {return PhpConstants.T_HTML_TAG_DEFAULT;}

	"<"{ANY_TAG}([^ \n\t\r>]+)          {return PhpConstants.T_UNKNOWN_TOKEN;} /* handle an unknown tag */
	(<\/){ANY_TAG}([^ \n\t\r>]+)        {return PhpConstants.T_UNKNOWN_TOKEN;} /* handle an unknown close tag */

	"<!--"                              {beginState(ST_HTML_IN_COMMENT);    return PhpConstants.T_HTML_COMMENT;}

	{LABEL}                             {return PhpConstants.T_HTML_TEXT;}
	.                                   {return PhpConstants.T_HTML_TEXT;}
}

<ST_HTML_IN_TAG_HEADER> {
	([^ \n\r\t>=<]+)({WHITESPACE}?)"="  {beginState(ST_HTML_ATTRVAL); return defaultReturnValue; }
	([^ \n\r\t>=<]+)({WHITESPACE}?)     {return ST_HTML_IN_TAG_HEADER;}
	>|(\/>)                             {beginState(ST_HTML_IN_HTML); return defaultReturnValue;}
}

<ST_HTML_ATTRVAL> {
	([^ \n\r\t'\"><]+)({WHITESPACE})?   {beginState(ST_HTML_IN_TAG_HEADER);return  PhpConstants.T_HTML_ATTRVAL;}
	>|(\/>)                             {beginState(ST_HTML_IN_HTML); return defaultReturnValue;}
	"'"                                 {beginState(ST_HTML_SINGLE_QUOTE_ATTRVAL) ;return  PhpConstants.T_HTML_ATTRVAL;}
	"\""                                {beginState(ST_HTML_DOUBLE_QUOTES_ATTRVAL);return  PhpConstants.T_HTML_ATTRVAL;}
}

<ST_HTML_DOUBLE_QUOTES_ATTRVAL> {
	"\""                                {beginState(ST_HTML_IN_TAG_HEADER); return  PhpConstants.T_HTML_ATTRVAL;}
	.                                   {return  PhpConstants.T_HTML_ATTRVAL;}
}

<ST_HTML_SINGLE_QUOTE_ATTRVAL> {
	"'"                                 {beginState(ST_HTML_IN_TAG_HEADER); return  PhpConstants.T_HTML_ATTRVAL;}
	.                                   {return  PhpConstants.T_HTML_ATTRVAL;}
}

<ST_HTML_IN_COMMENT> {
	"--"({WHITESPACE}?)>                {beginState(ST_HTML_IN_HTML); return PhpConstants.T_HTML_COMMENT;}
	.                                   {return PhpConstants.T_HTML_COMMENT;}
}

<ST_PHP_IN_SCRIPTING> {
    "exit" 		    {return PhpConstants.T_PHP_EXIT;}
	"die" 		    {return PhpConstants.T_PHP_DIE;}
	"old_function" 	{return PhpConstants.T_PHP_FUNCTION;}
	"function"	    {return PhpConstants.T_PHP_FUNCTION;}
	"cfunction" 	{return PhpConstants.T_PHP_FUNCTION;}
	"const" 	    {return PhpConstants.T_PHP_CONST;}
	"return" 	    {return PhpConstants.T_PHP_RETURN;}
	"if" 		    {return PhpConstants.T_PHP_IF;}
	"elseif" 	    {return PhpConstants.T_PHP_ELSEIF;}
	"endif" 	    {return PhpConstants.T_PHP_ENDIF;}
	"else" 		    {return PhpConstants.T_PHP_ELSE;}
	"while" 	    {return PhpConstants.T_PHP_WHILE;}
	"endwhile" 	    {return PhpConstants.T_PHP_ENDWHILE;}
	"do" 		    {return PhpConstants.T_PHP_DO;}
	"for" 		    {return PhpConstants.T_PHP_FOR;}
	"endfor" 	    {return PhpConstants.T_PHP_ENDFOR;}
	"foreach" 	    {return PhpConstants.T_PHP_FOREACH;}
	"endforeach" 	{return PhpConstants.T_PHP_ENDFOREACH;}
	"declare" 	    {return PhpConstants.T_PHP_DECLARE;}
	"enddeclare" 	{return PhpConstants.T_PHP_ENDDECLARE;}
	"as" 		    {return PhpConstants.T_PHP_AS;}
	"switch" 	    {return PhpConstants.T_PHP_SWITCH;}
	"endswitch" 	{return PhpConstants.T_PHP_ENDSWITCH;}
	"case" 		    {return PhpConstants.T_PHP_CASE;}
	"default" 	    {return PhpConstants.T_PHP_DEFAULT;}
	"break" 	    {return PhpConstants.T_PHP_BREAK;}
	"continue" 	    {return PhpConstants.T_PHP_CONTINUE;}
	"echo" 		    {return PhpConstants.T_PHP_ECHO;}
	"print" 	    {return PhpConstants.T_PHP_PRINT;}
	"class" 	    {return PhpConstants.T_PHP_CLASS;}
	"extends" 	    {return PhpConstants.T_PHP_EXTENDS;}
	"::" 		    {return PhpConstants.T_PHP_RESERVED_WORD;}
	"new" 		    {return PhpConstants.T_PHP_NEW;}
	"var" 		    {return PhpConstants.T_PHP_VAR;}
	"eval" 		    {return PhpConstants.T_PHP_EVAL;}
	"include" 	    {return PhpConstants.T_PHP_INCLUDE;}
	"include_once" 	{return PhpConstants.T_PHP_INCLUDE_ONCE;}
	"require" 	    {return PhpConstants.T_PHP_REQUIRE;}
	"require_once" 	{return PhpConstants.T_PHP_REQUIRE_ONCE;}
	"use" 		    {return PhpConstants.T_PHP_USE;}
	"global" 	    {return PhpConstants.T_PHP_GLOBAL;}
	"isset" 	    {return PhpConstants.T_PHP_ISSET;}
	"empty" 	    {return PhpConstants.T_PHP_EMPTY;}
	"static" 	    {return PhpConstants.T_PHP_STATIC;}
	"unset" 	    {return PhpConstants.T_PHP_UNSET;}
	"list" 		    {return PhpConstants.T_PHP_LIST;}
	"array" 	    {return PhpConstants.T_PHP_ARRAY;}
	"parent" 	    {return PhpConstants.T_PHP_PARENT;}
	"__LINE__" 	    {return PhpConstants.T_PHP_LINE;}
	"__FILE__" 	    {return PhpConstants.T_PHP_FILE;}
	"try"           {return PhpConstants.T_PHP_TRY;}
	"catch" 	    {return PhpConstants.T_PHP_CATCH;}
	"throw" 	    {return PhpConstants.T_PHP_THROW;}
	"instanceof"    {return PhpConstants.T_PHP_INSTANCEOF;}
	"interface"     {return PhpConstants.T_PHP_INTERACE;}
	"implements" 	{return PhpConstants.T_PHP_IMPLEMENTS;}
	"from" 	        {return PhpConstants.T_PHP_FROM;}
	"abstract" 	    {return PhpConstants.T_PHP_ABSTRACT;}
	"final" 	    {return PhpConstants.T_PHP_FINAL;}
	"private" 	    {return PhpConstants.T_PHP_PRIVATE;}
	"protected" 	{return PhpConstants.T_PHP_PROTECTED;}
	"public" 	    {return PhpConstants.T_PHP_PUBLIC;}

    {PHP_CASTING} 	{return PhpConstants.T_PHP_CASTING;}
	{PHP_OPERATOR} 	{return PhpConstants.T_PHP_OPERATOR;}

	{TOKENS}        {return PhpConstants.T_PHP_TOKEN;}

	"{" 	        {pushState(ST_PHP_IN_SCRIPTING); return PhpConstants.T_PHP_CURLYOPEN;}

    "}" {
        if (!stack.isEmpty()) {
            popState();
            return  PhpConstants.T_PHP_CURLYCLOSE;
        }
        return  PhpConstants.T_UNKNOWN_TOKEN;
    }

	{LNUM} 			{return PhpConstants.T_PHP_NUMBER;}
	{HNUM} 			{return PhpConstants.T_PHP_NUMBER;}
	{DNUM}|{EXPONENT_DNUM} 	{return PhpConstants.T_PHP_NUMBER;}

	"/*"            {beginState(ST_PHP_COMMENT);return PhpConstants.T_PHP_COMMENT;}

	[\"]            {beginState(ST_PHP_DOUBLE_QUOTES);  return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	[`]             {beginState(ST_PHP_BACKQUOTE);      return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	[']             {beginState(ST_PHP_SINGLE_QUOTE);   return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}

	"<<<"{TABS_AND_SPACES}{LABEL}("\r")?"\n"
	{
        int startString=3;
		heredoc_len = yylength()-3-1-(yytext().charAt(yylength()-2)=='\r'?1:0);
		while ((yytext().charAt(startString) == ' ') || (yytext().charAt(startString) == '\t')) {
		    startString++;
			heredoc_len--;
		}
		heredoc = yytext().substring(startString,heredoc_len+startString);
		beginState(ST_PHP_HEREDOC);
		return    PhpConstants.T_PHP_HEREDOC_TAG;
	}

	{LABEL} 	    {return  PhpConstants.T_PHP_STRING;}
	{WHITESPACE} 	{return  PhpConstants.T_WHITESPACE;}
    ([#]|"//")      {beginState(ST_PHP_LINE_COMMENT); return    PhpConstants.T_PHP_COMMENT;}

    ("</script"{WHITESPACE}*">")([\n]|"\r\n")? {exitPHPState(); return PhpConstants.T_PHP_CLOSE_TAG;}
}

<ST_PHP_IN_SCRIPTING,ST_PHP_LOOKING_FOR_PROPERTY,ST_HIGHLIGHTING_ERROR> {
    "?>"                {exitPHPState(); return PhpConstants.T_PHP_CLOSE_TAG;}
	"%>"([\n]|"\r\n")? {
		if (asp_tags) {
			exitPHPState();
			return PhpConstants.T_PHP_CLOSE_TAG;
		} else {
            // yypushback(2);
            // pushState(ST_HIGHLIGHTING_ERROR);
            return PhpConstants.T_UNKNOWN_TOKEN;
        }
	}
}

<ST_PHP_LINE_COMMENT> {
    "?>" {
        beginState(ST_PHP_IN_SCRIPTING);
        exitPHPState();
        return PhpConstants.T_PHP_CLOSE_TAG;
    }
	"%>"([\n]|"\r\n")? {
		if (asp_tags) {
			exitPHPState();
			return PhpConstants.T_PHP_CLOSE_TAG;
		} else {
            // yypushback(2);
            // pushState(ST_HIGHLIGHTING_ERROR);
            return PhpConstants.T_UNKNOWN_TOKEN;
        }
	}
    ("\n"|"\r\n")       {beginState(ST_PHP_IN_SCRIPTING);return  PhpConstants.T_PHP_COMMENT;}
    .                   {return    PhpConstants.T_PHP_COMMENT;}
}

<ST_PHP_IN_SCRIPTING> {
	"<?"	{return PhpConstants.T_UNKNOWN_TOKEN;}
	.       {return PhpConstants.T_UNKNOWN_TOKEN;}
}

<ST_PHP_HEREDOC>{
	[`]+ 	{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\\"" 	{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	[\"] 	{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	.	    {return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
}

<ST_PHP_SINGLE_QUOTE>{
	([^'\\]|\\[^'\\])+ 	{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\'" 			{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\\\" 			{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	[']		 	    {beginState(ST_PHP_IN_SCRIPTING);return   PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	.			    {return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
}

<ST_PHP_DOUBLE_QUOTES>{
	[`]+ 	{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\\"" 	{return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	[\"] 	{beginState(ST_PHP_IN_SCRIPTING);return   PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	.	    {return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
}

<ST_PHP_BACKQUOTE>{
	[\"]+ 	{return   PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\`" 	{return   PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	[`] 	{beginState(ST_PHP_IN_SCRIPTING);return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	.	    {return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
}

<ST_PHP_IN_SCRIPTING,ST_PHP_DOUBLE_QUOTES,ST_PHP_BACKQUOTE,ST_PHP_HEREDOC>{
	"->" 		{pushState(ST_PHP_LOOKING_FOR_PROPERTY); return PhpConstants.T_PHP_OPERATOR;}
	"$"{LABEL} 	{return PhpConstants.T_PHP_VARIABLE;}
}

<ST_PHP_DOUBLE_QUOTES,ST_PHP_BACKQUOTE,ST_PHP_HEREDOC>{
	"${" 			            {pushState(ST_PHP_IN_SCRIPTING); return PhpConstants.T_PHP_TOKEN;}
	{LNUM}|{HNUM} 		        {return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	{LABEL} 		            {return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	{ESCAPED_AND_WHITESPACE}    {return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"{$" 			            {yypushback(1); pushState(ST_PHP_IN_SCRIPTING); return PhpConstants.T_PHP_TOKEN;}
	"\\". 			            {return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\"[0-7]{1,3} 		        {return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
	"\\x"[0-9A-Fa-f]{1,2} 	    {return    PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}
}

<ST_PHP_LOOKING_FOR_PROPERTY>{
    {LABEL} 	{popState(); return PhpConstants.T_PHP_STRING;}
    "$"{LABEL} 	{popState(); return PhpConstants.T_PHP_VARIABLE;}
	"{" 	    {popState(); pushState(ST_PHP_IN_SCRIPTING); return PhpConstants.T_PHP_CURLYOPEN;}
	[\"] 		{popState(); beginState(ST_PHP_IN_SCRIPTING); return PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;}

	"/*"            {beginState(ST_PHP_COMMENT);return PhpConstants.T_PHP_COMMENT;}
    ([#]|"//")      {beginState(ST_PHP_LINE_COMMENT); return    PhpConstants.T_PHP_COMMENT;}
}

<ST_PHP_COMMENT>{
	"*/" 	{beginState(ST_PHP_IN_SCRIPTING);return    PhpConstants.T_PHP_COMMENT;}
	[^*]+ 	{return    PhpConstants.T_PHP_COMMENT;}
	"*" 	{return    PhpConstants.T_PHP_COMMENT;}
}

<ST_PHP_HEREDOC>{
	^{LABEL}(";")?("\r")?"\n"
	{
		int label_len;
		int length=yylength();
		if (yytext().charAt(length-2)=='\r') {
			label_len = length-2;
		} else {
			label_len = length-1;
		}

		if (yytext().charAt(label_len-1)==';') {
			label_len--;
		}

		if (label_len==heredoc_len && yytext().substring(0,label_len).equals(heredoc)) {
			heredoc=null;
			heredoc_len=0;
			beginState(ST_PHP_IN_SCRIPTING);
			return  PhpConstants.T_PHP_HEREDOC_TAG;
		} else {
            return  PhpConstants.T_PHP_CONSTANT_ENCAPSED_STRING;
		}
	}

	[\"'`]+ {return    PhpConstants.T_PHP_ENCAPSED_AND_WHITESPACE;}
}

<ST_JSCRIPT_IN_SCRIPTING> {
    "abstract"      {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_ABSTRACT;}
    "boolean"       {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_BOOLEAN;}
    "break"         {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_BREAK;}
    "byte"          {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_BYTE;}
    "case" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_CASE;}
    "catch"		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_CATCH;}
    "char" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_CHAR;}
    "class"		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_CLASS;}
    "const" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_CONST;}
    "continue" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_CONTINUE;}
    "default" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_DEFAULT;}
    "delete" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_DELETE;}
    "do" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_DO;}
    "double" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_DOUBLE;}
    "else" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_ELSE;}
    "extends" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_EXTENDS;}
    "final" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_FINAL;}
    "finally" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_FINALLY;}
    "float" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_FLOAT;}
    "for" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_FOR;}
    "function" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_FUNCTION;}
    "goto" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_GOTO;}
    "if" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_IF;}
    "implements"    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_IMPLEMENTS;}
    "import" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_IMPORT;}
    "in" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_IN;}
    "instanceof"    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_INSTANCEOF;}
    "int" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_INT;}
    "interface"     {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_INTERFACE;}
    "long" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_LONG;}
    "native" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_NATIVE;}
    "new" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_NEW;}
    "package" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_PACKAGE;}
    "private" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_PRIVATE;}
    "protected"     {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_PROTECTED;}
    "public" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_PUBLIC;}
    "return" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_RETURN;}
    "short" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_SHORT;}
    "static" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_STATIC;}
    "super" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_SUPER;}
    "switch" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_SWITCH;}
    "synchronized"  {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_SYNCHRONIZED;}
    "this" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_THIS;}
    "throw" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_THROW;}
    "throws" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_THROWS;}
    "transient"     {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_TRANSIENT;}
    "try" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_TRY;}
    "typeof" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_TYPEOF;}
    "var" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_VAR;}
    "void" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_VOID;}
    "volatile" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_VOLATILE;}
    "while" 	    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_WHILE;}
    "with" 		    {return (!isLowerCase(yytext())) ? PhpConstants.T_JSCRIPT_STRING: PhpConstants.T_JSCRIPT_WITH;}

	{TOKENS}        {return PhpConstants.T_JSCRIPT_TOKEN;}

	"{" 	        {pushState(ST_JSCRIPT_IN_SCRIPTING); return PhpConstants.T_JSCRIPT_CURLYOPEN;}
	"}"
    {
        if (stack.contains(ST_JSCRIPT_IN_SCRIPTING)) {
            popState();
            return  PhpConstants.T_JSCRIPT_CURLYCLOSE;
        }
        return  PhpConstants.T_UNKNOWN_TOKEN;
    }

	{LNUM} 			{return PhpConstants.T_JSCRIPT_NUMBER;}
	{HNUM} 			{return PhpConstants.T_JSCRIPT_NUMBER;}
	{DNUM}|{EXPONENT_DNUM} 	{return PhpConstants.T_JSCRIPT_NUMBER;}

	[\"]            {beginState(ST_JSCRIPT_DOUBLE_QUOTES);  return PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
	[']             {beginState(ST_JSCRIPT_SINGLE_QUOTE);   return PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}

	{LABEL} 	    {return  PhpConstants.T_JSCRIPT_STRING;}
	{WHITESPACE} 	{return  PhpConstants.T_WHITESPACE;}

	"/*"            {beginState(ST_JSCRIPT_COMMENT)         ;return PhpConstants.T_JSCRIPT_COMMENT;}
    ([#]|"//")      {beginState(ST_JSCRIPT_LINE_COMMENT)    ;return PhpConstants.T_JSCRIPT_COMMENT;}

    ("</script"{WHITESPACE}*">")([\n]|"\r\n")? {exitJavaScriptState(); return PhpConstants.T_JSCRIPT_CLOSE_TAG;}
}

<ST_JSCRIPT_COMMENT>{
	"*/" 	        {beginState(ST_JSCRIPT_IN_SCRIPTING);return    PhpConstants.T_JSCRIPT_COMMENT;}
	[^*]+ 	        {return PhpConstants.T_JSCRIPT_COMMENT;}
	"*" 	        {return PhpConstants.T_JSCRIPT_COMMENT;}
}

<ST_JSCRIPT_LINE_COMMENT> {
    ("\n"|"\r\n")   {beginState(ST_JSCRIPT_IN_SCRIPTING);return  PhpConstants.T_JSCRIPT_COMMENT;}
    .               {return PhpConstants.T_JSCRIPT_COMMENT;}
}

<ST_JSCRIPT_DOUBLE_QUOTES>{
	"\\\"" 	        {return PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
	[\"]            {beginState(ST_JSCRIPT_IN_SCRIPTING);return   PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
	.               {return PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
}

<ST_JSCRIPT_SINGLE_QUOTE>{
	"\\'" 	        {return PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
	[']             {beginState(ST_JSCRIPT_IN_SCRIPTING);return   PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
	.               {return PhpConstants.T_JSCRIPT_CONSTANT_ENCAPSED_STRING;}
}

/* ============================================
   Stay in this state until we find a whitespace.
   After we find a whitespace we go the the prev state and try again from the next token.
   ============================================ */
<ST_HIGHLIGHTING_ERROR> {
	{WHITESPACE}	{popState();return PhpConstants.T_WHITESPACE;}
    .   	        {return PhpConstants.T_UNKNOWN_TOKEN;}
}

/* ============================================
   This rule must be the last rule in the file !!
   it should contain all the states.
   ============================================ */
<ST_PHP_IN_SCRIPTING,ST_PHP_DOUBLE_QUOTES,ST_PHP_SINGLE_QUOTE,ST_PHP_BACKQUOTE,ST_PHP_HEREDOC,ST_PHP_LOOKING_FOR_PROPERTY,ST_PHP_COMMENT,ST_HTML_IN_HTML,ST_HTML_TAG_ATTLIST,ST_HTML_IN_COMMENT,ST_HTML_ATTRVAL,ST_HTML_IN_TAG_HEADER,ST_HTML_TAG_NAME,ST_HTML_DOUBLE_QUOTES_ATTRVAL,ST_HTML_SINGLE_QUOTE_ATTRVAL,ST_HIGHLIGHTING_ERROR,ST_JSCRIPT_IN_SCRIPTING,ST_JSCRIPT_COMMENT,ST_JSCRIPT_DOUBLE_QUOTES,ST_JSCRIPT_SINGLE_QUOTE> {
    {WHITESPACE}	{return PhpConstants.T_WHITESPACE;}
    .       		{yypushback(1); pushState(ST_HIGHLIGHTING_ERROR);}
}