/*
    Interrupt Descriptor Table (IDT) Guard.
    VERSiON 1.0 - (c) December, 2005 - 

    Copyright (C) 2005 Matthieu Suiche <msuiche@gmail.com> www.msuiche.net

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the Free
    Software Foundation; either version 2 of the License, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along with
    this program; if not, write to the Free Software Foundation, Inc., 59 Temple
    Place, Suite 330, Boston, MA 02111-1307 USA
*/ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include <memory.h>
#include <aclapi.h>

#include "header.h"

HANDLE					hPhysMem;
BYTE					KernelPath[LEN_PATH];

int main(int argc, char **argv) {
KIDTRAWENTRY			IdtRaw[MAX_INTERRUPTION], IdtMem[MAX_INTERRUPTION]; 
BYTE					lpCmd[12];
DWORD					VirtualAddress, Counter, Status,
						Index, pCmd;

	printf("Interrupt Descriptor Table(IDT) Guard\n");
	printf("Matthieu Suiche <matt@msuiche.net> www.msuiche.net\n");
	printf("Version 0.1 - (c) December, 2005 - \n\n\n");

	if(!CheckOsVersion())
    {
        printf("\nError: Bad OS Version\n");
		return FALSE;
    }

	if(!UpdateAffinityMask())
    {
        printf("\nError: Affinity Mask\n");
		return FALSE;
    }
	if(!GetNtdllFunction())
    {
        printf("\nError: Getting Ntdll's functions\n");
		return FALSE;
    }

	if(!GetRawInterruption((PKIDTRAWENTRY)&IdtRaw))
    {
        printf("\nError: Getting Raw interruptions\n");
		return FALSE;
    }

	if(!(VirtualAddress = StartUp()))
    {
        printf("\nError: Bad Virtual Address\n");
		return FALSE;
    }
	if(!GetMemoryInterruption((PKIDTRAWENTRY)&IdtMem, VirtualAddress+0x400)) // IDT Offset = GDT Offset + 0x400
    {
        printf("\nError: Cannot get interruption. Is access to \\Device\\PhysicalMemory.\n");
		return FALSE;
    }
	Counter = CompareInterruption((PKIDTRAWENTRY)&IdtRaw, (PKIDTRAWENTRY)&IdtMem);

	printf("\n%d Interruptions have been modified.\n\n",Counter);

	GetHelp();
	Status = 1;
	while(Status) {

		printf("\ncmd>");

		switch(getchar()) {

		case 'q':
			printf("I hope you like this tool ;)\n");
			Status = 0;
		break;
		
		case 'h':
			GetHelp();
		break;

		case 's':
			getchar();
			GetMemoryInterruption((PKIDTRAWENTRY)&IdtMem, VirtualAddress+0x400); // IDT Offset = GDT Offset + 0x400
			CompareInterruption((PKIDTRAWENTRY)&IdtRaw, (PKIDTRAWENTRY)&IdtMem);
		break;

		case 'r':

			memset(lpCmd,0,sizeof(lpCmd));
			fgets(lpCmd,sizeof(lpCmd)-1,stdin);
			pCmd	= (DWORD)&lpCmd;
			memset((PDWORD)(pCmd+5), 0, 1);
			Index = strtol((PBYTE)pCmd+3,0,16);

			printf("Are you sure that you want to restore the Interruption 0x%02X(%d)? (y/n)",Index,Index);
			
			if(getchar() == 'y') {

				getchar();
				printf("\nLet's restore it !\n");
				RestoreInterruptionByHandle(Index, IdtRaw, IdtMem, (PKIDTENTRY)(VirtualAddress+0x400));
				GetMemoryInterruption((PKIDTRAWENTRY)&IdtMem, VirtualAddress+0x400);
				
			} else getchar();

		break;
		default:
			printf("Invalid Command\n");
		break;

		}
	}

	CleanUp(VirtualAddress);

	return TRUE;

}

DWORD CheckOsVersion(void) {
OSVERSIONINFO			OsVersion;

	OsVersion.dwOSVersionInfoSize = sizeof(OsVersion);
	GetVersionEx(&OsVersion);

	if( (OsVersion.dwMajorVersion != 5) && 
		(OsVersion.dwMinorVersion != 0) && (OsVersion.dwMinorVersion != 1)
		){
		printf("Sorry, but this tool works only with Win2K and WinXP.\n");
		return FALSE;
	}

	return TRUE;
}

DWORD UpdateAffinityMask(void) {
DWORD lpProcessAffinityMask, lpSystemAffinityMask;

	GetProcessAffinityMask((HANDLE)-1, &lpProcessAffinityMask, &lpSystemAffinityMask);

	if(lpProcessAffinityMask == 0)	{
		lpProcessAffinityMask = 1;

		if(!SetProcessAffinityMask((HANDLE)-1, lpProcessAffinityMask)) {
			printf("Cannot update process affinity mask.\n");
			return FALSE;
		}
	}

	return TRUE;

}

DWORD RestoreInterruptionByHandle(DWORD Index, PKIDTRAWENTRY IdtRaw, PKIDTRAWENTRY IdtMem, PKIDTENTRY IdtEntry) {
BYTE					TypeRaw[10], TypeMem[10];

	if (IdtRaw[Index].Type == GATE_TASK_16) strcpy(TypeRaw, "TaskG16 ");
	else if (IdtRaw[Index].Type == GATE_TASK_32) strcpy(TypeRaw, "TaskG32 ");

	else if (IdtRaw[Index].Type == GATE_INTERRUPT_32) strcpy(TypeRaw, "IntG32  ");
	else if (IdtRaw[Index].Type == GATE_INTERRUPT_16) strcpy(TypeRaw, "IntG16  ");

	else if (IdtRaw[Index].Type == GATE_TRAP_32) strcpy(TypeRaw, "TrapG32 ");
	else if (IdtRaw[Index].Type == GATE_TRAP_16) strcpy(TypeRaw, "TrapG16 ");
	else strcpy(TypeRaw, "Reserved");

	if (IdtMem[Index].Type == GATE_TASK_16) strcpy(TypeMem, "TaskG16 ");
	else if (IdtMem[Index].Type == GATE_TASK_32) strcpy(TypeMem, "TaskG32 ");

	else if (IdtMem[Index].Type == GATE_INTERRUPT_32) strcpy(TypeMem, "IntG32  ");
	else if (IdtMem[Index].Type == GATE_INTERRUPT_16) strcpy(TypeMem, "IntG16  ");

	else if (IdtMem[Index].Type == GATE_TRAP_32) strcpy(TypeMem, "TrapG32 ");
	else if (IdtMem[Index].Type == GATE_TRAP_16) strcpy(TypeMem, "TrapG16 ");
	else strcpy(TypeMem, "Reserved");


	printf("I will do that : \n");
	printf("\tOffset   : 0x%08X => 0x%08X\n", IdtMem[Index].Offset,IdtRaw[Index].Offset);
	printf("\tDpl      :       0x%02X =>       0x%02X\n", IdtMem[Index].Dpl,IdtRaw[Index].Dpl);
//	printf("\tSelector :     0x%04X =>     0x%04X\n", IdtMem[Index].Selector,IdtRaw[Index].Selector);
//	printf("\tPresent  :        0x%01d =>        0x%01d\n", IdtMem[Index].Present,IdtRaw[Index].Present);
	printf("\tType     : %08s   => %08s\n\n",TypeMem, TypeRaw);

	if(
		   (IdtMem[Index].Offset	== IdtRaw[Index].Offset) 
		&& (IdtMem[Index].Dpl		== IdtRaw[Index].Dpl)
		&& (IdtMem[Index].Type		== IdtRaw[Index].Type)
	//	&& (IdtMem[Index].Present	== IdtRaw[Index].Present)
	) return TRUE;

	printf("Are you sure?(y/n)");
	
	if(getchar() == 'y') {

		getchar();

		/*
		Dpl: Descriptor Privilege Level
		Offset: Offset To Procedure Entry Point
		P: Segment Present Bit
		Reserved: Do not use
		Selector: Segment Selector For Destination Code Segment
		*/
		printf("\nReconstrution of the INT 0x%02X\n",Index);
		
		printf("\tOffset value...");
		IdtEntry[Index].OffsetLow = IdtRaw[Index].OffsetLow;
		IdtEntry[Index].OffsetHigh = IdtRaw[Index].OffsetHigh;
		printf("Done\n");

		printf("\tDpl(Descriptor Privilege Level) value...");
		IdtEntry[Index].Dpl = IdtRaw[Index].Dpl;
		printf("Done\n");

		printf("\tType value...");
		IdtEntry[Index].Type = IdtRaw[Index].Type;
		printf("Done\n");

		printf("\nOKiE\n");

	} else getchar();

	return TRUE;
}

DWORD GetRawInterruption(PKIDTRAWENTRY IdtRaw) {
HANDLE					hFile, hMapping, pMapping;

PIMAGE_DOS_HEADER		DOSHeader;
PIMAGE_NT_HEADERS		NTHeaders;

PIMAGE_SECTION_HEADER	pSecHdr;

DWORD					NumberOfSections, RawSize, IntAddr, ImageBase = 0;
DWORD					INIT_Sex, KernelBaseAddr, i;

PKIDTRAWENTRY			IDTr;
BYTE					Type[10];

	KernelBaseAddr = GetKernelInformation();
	if(!KernelBaseAddr)
		return FALSE;

	//printf("%08X:%s\n",KernelBaseAddr,KernelPath);

	hFile = CreateFile(KernelPath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);

	if(hFile == INVALID_HANDLE_VALUE) {
			printf("\nCannot open the kernel. Please check if the file is correct\n");
			return FALSE;
	}

	hMapping = CreateFileMapping(hFile, 0, PAGE_READONLY, 0, 0, 0);

	if(!hMapping){

			printf("Cannot create file mapping.\n");
			CloseHandle(hFile);

			return FALSE;
	}

	pMapping = MapViewOfFile(hMapping, FILE_MAP_READ , 0, 0, 0);

	if(!pMapping){

			printf("Cannot view mapping.\n");
			CloseHandle(hMapping);
			CloseHandle(hFile);

			return FALSE;
	}


	DOSHeader = (PIMAGE_DOS_HEADER)pMapping;
	if (DOSHeader->e_magic != IMAGE_DOS_SIGNATURE) {

			printf("Bad DOS Signature.\n");
			UnmapViewOfFile(pMapping);
			CloseHandle(hMapping);
			CloseHandle(hFile);

			return FALSE;
	}

	NTHeaders = (PIMAGE_NT_HEADERS)((BYTE *)pMapping + DOSHeader->e_lfanew);
	if (NTHeaders->Signature != IMAGE_NT_SIGNATURE){

			printf("Bad PE Signature\n");
			UnmapViewOfFile(pMapping);
			CloseHandle(hMapping);
			CloseHandle(hFile);

			return FALSE;
	}

	ImageBase = NTHeaders->OptionalHeader.ImageBase;

	KernelBaseAddr -= ImageBase;

	pSecHdr = IMAGE_FIRST_SECTION(NTHeaders);

		for(NumberOfSections = NTHeaders->FileHeader.NumberOfSections;
			NumberOfSections;
			(DWORD)pSecHdr += sizeof(IMAGE_SECTION_HEADER), NumberOfSections--
		)
			if(strcmp(pSecHdr->Name,"INIT") == 0)
				break;

		if(!NumberOfSections) {

			printf("Cannot found the section \"INIT\".\n");
			UnmapViewOfFile(pMapping);
			CloseHandle(hMapping);
			CloseHandle(hFile);

			return FALSE;
		}

		INIT_Sex =	(DWORD)((DWORD)pMapping + (DWORD)pSecHdr->PointerToRawData);
		RawSize = pSecHdr->SizeOfRawData - 12;

		while(RawSize) {
			/*
			INIT:00558B36                 mov     ecx, 800h
			INIT:00558B3B                 shr     ecx, 2
			INIT:00558B3E                 rep movsd               ; Copy To IDT !
			*/

				if(!memcmp((PBYTE)INIT_Sex,"\xB9\x00\x08\x00\x00\xC1\xE9\x02\xF3\xA5",10))
					break; // Good Boy ;)

				(DWORD)INIT_Sex += 1;
				(DWORD)RawSize -= 1;

		}
	
		if(!RawSize) {

			printf("Cannot found the magic part of code.\n");
			UnmapViewOfFile(pMapping);
			CloseHandle(hMapping);
			CloseHandle(hFile);

			return FALSE;
		}

		(DWORD)INIT_Sex -= 4; // INIT:00558B31       mov esi, offset xxxxx ; _IDT

		// IDT Offset in ntoskrnl.exe
		IDTr = (PKIDTRAWENTRY)( ((DWORD *)(INIT_Sex)) [0]); // Sexy line ^^

		(DWORD)IDTr -= (DWORD)ImageBase;
		(DWORD)IDTr += (DWORD)pMapping;

		for(i=0;i<MAX_INTERRUPTION;i++,IDTr) {

				if (IDTr[i].Type == GATE_TASK_16) strcpy(Type, "TaskG16 ");
				else if (IDTr[i].Type == GATE_TASK_32) strcpy(Type, "TaskG32 ");

				else if (IDTr[i].Type == GATE_INTERRUPT_32) strcpy(Type, "IntG32  ");
				else if (IDTr[i].Type == GATE_INTERRUPT_16) strcpy(Type, "IntG16  ");

				else if (IDTr[i].Type == GATE_TRAP_32) strcpy(Type, "TrapG32 ");
				else if (IDTr[i].Type == GATE_TRAP_16) strcpy(Type, "TrapG16 ");
				else strcpy(Type, "Reserved");


			if(!(DWORD)IDTr[i].Offset)
				IntAddr = 0;
			else
				IntAddr = ((DWORD)IDTr[i].Offset + KernelBaseAddr);

			// ---------------------------------------

			IdtRaw[i].Offset	= IntAddr;
			IdtRaw[i].Reserved	= IDTr[i].Reserved;
			IdtRaw[i].Type		= IDTr[i].Type;
			IdtRaw[i].Always0	= IDTr[i].Always0;
			IdtRaw[i].Dpl		= IDTr[i].Present;
			IdtRaw[i].Selector	= 0;

			// ---------------------------------------
/*
			printf("%04X  %s  %08X DPL=%x  %-2s\n",
					i,
					Type,
					IntAddr,
					IDTr[i].Dpl,
					IDTr[i].Present ? "P" : "NP"
			);
*/
		}

	UnmapViewOfFile(pMapping);
	CloseHandle(hMapping);
	CloseHandle(hFile);
	return TRUE;
}

DWORD StartUp(void) {
DWORD					PhysicalAddress, VirtualAddress, Lenght;
KGDT 					GDT;
PHYSICAL_ADDRESS		pAddress;
NTSTATUS				Status;

	if(!(hPhysMem = OpenPhysicalMemory()))
		return FALSE;

	_asm {
		sgdt [GDT]
		mov eax, dword ptr [GDT+2]
		mov dword ptr [PhysicalAddress], eax
	}

	if((PhysicalAddress >> 24) != 0x80) {
		printf("You are using a VMM(VMWare, VirtualPC,...) so I cannot get the GDT BaseAddr from ring3.\n");
	}

	pAddress	= GetPhysicalAddress(PhysicalAddress);
	Lenght = 0x1000;
	VirtualAddress = 0;

	Status = NtMapViewOfSection(hPhysMem,
					(HANDLE)-1,
					(PVOID)&VirtualAddress,
					0L,
					Lenght,
					&pAddress,
					(PDWORD)&Lenght,
					ViewShare,
					0,
					PAGE_READWRITE);

	if(!NT_SUCCESS(Status)) {

		printf("Cannot NtMapViewOfSection.");
		NtClose(hPhysMem);
		return FALSE;

	}

	return VirtualAddress;
}

DWORD GetMemoryInterruption(PKIDTRAWENTRY IdtMem, DWORD VirtualAddress) {
DWORD					i;
PKIDTENTRY				IDTm;
BYTE					Type[10];

	IDTm = (PKIDTENTRY)(VirtualAddress);

			for(i=0;i<MAX_INTERRUPTION;i++,IDTm) {

				if (IDTm[i].Type == GATE_TASK_16) strcpy(Type, "TaskG16 ");
				else if (IDTm[i].Type == GATE_TASK_32) strcpy(Type, "TaskG32 ");

				else if (IDTm[i].Type == GATE_INTERRUPT_32) strcpy(Type, "IntG32  ");
				else if (IDTm[i].Type == GATE_INTERRUPT_16) strcpy(Type, "IntG16  ");

				else if (IDTm[i].Type == GATE_TRAP_32) strcpy(Type, "TrapG32 ");
				else if (IDTm[i].Type == GATE_TRAP_16) strcpy(Type, "TrapG16 ");
				else strcpy(Type, "Reserved");


			// ---------------------------------------

			IdtMem[i].Reserved		= IDTm[i].Reserved;
			IdtMem[i].Type			= IDTm[i].Type;
			IdtMem[i].Always0		= IDTm[i].Always0;
			IdtMem[i].Dpl			= IDTm[i].Present;
			IdtMem[i].Selector		= IDTm[i].Selector;
			IdtMem[i].OffsetHigh	= IDTm[i].OffsetHigh;
			IdtMem[i].OffsetLow		= IDTm[i].OffsetLow;

			// ---------------------------------------
/*
			printf("%04X  %s  %08X DPL=%x  %-2s\n",
					i,
					Type,
					IdtMem[i].Offset,
					IDTm[i].Dpl,
					IDTm[i].Present ? "P" : "NP"
			);
*/

			}	

	return TRUE;
}

DWORD CompareInterruption(PKIDTRAWENTRY IdtRaw, PKIDTRAWENTRY IdtMem) {
ULONG						cbBuffer = 0x8000;
LPVOID						pBuffer = NULL;

NTSTATUS					Status;
PSYSTEM_MODULE_INFORMATION	pInfo;

DWORD						i, j, Counter = 0;
	    do {
		pBuffer = HeapAlloc(GetProcessHeap(), 0, cbBuffer);
		if (!pBuffer) {
			printf("HeapAlloc\n");
			return FALSE;
		}
		Status = NtQuerySystemInformation(SystemModuleInformation, pBuffer, cbBuffer, NULL);

		if(Status == STATUS_INFO_LENGTH_MISMATCH) {
			HeapFree(GetProcessHeap(), 0, pBuffer);
			cbBuffer *= 2;
		}
		else if(Status != STATUS_SUCCESS) {
			HeapFree(GetProcessHeap(), 0, pBuffer);
			printf("HeapFree\n");
			return FALSE;
		}
    }
    while (Status == STATUS_INFO_LENGTH_MISMATCH);
		
		// Compare INT
			for(i=0; i < MAX_INTERRUPTION; i++) {
				if(IdtRaw[i].Offset != IdtMem[i].Offset) {

						pInfo = (PSYSTEM_MODULE_INFORMATION)pBuffer;
						Status=0;
						Counter++;

					// -----
					for(j = pInfo->ModulesCount;j > 0; j--) {

						if(	(IdtMem[i].Offset > pInfo->ImageBaseAddress) 
							&& (IdtMem[i].Offset < (pInfo->ImageBaseAddress+pInfo->ImageSize)) ) {

							printf("INT 0x%02X has been hooked at 0x%08X (Org INT = 0x%08X) by %s\n",
									i,
									IdtMem[i].Offset,
									IdtRaw[i].Offset,
									pInfo->Name+pInfo->NameOffset
							);
							j=1;
							Status=1;
						} 

						pInfo++;
					} // Module List
					// -----

					if(!j && !Status) {
						printf("INT 0x%02X has been hooked at 0x%08X (Org INT = 0x%08X) by Unknow\n",
								i,
								IdtMem[i].Offset,
								IdtRaw[i].Offset
						);
					}

				} // Idt compare

			} // Idt Count

		return Counter;
}

DWORD	CleanUp(DWORD VirtualAddress) {
NTSTATUS Status;

	Status = NtUnmapViewOfSection((HANDLE) -1, (PVOID)VirtualAddress);

		if(!NT_SUCCESS(Status)) {
			printf("Unable to NtUnmapViewOfSection");
		}

	NtClose(hPhysMem);

	return TRUE;
}

DWORD GetKernelInformation(void) {

ULONG cbBuffer = 0x8000;
LPVOID pBuffer = NULL;
NTSTATUS Status;

DWORD BaseAddr;

PSYSTEM_MODULE_INFORMATION pInfo;

    do {
		pBuffer = HeapAlloc(GetProcessHeap(), 0, cbBuffer);
		if (!pBuffer)
			return FALSE;

		Status = NtQuerySystemInformation(SystemModuleInformation, pBuffer, cbBuffer, NULL);

		if(Status == STATUS_INFO_LENGTH_MISMATCH) {
			HeapFree(GetProcessHeap(), 0, pBuffer);
			cbBuffer *= 2;
		}
		else if(Status != STATUS_SUCCESS) {
			HeapFree(GetProcessHeap(), 0, pBuffer);
			return FALSE;
		}
    }
    while (Status == STATUS_INFO_LENGTH_MISMATCH);

	pInfo = (PSYSTEM_MODULE_INFORMATION)pBuffer;

	BaseAddr = pInfo->ImageBaseAddress;
	strncpy(KernelPath,pInfo->Name,sizeof(KernelPath)-1);
	HeapFree(GetProcessHeap(), 0, pBuffer);

	return BaseAddr;
}

DWORD GetNtdllFunction(void) {

	if(!(*(FARPROC *)&NtQuerySystemInformation = GetProcAddress(LoadLibrary("ntdll"), "ZwQuerySystemInformation")))
		return FALSE;

	if(!(*(FARPROC *)&NtOpenSection = GetProcAddress(LoadLibrary("ntdll"), "ZwOpenSection")))
		return FALSE;

	if(!(*(FARPROC *)&NtMapViewOfSection = GetProcAddress(LoadLibrary("ntdll"), "ZwMapViewOfSection")))
		return FALSE;

	if(!(*(FARPROC *)&NtUnmapViewOfSection = GetProcAddress(LoadLibrary("ntdll"), "ZwUnmapViewOfSection")))
		return FALSE;

	if(!(*(FARPROC *)&NtClose = GetProcAddress(LoadLibrary("ntdll"), "ZwClose")))
		return FALSE;

	if(!(*(FARPROC *)&RtlInitUnicodeString = GetProcAddress(LoadLibrary("ntdll"), "RtlInitUnicodeString")))
		return FALSE;

	if(!(*(FARPROC *)&RtlNtStatusToDosError = GetProcAddress(LoadLibrary("ntdll"), "RtlNtStatusToDosError")))
		return FALSE;

	return TRUE;
}

// Set Write Access to physicalmemory and return physicalmemory handle.
HANDLE OpenPhysicalMemory(void) {
NTSTATUS			Status;
HANDLE				hPhysMem;
UNICODE_STRING		lpString;
OBJECT_ATTRIBUTES	Attributes;
WCHAR				lpName[] = L"\\device\\physicalmemory";
PACL				lpACL, lpACLNew;
PSECURITY_DESCRIPTOR lpSecDesc;
EXPLICIT_ACCESS		ExpAccess;
BYTE				lpUserName[MAX_PATH];
DWORD				dwNameSize = sizeof(lpUserName)-1;

	RtlInitUnicodeString(&lpString, lpName );	

	InitializeObjectAttributes(Attributes, &lpString, OBJ_CASE_INSENSITIVE | OBJ_KERNEL_HANDLE, NULL, NULL );
	
	Status = NtOpenSection(&hPhysMem, WRITE_DAC | READ_CONTROL, &Attributes );

	if(!NT_SUCCESS(Status)) {
		printf( "Cannot open \\device\\physicalmemory");
		return FALSE;
	} else	{
		
		if(GetSecurityInfo(hPhysMem, SE_KERNEL_OBJECT, DACL_SECURITY_INFORMATION, NULL, NULL,
						&lpACL, NULL, &lpSecDesc) == ERROR_SUCCESS){

			GetUserName(lpUserName, &dwNameSize);

			InitializeUserAccess(ExpAccess, lpUserName);

			if(SetEntriesInAcl(1, &ExpAccess, lpACL, &lpACLNew) == ERROR_SUCCESS) {

				if(SetSecurityInfo(hPhysMem, SE_KERNEL_OBJECT, DACL_SECURITY_INFORMATION, NULL, NULL,
								lpACLNew, NULL) != ERROR_SUCCESS)	{		
					NtClose(hPhysMem);
					return FALSE;
				}

				LocalFree(lpACLNew);
			} else { NtClose(hPhysMem); return FALSE; }

		} else { NtClose(hPhysMem); return FALSE; }

	}

	Status = NtOpenSection(&hPhysMem, SECTION_MAP_READ | SECTION_MAP_WRITE, &Attributes );

	if(!NT_SUCCESS(Status)) {
		printf( "Cannot open \\device\\physicalmemory");
		return FALSE;
	}

	return hPhysMem;
}

PHYSICAL_ADDRESS GetPhysicalAddress(ULONG vAddress) {
   PHYSICAL_ADDRESS  add;
   
   if (vAddress < 0x80000000L || vAddress >= 0xA0000000L)
      add.QuadPart = (ULONGLONG) vAddress & 0xFFFF000;
   else
      add.QuadPart = (ULONGLONG) vAddress & 0x1FFFF000;
   return(add);
}

DWORD GetHelp(void) {
		printf( "Help:\n"
			"\tq    :quit\n"
			"\ts    :reshow list of modified interrupt\n"
			"\tr X  :restore interruption X in IDT(sample: r 0xA1)\n"
			"\th    :show this help\n"
			"\n"
			);
return TRUE;
}
