/*++
SandMan framework.
Copyright 2008 (c) Matthieu Suiche. <msuiche[at]gmail.com>

This file is part of SandMan.

SandMan is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SandMan is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SandMan.  If not, see <http://www.gnu.org/licenses/>.

Module Name:

    hiber.h

Abstract:

    - The most interesting file. The whole hibernation file is documented here.
    - References:
        http://www.x86.org/secrets/dr7.htm
        http://www.x86.org/secrets/opcodes/icebp.htm
        http://www.x86.org/articles/pmcr/probemodecontrolregister.htm

Environment:

    - User mode

Revision History:

    - Matthieu Suiche

--*/

#ifndef _STRUCTS_
#define _STRUCTS_

#include <windows.h>

#ifdef __cplusplus
extern "C"
{
#endif

//
// Various defines
//

#define WAKE_MAGIC              "wake"      // If loaded 
#define LINK_MAGIC              "link"

//
// For Windows Vista and above, the signature tag is HIBR.
//
#define HIBR_MAGIC              "hibr"      // If unloaded
#define HIBR_VISTA_MAGIC        "HIBR"

#define RESTORED_MAGIC          0x00000000

#define MM_HIBER_PAGES          0x00000300
#define MM_HIBER_WIN2K_PAGES32  0x00000200

// 32-bit systems
#define PAGE_SIZE               0x00001000

#define MAX_ARRAY_ENTRY         0x000000FF

#define WIN32_WINNT_NT4         0x0400
#define WIN32_WINNT_WIN2K       0x0500
#define WIN32_WINNT_WINXP       0x0501
#define WIN32_WINNT_WS03        0x0502
#define WIN32_WINNT_VISTA       0x0600

/*++

            32bits Hibernation file looks like this : (Windows XP)

                 +------------------------------------+
        00000000 |                                    |
                 |       IMAGE_HIBER_NTX_HEADER32     |
                 |                                    |
                 +------------------------------------+
        00001000 |                                    |
                 |            FreeMapPages            |
                 |        (size = MM_HIBER_PAGES)     |
                 |           + page padding           |
                 +------------------------------------+
        00002000 |                                    |
                 |           KPROCESSOR_STATE32       |
                 |            (size = 0x320)          |
                 |            + page padding          |
                 |                                    |
                 +------------------------------------+
        00003000 |                                    |
                 |       MEMORY_RANGE_ARRAY_LINK      | ->---\
                 |                                    |      |
        00003010 |                                    |      |
                 |       MEMORY_RANGE_ARRAY_RANGE     | -----|-----\
                 |       (size = MAX_ARRAY_ENTRY)     |      |     |
                 |                                    |      |     |
                 +------------------------------------+      |     |
        00004000 |                                    |      |     |
                 |       XPRESS_BLOCK_HEADER          |      |     |     /----> \Device\PhysicalMemory
                 |                                    |      |     |     |
        00004020 |                                    |      |     |     |
                 |         COMPRESSED_DATA            | <----------/     |
                 |                                    | -----|-----------/
                 +------------------------------------+      |
        XXXXX000 |                                    |      |
                 |       MEMORY_RANGE_ARRAY_LINK      | <----/
                 |                                    |
        XXXXX010 |                                    |
                 |       MEMORY_RANGE_ARRAY_RANGE     |
                 |       (size = MAX_ARRAY_ENTRY)     |
                 |                                    |
                 +------------------------------------+
        XXXXY000 |                                    |
                 |       XPRESS_BLOCK_HEADER          |
                 |                                    |
        XXXXY020 |                                    |
                 |         COMPRESSED_DATA            |
                 |                                    |
                 +------------------------------------+
        ZZZZZZZZ |                                    |
                 |              PADDING               |
                 |                                    |
                 +------------------------------------+
                                --EOF--               
--*/

//
// HIBER_PERF struct is present in headers starting from Windows XP
//

typedef struct _HIBER_PERF
{
    LARGE_INTEGER IoTicks;
    LARGE_INTEGER InitTicks;
    LARGE_INTEGER CopyTicks;
    LARGE_INTEGER StartCount;
    ULONG ElapsedTime;
    ULONG IoTime;
    ULONG CopyTime;
    ULONG InitTime;
    ULONG PagesWritten;
    ULONG PagesProcessed;
    ULONG BytesCopied;
    ULONG DumpCount;
    ULONG FileRuns;
    LARGE_INTEGER ResumeAppStartTime;
    LARGE_INTEGER ResumeAppEndTime;
    LARGE_INTEGER HiberfilResumeTime;
} HIBER_PERF, *PHIBER_PERF;

//
// Master file header (offset 0)
//
typedef struct _IMAGE_HIBER_HEADER
{
    UCHAR Signature[4];
    ULONG Version;
    ULONG CheckSum;
    ULONG LengthSelf;
    ULONG PageSelf;
    ULONG PageSize;
#ifndef VISTA_TARGET
    ULONG64 ImageType;        // Field disappeared on Windows Vista and above !!!
#endif
    LARGE_INTEGER SystemTime;
    ULONG64 InterruptTime;
    ULONG FeatureFlags;
    ULONG HiberFlags;
    ULONG NoHiberPtes;
    ULONG HiberVa;
    LARGE_INTEGER HiberPte;
    ULONG NoFreePages;
    ULONG FreeMapCheck;
    ULONG WakeCheck;
    ULONG TotalPages;
    ULONG FirstTablePage;
    ULONG LastFilePage;
    //
    // Windows XP and above.
    //
    HIBER_PERF PerfInfo;
    //
    // Windows Vista
    //
    ULONG NoBootLoaderLogPages;
    ULONG BootLoaderLogPages[8];
    ULONG TotalPhysicalMemoryCount;
} IMAGE_HIBER_HEADER, IMAGE_HIBER_HEADER32, 
*PIMAGE_HIBER_HEADER, *PIMAGE_HIBER_HEADER32;

//
// Below are utility structures
//

typedef struct _PFN_NUMBER_TABLE
{
    ULONG PageFrameIndex[MM_HIBER_PAGES];
} PFN_NUMBER_TABLE, *PPFN_NUMBER_TABLE;

/*++
// included from "winnt.h"
typedef struct _FLOATING_SAVE_AREA
{
    ULONG ControlWord;
    ULONG StatusWord;
    ULONG TagWord;
    ULONG ErrorOffset;
    ULONG ErrorSelector;
    ULONG DataOffset;
    ULONG DataSelector;
    UCHAR RegisterArea[80];
    ULONG Cr0NpxState;
} FLOATING_SAVE_AREA, *PFLOATING_SAVE_AREA;


typedef struct _CONTEXT
{
    ULONG ContextFlags;
    ULONG Dr0;
    ULONG Dr1;
    ULONG Dr2;
    ULONG Dr3;
    ULONG Dr6;
    ULONG Dr7;
    FLOATING_SAVE_AREA FloatSave;
    ULONG SegGs;
    ULONG SegFs;
    ULONG SegEs;
    ULONG SegDs;
    ULONG Edi;
    ULONG Esi;
    ULONG Ebx;
    ULONG Edx;
    ULONG Ecx;
    ULONG Eax;
    ULONG Ebp;
    ULONG Eip;
    ULONG SegCs;
    ULONG EFlags;
    ULONG Esp;
    ULONG SegSs;
    UCHAR ExtendedRegisters[512];
} CONTEXT, *PCONTEXT;
--*/

typedef struct _DESCRIPTOR
{
    USHORT Pad;
    USHORT Limit;
    ULONG Base;
} DESCRIPTOR, *PDESCRIPTOR;

typedef struct _KSPECIAL_REGISTERS
{
    union 
    {
    ULONG Cr0;
        struct
        {
            ULONG u00:31;
            ULONG Paging:1;             // 31
        };
    } u_cr0;
    ULONG Cr2;
    //
    // Magic Forensic register.
    //
    ULONG Cr3;
    union
    {
    ULONG Cr4;
        struct
        {
            ULONG u00:4;
            ULONG PageSizeExtensions:1;             // 4
            ULONG PhysicalAddressExtension:1;       // 5
            ULONG MachineCheck:1;                   // 6
            ULONG PageGlobal:1;                     // 7
            ULONG PerformanceMonitoringCounter:1;   // 8
            ULONG FxSupport:1;                      // 9
            ULONG MmxException:1;                   // 10
            ULONG u11b:2;                           // 11
            ULONG VirtualMachineExtension:1;        // 12
        };
    } u_cr4;
    ULONG KernelDr0;
    ULONG KernelDr1;
    ULONG KernelDr2;
    ULONG KernelDr3;
    ULONG KernelDr6;
    union
    {
    ULONG KernelDr7;
        struct
        {
            UCHAR    LocalDr0:1;             //  0
            UCHAR    GlobalDr0:1;            //  1
            UCHAR    LocalDr1:1;             //  2
            UCHAR    GlobalDr1:1;            //  3
            UCHAR    LocalDr2:1;             //  4
            UCHAR    GlobalDr2:1;            //  5
            UCHAR    LocalDr3:1;             //  6
            UCHAR    GlobalDr3:1;            //  7
            UCHAR    LocalExact:1;           //  8
            UCHAR    GlobalExact:1;          //  9
            UCHAR    Reserved0:2;            // 10
            /*++
            - IceBp  1=INT01 causes emulator to break emulation / 0=CPU handles INT01
            Interrupt Redirection = When set, causes the emulator to break execution when any 
                breakpoint condition occurs. These conditions include debug register breakpoints, 
            TSS breakpoints, and the undocumented instruction ICEBP.
            --*/
            UCHAR    InterruptRedirection:1; // 12
            /*++
            - Global Debug. This bit has enjoyed an on-again, off-again relationship with Intel 
                documenteers. It all depends on which data book you get, and where you look. 
            If you look in early 80386 data books, it is described. But if you look in the 
                80386 Programmer's Reference Manual, it is omitted. Suit yourself, Intel.
            --*/
            UCHAR    GlobalDebug:1;          // 13
            /*++
            - TRace1 1=Generate special address cycles after code dis-continuities.
            On Pentium, these cycles are called Branch Trace Messages.
            Trace1= When set, the CPU generates a special cycle each time a code discontinuity 
                occurs.
            The ICE reads this special cycle and stores it in the trace data. This address helps
                the CPU reconstruct code sequences from the trace data -- since all fetch 
                discontinuity are logged by the setting of this bit. See also Branch Trace Messages 
                in Pentium TR12.
            --*/
            UCHAR    TRace1:1;               // 14
            /*++
            - Trace2 1=Unknown.
            Trace2= I don't know the exact purpose of this bit, but somehow it governs trace collection.
            By clearing this bit and resuming emulation (on an ICE) until the trace buffer is full, one 
            can observe that the ICE is unable to reconstruct the execution trace for many hundreds, and 
            sometimes thousands of CPU cycles.
            --*/
            UCHAR    TRace2:1;               // 15
            UCHAR    ConditionDr0:2;
            UCHAR    LenghtDr0:2;
            UCHAR    ConditionDr1:2;
            UCHAR    LenghtDr1:2;
            UCHAR    ConditionDr2:2;
            UCHAR    LenghtDr2:2;
            UCHAR    ConditionDr3:2;
            UCHAR    LenghtDr3:2;
        };
    };
    DESCRIPTOR Gdtr;
    DESCRIPTOR Idtr;
    USHORT Tr;
    USHORT Ldtr;
    ULONG Reserved[6];
} KSPECIAL_REGISTERS, *PKSPECIAL_REGISTERS;

typedef struct _KPROCESSOR_STATE32
{
    CONTEXT ContextFrame;
    KSPECIAL_REGISTERS SpecialRegisters;
} KPROCESSOR_STATE32, *PKPROCESSOR_STATE32;

typedef struct _MEMORY_RANGE_ARRAY_LINK
{
    ULONG Next;         // ???
    ULONG NextTable;    // (NextTable * PAGE_SIZE) is file offset of next array
    ULONG CheckSum;     // Always zero
    ULONG EntryCount;   // 255 in practice, for the whole thing to fit in 1 page (254 entries + header)
} MEMORY_RANGE_ARRAY_LINK, *PMEMORY_RANGE_ARRAY_LINK;

typedef struct _MEMORY_RANGE_ARRAY_RANGE
{
    ULONG PageNo;       // ???
    ULONG StartPage;    // Block start (physical address)
    ULONG EndPage;      // Block stop (physical address)
    ULONG CheckSum;     // Always zero, but used under Windows 2000.
} MEMORY_RANGE_ARRAY_RANGE, *PMEMORY_RANGE_ARRAY_RANGE;


typedef struct _MEMORY_RANGE_ARRAY
{
    MEMORY_RANGE_ARRAY_LINK MemArrayLink;
    MEMORY_RANGE_ARRAY_RANGE MemArrayRange[MAX_ARRAY_ENTRY];
} MEMORY_RANGE_ARRAY, *PMEMORY_RANGE_ARRAY;

//
// Xpress block definition.
//
typedef struct _IMAGE_XPRESS_HEADER {   /* sizeof() = 0x20 */

    UCHAR Signature[0x08];           // 0x000
    struct {
        UCHAR u08;                   // 0x008
        UCHAR u09;
        UCHAR u0A;
        UCHAR u0B;
    } Size;
    ULONG u0C;                       // 0x00C
    ULONG Checksum;                  // 0x010
    ULONG u14;                       // 0x014
    ULONG u18;                       // 0x018
    ULONG u1C;                       // 0x01C

} IMAGE_XPRESS_HEADER, *PIMAGE_XPRESS_HEADER;

#ifdef __cplusplus
}
#endif

#endif