/*++
SandMan framework.
Copyright 2008 (c) Matthieu Suiche. <msuiche[at]gmail.com>

This file is part of SandMan.

SandMan is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SandMan is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SandMan.  If not, see <http://www.gnu.org/licenses/>.

Module Name:

    compression.c

Abstract:

    - Windows hibernation functions implementation.
    - BEWARE: Very ugly code. Sorry.

Environment:

    - User mode

Revision History:

    - 04-08-2008 Final version.
    - 06-23-2008 New implementation of Xpress encode and decode.(msuiche)
    - Matthieu Suiche

--*/

#include "compression.h"

//
// Variable for unimplemented use.
//
#ifndef _XPRESS_DEPENDENCY
#define _XPRESS_DEPENDENCY
#endif

/*++
Function Name: XpressWriteMetaData

Overview:

Parameters:
        -

Return Values:
        -
--*/
ULONG XpressWriteMetaData(IN ULONG Size, 
                          IN ULONG Offset,
                          OUT PUCHAR Buffer,
                          IN ULONG Index,
                          IN ULONG Nibble)
{
ULONG NumberOfBytesWritten;
USHORT Metadata;
ULONG Length;
PSHORT Destination;

    NumberOfBytesWritten = 0;
    Destination = (PUSHORT)&Buffer[Index];
    Length = Size;
    Length -= 3;
    Offset--;

    if (Length < 7)
    {
        //
        // Classical meta-data
        //
        Metadata = (USHORT)((Offset << 3) | Length);
        Destination[NumberOfBytesWritten / sizeof(USHORT)] = Metadata;
        NumberOfBytesWritten += sizeof(USHORT);
    }
    else
    {
        Metadata = (USHORT)((Offset << 3) | 7);

        Destination[NumberOfBytesWritten / sizeof(USHORT)] = Metadata;
        NumberOfBytesWritten = sizeof(USHORT);
        
        if ((Length + 3) < (15 + 7 + 3))
        {
            //
            // Shared byte
            //
            if (!Nibble) 
            {
                Buffer[Index + NumberOfBytesWritten] |= LOBYTE(Length - 7);
                NumberOfBytesWritten += sizeof(UCHAR);
            }
            else
            {
                Buffer[Nibble] |= LOBYTE(Length - 7) << 4;
            }
            

        }
        else if ((Length + 3) < (3 + 7 + 15 + 255))
        {
            //
            // Shared byte
            //
            if (!Nibble) 
            {
                Buffer[Index + NumberOfBytesWritten] |= 15;
                NumberOfBytesWritten += sizeof(UCHAR);
            }
            else 
            {
                Buffer[Nibble] |= 15 << 4;
            }

            //
            // Additionnal len
            //
            if (!Nibble) Buffer[Index + NumberOfBytesWritten] = LOBYTE(Length - (7 + 15));
            else Buffer[Index + NumberOfBytesWritten] = LOBYTE(Length - (7 + 15));

            NumberOfBytesWritten += sizeof(UCHAR);

        }
        else
        {
            //
            // Shared byte
            //
            if (!Nibble)
            {
                Buffer[Index + NumberOfBytesWritten] |= 15;
                NumberOfBytesWritten += sizeof(UCHAR);
            }
            else 
            {
                Buffer[Nibble] |= 15 << 4;
            }

            //
            // Additionnal len
            //
            Buffer[Index + NumberOfBytesWritten] = 255;

            NumberOfBytesWritten += sizeof(UCHAR);

            //Destination[NumberOfBytesWritten / sizeof(USHORT)] = LOWORD(Length);
            //printf("LENLEN : %d (%08X) -> 0x%02X%02X\n", Length, LOBYTE(Length), HIBYTE(Length));
            Buffer[Index + NumberOfBytesWritten] = LOBYTE(Length);
            Buffer[Index + NumberOfBytesWritten + 1] = HIBYTE(Length);

            NumberOfBytesWritten += sizeof(USHORT);
        }

    }
    return NumberOfBytesWritten;
}


/*++
Function Name: XpressWriteIndicator

Overview:

Parameters:
        -

Return Values:
        -
--*/
VOID XpressWriteIndicator(OUT PUCHAR *IndicatorOffset,
                          OUT PULONG Indicator,
                          OUT PUCHAR OutputBuffer,
                          OUT PULONG OutputPosition)
{
    *(PULONG)(*IndicatorOffset) = *Indicator;
    *Indicator = 0;
    *IndicatorOffset = &OutputBuffer[*OutputPosition];
    *OutputPosition += sizeof(ULONG);

    if (*Indicator != XPRESS_ENCODE_MAGIC) return;
}

/*++
Function Name: Xpress_Compress

Overview:

Parameters:
        -

Return Values:
        -
--*/
ULONG Xpress_Compress(PUCHAR UncompressedBuffer,
                      ULONG UncompressedSize,
                      PUCHAR CompressedBuffer,
                      ULONG CompressedSize)
{
ULONG UncompressedPosition, CompressedPosition, BytesLeft;
ULONG MaxOffset, BestOffset;
LONG Offset;
ULONG MaxLength, Length, BestLength;
PUCHAR String1, String2;
ULONG Indicator;
PUCHAR IndicatorPosition;
ULONG IndicatorBit, NibbleIndex;
ULONG MetadataSize;

    if (!UncompressedSize) return 0;

    UncompressedPosition = 0;
    Indicator = 0;
    CompressedPosition = sizeof(ULONG);
    IndicatorPosition = &CompressedBuffer[0];

    BytesLeft = UncompressedSize;
    IndicatorBit = 0;
    NibbleIndex = 0;

    while (BytesLeft > 3) 
    {
        if (UncompressedPosition > UNCOMPRESSED_BLOCK_SIZE) MaxOffset = UNCOMPRESSED_BLOCK_SIZE;
        else MaxOffset = UncompressedPosition;

        String1 = &UncompressedBuffer[UncompressedPosition];

        BestLength = 2;
        BestOffset = 0;

        for (Offset = 1; (ULONG)Offset <= MaxOffset; Offset++)
        {
            String2 = &String1[-Offset];

            if ((String1[0] == String2[0]) &&
                (String1[BestLength] == String2[BestLength]))
            {
                MaxLength = min(BytesLeft,(ULONG)Offset);

                if (Offset == 1)
                {
                    if ((String1[0] == String1[1]) && (String1[0] == String1[2]))
                    {
                        for (Length = 0; (Length < BytesLeft) && (String1[0] == String1[Length]); ++Length);

                        if (Length > BestLength)
                        {
                            BestLength = Length;
                            BestOffset = 1;
                        }
                    }
                }

                for (Length = 0; (Length < MaxLength) && (String1[Length] == String2[Length]); Length++);

                if (Length > BestLength )
                {
                    BestLength = Length;
                    BestOffset = Offset;
                }
            }
         }

        if( (BestLength >= 3) && (BestOffset <= 0x1FFF) && (BestLength < BytesLeft))
        {
            //if (BestLength > 280) printf("COMP DEBUG: [0x%08X]->[0x%08X] Len: %d Offset: %08X\n", 
            //    UncompressedPosition, CompressedPosition, BestLength, BestOffset);
            MetadataSize = XpressWriteMetaData(BestLength, 
                                               BestOffset, 
                                               CompressedBuffer, 
                                               CompressedPosition, 
                                               NibbleIndex);
            
            Indicator |= 1 << (32 - ((IndicatorBit % 32) + 1));

            if (BestLength > 9) 
            {
                if (NibbleIndex == 0) NibbleIndex = CompressedPosition + sizeof(USHORT);
                else NibbleIndex = 0;
            }

            CompressedPosition += MetadataSize;

            UncompressedPosition += BestLength;
            BytesLeft -= BestLength;
        }
        else
        {
            CompressedBuffer[CompressedPosition++] = UncompressedBuffer[UncompressedPosition++];
            BytesLeft--;
        }

        IndicatorBit++;

        if ((IndicatorBit - 1) % 32 > (IndicatorBit % 32))
        {
            XpressWriteIndicator(&IndicatorPosition,
                                 &Indicator,
                                 CompressedBuffer,
                                 &CompressedPosition);
        }
    }

    while (UncompressedPosition < UncompressedSize)
    {
        CompressedBuffer[CompressedPosition] = UncompressedBuffer[UncompressedPosition];
        IndicatorBit++;
        UncompressedPosition++;
        CompressedPosition++;
    }

    if ((IndicatorBit % 32) > 0) 
    {
        for (IndicatorBit; (IndicatorBit % 32) != 0; IndicatorBit++)
        {
            Indicator |= 1 << (32 - ((IndicatorBit % 32) + 1));
        }
        XpressWriteIndicator(&IndicatorPosition,
                              &Indicator,
                              CompressedBuffer,
                              &CompressedPosition);
    }

    return CompressedPosition;
}

/*++
Function Name: Xpress_Decompress

Overview:

Parameters:
        -

Return Values:
        -
--*/
ULONG Xpress_Decompress(PUCHAR InputBuffer,
                      PUCHAR OutputBuffer,
                      ULONG OutputSize)
{
ULONG OutputIndex, InputIndex;
ULONG Indicator, IndicatorBit;
ULONG Length;
ULONG Offset;
ULONG NibbleIndex;
ULONG NibbleIndicator;

OutputIndex = 0; 
InputIndex = 0; 
Indicator = 0; 
IndicatorBit = 0; 
Length = 0; 
Offset = 0;
NibbleIndex = 0; 
NibbleIndicator = XPRESS_ENCODE_MAGIC;

    while (OutputIndex < OutputSize) 
    {
        if (IndicatorBit == 0)
        {
            Indicator = (InputBuffer[InputIndex + 3] << 24) | (InputBuffer[InputIndex + 2] << 16) |
                        (InputBuffer[InputIndex + 1] <<  8) | InputBuffer[InputIndex];
            InputIndex += sizeof(ULONG);
            IndicatorBit = 32; 
        }
        IndicatorBit--;

    //* check whether the bit specified by IndicatorBit is set or not 
    //* set in Indicator. For example, if IndicatorBit has value 4 
    //* check whether the 4th bit of the value in Indicator is set 

        if (((Indicator >> IndicatorBit) & 1) == 0)
        {
            OutputBuffer[OutputIndex] = InputBuffer[InputIndex]; 
            InputIndex += sizeof(UCHAR);
            OutputIndex += sizeof(UCHAR);
        }
        else 
        {
            Length = (InputBuffer[InputIndex + 1] << 8) | InputBuffer[InputIndex];
            
            /*
            if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0))
            {
                printf("DECOMP: READ AT [0x%08X] = %04X \n", InputIndex, Length);
            }
            */
            InputIndex += sizeof(USHORT); 
            Offset = Length / 8;
            Length = Length % 8;
            //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--1 Len: %02X (%d)\n", Length, Length);
            if (Length == 7)
            {
                if (NibbleIndex == 0)
                {
                    NibbleIndex = InputIndex;
                    Length = InputBuffer[InputIndex] % 16; 
                    //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--2 Len: %02X (%d)\n", Length, Length);
                    InputIndex += sizeof(UCHAR);
                }
                else 
                {
                    Length = InputBuffer[NibbleIndex] / 16;
                    //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--3 Len: %02X (%d)\n", Length, Length);
                    NibbleIndex = 0;
                }

                if (Length == 15)
                {
                    Length = InputBuffer[InputIndex];
                    //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--4 Len: %02X (%d)\n", Length, Length);
                    InputIndex += sizeof(UCHAR); 
                        if (Length == 255)
                        {
                            Length = (InputBuffer[InputIndex + 1] << 8) | InputBuffer[InputIndex];
                            InputIndex += sizeof(USHORT);
                            Length -= (15 + 7); 
                        }
                    Length += 15; 
                    //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--5 Len: %02X (%d)\n", Length, Length);
                }
                Length += 7;
                //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--6 Len: %02X (%d)\n", Length, Length);
            }

            Length += 3;
            //if ((OutputIndex > 0xD0) && (OutputIndex < 0xF0)) printf("--7 Len: %02X (%d)\n", Length, Length);
            //if (Length > 280) printf("DECOMP DEBUG: [0x%08X]->[0x%08X] Len: %d Offset: %08X\n", 
            //    OutputIndex, InputIndex, Length, Offset);
            while (Length != 0) 
            {
                if ((OutputIndex >= OutputSize) || ((Offset + 1) >= OutputIndex)) break;
                OutputBuffer[OutputIndex] = OutputBuffer[OutputIndex - Offset - 1];
                OutputIndex += sizeof(UCHAR);
                Length -= sizeof(UCHAR);
            }
        }

    }

    return OutputIndex;
}

/*++
Function Name: XpressEncode

Overview:

Parameters:
            -

Return Values:
            -

--*/
ULONG
XpressEncode(PUCHAR InputBuffer, 
             ULONG InputSize, 
             PUCHAR OutputBuffer, 
             ULONG OutputSize)
{
    return Xpress_Compress(InputBuffer, 
                           InputSize, 
                           OutputBuffer, 
                           OutputSize);
}

/*++
Function Name: XpressDecode

Overview:

Parameters:
         -

Return Values:
        - Uncompressed bytes.

--*/
ULONG
XpressDecode(PUCHAR InputBuffer,
             ULONG InputSize, 
             PUCHAR OutputBuffer, 
             ULONG OutputSize)
{
    //
    // Xpress = LZ77 + DIRECT2
    // LZ77 Algorithm is a bit modified, because of DIRECT2 metadata only identical
    // string of a minimum len of 3 are considered.
    // This algorithm has been initially implemented by the Microsoft Exchange Team.
    //

    //
    // We wipe the buffer.
    //
    RtlZeroMemory(OutputBuffer, OutputSize);

    return Xpress_Decompress(InputBuffer, OutputBuffer, OutputSize);
}

/*
Function Name: GetXpressBlockSize

Overview:
            - This functions aims to read the Size dword, and convert it
              into an human readable size format.

Parameters:
            - PIMAGE_XPRESS_HEADER pIXH

Return Values:
            -
*/
ULONG
GetXpressBlockSize(PIMAGE_XPRESS_HEADER pIXH
                   )
{
ULONG Size;

    Size = (pIXH->Size.u0B << 24)
         + (pIXH->Size.u0A << 16)
         + (pIXH->Size.u09 << 8);
    Size >>= 10;
    Size += 1;


    if ((Size % 8) == 0)
        return Size;
    return (Size & ~7) + 8;
}

/*
Function Name: GetXpressBlockSize

Overview:
            - This functions aims to write the Size dword

Parameters:
            - PIMAGE_XPRESS_HEADER pIXH
            - ULONG

Return Values:
            -
*/
VOID
SetXpressBlockSize(PIMAGE_XPRESS_HEADER pIXH, 
                   ULONG Size)
{
    Size--;
    Size <<= 10;
    pIXH->Size.u0B = LOBYTE(Size >> 24);
    pIXH->Size.u0A = LOBYTE(Size >> 16);
    pIXH->Size.u09 = LOBYTE(Size >> 8);
}
