/*++                                                
SandMan framework.
Copyright 2008 (c) Matthieu Suiche. <msuiche[at]gmail.com>

This file is part of SandMan.

SandMan is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SandMan is distributed in the hope that it will be usef,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You shod have received a copy of the GNU General Public License
along with SandMan.  If not, see <http://www.gnu.org/licenses/>.

Mode Name:

    hiber.c

Abstract:

    - Main implementation.

Environment:

    - User mode

Revision History:
    - 23-06-2008: Implementation of new xpress functions (msuiche)
    - Matthieu Suiche, Nicolas Ruff

--*/
#include "sandman.h"
#include "compression.h"
#include "hiber.h"
#include "mm.h"
#include "checksum.h"
#include "internals.h"

UCHAR NlPage[PAGE_SIZE] = {0};

/*++
Function Name: HiberOpen

Overview:
    - Opens an hibernation file and fills a SANDMAN_OBJECT struct.
    - Function will fail if file cannot be opened *or* contains invalid data

Parameters:
    - PUCHAR Filename[]

Return Values:
    - NULL if open failed. Use GetLastError() to retrieve last error.
    - Pointer to an allocated SANDMAN_OBJECT otherwise. The caller must call HiberClose() to free memory.

--*/

PSANDMAN_OBJECT 
HiberOpen( UCHAR FileName[] )
{
PSANDMAN_OBJECT s;
ULONG FirstPageTable;
ULONG KProcPage;

    if (FileName == NULL) return NULL;

    //
    // Allocates a new, empty SANDMAN_OBJECT
    //
    s = VirtualAlloc(NULL,
            sizeof(SANDMAN_OBJECT),
            MEM_COMMIT | MEM_RESERVE,
            PAGE_READWRITE);

    if (s == NULL) return NULL;

    RtlZeroMemory(s, sizeof(SANDMAN_OBJECT));

    #if (_MSC_VER >= 1400) 
        strncpy_s( s->FileName, _countof( s->FileName), FileName, MAX_PATH-1 );
    #else
        strncpy( s->FileName, FileName, MAX_PATH-1 );
    #endif

    s->FileName[MAX_PATH] = '\0';

    //
    // Try to open file
    //
    s->hFile = CreateFileA(FileName, 
                     GENERIC_READ|GENERIC_WRITE, 
                     FILE_SHARE_READ,
                     NULL,
                     OPEN_ALWAYS,
                     FILE_ATTRIBUTE_NORMAL,
                     NULL);

    if (s->hFile == INVALID_HANDLE_VALUE)
    {
        //
        // Catch error.
        //
        VirtualFree( s, 0, MEM_RELEASE );
        return FALSE;
    }

    //
    // Create file mapping.
    //
    s->hMap = CreateFileMapping(s->hFile,
                                0,
                                PAGE_READWRITE,
                                0,
                                0,
                                0);

    if (s->hMap == NULL)
    {
        //
        // Catch error.
        //
        CloseHandle( s->hFile );

        VirtualFree( s, 0, MEM_RELEASE );
        return FALSE;
    }

    s->Map = MapViewOfFile(s->hMap,
                            FILE_MAP_ALL_ACCESS,
                            0,
                            0,
                            0);

    if (s->Map == NULL)
    {
        CloseHandle( s->hMap );
        CloseHandle( s->hFile );
        VirtualFree( s, 0, MEM_RELEASE );
        return FALSE;
    }

    //
    // Create an alias to file header
    //
    s->FileHdr = (PIMAGE_HIBER_HEADER)s->Map;

    //
    // Yes, it's correct of the 4 first bytes are erased, Sandman will not open
    // the hibernation file.
    //
    if (_strnicmp(s->FileHdr->Signature, "hibr", 4) == 0)
    {
        //
        // Create an alias to 1st element in Memory Array list
        // There is field called FirstPageTable in every Hibernation file header structure,
        // the problem is that the structure is most of time different. Then, we have to 
        // execute a home made routine that determine this field manually.
        //
        for (FirstPageTable = 0; FirstPageTable < 10; FirstPageTable++)
        {
            if (memcmp(s->Map + (FirstPageTable * PAGE_SIZE), XPRESS_MAGIC, 8) == 0) break;
        }

        //
        // Note: We only scan the 10 first pages. It's enough for 32bits hibernation file.
        //
        if (FirstPageTable == 10) return FALSE;

        s->MemArray = (PMEMORY_RANGE_ARRAY) (s->Map + (FirstPageTable - 1) * PAGE_SIZE);

        for (KProcPage = 1; KProcPage < FirstPageTable - 1; KProcPage++)
        {
            //
            // Create an alias to processor state
            //
            s->ProcState = (PKPROCESSOR_STATE32) (s->Map + KProcPage * PAGE_SIZE);

            //
            // Kernel registers finger printing
            //
            if ((s->ProcState->ContextFrame.SegDs == 0x23) &&
                (s->ProcState->ContextFrame.SegEs == 0x23) &&
                (s->ProcState->ContextFrame.SegFs == 0x30))
            {
                break;
            }
        }

        if (KProcPage == (FirstPageTable - 1)) return FALSE;

        //
        // Create the tree to be able to go back to the roots.
        //
        HiberCreateTree(s);
    }

    return s;
}


/*++
Function Name: HiberClose

Overview:
    - Frees all resources allocated by HiberOpen()

Parameters:
    - PSANDMAN_OBJECT SandmanObject

Return Values:
    - None. SANDMAN_OBJECT is released and shall not be reused after close.

--*/
VOID HiberClose(PSANDMAN_OBJECT Hiberfil)
{
    if (Hiberfil == NULL) return;

    if (Hiberfil->Map != NULL) UnmapViewOfFile(Hiberfil->Map);

    if (Hiberfil->hMap != NULL) CloseHandle(Hiberfil->hMap);

    if (Hiberfil->hFile != INVALID_HANDLE_VALUE) CloseHandle(Hiberfil->hFile);

    VirtualFree( Hiberfil, 0, MEM_RELEASE );
}

/*++
Function Name: HiberCreateFile

Overview:
         - Used by HiberBuildPhysicalMemoryDump()

Parameters:
         - PUCHAR FileName
         - ULONG Size

Return Values:

--*/
ULONG
HiberCreateFile(PUCHAR Filename,
                ULONG Size
                )
{
ULONG NumberOfBytesWritten;
ULONG Page;
PUCHAR pBuffer;
HANDLE hFile;

    if ((Size % PAGE_SIZE) !=  0) 
    {
        printf("SANDY: Bad file size (HiberCreateFile)\n");
        return FALSE;
    }

    hFile = CreateFileA(Filename,
                 GENERIC_READ|GENERIC_WRITE, 
                 FILE_SHARE_READ,
                 NULL,
                 OPEN_ALWAYS,
                 FILE_ATTRIBUTE_NORMAL,
                 NULL);

    if (!hFile)
    {
        printf("SANDY: CreateFile(%s)\n", Filename);
        return FALSE;
    }

    pBuffer = VirtualAlloc(NULL,
                  PAGE_SIZE,
                  MEM_COMMIT | MEM_RESERVE,
                  PAGE_READWRITE);

    for (Page = 0; Page < (Size / PAGE_SIZE); Page++)
    {
        WriteFile(hFile, pBuffer, PAGE_SIZE, &NumberOfBytesWritten, NULL);
    }

    VirtualFree(pBuffer, 0, MEM_RELEASE );

    CloseHandle(hFile);

    return TRUE;
}

/*++
Function Name: HiberBuildPhysicalMemoryDump

Overview:
        - Build the physical memory dump.

Parameters:
        - PSANDMAN_OBJECT Hiberfil,
        - UCHAR DumpFilename[]

Return Values:
        - TRUE/FALSE
--*/
ULONG
HiberBuildPhysicalMemoryDump(PSANDMAN_OBJECT Hiberfil,
                             UCHAR DumpFilename[])
{
PSANDMAN_OBJECT pDumpfile;
PSANDMAN_TREE PageTree;

PUCHAR pPage;

ULONG BlockSize;

    HiberCreateFile(DumpFilename, HiberGetPhysicalMemorySize(Hiberfil));

    pDumpfile = HiberOpen(DumpFilename);

    PageTree = HiberGetPageFirst(Hiberfil);

    pPage = VirtualAlloc(NULL,
                 UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE,
                 MEM_COMMIT | MEM_RESERVE,
                 PAGE_READWRITE);

    if (!pPage) return FALSE;

    do
    {
        if (PageTree->XpressPage == 0)
        {
            RtlZeroMemory(pPage, UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE);

            BlockSize = GetXpressBlockSize(PageTree->ImgXpressHeader);

            XpressDecode((PUCHAR)PageTree->ImgXpressHeader + sizeof(IMAGE_XPRESS_HEADER),
                         BlockSize,
                         pPage,
                         UNCOMPRESSED_BLOCK_SIZE);

        }

        memcpy(&pDumpfile->Map[ PageTree->PageNumber * PAGE_SIZE ],
               &pPage[ PageTree->XpressPage * PAGE_SIZE ],
               PAGE_SIZE);

    } while (PageTree = HiberGetPageNext(PageTree));

    VirtualFree(pPage, 0, MEM_RELEASE);

    return TRUE;
}

/*++
Function Name: HiberGetPhysicalMemorySize

Overview:

Parameters:
         - PSANDMAN_OBJECT Hiberfil

Return Values:
        - Physical memory size.
--*/
ULONG
HiberGetPhysicalMemorySize(PSANDMAN_OBJECT Hiberfil
                           )
{
    return Hiberfil->HighestPage * PAGE_SIZE;
}

/*++
Function Name: HiberGetVersion

Overview:
       - Retrieves the version number of the target operating system.

Parameters:
       - PSANDMAN_OBJECT Hiberfil

Return Values:
       - If the function succeeds, the return value includes the major and minor
         version numbers of the operating system in the low-order word, and information
         about the operating system platform in the high-order word.

         For all platforms, the low-order word contains the version number of the operating
         system. The low-order byte of this word specifies the major version number, in 
         hexadecimal notation. The high-order byte specifies the minor version (revision)
         number, in hexadecimal notation. The high-order bit is zero, the next 7 bits 
         represent the build number, and the low-order byte is 5.
--*/
ULONG
HiberGetVersion(PSANDMAN_OBJECT Hiberfil)
{
ULONG NtTibAddress, PebAddress;
PPEB pPEB;

KGDTENTRY NtTibEntry;
PKGDTENTRY GdtTable;
ULONG GdtIndex;
DESCRIPTOR Gdtr;

PKPROCESSOR_STATE32 pPS;

PUCHAR pData;

ULONG Version;

    pPS = Hiberfil->ProcState;

    // Ring3 FS is 3B, (> Windows 2000)
    //ProcessorState->ContextFrame.SegFs = 0x3B;

    GdtIndex = (0x3B >> 3);

    Gdtr = pPS->SpecialRegisters.Gdtr;

    GdtTable = VirtualAlloc(NULL,
                            PAGE_SIZE,
                            MEM_COMMIT | MEM_RESERVE,
                            PAGE_READWRITE);

    GdtTable = (PKGDTENTRY)HiberGetPageAtVirtualAddress(
                                Hiberfil, 
                                Gdtr.Base);

    memcpy(&NtTibEntry, &GdtTable[ GdtIndex ], 8);

    NtTibAddress = (NtTibEntry.BaseLow)
                 | (NtTibEntry.BaseMid << (2 * 8))
                 | (NtTibEntry.BaseHigh << (3 * 8));

    //
    // Check if TIB Address belong to user-space.
    //
    if ((NtTibAddress == 0) || (NtTibAddress > 0x80000000))
    {
        return FALSE;
    }

    pData = HiberGetPageAtVirtualAddress(Hiberfil, NtTibAddress);

    PebAddress = *(PULONG)(pData + 0x30);

    pPEB = (PPEB)HiberGetPageAtVirtualAddress(Hiberfil, PebAddress);

    Version = MAKEWORD(pPEB->OSMajorVersion, pPEB->OSMinorVersion);
    Version |= ((ULONG)(pPEB->OSBuildNumber)) << 16;

    return Version;
}

/*++
Function Name: HiberReadFileHeader

Overview:

Parameters:
         - PSANDMAN_OBJECT Hiberfil,
           PIMAGE_HIBER_HEADER32 HiberHeader

Return Values:

--*/
ULONG
HiberReadFileHeader(PSANDMAN_OBJECT Hiberfil,
                    PIMAGE_HIBER_HEADER32 HiberHeader)
{
    memcpy(HiberHeader, Hiberfil->Map, sizeof(IMAGE_HIBER_HEADER32));

    return TRUE;
}

/*++
Function Name: HiberWriteFileHeader

Overview:

Parameters:
         - PSANDMAN_OBJECT Hiberfil,
           PIMAGE_HIBER_HEADER32 HiberHeader

Return Values:

--*/
ULONG
HiberWriteFileHeader(PSANDMAN_OBJECT Hiberfil,
                     PIMAGE_HIBER_HEADER32 HiberHeader
                    )
{

    memcpy(Hiberfil->FileHdr, HiberHeader, sizeof(IMAGE_HIBER_HEADER32));

    //
    // Don't forget the checksum in the file header.
    //
    Hiberfil->FileHdr->CheckSum = tcpxsum(0, 
                                    (PUCHAR)Hiberfil->FileHdr, 
                                    Hiberfil->FileHdr->LengthSelf);
    return TRUE;
}

/*++
Function Name: HiberReadProcState

Overview:

Parameters:
         - PSANDMAN_OBJECT Hiberfil
         - PKPROCESSOR_STATE32 ProcessorState

Return Values:

--*/
ULONG
HiberReadProcState(PSANDMAN_OBJECT Hiberfil,
                    PKPROCESSOR_STATE32 ProcessorState
                    )
{

    memcpy(ProcessorState, Hiberfil->ProcState, sizeof(KPROCESSOR_STATE32));

    return TRUE;
}

/*++
Function Name: HiberWriteProcState

Overview:

Parameters:
         - PSANDMAN_OBJECT Hiberfil
         - PKPROCESSOR_STATE32 ProcessorState

Return Values:

--*/
ULONG
HiberWriteProcState(PSANDMAN_OBJECT Hiberfil,
                    PKPROCESSOR_STATE32 ProcessorState
                    )
{

    //
    // WakeCheck field. = Checksum
    //
    memcpy(Hiberfil->ProcState, ProcessorState, sizeof(KPROCESSOR_STATE32));

    return TRUE;
}

/*++
Function Name: HiberGetPageFirst

Overview:

Parameters:
         - PSANDMAN_OBJECT Hiberfil

Return Values:

--*/
PSANDMAN_TREE
HiberGetPageFirst(
    PSANDMAN_OBJECT Hiberfil 
)
{
    return &Hiberfil->PagesListHead[0];
}

/*++
Function Name: HiberGetPageNext

Overview:

Parameters:
         - PSANDMAN_TREE PagesList

Return Values:
        - NULL if there is no next entry.
--*/
PSANDMAN_TREE
HiberGetPageNext(
    PSANDMAN_TREE PagesList
)
{
    if (PagesList[1].PageNumber == 0) return NULL;

    return &PagesList[1];
}

/*++
Function Name: HiberIsPagePresent

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil
            - PHYSICAL_ADDRESS PhysicalAddress

Return Values:

--*/
BOOLEAN 
HiberIsPagePresent(
    PSANDMAN_OBJECT Hiberfil,
    PHYSICAL_ADDRESS PhysicalAddress
)
{
PSANDMAN_TREE PageTree;

ULONG PageNumber;

    PageNumber = GetPage(PhysicalAddress);

    PageTree = HiberGetPageFirst(Hiberfil);

    do
    {

        if (PageTree->PageNumber == PageNumber) return TRUE;

    } while (PageTree = HiberGetPageNext(PageTree));

    return FALSE;
}

/*++
Function Name: HiberGetPageAt

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil
            - PHYSICAL_ADDRESS PhysicalAddress

Return Values:

--*/
PUCHAR
HiberGetPageAt(
    PSANDMAN_OBJECT Hiberfil,
    PHYSICAL_ADDRESS PhysicalAddress
)
{
PSANDMAN_TREE PageTree;

ULONG PageNumber, BlockSize;

PUCHAR pPage;

    PageNumber = GetPage(PhysicalAddress);

    PageTree = HiberGetPageFirst(Hiberfil);

    do
    {

        if (PageTree->PageNumber == PageNumber) break;

    } while (PageTree = HiberGetPageNext(PageTree));

    if (!PageTree) return 0;

    //
    // Uncompress and return pointer to page.
    //
    pPage = VirtualAlloc(NULL,
                         UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE,
                         MEM_COMMIT | MEM_RESERVE,
                         PAGE_READWRITE);

    BlockSize = GetXpressBlockSize(PageTree->ImgXpressHeader);

    RtlZeroMemory(pPage, UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE);

    XpressDecode((PUCHAR)PageTree->ImgXpressHeader + sizeof(IMAGE_XPRESS_HEADER),
                    BlockSize,
                    pPage,
                    UNCOMPRESSED_BLOCK_SIZE);


    pPage += (PageTree->XpressPage * PAGE_SIZE);

    //
    // We don't align the page.
    //
    pPage += PhysicalAddress.LowPart & 0xFFF;

    return pPage;
}

/*++
Function Name: HiberCountMemoryRanges

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil

Return Values:

--*/
ULONG
HiberCountMemoryRanges(
    PSANDMAN_OBJECT Hiberfil
)
{
    return Hiberfil->MemRangesCount;
}

/*++
Function Name: HiberPatch

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil,
            - PHYSICAL_ADDRESS PhysicalAddress
            - PUCHAR BytesArray
            - ULONG SizeOfBytesArray

Return Values:

--*/
BOOLEAN
HiberPatch(PSANDMAN_OBJECT Hiberfil,
           PHYSICAL_ADDRESS PhysicalAddress,
           PUCHAR BytesArray,
           ULONG SizeOfBytesArray)
{
PUCHAR pPage;
ULONG Limit;

    pPage = HiberGetPageAt(Hiberfil, PhysicalAddress);

    Limit = ((PtrToUlong(pPage) & 0xFFFFF000) + PAGE_SIZE) - PtrToUlong(pPage);

    if (SizeOfBytesArray >= Limit)
    {
        printf("SANDY: Cannot patch, bytes array is too large.\n");
        return FALSE;
    }
    memcpy(pPage, BytesArray, SizeOfBytesArray);

    //
    // Align pages!
    //
    PhysicalAddress.LowPart &= 0xFFFFF000;
    PtrToUlong(pPage) &= 0xFFFFF000;

    HiberPageReplace(Hiberfil, PhysicalAddress, pPage);

    return TRUE;
}

/*++
Function Name: HiberPageReplace

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil,
            - PHYSICAL_ADDRESS PhysicalAddress,
            - PUCHAR Source

Return Values:

--*/
BOOLEAN
HiberPageReplace(
    PSANDMAN_OBJECT Hiberfil,
    PHYSICAL_ADDRESS PhysicalAddress,
    PUCHAR Source)
{
ULONG PrevCompressedSize, NewCompressedSize;
ULONG UncompressedSize;
PUCHAR CompressedBuffer, DecompressedBuffer;
PSANDMAN_TREE PageTree;
ULONG PageNumber;

    //
    // Page is unavailable.
    //
    if (!HiberIsPagePresent(Hiberfil, PhysicalAddress)) return FALSE;

    PageNumber = GetPage(PhysicalAddress);

    PageTree = HiberGetPageFirst(Hiberfil);

    while (PageTree = HiberGetPageNext(PageTree))
    {

        if (PageTree->PageNumber == PageNumber) break;

    }

    if (!PageTree) return FALSE;

    //
    // ...
    //
    DecompressedBuffer = VirtualAlloc(NULL,
                                     UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE,
                                     MEM_COMMIT | MEM_RESERVE,
                                     PAGE_READWRITE);

    PrevCompressedSize = GetXpressBlockSize(PageTree->ImgXpressHeader);

    RtlZeroMemory(DecompressedBuffer, UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE);

    //
    // Decompress Xpress header.
    //
    UncompressedSize = XpressDecode((PUCHAR)PageTree->ImgXpressHeader + sizeof(IMAGE_XPRESS_HEADER),
                                     PrevCompressedSize,
                                     DecompressedBuffer,
                                     UNCOMPRESSED_BLOCK_SIZE);


    //
    // ...
    //
    CompressedBuffer = VirtualAlloc(NULL,
                                    UNCOMPRESSED_BLOCK_SIZE + PAGE_SIZE,
                                    MEM_COMMIT | MEM_RESERVE,
                                    PAGE_READWRITE);

    //
    // Apply the new page.
    //
    memcpy(DecompressedBuffer + (PageTree->XpressPage * PAGE_SIZE),
            Source,
            PAGE_SIZE);

    NewCompressedSize = XpressEncode(DecompressedBuffer,
                                     UNCOMPRESSED_BLOCK_SIZE,
                                     CompressedBuffer,
                                     PrevCompressedSize);

    if (NewCompressedSize <= PrevCompressedSize)
    {
        printf("DEBUG: Previous compressed size 0x%08X\n", PrevCompressedSize);
        SetXpressBlockSize(PageTree->ImgXpressHeader,
                           NewCompressedSize);
        printf("DEBUG: New compressed size 0x%08X\n", GetXpressBlockSize(PageTree->ImgXpressHeader));

        //
        // Wipe the previous buffer
        //
        memset((PUCHAR)PageTree->ImgXpressHeader + sizeof(IMAGE_XPRESS_HEADER),
               0,
               PrevCompressedSize);
        memcpy((PUCHAR)PageTree->ImgXpressHeader + sizeof(IMAGE_XPRESS_HEADER),
           CompressedBuffer,
           NewCompressedSize);
    }
    else
    {
        printf("SANDY: Compressed size is higher than the previous one\n");
        return FALSE;
    }

    VirtualFree(CompressedBuffer, 0, MEM_RELEASE );

    VirtualFree(DecompressedBuffer, 0, MEM_RELEASE );

    return TRUE;
}

/*++
Function Name: HiberPageRemove

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil
            - PHYSICAL_ADDRESS PhysicalAddress

Return Values:

--*/
BOOLEAN
HiberPageRemove(
    PSANDMAN_OBJECT Hiberfil,
    PHYSICAL_ADDRESS PhysicalAddress)
{

    return HiberPageReplace(Hiberfil, PhysicalAddress, NlPage);

}

/*++
Function Name: HiberCreateTree

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil

Return Values:
            - Pointer to First entry.

--*/
PSANDMAN_TREE HiberCreateTree(
                              PSANDMAN_OBJECT Hiberfil)
{
PSANDMAN_TREE PagesListHead;
PIMAGE_XPRESS_HEADER pIXH;
PMEMORY_RANGE_ARRAY pMRA;

ULONG MemRangeCnt;
ULONG LocalPageCnt;
ULONG EntryCount;
ULONG PageIndex, XpressIndex;
ULONG HighestPage;

ULONG i, j;

    PagesListHead = VirtualAlloc(NULL,
                          (Hiberfil->FileHdr->TotalPages + 1) * sizeof(SANDMAN_TREE),
                          MEM_COMMIT | MEM_RESERVE,
                          PAGE_READWRITE);

    PageIndex = 0;
    XpressIndex = 0;
    MemRangeCnt = 1;
    HighestPage = 0;

    pMRA = Hiberfil->MemArray;

    pIXH = (PIMAGE_XPRESS_HEADER)(Hiberfil->Map); 
    //     + (ULONG)( (Hiberfil->FileHdr->FirstTablePage + 1) * PAGE_SIZE ));

    //
    // Get first xpress block.
    //
    while (memcmp(pIXH, XPRESS_MAGIC, 8) != 0)
    {
         PtrToUlong(pIXH) += (ULONG)(sizeof(XPRESS_MAGIC) - 1);
    }

    do
    {
        EntryCount = pMRA->MemArrayLink.EntryCount;

        for (i = 0; i < EntryCount; i++)
        {
            if (pMRA->MemArrayRange[i].EndPage > HighestPage) 
                HighestPage = pMRA->MemArrayRange[i].EndPage;

            LocalPageCnt = (pMRA->MemArrayRange[i].EndPage -
                              pMRA->MemArrayRange[i].StartPage);

            for (j = 0; j < LocalPageCnt; j++)
            {
                //
                // 0x10 pages used.
                //
                if (XpressIndex && ((XpressIndex % 0x10) == 0)) NextXpressBlock(pIXH)

                PagesListHead[PageIndex].PageNumber = pMRA->MemArrayRange[i].StartPage + j;
                PagesListHead[PageIndex].XpressPage = XpressIndex % 0x10;
                PagesListHead[PageIndex].ImgXpressHeader = pIXH;

                PageIndex++;
                XpressIndex++;
            }
        }

        //
        // Next Memory Array Table, then next Xpress header.
        //
        if ((pMRA->MemArrayLink.NextTable != 0) && (EntryCount == 0xFF))
        {
            pMRA = (PMEMORY_RANGE_ARRAY) &(Hiberfil->Map[ pMRA->MemArrayLink.NextTable * PAGE_SIZE ]);
            MemRangeCnt++;

            NextXpressBlock(pIXH)

            XpressIndex = 0;
        }
        else pMRA = NULL;

    } while (pMRA != NULL);

    //
    // Last entry is nl.
    //
    PagesListHead[PageIndex].PageNumber = 0;
    PagesListHead[PageIndex].XpressPage = 0;
    PagesListHead[PageIndex].ImgXpressHeader = NULL;

    //
    // Store tree and stats field.
    //
    Hiberfil->PagesListHead = PagesListHead;
    Hiberfil->TotalListEntries = PageIndex;
    Hiberfil->MemRangesCount = MemRangeCnt;
    Hiberfil->HighestPage = HighestPage;

    return PagesListHead;
}

/*++
Function Name: HiberDestroyTree

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil

Return Values:
            - 

--*/
ULONG
HiberDestroyTree(PSANDMAN_OBJECT pHiberfil)
{
    VirtualFree(pHiberfil->PagesListHead, 0, MEM_RELEASE );

    return 1;
}

/*++
Function Name: HiberFreePage

Overview:

Parameters:
            - PUCHAR Page

Return Values:
            - 

--*/
ULONG
HiberFreePage(PUCHAR Page)
{
    Page = UlongToPtr(PtrToUlong(Page) & 0xFFFF0000);

    return VirtualFree(Page, 0, MEM_RELEASE);
}

/*++
Function Name: HiberGetPageAtVirtualAddress

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil
            - ULONG VirtualAddress

Return Values:
            - 

--*/
PUCHAR
HiberGetPageAtVirtualAddress(PSANDMAN_OBJECT Hiberfil,
                             ULONG VirtualAddress)
{
PHYSICAL_ADDRESS PhysicalPage;
PUCHAR pPage;

    PhysicalPage = MmGetPhysicalAddress(Hiberfil, VirtualAddress);
    pPage = HiberGetPageAt(Hiberfil, PhysicalPage);

    return pPage;
}

/*++
Function Name: HiberGetPageAtVirtualAddress

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil
            - ULONG VirtualAddress

Return Values:
            - 

--*/
PUCHAR
HiberGetFastPageAtVirtualAddress(PSANDMAN_OBJECT Hiberfil,
                                 ULONG VirtualAddress)
{
PHYSICAL_ADDRESS PhysicalPage;
PUCHAR pPage;

    if (VirtualAddress > 0x80000000 || VirtualAddress < 0xA0000000)
    {
      PhysicalPage.LowPart = (VirtualAddress & 0x1FFFFFFF);
      PhysicalPage.HighPart = 0;
    }
    else return FALSE;

    pPage = HiberGetPageAt(Hiberfil, PhysicalPage);

    return pPage;
}

/*++
Function Name: HiberGetPhysicalAddress

Overview:

Parameters:
            - PSANDMAN_OBJECT Hiberfil
            - ULONG BaseAddress

Return Values:
            - 

--*/
PHYSICAL_ADDRESS
HiberGetPhysicalAddress(PSANDMAN_OBJECT Hiberfil,
                        ULONG BaseAddress)
{
    return MmGetPhysicalAddress(Hiberfil, BaseAddress);
}