/*++
SandMan framework.
Copyright 2008 (c) Matthieu Suiche. <msuiche[at]gmail.com>

This file is part of SandMan.

SandMan is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SandMan is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SandMan.  If not, see <http://www.gnu.org/licenses/>.


Module Name:

    py.c

Abstract:

    - Python protocols.

Environment:

    - User mode

Revision History:

    - Matthieu Suiche

TODO: Improve memory management between pages. Actual version is very ugly.
      Improve functions which use structure like IMAGE_HIBR_HEADER or KPROC_STATE.

--*/
#include <sandman.h>

#ifdef _PYTHON_BUILD_
#include <Python.h>


PyObject * hiber_open(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
UCHAR *Filename;

    if (PyArg_ParseTuple(args, "s", &Filename))
    {

        so = HiberOpen(Filename);
        return Py_BuildValue("l", so);

    }
    else
    {
        printf("Error: hiber_open().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_close(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        HiberClose(so);
    }
    else
    {
        printf("Error: hiber_close().\n");
    }

    return Py_BuildValue("");
}

PyObject * hiber_get_memsize(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
ULONG ulSize;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        ulSize = HiberGetPhysicalMemorySize(so);
        return Py_BuildValue("l", ulSize);
    }
    else
    {
        printf("Error: hiber_get_memsize().\n");
        return Py_BuildValue("l", 0);
    }
}


PyObject * hiber_patch(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PHYSICAL_ADDRESS paAddr;
ULONGLONG ullAddr;
PUCHAR pBytesArray;
ULONG ulSizeOfBytesArray;
ULONG ulStatus;

    if (PyArg_ParseTuple(args, "lLs#", &so, &ullAddr, &pBytesArray, &ulSizeOfBytesArray))
    {
        paAddr.LowPart = (ULONG)((ullAddr << 32 ) >> 32);
        paAddr.HighPart = (ULONG)(ullAddr >> 32);

        if ((((paAddr.LowPart & 0xFFF) + PAGE_SIZE) - paAddr.LowPart) > ulSizeOfBytesArray)
        {
            printf("Error: hiber_patch(INVALID_SIZE).\n");
            return Py_BuildValue("l", 0);
        }
        else
        {
            ulStatus = HiberPatch(so, paAddr, pBytesArray, ulSizeOfBytesArray);
            return Py_BuildValue("i", ulStatus);
        }
    }
    else
    {
        printf("Error: hiber_patch().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_dump(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
UCHAR *dumpFilename;

ULONG ulStatus;

    if (PyArg_ParseTuple(args, "ls", &so, &dumpFilename))
    {
        ulStatus = HiberBuildPhysicalMemoryDump(so, dumpFilename);
        return Py_BuildValue("i", ulStatus);
    }
    else
    {
        printf("Error: hiber_dump().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_read_file_header(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PIMAGE_HIBER_HEADER32 pHiberHeader;

    pHiberHeader = VirtualAlloc(NULL,
                    sizeof(IMAGE_HIBER_HEADER32),
                    MEM_COMMIT | MEM_RESERVE,
                    PAGE_READWRITE);

    if (PyArg_ParseTuple(args, "l", &so))
    {
        HiberReadFileHeader(so, (PIMAGE_HIBER_HEADER32)&pHiberHeader);
        return Py_BuildValue("z#", pHiberHeader, sizeof(IMAGE_HIBER_HEADER32));
    }
    else
    {
        printf("Error: hiber_read_file_header().\n");
        return Py_BuildValue("l", 0);
    }
}


PyObject * hiber_write_file_header(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PIMAGE_HIBER_HEADER32 pHiberHeader;
ULONG ulSize, ulStatus;

    pHiberHeader = VirtualAlloc(NULL,
                    sizeof(IMAGE_HIBER_HEADER32),
                    MEM_COMMIT | MEM_RESERVE,
                    PAGE_READWRITE);

    if (PyArg_ParseTuple(args, "lz#", &so, &pHiberHeader, &ulSize))
    {
        //
        // if (ulSize != sizeof(IMAGE_HIBER_HEADER32)
        //
        ulStatus = HiberWriteFileHeader(so, (PIMAGE_HIBER_HEADER32)&pHiberHeader);
        return Py_BuildValue("l", ulStatus);
    }
    else
    {
        printf("Error: hiber_write_file_header().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_read_proc_state(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PKPROCESSOR_STATE32 pProcState;

    pProcState = VirtualAlloc(NULL,
                    sizeof(KPROCESSOR_STATE32),
                    MEM_COMMIT | MEM_RESERVE,
                    PAGE_READWRITE);

    if (PyArg_ParseTuple(args, "l", &so))
    {
        HiberReadProcState(so, (PKPROCESSOR_STATE32)&pProcState);
        return Py_BuildValue("z#", pProcState, sizeof(KPROCESSOR_STATE32));
    }
    else
    {
        printf("Error: hiber_read_proc_state().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_write_proc_state(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PKPROCESSOR_STATE32 pProcState;
ULONG ulSize, ulStatus;

    pProcState = VirtualAlloc(NULL,
                    sizeof(KPROCESSOR_STATE32),
                    MEM_COMMIT | MEM_RESERVE,
                    PAGE_READWRITE);

    if (PyArg_ParseTuple(args, "lz#", &so, &pProcState, &ulSize))
    {
        //
        // if (ulSize != sizeof(KPROCESSOR_STATE32)
        //
        ulStatus = HiberWriteProcState(so, (PKPROCESSOR_STATE32)&pProcState);
        return Py_BuildValue("l", ulStatus);
    }
    else
    {
        printf("Error: hiber_write_proc_state().\n");
        return Py_BuildValue("l", 0);
    }

}

PyObject *hiber_get_cr0(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        //
        // Little check before!
        //
        if (!so) return Py_BuildValue("i", 0);

        return Py_BuildValue("l",
            so->ProcState->SpecialRegisters.u_cr0.Cr0);
    }
    else
    {
        printf("Error: hiber_get_cr0().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject *hiber_get_cr3(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        //
        // Little check before!
        //
        if (!so) return Py_BuildValue("i", 0);

        return Py_BuildValue("l",
            so->ProcState->SpecialRegisters.Cr3);
    }
    else
    {
        printf("Error: hiber_get_cr0().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject *hiber_get_cr4(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        //
        // Little check before!
        //
        if (!so) return Py_BuildValue("i", 0);

        return Py_BuildValue("l",
            so->ProcState->SpecialRegisters.u_cr4.Cr4);
    }
    else
    {
        printf("Error: hiber_get_cr0().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_get_version(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

ULONG ulVersion;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        ulVersion = HiberGetVersion(so);
        return Py_BuildValue("i", ulVersion);
    }
    else
    {
        printf("Error: hiber_get_version().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_get_page_first(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PSANDMAN_TREE st;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        st = HiberGetPageFirst(so);
        return Py_BuildValue("{s:l,s:l,s:l}",
            "page_number", st->PageNumber,
            "xheader_offset", st->ImgXpressHeader,
            "xpage", st->XpressPage);
    }
    else
    {
        printf("Error: hiber_get_page_first().\n");
        return Py_BuildValue("l", 0);
    }
}


PyObject * hiber_get_page_next(PyObject *self, PyObject *args)
{
ULONG ulPageNumber, ulXpressPage; 
PIMAGE_XPRESS_HEADER pImgXpressHeader;

SANDMAN_TREE st;
PSANDMAN_TREE pst;

    if (PyArg_ParseTuple(args, "lll", 
            &ulPageNumber, &pImgXpressHeader, &ulXpressPage))
    {
        st.PageNumber = ulPageNumber;
        st.ImgXpressHeader = pImgXpressHeader;
        st.XpressPage = ulXpressPage;

        pst = HiberGetPageNext(&st);
        return Py_BuildValue("{s:l,s:l,s:l}",
            "page_number", pst->PageNumber,
            "xheader_offset", pst->ImgXpressHeader,
            "xpage", pst->XpressPage);
    }
    else
    {
        printf("Error: hiber_get_page_next().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_is_page_present(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PHYSICAL_ADDRESS paAddr;
ULONGLONG ullAddr;

ULONG ulStatus;

    if (PyArg_ParseTuple(args, "lL", &so, &ullAddr))
    {
        paAddr.LowPart = (ULONG)((ullAddr << 32 ) >> 32);
        paAddr.HighPart = (ULONG)(ullAddr >> 32);

            ulStatus = HiberIsPagePresent(so, paAddr);
            return Py_BuildValue("i", ulStatus);

    }
    else
    {
        printf("Error: hiber_is_page_present().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_get_page_at(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PHYSICAL_ADDRESS paAddr;
ULONGLONG ullAddr;

PUCHAR pPage;

    if (PyArg_ParseTuple(args, "lL", &so, &ullAddr))
    {
        paAddr.LowPart = (ULONG)((ullAddr << 32 ) >> 32);
        paAddr.HighPart = (ULONG)(ullAddr >> 32);

        pPage = HiberGetPageAt(so, paAddr);
        return Py_BuildValue("{s:l, s:z#}", "raw_offset", PtrToUlong(pPage), "buffer", pPage, PAGE_SIZE);
    }
    else
    {
        printf("Error: hiber_get_page_at().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_count_memory_ranges(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

ULONG ulCount;

    if (PyArg_ParseTuple(args, "l", &so))
    {
        ulCount = HiberCountMemoryRanges(so);
        return Py_BuildValue("i", ulCount);
    }
    else
    {
        printf("Error: hiber_count_memory_ranges().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_page_replace(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
PHYSICAL_ADDRESS paAddr;
ULONGLONG ullAddr;
PUCHAR pBytesArray;
ULONG ulSizeOfBytesArray;
ULONG ulStatus;

    if (PyArg_ParseTuple(args, "lLs#", &so, &ullAddr, &pBytesArray, &ulSizeOfBytesArray))
    {
        paAddr.LowPart = (ULONG)((ullAddr << 32 ) >> 32);
        paAddr.HighPart = (ULONG)(ullAddr >> 32);

        //
        // Copy 0x1000 bytes. (one page.)
        //
        ulStatus = HiberPageReplace(so, paAddr, pBytesArray);
        return Py_BuildValue("i", ulStatus);
    }
    else
    {
        printf("Error: hiber_page_replace().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_page_remove(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;

PHYSICAL_ADDRESS paAddr;
ULONGLONG ullAddr;

ULONG ulStatus;

    if (PyArg_ParseTuple(args, "lL", &so, &ullAddr))
    {
        paAddr.LowPart = (ULONG)((ullAddr << 32 ) >> 32);
        paAddr.HighPart = (ULONG)(ullAddr >> 32);

        //
        // Copy 0x1000 bytes. (one page.)
        //
        ulStatus = HiberPageRemove(so, paAddr);
        return Py_BuildValue("i", ulStatus);
    }
    else
    {
        printf("Error: hiber_page_remove().\n");
        return Py_BuildValue("l", 0);
    }
}

PyObject * hiber_free_page(PyObject *self, PyObject *args)
{
PUCHAR pPage;
PUCHAR pyPage;

    if (PyArg_ParseTuple(args, "(lz#)", &pPage, &pyPage))
    {
        HiberFreePage(pPage);

        //
        // Destroy pyPage ?
        //
    }
    else
    {
        printf("Error: hiber_free_page().\n");
        return Py_BuildValue("l", 0);
    }

}

PyObject * hiber_get_page_at_va(PyObject *self, PyObject *args)
{
PSANDMAN_OBJECT so;
ULONG ulAddr;

PUCHAR pPage;

    if (PyArg_ParseTuple(args, "ll", &so, &ulAddr))
    {

        pPage = HiberGetPageAtVirtualAddress(so, ulAddr);
        return Py_BuildValue("{s:l, s:z#}", 
            "raw_offset", PtrToUlong(pPage), 
            "buffer", pPage, PAGE_SIZE);

    }
    else
    {

        printf("Error: hiber_get_page_at().\n");
        return Py_BuildValue("l", 0);

    }
}



static PyMethodDef functions[] = {
    {"hiber_open", hiber_open, METH_VARARGS, "Open hibernation file."},
    {"hiber_close", hiber_close, METH_VARARGS, "Close hibernation file."},
    {"hiber_get_memsize", hiber_get_memsize, METH_VARARGS, "Get physical memory size."},
    {"hiber_patch", hiber_patch, METH_VARARGS, "Patch bytes in a physical page."},
    {"hiber_dump", hiber_dump, METH_VARARGS, "Generate a physical memory dump from hibernation file."},
    {"hiber_read_file_header", hiber_read_file_header, METH_VARARGS, "Read hibernation file header."},
    {"hiber_write_file_header", hiber_write_file_header, METH_VARARGS, "Write hibernation file header."},
    {"hiber_read_proc_state", hiber_read_proc_state, METH_VARARGS, "Read hibernation processor state."},
    {"hiber_write_proc_state", hiber_write_proc_state, METH_VARARGS, "Write hibernation processor state."},
    {"hiber_get_version", hiber_get_version, METH_VARARGS, "Return target version."},
    {"hiber_get_page_first", hiber_get_page_first, METH_VARARGS, "Return a pointer to the first page."},
    {"hiber_get_page_next", hiber_get_page_next, METH_VARARGS, "Return a pointer to the next page."},
    {"hiber_is_page_present", hiber_is_page_present, METH_VARARGS, "Returned DWORD indicates if page is present or not."},
    {"hiber_get_page_at", hiber_get_page_at, METH_VARARGS, "Return a pointer to a physical page."},
    {"hiber_count_memory_ranges", hiber_count_memory_ranges, METH_VARARGS, "Return the number of memory ranges array."},
    {"hiber_page_replace", hiber_page_replace, METH_VARARGS, "Replace a specific page by another one."},
    {"hiber_page_remove", hiber_page_remove, METH_VARARGS, "Remove a specific page."},
    {"hiber_free_page", hiber_free_page, METH_VARARGS, "Unallocate a buffer created by Sandman for an existing page."},
    {"hiber_get_page_at_va", hiber_get_page_at_va, METH_VARARGS, "Return a pointer to a virtual page."},
    {"hiber_get_cr4", hiber_get_cr4, METH_VARARGS, "Return CR4 register."},
    {"hiber_get_cr3", hiber_get_cr3, METH_VARARGS, "Return CR3 register."},
    {"hiber_get_cr0", hiber_get_cr0, METH_VARARGS, "Return CR0 register."},
    {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC initsandman()
{
    Py_InitModule("sandman", functions);
}
#endif