/*
 *	Advantech Single Board Computer WDT driver for Linux 2.4.x
 *
 *	(c) Copyright 2000-2001 Marek Michalkiewicz <marekm@linux.org.pl>
 *
 *	Based on acquirewdt.c which is based on wdt.c.
 *	Original copyright messages:
 *
 *	(c) Copyright 1996 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *				http://www.redhat.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *	Neither Alan Cox nor CymruNet Ltd. admit liability nor provide
 *	warranty for any of this software. This material is provided
 *	"AS-IS" and at no charge.
 *
 *	(c) Copyright 1995    Alan Cox <alan@redhat.com>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/watchdog.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/init.h>

/*
 *	You must set these - there is no sane way to probe for this board.
 *
 *	To enable or restart, write the timeout value in seconds (1 to 63)
 *	to I/O port wdt_start.  To disable, read I/O port wdt_stop.
 *	Both are 0x443 for most boards (tested on a PCA-6276VE-00B1), but
 *	check your manual (at least the PCA-6159 seems to be different -
 *	the manual says wdt_stop is 0x43, not 0x443).
 *	(0x43 is also a write-only control register for the 8254 timer!)
 */
 
static int wdt_stop = 0x443;
static int wdt_start = 0x443;

/*
 *	Kernel methods.
 */

#ifndef MODULE

static int __init adv_setup(char *str)
{
	int ints[4];

	str = get_options(str, ARRAY_SIZE(ints), ints);

	if(ints[0] > 0){
		wdt_stop = ints[1];
		if(ints[0] > 1)
			wdt_start = ints[2];
	}

	return 1;
}

__setup("advwdt=", adv_setup);

#endif /* !MODULE */

MODULE_PARM(wdt_stop, "i");
MODULE_PARM_DESC(wdt_stop, "Advantech WDT 'stop' io port (default 0x443)");
MODULE_PARM(wdt_start, "i");
MODULE_PARM_DESC(wdt_start, "Advantech WDT 'start' io port (default 0x443)");

static void
advwdt_ping(struct watchdog_driver_s *driver)
{
	/* Write a watchdog value */
	outb_p(driver->timeout, wdt_start);
}

static void
advwdt_disable(struct watchdog_driver_s *driver)
{
	inb_p(wdt_stop);
}

/*
 *	Kernel Interfaces
 */
 
static struct watchdog_device_s watchdog_advantech = {
	enable:		advwdt_ping,
	disable:	advwdt_disable,
	min_timeout:	1,
	max_timeout:	63,
	max_write:	PAGE_SIZE,
	timeout:	60,
	ping:		advwdt_ping,
	info:	{ WDIOF_KEEPALIVEPING | WDIOF_SETTIMEOUT, 0, "Advantech WDT" },
	device:	{ WATCHDOG_MINOR, "watchdog" }

};

static int __init
advwdt_init(void)
{
	if(wdt_stop != wdt_start)
		request_region(wdt_stop, 1, "Advantech WDT");
	request_region(wdt_start, 1, "Advantech WDT");
	if (watchdog_register(&watchdog_advantech) != 0) {
		if(wdt_stop != wdt_start)
			release_region(wdt_stop,1);
		release_region(wdt_stop,1);
	}
	return 0;
}

static void __exit
advwdt_exit(void)
{
	watchdog_unregister(&advwdt_miscdev);
	if(wdt_stop != wdt_start)
		release_region(wdt_stop,1);
	release_region(wdt_start,1);
}

module_init(advwdt_init);
module_exit(advwdt_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Marek Michalkiewicz <marekm@linux.org.pl>");
MODULE_DESCRIPTION("Advantech Single Board Computer WDT driver");
EXPORT_NO_SYMBOLS;

/* end of advantechwdt.c */
