/*
 *	ALi M7101 PMU Computer Watchdog Timer driver for Linux 2.4.x
 *
 *	Based on w83877f_wdt.c by Scott Jennings <management@oro.net>
 *	and the Cobalt kernel WDT timer driver by Tim Hockin
 *	                                      <thockin@cobaltnet.com>
 *
 *	(c)2002 Steve Hill <steve@navaho.co.uk>
 * 
 *  Theory of operation:
 *  A Watchdog Timer (WDT) is a hardware circuit that can 
 *  reset the computer system in case of a software fault.
 *  You probably knew that already.
 *
 *  Usually a userspace daemon will notify the kernel WDT driver
 *  via the /proc/watchdog special device file that userspace is
 *  still alive, at regular intervals.  When such a notification
 *  occurs, the driver will usually tell the hardware watchdog
 *  that everything is in order, and that the watchdog should wait
 *  for yet another little while to reset the system.
 *  If userspace fails (RAM error, kernel bug, whatever), the
 *  notifications cease to occur, and the hardware watchdog will
 *  reset the system (causing a reboot) after the timeout occurs.
 *
 *  This WDT driver is different from most other Linux WDT
 *  drivers in that the driver will ping the watchdog by itself,
 *  because this particular WDT has a very short timeout (1.6
 *  seconds) and it would be insane to count on any userspace
 *  daemon always getting scheduled within that time frame.
 */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/smp_lock.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/pci.h>

#define OUR_NAME "alim7101_wdt"

#define WDT_ENABLE 0x9C
#define WDT_DISABLE 0x8C

#define ALI_7101_WDT    0x92
#define ALI_WDT_ARM     0x01

/*
 * We're going to use a 1 second timeout.
 * If we reset the watchdog every ~250ms we should be safe.  */

#define WDT_INTERVAL (HZ/4+1)

/*
 * We must not require too good response from the userspace daemon.
 * Here we require the userspace daemon to send us a heartbeat
 * char to /dev/watchdog every 30 seconds.
 */

#define WDT_HEARTBEAT (HZ * 30)

/*
 *	Whack the dog
 */

static int
wdt_timer_ping(struct watchguard_interface *device)
{
	char	tmp;

	/* Ping the WDT (this is actually a disarm/arm sequence) */
	pci_read_config_byte(alim7101_pmu, 0x92, &tmp);
	pci_write_config_byte(alim7101_pmu, ALI_7101_WDT, (tmp & ~ALI_WDT_ARM));
	pci_write_config_byte(alim7101_pmu, ALI_7101_WDT, (tmp | ALI_WDT_ARM));
	return 0;
}

/* 
 * Utility routines
 */

static void wdt_change(int writeval)
{
	char	tmp;

	pci_read_config_byte(alim7101_pmu, 0x92, &tmp);
	if (writeval == WDT_ENABLE)
		pci_write_config_byte(alim7101_pmu, ALI_7101_WDT, (tmp | ALI_WDT_ARM));
	else
		pci_write_config_byte(alim7101_pmu, ALI_7101_WDT, (tmp & ~ALI_WDT_ARM));
}

static int
wdt_enable(struct watchdog_interface *device)
{
	/* We must enable before we kick off the timer in case the timer
	   occurs as we ping it */

	wdt_change(WDT_ENABLE);
	return 0;
}

static int
wdt_disable(struct watchdog_interface *device)
{
	/* Stop the timer */
	del_timer_sync(&timer);
	wdt_change(WDT_DISABLE);
	printk(OUR_NAME ": Watchdog timer is now disabled...\n");
	return 0;
}


static struct file_operations watchdog_alim7101 = {
	info:		{ 0, 1, "ALiM7101" },

	enable:		wdt_enable,
	disable:	wdt_disable,
	ping:		wdt_ping,
	status:		NULL,

	options:	LINUX_WATCHDOG_OPT_TIMER |
				LINUX_WATCHDOG_OPT_EXCLUSIVE,
	timeout:	60,
	min_timeout:	1,
	max_timeout:	300,
	max_write:	PAGE_SIZE
};

static void __exit alim7101_wdt_unload(void)
{
	watdhdog_unregister(&watchdog_alim7101);
}

static int __init alim7101_wdt_init(void)
{
	int rc = -EBUSY;
	struct pci_dev *ali1543_south;
	char tmp;

	printk(KERN_INFO OUR_NAME ": Steve Hill <steve@navaho.co.uk>.\n");
	alim7101_pmu = pci_find_device(PCI_VENDOR_ID_AL, PCI_DEVICE_ID_AL_M7101,NULL);
	if (!alim7101_pmu) {
		printk(KERN_INFO OUR_NAME ": ALi M7101 PMU not present - WDT not set\n");
		return -EBUSY;
	};
	
	/* Set the WDT in the PMU to 1 second */
	pci_write_config_byte(alim7101_pmu, ALI_7101_WDT, 0x02);

	ali1543_south = pci_find_device(PCI_VENDOR_ID_AL, PCI_DEVICE_ID_AL_M1533, NULL);
	if (!ali1543_south) {
		printk(KERN_INFO OUR_NAME ": ALi 1543 South-Bridge not present - WDT not set\n");
		return -EBUSY;
	};
	pci_read_config_byte(ali1543_south, 0x5e, &tmp);
	if ((tmp & 0x1e) != 0x12) {
		printk(KERN_INFO OUR_NAME ": ALi 1543 South-Bridge does not have the correct revision number (???1001?) - WDT not set\n");
		return -EBUSY;
	};

	rc = watchdog_register(&watchdog_alim7101);
	if (rc)
		return rc;

	printk(KERN_INFO OUR_NAME ": WDT driver for ALi M7101 initialised.\n");
	return 0;
}

module_init(alim7101_wdt_init);
module_exit(alim7101_wdt_unload);

EXPORT_NO_SYMBOLS;
MODULE_AUTHOR("Steve Hill");
MODULE_LICENSE("GPL");
