/*
 *	Eurotech CPU-1220/1410 on board WDT driver for Linux 2.4.x
 *
 *	(c) Copyright 2001 Ascensit <support@ascensit.com>
 *	(c) Copyright 2001 Rodolfo Giometti <giometti@ascensit.com>
 *	(c) Copyright 2002 Rob Radez <rob@osinvestor.com>
 *
 *	Based on wdt.c.
 *	Original copyright messages:
 *
 *      (c) Copyright 1996-1997 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *                              http://www.redhat.com
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *      Neither Alan Cox nor CymruNet Ltd. admit liability nor provide
 *      warranty for any of this software. This material is provided
 *      "AS-IS" and at no charge.
 *
 *      (c) Copyright 1995    Alan Cox <alan@lxorguk.ukuu.org.uk>*
 */

/* Changelog:
 *
 * 2002/04/25 - Rob Radez
 *	clean up #includes
 *	clean up locking
 *	make __setup param unique
 *	proper options in watchdog_info
 *	add WDIOC_GETSTATUS and WDIOC_SETOPTIONS ioctls
 *	add expect_close support
 *
 * 2001 - Rodolfo Giometti
 *	Initial release
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
 
/*
 *      You must set these - there is no sane way to probe for this board.
 *      You can use wdt=x,y to set these now.
 */
 
static char *ev = "int";
 
#define WDT_TIMEOUT		60                /* 1 minute */


/*
 * Some symbolic names 
 */

#define WDT_CTRL_REG		0x30
#define WDT_OUTPIN_CFG		0xe2
   #define WDT_EVENT_INT	   0x00
   #define WDT_EVENT_REBOOT	   0x08
#define WDT_UNIT_SEL		0xf1
   #define WDT_UNIT_SECS	   0x80
#define WDT_TIMEOUT_VAL		0xf2
#define WDT_TIMER_CFG		0xf3
 

#ifndef MODULE

/**
 *      eurwdt_setup:
 *      @str: command line string
 *
 *      Setup options. The board isn't really probe-able so we have to
 *      get the user to tell us the configuration. Sane people build it
 *      modular but the others come here.
 */
 
static int __init eurwdt_setup(char *str)
{
   int ints[4];
 
   str = get_options (str, ARRAY_SIZE(ints), ints);
 
   if (ints[0] > 0) {
      io = ints[1];
      if (ints[0] > 1)
         irq = ints[2];
   }
 
   return 1;
}
 
__setup("eurwdt=", eurwdt_setup);

#endif /* !MODULE */
 
MODULE_PARM(io, "i");
MODULE_PARM_DESC(io, "Eurotech WDT io port (default=0x3f0)");
MODULE_PARM(irq, "i");
MODULE_PARM_DESC(irq, "Eurotech WDT irq (default=10)");
MODULE_PARM(ev, "s");
MODULE_PARM_DESC(ev, "Eurotech WDT event type (default is `int')");


/*
 *      Programming support
 */

static inline void eurwdt_write_reg(u8 index, u8 data, int io)
{
   outb(index, io);
   outb(data, io+1);
}

static inline void eurwdt_lock_chip(int io)
{
   outb(0xaa, io);
}

static inline void eurwdt_unlock_chip(int io)
{
   outb(0x55, io);
   eurwdt_write_reg(0x07, 0x08);   /* set the logical device */
}

static inline eurwdt_set_timeout(int timeout, int io)
{
   eurwdt_write_reg(WDT_TIMEOUT_VAL, (u8)timeout, io);
}

static int 
eurwdt_ping(struct watchdog_interface *driver)
{
   eurwdt_set_timeout(driver->timeout, driver->io);
   return 0;
}

static int 
eurwdt_disable(struct watchdog_interface *device)
{
   eurwdt_set_timeout(0);
}
 
static int
eurwdt_enable(struct watchdog_interface *driver)
{
   eurwdt_disable_timer();
   eurwdt_write_reg(WDT_CTRL_REG, 0x01);      /* activate the WDT */
   eurwdt_write_reg(WDT_OUTPIN_CFG, !strcmp("int", ev) ?
                                    WDT_EVENT_INT : WDT_EVENT_REBOOT);
   /* Setting interrupt line */
   if (driver->irq == 2 || driver->irq > 15 || driver->irq < 0) {
      printk(KERN_ERR ": invalid irq number\n");
      driver->irq = 0;   /* if invalid we disable interrupt */
   }
   if (driver->irq == 0)
      printk(KERN_INFO ": interrupt disabled\n");
   eurwdt_write_reg(WDT_TIMER_CFG, driver->irq <<4);

   eurwdt_write_reg(WDT_UNIT_SEL, WDT_UNIT_SECS);   /* we use seconds */
   eurwdt_set_timeout(0);                           /* the default timeout */ 
}

static struct watchdog_interface watchdog_eurotech = {
	info: {		WDIOF_KEEPALIVEPING | WDIOF_SETTIMEOUT, 0,
				"WDT Eurotech CPU-1220/1410"
	},

	enable:
	disable:
	ping:
	status:
	interrupt:	NULL,
	options:	LINUX_WATCHDOG_OPT_EXCLUSIVE |
				LINUX_WATCHDOG_OPT_INTERRUPT
	irq:		10,
	irq_flags:	SA_INTERRUPT,
	io:		0x3f0
	io_region_size:	2,
	timeout:	60,
	min_timeout:	1,
	max_timeout:	300,
	max_write:	PAGE_SIZE
};

/**
 *      cleanup_module:
 *
 *      Unload the watchdog. You cannot do this with any file handles open.
 *      If your watchdog is set to continue ticking on close and you unload
 *      it, well it keeps ticking. We won't get the interrupt but the board
 *      will not touch PC memory so all is fine. You just have to load a new
 *      module in 60 seconds or reboot.
 */
 
static void __exit eurwdt_exit(void)
{
   eurwdt_lock_chip();
   watchdog_unregister(&watchdog_eurotech);
}
 
/**
 *      eurwdt_init:
 *
 *      Set up the WDT watchdog board. After grabbing the resources 
 *      we require we need also to unlock the device.
 *      The open() function will actually kick the board off.
 */
 
static int __init eurwdt_init(void)
{
   int ret;
 
   ret = watchdog_register(&watchdog_eurotech);
   if (ret) {
      printk(KERN_ERR "eurwdt: can't misc_register on minor=%d\n",
            WATCHDOG_MINOR);
      return ret;
   }

   eurwdt_unlock_chip();
 
   return 0;
}
 
module_init(eurwdt_init);
module_exit(eurwdt_exit);
 
MODULE_AUTHOR("Rodolfo Giometti");
MODULE_DESCRIPTION("Driver for Eurotech CPU-1220/1410 on board watchdog");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
