/*
 *	60xx Single Board Computer Watchdog Timer driver for Linux 2.2.x
 *
 *      Based on acquirewdt.c by Alan Cox.
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 *	The author does NOT admit liability nor provide warranty for 
 *	any of this software. This material is provided "AS-IS" in 
 *      the hope that it may be useful for others.
 *
 *	(c) Copyright 2000    Jakob Oestergaard <jakob@ostenfeld.dk>
 *
 *           12/4 - 2000      [Initial revision]
 *           25/4 - 2000      Added /dev/watchdog support
 *           09/5 - 2001      [smj@oro.net] fixed fop_write to "return 1" on success
 *
 *
 *  Theory of operation:
 *  A Watchdog Timer (WDT) is a hardware circuit that can 
 *  reset the computer system in case of a software fault.
 *  You probably knew that already.
 *
 *  Usually a userspace daemon will notify the kernel WDT driver
 *  via the /proc/watchdog special device file that userspace is
 *  still alive, at regular intervals.  When such a notification
 *  occurs, the driver will usually tell the hardware watchdog
 *  that everything is in order, and that the watchdog should wait
 *  for yet another little while to reset the system.
 *  If userspace fails (RAM error, kernel bug, whatever), the
 *  notifications cease to occur, and the hardware watchdog will
 *  reset the system (causing a reboot) after the timeout occurs.
 *
 *  This WDT driver is different from the other Linux WDT 
 *  drivers in several ways:
 *  *)  The driver will ping the watchdog by itself, because this
 *      particular WDT has a very short timeout (one second) and it
 *      would be insane to count on any userspace daemon always
 *      getting scheduled within that time frame.
 *  *)  This driver expects the userspace daemon to send a specific
 *      character code ('V') to /dev/watchdog before closing the
 *      /dev/watchdog file.  If the userspace daemon closes the file
 *      without sending this special character, the driver will assume
 *      that the daemon (and userspace in general) died, and will
 *      stop pinging the WDT without disabling it first.  This will
 *      cause a reboot.
 *
 *  Why `V' ?  Well, `V' is the character in ASCII for the value 86,
 *  and we all know that 86 is _the_ most random number in the universe.
 *  Therefore it is the letter that has the slightest chance of occuring
 *  by chance, when the system becomes corrupted.
 *
 */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/smp_lock.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>

#define OUR_NAME "sbc60xxwdt"

/*
 * You must set these - The driver cannot probe for the settings
 */
 
#define WDT_STOP 0x45
#define WDT_START 0x443

/*
 * The 60xx board can use watchdog timeout values from one second
 * to several minutes.  The default is one second, so if we reset
 * the watchdog every ~250ms we should be safe.
 */

#define WDT_INTERVAL (HZ/4+1)

/*
 * We must not require too good response from the userspace daemon.
 * Here we require the userspace daemon to send us a heartbeat
 * char to /dev/watchdog every 10 seconds.
 * If the daemon pulses us every 5 seconds, we can still afford
 * a 5 second scheduling delay on the (high priority) daemon. That
 * should be sufficient for a box under any load.
 */

#define WDT_HEARTBEAT (HZ * 10)

/*
 *	Whack the dog
 */

static int
wdt_timer_ping(struct watchdog_interface *device)
{
	/* Ping the WDT by reading from WDT_START */
	inb_p(WDT_START);
	return 0;
}

/* 
 * Utility routines
 */

static int
wdt_enable(struct watchdog_interface *device)
{
	inb_p(WDT_START);
	printk(OUR_NAME ": Watchdog timer is now enabled.\n");  
	return 0;
}

static int
wdt_disable(struct watchdog_interface *device)
{
	inb_p(WDT_STOP);
	printk(OUR_NAME ": Watchdog timer is now disabled...\n");
	return 0;
}


static struct watchdog_interface watchdog_sbc66xx = {
	enable:		wdt_enable,
	disable:	wdt_disable,
	ping:		wdt_ping,
	status:		wdt_status,
	options:	LINUX_WATCHDOG_OPT_TIMER,
	timeout:	60,
	min_timeout:	1,
	max_timeout:	300,
	max_write:	PAGE_SIZE,
	info:		{ 0, 1, "SB60xx" };
};

static void __exit sbc60xxwdt_unload(void)
{
	watchdog_unregister(&watchdog_sbc66xx);

	release_region(WDT_START,1);
	release_region(WDT_STOP,1);
}

static int __init sbc60xxwdt_init(void)
{
	int rc = -EBUSY;

	if (!request_region(WDT_STOP, 1, "SBC 60XX WDT"))
		goto err_out;
	if (!request_region(WDT_START, 1, "SBC 60XX WDT"))
		goto err_out_region1;

	rc = watchdog_register(&watchdog_sbc66xx);
	if (rc)
		goto err_out_region2;

	printk(KERN_INFO OUR_NAME ": WDT driver for 60XX single board computer initialised.\n");
	
	return 0;

err_out_miscdev:
	misc_deregister(&wdt_miscdev);
err_out_region2:
	release_region(WDT_START,1);
err_out_region1:
	release_region(WDT_STOP,1);
err_out:
	return rc;
}

module_init(sbc60xxwdt_init);
module_exit(sbc60xxwdt_unload);

MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
