/*
 *	AMD Elan SC520 processor Watchdog Timer driver for Linux 2.4.x
 *
 *      Based on acquirewdt.c by Alan Cox,
 *           and sbc60xxwdt.c by Jakob Oestergaard <jakob@ostenfeld.dk>
 *     
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 *	The authors do NOT admit liability nor provide warranty for 
 *	any of this software. This material is provided "AS-IS" in 
 *      the hope that it may be useful for others.
 *
 *	(c) Copyright 2001    Scott Jennings <linuxdrivers@oro.net>
 *           9/27 - 2001      [Initial release]
 *	
 *	Additional fixes Alan Cox
 *	-	Fixed formatting
 *	-	Removed debug printks
 *	-	Fixed SMP built kernel deadlock
 *	-	Switched to private locks not lock_kernel
 *	-	Used ioremap/writew/readw
 *	-	Added NOWAYOUT support
 *
 *  Theory of operation:
 *  A Watchdog Timer (WDT) is a hardware circuit that can 
 *  reset the computer system in case of a software fault.
 *  You probably knew that already.
 *
 *  Usually a userspace daemon will notify the kernel WDT driver
 *  via the /proc/watchdog special device file that userspace is
 *  still alive, at regular intervals.  When such a notification
 *  occurs, the driver will usually tell the hardware watchdog
 *  that everything is in order, and that the watchdog should wait
 *  for yet another little while to reset the system.
 *  If userspace fails (RAM error, kernel bug, whatever), the
 *  notifications cease to occur, and the hardware watchdog will
 *  reset the system (causing a reboot) after the timeout occurs.
 *
 *  This WDT driver is different from most other Linux WDT
 *  drivers in that the driver will ping the watchdog by itself,
 *  because this particular WDT has a very short timeout (1.6
 *  seconds) and it would be insane to count on any userspace
 *  daemon always getting scheduled within that time frame.
 *
 *  This driver uses memory mapped IO, and spinlock.
 */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/smp_lock.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>

/*
 * The SC520 can timeout anywhere from 492us to 32.21s.
 * If we reset the watchdog every ~250ms we should be safe.
 */

#define WDT_INTERVAL (HZ/4+1)

/*
 * We must not require too good response from the userspace daemon.
 * Here we require the userspace daemon to send us a heartbeat
 * char to /dev/watchdog every 30 seconds.
 */

#define WDT_HEARTBEAT (HZ * 30)

/*
 * AMD Elan SC520 timeout value is 492us times a power of 2 (0-7)
 *
 *   0: 492us    2: 1.01s    4: 4.03s   6: 16.22s
 *   1: 503ms    3: 2.01s    5: 8.05s   7: 32.21s
 */

#define TIMEOUT_EXPONENT ( 1 << 3 )  /* 0x08 = 2.01s */

/* #define MMCR_BASE_DEFAULT 0xfffef000 */
#define MMCR_BASE_DEFAULT ((__u16 *)0xffffe)
#define OFFS_WDTMRCTL ((unsigned int)0xcb0)
#define WDT_ENB 0x8000		/* [15] Watchdog Timer Enable */
#define WDT_WRST_ENB 0x4000	/* [14] Watchdog Timer Reset Enable */

#define OUR_NAME "sc520_wdt"

#define WRT_DOG(data) *wdtmrctl=data

static __u16 *wdtmrctl;

static spinlock_t wdt_spinlock;
/*
 *	Whack the dog
 */

static int
wdt_timer_ping(struct watchdog_device_s *device)
{
	spin_lock(&wdt_spinlock);
	writew(0xAAAA, wdtmrctl);
	writew(0x5555, wdtmrctl);
	spin_unlock(&wdt_spinlock);
	return 0;
}

/* 
 * Utility routines
 */

static void wdt_config(int writeval)
{
	__u16 dummy;
	unsigned long flags;

	/* buy some time (ping) */
	spin_lock_irqsave(&wdt_spinlock, flags);
	dummy=readw(wdtmrctl);  /* ensure write synchronization */
	writew(0xAAAA, wdtmrctl);
	writew(0x5555, wdtmrctl);
	/* make WDT configuration register writable one time */
	writew(0x3333, wdtmrctl);
	writew(0xCCCC, wdtmrctl);
	/* write WDT configuration register */
	writew(writeval, wdtmrctl);
	spin_unlock_irqrestore(&wdt_spinlock, flags);
}

static int
wdt_enable(struct watchdog_device_s *device)
{
	wdt_config(WDT_ENB | WDT_WRST_ENB | TIMEOUT_EXPONENT);
	return 0;
}

static int
wdt_disable(struct watchdog_device_s *device)
{
	wdt_config(0);
	return 0;
}


static struct watchdog_device_s watchdog_sc520 = {
	enable:		wdt_enable,
	disable:	wdt_disable,
	ping:		wdt_ping,
	status:		NULL,
	timeout:	60,
	min_timeout:	1,
	max_timeout:	300,
	max_write:	PAGE_SIZE,
	options:	LINUX_WATCHDOG_OPT_TIMER,
	info:		{ 0, 1, "SC520" };
};

static void __exit sc520_wdt_unload(void)
{
	watchdog_unregister(&watchdog_sc520);
	iounmap(wdtmrctl);
}

static int __init sc520_wdt_init(void)
{
	int rc;
	unsigned long cbar;


	rc = watchdog_register(&watchdog_sc520);
	if (rc)
		return rc;

	spin_lock_init(&wdt_spinlock);

	/* get the Base Address Register */
	cbar = inl_p(0xfffc);
	printk(OUR_NAME ": CBAR: 0x%08lx\n", cbar);
	/* check if MMCR aliasing bit is set */
	if (cbar & 0x80000000) {
		printk(OUR_NAME ": MMCR Aliasing enabled.\n");
		wdtmrctl = (__u16 *)(cbar & 0x3fffffff);
	} else {
		printk(OUR_NAME "!!! WARNING !!!\n"
		  "\t MMCR Aliasing found NOT enabled!\n"
		  "\t Using default value of: %p\n"
		  "\t This has not been tested!\n"
		  "\t Please email Scott Jennings <smj@oro.net>\n"
		  "\t  and Bill Jennings <bj@oro.net> if it works!\n"
		  , MMCR_BASE_DEFAULT
		  );
	  wdtmrctl = MMCR_BASE_DEFAULT;
	}

	wdtmrctl = (__u16 *)((char *)wdtmrctl + OFFS_WDTMRCTL);
	wdtmrctl = ioremap((unsigned long)wdtmrctl, 2);
	printk(KERN_INFO OUR_NAME ": WDT driver for SC520 initialised.\n");

	return 0;
}

module_init(sc520_wdt_init);
module_exit(sc520_wdt_unload);

MODULE_AUTHOR("Scott and Bill Jennings");
MODULE_DESCRIPTION("Driver for watchdog timer in AMD \"Elan\" SC520 uProcessor");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
