/*
 *	ICP Wafer 5823 Single Board Computer WDT driver for Linux 2.4.x
 *      http://www.icpamerica.com/wafer_5823.php
 *      May also work on other similar models
 *
 *	(c) Copyright 2002 Justin Cormack <justin@street-vision.com>
 *
 *      Release 0.02
 *
 *	Based on advantechwdt.c which is based on wdt.c.
 *	Original copyright messages:
 *
 *	(c) Copyright 1996-1997 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *				http://www.redhat.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 *	Neither Alan Cox nor CymruNet Ltd. admit liability nor provide 
 *	warranty for any of this software. This material is provided 
 *	"AS-IS" and at no charge.	
 *
 *	(c) Copyright 1995    Alan Cox <alan@lxorguk.ukuu.org.uk>
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/watchdog.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/init.h>
#include <linux/spinlock.h>

static spinlock_t wafwdt_lock;

/*
 *	You must set these - there is no sane way to probe for this board.
 *
 *      To enable, write the timeout value in seconds (1 to 255) to I/O
 *      port WDT_START, then read the port to start the watchdog. To pat
 *      the dog, read port WDT_STOP to stop the timer, then read WDT_START
 *      to restart it again.
 */

#define WDT_START 0x443
#define WDT_STOP 0x843

static int
wafwdt_ping(struct watchdog_driver_s *driver)
{
	/* pat watchdog */
        spin_lock(&wafwdt_lock);
	inb_p(WDT_STOP);
	inb_p(WDT_START);
        spin_unlock(&wafwdt_lock);
	return 0;
}

static void
wafwdt_start(struct watchdog_driver_s *driver)
{
	/* start up watchdog */
        spin_lock(&wafwdt_lock);
	outb_p(driver->timeout, WDT_START);
	inb_p(WDT_START);
        spin_unlock(&wafwdt_lock);
	return 0;
}

static int
wafwdt_stop(struct watchdog_driver_s *driver)
{
	/* stop watchdog */
	inb_p(WDT_STOP);
}

/*
 *	Kernel Interfaces
 */

static struct watchdog_driver_s watchdog_wafer5823 = {
	enable:		wafwdt_start,
	disable:	wafwdt_stop,
	timeout:	60,
	min_timeout:	1,
	max_timeout:	,
	max_write:	PAGE_SIZE,
	ping:		wafwdt_ping,
	info:		{ WDIOF_KEEPALIVEPING, 1, "Wafer 5823 WDT" },
	device:		{ WATCHDOG_MINOR, "watchdog" }
};

static int __init wafwdt_init(void)
{
	spin_lock_init(&wafwdt_lock);
	if(!request_region(WDT_STOP, 1, "Wafer 5823 WDT"))
		goto error;
	if(!request_region(WDT_START, 1, "Wafer 5823 WDT"))
		goto error2;
	if (watchdog_register(&watchdg_wafer5823) != 0)
		goto error3;
	return 0;
error3:
	release_region(WDT_START, 1);
error2:
	release_region(WDT_STOP, 1);
error:
	return -ENODEV;
}

static void __exit wafwdt_exit(void)
{
	watchdog_unregister(&wafter_8523);
	release_region(WDT_STOP, 1);
	release_region(WDT_START, 1);
}

module_init(wafwdt_init);
module_exit(wafwdt_exit);

MODULE_AUTHOR("Justin Cormack");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;

/* end of wafer5823wdt.c */
