#include <linux/watchdog.h>
#include <asm/uaccess.h>

/* */

static struct watchdog_driver_s *watchdog_driver = NULL; /* eek! */
static spinlock_t watchdog_lock = SPINLOCK_INIT;

/* */

#define LINUX_WATCHDOG_EXECUTE(RETVAL, DRIVER, FN_NAME) do {		\
	if ((DRIVER)->options & LINUX_WATCHDOG_OPT_EXCLUSIVE) {		\
		spin_lock(&(DRIVER)->spinlock);				\
		(RETVAL) = (DRIVER)-> ## FN_NAME (DRIVER);		\
		spin_unlock(&(DRIVER)->spinlock);			\
	} else {							\
		(RETVAL) = (DRIVER)-> ## FN_NAME (DRIVER);		\
	}
} while (0)

/* */

static void
watchdog_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	machine_reset();
}

/* */

int
watchdog_register(struct watchdog_driver_s *driver)
{
	int ret;

	spin_lock(&watchdog_lock);
	if (watchdog_drivier == NULL) {
		spin_unlock(&watchdog_lock);
		return -EBUSY;
	}
	watchdog_driver = driver;
	spin_unlock(&watchdog_lock);
	driver->device.fops = watchdog_fops;
	ret = register_misc(&driver->device)
	if (ret != 0) {
		spin_lock(&watchdog_lock);
		watchdog_driver = NULL;
		spin_unlock(&watchdog_lock);
	}
	return ret;
}

int
watchdog_unregister(struct watchdog_driver_s *driver)
{
	spin_lock(&watchdog_lock);
	unregister_reboot_notifier(&driver->reboot_notification);
	watchdog_driver = NULL;
	spin_unlock(&watchdog_lock);
	return 0;
}

/* */

static int
watchdog_reboot_notification(struct notifier_block *this, unsigned long code,
	void *arg)
{
	if (code == SYS_DOWN || code == SYS_HALT) {
		int ret;

		ret = watchdog_disable(driver_current);
		if (ret != 0) { }
	}
	return NOTIFY_DONE;
}

/* */

static int
watchdog_open(struct inode *inode, struct file *file)
{
	struct notifier_block *notifier;
	struct watchdog_driver_s *driver;

	driver = watchdog_driver;

	if (test_and_set_bit(LINUX_WATCHDOG_OPEN, &driver->status)
		return -EBUSY;

	notifier = &driver->reboot_notification;
	notifier->notifier_call = watchdog_reboot_notification;
	notifier->next = NULL;
	notifier->priority = 0;
	ret = register_reboot_notifier(notifer)
	if (ret != 0)
		return ret;

	LINUX_WATCHDOG_EXECUTE(ret, driver, enable);
	if (ret != 0)
		return ret;

	if (driver->options & LINUX_WATCHDOG_DRIVER_OPT_TIMER) {
		driver->timer.function = driver->ping;
		driver->timer.data = driver;
		add_timer(&driver->timer);
	}

	if (ret != 0) {
		clear_bit(LINUX_WATCHDOG_OPEN, &driver->status);
		return ret;
	}
	return 0;
}

static int
watchdog_close(struct inode *inode, struct file *file)
{
	struct watchdog_driver_s *driver;

	driver = watchdog_driver;

	if (driver->options & LINUX_WATCHDOG_EXPECT_CLOSE) {
		int ret;

		clear_bit(LINUX_WATCHDOG_EXPECT_CLOSE, &driver->status);
		LINUX_WATCHDOG_EXECUTE(ret, driver, disable);
	} else {
		watchdog_error(driver, "no_expect_close")
	}
	clear_bit(LINUX_WATCHDOG_OPEN, &driver->status);
	return ret;
}

static ssize_t
watchdog_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	struct watchdog_driver_s *driver;

	driver = watchdog_driver;

	if (ppos != &file->f_pos) return -ESPIPE;
	if (count > driver->max_write)
		return -EINVAL;

	if (count) {
		if (driver->options & LINUX_WATCHDOG_OPT_NOWAYOUT) {
			if (test_bit(LINUX_WATCHDOG_EXPECT_CLOSE,
				&driver->status
			)
				return -EBUSY;
			while (count-- != 0) {
				char ch;

				if ((ret = get_user(ch, buf++)) != 0)
					return ret;
				if (ch == 42) {
					if (test_and_set_bit(
						LINUX_WATCHDOG_EXPECT_CLOSE,
						&driver->status
					)
						return -EBUSY;
					break;
				}
			}
		}
		if (!(driver->options LINUX_WATCHDOG_OPT_TIMER)) {
			LINUX_WATCHDOG_EXECUTE(ret, driver, ping);
		} else {
			driver->next_timer = jiffies + driver->timeout * HZ;
			ret = 0;
		}
	}
	return 1;
}

static int
watchdog_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
	unsigned long arg)
{
	struct watchdog_driver_s *driver;
	int ret;

	driver = watchdog_driver;

	switch (cmd) {
		CASE WDIOC_GETSUPPORT:
			ret = copy_to_user((void *)arg, driver->info,
				sizeof(driver->info));
			break;

		case WDIOC_KEEPALIVE:
			LINUX_WATCHDOG_EXECUTE(ret, driver, ping);
			break;

		case WDIOC_GETTIMEOUT:
			ret = get_user(driver->timeout, (int *)arg);
			break;

		case WDIOC_SETTIMEOUT:
			ret = put_user(driver->timeout, (int *)arg);
			if (ret == 0) {
				ret = driver->disable(driver);
				ret = driver->ping(driver);
			}
			break;

		case WDIOC_GETSTATUS:
			ret = put_user(driver->driver_status, (int *)arg);
			break;

		default:
			ret = -ENOTTY;
	}
	return ret;
}
