/*
 * "abstract.h"
 * Physical memory and I/O functions for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */

#ifndef _CCD_H
#define _CCD_H

#include <sys/types.h>

typedef struct {
	int *LinAddr;
	int *PhysAddr;
	int size;
	int __reserved;
} MEMBLK;				/* physical memory block structure */

#define BT848	0x0350109e		/* Brooktree BT848 VIDCAP */
#define BT849	0x0351109e		/* Brooktree BT849 VIDCAP */
#define BT878	0x036e109e		/* Brooktree BT878 VIDCAP */
#define BT879	0x036f109e		/* Brooktree BT879 VIDCAP */

#define PCI_VENDOR_ID_BROOKTREE         0x109e
#define PCI_DEVICE_ID_BROOKTREE_848     0x0350
#define PCI_DEVICE_ID_BROOKTREE_849A    0x0351
#define PCI_DEVICE_ID_BROOKTREE_878_1   0x036e
#define PCI_DEVICE_ID_BROOKTREE_879     0x036f

typedef u_int64_t	uint64;
typedef u_int64_t	int64;

#ifndef __cplusplus
#define bool	int
#define true	1
#define false	0
#endif

#ifndef null
#ifdef __cplusplus
#define null	0
#else
#define null	((void *)0)
#endif
#endif

void *getBTCAPmembase();	/* returns BTCAP memory base, null if failed */
int   getBTCAPtype();		/* returns BTCAP type, '0' if failed */

void  allocMemBlk( MEMBLK * );	/* allocate physical RAM memory block */
void  freeMemBlk( MEMBLK * );	/* free memory block */
void  mapMemBlk( MEMBLK * );	/* map physical I/O memory block */

int64 GetMicroSec();		/* returns microsecond count */
void  DelayMicroSec( int64 );	/* delays (at least) specified number of microseconds */


/*
 * "adcntrl.h"
 * A/D output control functions for BT848 - BT879A.
 * Author :  Ewald Snel, 1998 - 1999.
 */
typedef enum { SVIDEO=1, TUNER=2, COMPOSITE=3 } VideoInput;
typedef enum { AUTODETECT=0, NTSC, NTSC_JAPAN, PAL_BDGHI, PAL_M, PAL_N,
               SECAM, PAL_NCOMB } VideoFormat;

#define XT0FREQ		28636360
#define XT1FREQ		35468950


#define XT0             1
#define XT1             2
#define XT_AUTOSELECT   3


VideoInput  getVideoInput();	/* video source connector */
VideoFormat getVideoFormat();	/* video input format */
int  getCrystal();		/* use video crystal */
int  getAGCDelay();		/* AGC delay */
int  getBurstDelay();		/* color burst delay */
int  getVerticalTotal();	/* number of scanlines, 1=default 525/625 */
int  getTest();			/* test control register */
int  getPllFrequency();		/* PLL reference multiplier with XT0 */
bool isPllEnabled();		/* PLL multiplier enabled? */
bool isCrystal0();		/* crystal 0 selected? */
bool is525Lines();	        /* is 525 lines video?  (NTSC) */
bool isDevHLock();		/* device in H-LOCK? */
bool isVideoPresent();		/* video signal present? */
bool isPowerDown();		/* normal or shut down clock operation? */
bool isAnalogThresholdLow();	/* sync. pulse threshold is 75mv instead of 125mv? */
bool isComponentVideo();	/* Y/C component video signal?  (composite otherwise) */

void setVideoInput( VideoInput );
void setVideoFormat( VideoFormat);
void setCrystal (int);
void setAGCDelay( int );
void setBurstDelay( int );
void setVerticalTotal( int );
void setTest( int );
void setPllFrequency( int );
void setPllEnabled( bool );
void setPowerDown( bool );
void softwareReset();
void setAnalogThresholdLow( bool );
void setComponentVideo( bool );


/*
 * "audio.h"
 * Audio functions for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
typedef enum { TVTUNER=0x00, CDROM=0x02, EXTAUDIO=0x01, MUTE=0x03 } AudioSource;


AudioSource getAudio();		/* returns current audio source */
void setAudio( AudioSource );	/* select audio source */


/*
 * "capture.h"
 * Capture control for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
bool isCaptureOddEnabled();		/* capture odd field enabled? */
bool isCaptureEvenEnabled();		/* capture even field enabled? */
bool isCaptureVBIOddEnabled();		/* VBI capture odd field enabled? */
bool isCaptureVBIEvenEnabled();		/* VBI capture even field enabled? */
bool isCaptureEnabled();		/* capture enabled? */
bool isCaptureVBIEnabled();		/* VBI capture enabled? */
bool isFifoEnabled();			/* FIFO unit enabled? */
bool isRiscEnabled();			/* RISC DMA processor enabled? */
bool isDecodeEvenField();		/* even field is being decoded? */
bool isDropFields();			/* drop fields? frames otherwise. */
bool isDropEvenField();			/* alignment, start decimation on even field? */
int  getDropRate();			/* field or frame drop rate */
int  getFieldCount();			/* lower 8 bits of frame count */

void setCaptureOddEnabled( bool );
void setCaptureEvenEnabled( bool );
void setCaptureVBIOddEnabled( bool );
void setCaptureVBIEvenEnabled( bool );
void setCaptureEnabled( bool );
void setCaptureVBIEnabled( bool );
void setFifoEnabled( bool );
void setRiscEnabled( bool );
void setDropFields( bool );
void setDropEvenField( bool );
void setDropRate( bool );


/*
 * "color.h"
 * Color level control for BT848 - BT879A.
 * Author :  Ewald Snel, 1998 - 1999.
 */
int getBrightness();			/* brightness level, 0..255 */
int getContrast();			/* contrast, 0..511 */
int getSaturation();			/* saturation, 0..511 */
int getHue();				/* hue, 0..255 */

void setBrightness( int );
void setContrast( int );
void setSaturation( int );
void setHue( int );


/*
 * "colorctl.h"
 * A/D color control functions for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
bool isGamma();				/* full gamma? */
bool isAGCFunction();			/* AGC function enabled? */
bool isAdaptiveAGC();			/* adaptive or non-adaptive AGC operation? */
bool isLowColorAutoRemoval();		/* low color detection and removal? */
bool isCbFirst();			/* invert Cb, Cr order? */
bool isChromaAGC();			/* chroma AGC compensation? */
bool isChromaComb();			/* ? */
bool isChromaADC();			/* normal or sleep chroma ADC operation? */
bool isChromaOverflow();		/* chroma overflow? */
bool isLumaOverflow();			/* luma overflow? */
bool isLumaADC();			/* normal or sleep luma ADC operation? */
bool isLumaNotchFilter();		/* luma notch filter enabled? */
bool isLumaPeakFilter();		/* luma peak filter enabled? */
bool isLumaDecimation();		/* luma decimation filter enabled? */

void setGamma( bool );
void setAGCFunction( bool );
void setAdaptiveAGC( bool );
void setLowColorAutoRemoval( bool );
void setCbFirst( bool );
void setChromaAGC( bool );
void setChromaComb( bool );
void setChromaADC( bool );
void resetChromaOverflow();
void resetLumaOverflow();
void setLumaADC( bool );
void setLumaNotchFilter( bool );
void setLumaPeakFilter( bool );
void setLumaDecimation( bool );


/*
 * "colorfmt.h"
 * Color format control for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
typedef enum {
    None_0,		/* `None' makes a compile error: `ccd.h:210: parse error before numeric constant'. Ansi C problem ? */
    Coring_8,
    Coring_16,
    Coring_32
} CoringLevel;

typedef enum {
    RGB32,
    RGB24,
    RGB16,
    RGB15,
    YUY,
    BtYUV,
    Y8,
    RGB8,
    YCrCb422,
    YCrCb411,
    Raw8X=14
} ColorFormat;


ColorFormat getColorFormat();		/* output color space */
CoringLevel getCoringLevel();		/* clip luma values to '0' if below this level */
bool isFullOutputRange();		/* luma full output range:  0=16-235, 1=0-255 */
bool isColorBars();			/* TV color bars (test pattern) */
bool isExtFrame();			/* EXT_FRAME bit, larger RAW capture if enabled */
bool isByteSwap();			/* byte swap output: ABCD <--> BADC */
bool isWordSwap();			/* word swap output: ABCD <--> CDAB */

void setColorFormat( ColorFormat );
void setCoringLevel( CoringLevel );
void setFullOutputRange( bool );
void setColorBars( bool );
void setExtFrame( bool );
void setByteSwap( bool );
void setWordSwap( bool );


/*
 * "gpio.h"
 * GPIO functions for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
int getGPOE();				/* GPIOOutputEnableControl */
int getGPData( int, int, int );

void setGPOE( int );
void setGPData( int, int, int, int );


/*
 * "i2c.h"
 * I2C interface for tuner devices connected to the SCL
 * and SDA pins of BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
void i2cInit();				/* initialize I2C */
void i2cStart();			/* signal 'START' */
void i2cStop();				/* signal 'STOP' */
void i2cWrite( int );			/* write byte */
int  i2cRead();					/* read byte */
void i2cSendNACK();				/* send 'NACK' */
void i2cSetSDA( int );			/* set data */
void i2cSetSCL( int );			/* set clock */
void i2cBitDelay();				/* I2C wait (~50KHz) */
void i2cWaitForACK();			/* wait for 'ACK' */


/*
 * "memmap.h"
 * Local registers memory map for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
typedef struct
{
    unsigned int
	sda:1,
        scl:1,
	w3b:1,
	sync:1,
	div:4,
	db2:8,
	db1:8,
	db0:8;
} I2C_t;


/*
 * description of local registers memory map
 */
volatile struct
{
	unsigned int			/* Device Status Register  (DSTATUS) */
		cof:1,
		lof:1,
		plock:1,	/* unused */
		csel:1,
		numl:1,
		evenfield:1,
		hloc:1,
		pres:1, :24;
	unsigned int			/* Input Format Register  (IFORM) */
		format:3,
		xtsel:2,
		muxsel:2,
		hactive:1, :24;
	unsigned int			/* Temporal Decimation Register  (TDEC) */
		dec_rate:6,
		dec_fieldalign:1,
		dec_field:1, :24;
	unsigned int			/* MSB Cropping Register  (E_CROP) */
		e_hactive_msb:2,
		e_hdelay_msb:2,
		e_vactive_msb:2,
		o_vdelay_msb:2,:24;
	unsigned int			/* Vertical Delay Register, Lower Byte */
		o_vdelay_lo:8, :24;
	unsigned int			/* Vertical Active Register, Lower Byte */
		e_vactive_lo:8, :24;
	unsigned int			/* Horizontal Delay Register, Lower Byte */
		e_hdelay_lo:8, :24;
	unsigned int			/* Horizontal Active Register, Lower Byte */
		e_hactive_lo:8, :24;
	signed int				/* Horizontal Scaling Register, Upper Byte */
		e_hscale_hi:8, :24;
	unsigned int			/* Horizontal Scaling Register, Lower Byte */
		e_hscale_lo:8, :24;
	signed int				/* Brightness Control Register */
		bright:8, :24;
	unsigned int			/* Miscellaneous Control Register  (E_CONTROL) */
		e_sat_v_hi:1,
		e_sat_u_hi:1,
		e_contrast_hi:1,
		:1,			/* (0) res */
		e_cbsense:1,
		e_ldec:1,
		e_comp:1,
		e_lnotch:1, :24;
	unsigned int			/* Luma Gain Register, Lower Byte */
		contrast_lo:8, :24;
	unsigned int			/* Chroma (U) Gain Register, Lower Byte */
		sat_u_lo:8, :24;
	unsigned int			/* Chroma (V) Gain Register, Lower Byte */
		sat_v_lo:8, :24;
	signed int				/* Hue Control Register */
		hue:8, :24;
	unsigned int			/* SC Loop Control Register */
		:3,			/* (000) res */
		e_hfilt:2,
		e_ckill:1,
		e_cagc:1,
		e_peak:1, :24;
	unsigned int			/* White Crush Up Register  (WC_UP) */
		UPCNT:6,	/* unused */
		MAJS:2,		/* unused */
		:24;
	unsigned int			/* Output Format Register  (OFORM) */
		:5,			/* (00000) res */
		core:2,
		range:1, :24;
	unsigned int			/* Vertical Scaling Register, Upper Byte  (E_VSCALE_HI) */
		e_vscale_hi:5,
		e_uint:1,	/* unused */
		e_comb:1,
		e_vsfldalign:1,/* unused */
		:24;
	unsigned int			/* Vertical Scaling Register, Lower Byte */
		e_vscale_lo:8, :24;
	unsigned int			/* Test Control Register */
		test:8, :24;
	int reg58:8, :24;/* (?) */
	int reg5C:8, :24;
	unsigned int			/* AGC Delay Register */
		adelay:8, :24;
	unsigned int			/* Burst Delay Register */
		bdelay:8, :24;
	unsigned int			/* ADC Interface Register */
		crush:1,
		c_sleep:1,
		y_sleep:1,
		clk_sleep:1,
		agc_en:1,
		sync_t:1,
		:2, :24;	/* (10) res */
	unsigned int			/* Video Timing Control Register  (E_VTC) */
		e_vfilt:3,
		:4,			/* (0000) res */
		e_hsfmt:1,	/* unused */
		:24;
	int reg70:32;	/* (?) */
	int reg74:32;	/* (?) */
	unsigned int			/* White Crush Down Register */
		DNCNT:6,	/* unused */
		WCFRAME:1,	/* unused */
		VERTEN:1,	/* unused */
		:24;
	unsigned int			/* Software Reset Register */
		sreset:8, :24;
	unsigned int			/* Timing Generator Load Byte */
		tglb:8,		/* nused */
		:24;
	unsigned int			/* Timing Generator Control  (TGCTRL) */
		tgc_vm:1,	/* unused */
		gpc_ar:1,	/* unused */
		tgc_ai:1,	/* unused */
		tgcki:2,	/* unused */
		tgcko:2,	/* unused */
		:1, :24;	/* (0) res */
	int reg88:8, :24;
	unsigned int			/* MSB Cropping Register  (O_CROP) */
		o_hactive_msb:2,
		o_hdelay_msb:2,
		o_vactive_msb:2,
		e_vdelay_msb:2,:24;
	unsigned int			/* Vertical Delay Register, Lower Byte  (E_VDELAY_LO) */
		e_vdelay_lo:8, :24;
	unsigned int			/* Vertical Active Register, Lower Byte  (O_VACTIVE_LO) */
		o_vactive_lo:8, :24;
	unsigned int			/* Horizontal Delay Register, Lower Byte  (O_HDELAY_LO) */
		o_hdelay_lo:8, :24;
	unsigned int			/* Horizontal Active Register, Lower Byte  (O_HACTIVE_LO) */
		o_hactive_lo:8, :24;
	signed int				/* Horizontal Scaling Register, Upper Byte  (O_HSCALE_HI) */
		o_hscale_hi:8, :24;
	unsigned int			/* Horizontal Scaling Register, Lower Byte  (O_HSCALE_LO) */
		o_hscale_lo:8, :24;
	int regA8:8, :24;
	unsigned int			/* Miscellaneous Control Register  (E_CONTROL) */
		o_sat_v_hi:1,
		o_sat_u_hi:1,
		o_contrast_hi:1,
		:1,			/* (0) res */
		o_cbsense:1,
		o_ldec:1,
		o_comp:1,
		o_lnotch:1, :24;
	unsigned int			/* Total Line Count Register */
		vtotal_lo:8, :24;
	unsigned int			/* Total Line Count Register */
		vtotal_hi:2,
		:6, :24;	/* (000000) res */
	int regB8:8, :24;
	int regBC:8, :24;
	unsigned int			/* SC Loop Control Register  (O_SCLOOP) */
		:3,			/* (000) res */
		o_hfilt:2,
		o_ckill:1,
		o_cagc:1,
		o_peak:1, :24;
	int regC4:8, :24;
	int regC8:8, :24;
	unsigned int			/* Vertical Scaling Register, Upper Byte  (O_VSCALE_HI) */
		o_vscale_hi:5,
		o_uint:1,	/* unused */
		o_comb:1,
		o_vsfldalign:1,/* unused */
		:24;
	unsigned int			/* Vertical Scaling Register, Lower Byte */
		o_vscale_lo:8, :24;
	unsigned int			/* Color Format Register  (COLOR_FMT) */
		color_even:4,
		color_odd:4, :24;
	unsigned int			/* Color Control Register  (COLOR_CTL) */
		bswap_even:1,
		bswap_odd:1,
		wswap_even:1,
		wswap_odd:1,
		gamma:1,
		rgb_ded:1,	/* unused */
		color_bars:1,
		ext_frmrate:1,/* unused */
		:24;
	unsigned int			/* Capture Control Register  (CAP_CTL) */
		capture_even:1,
		capture_odd:1,
		capture_vbi_even:1,
		capture_vbi_odd:1,
		dith_frame:1,/* unused */
		:3, :24;	/* (000) res */
	unsigned int			/* VBI Packet Size Register  (VBI_PACK_SIZE) */
		vbi_pkt_lo:8,/* unused */
		:24;
	unsigned int			/* VBI Packet Size / Delay Register  (VBI_PACK_DEL) */
		vbi_pkt_hi:1,/* unused */
		ext_frame:1,
		vbi_hdelay:6,/* unused */
		:24;
	unsigned int			/* Field Capture Counter Register  (FCAP) */
		fcntr:8,
		:24;
	unsigned int			/* Video Timing Control Register  (O_VTC) */
		o_vfilt:3,
		:4,			/* (0000) res */
		o_hsfmt:1,	/* unused */
		:24;
	unsigned int			/* PLL Reference Multiplier */
		pll_f_lo:8, :24;
	unsigned int			/* PLL Reference Multiplier */
		pll_f_hi:8, :24;
	unsigned int			/* Integer PLL-XCI Register */
		pll_i:6,
		pll_c:1,
		pll_x:1, :24;
	unsigned int			/* Digital Signal Interface Register  (DVSIF) */
		dvsif:8, :24;
	unsigned int			/* Interrupt Status Register  (INT_STAT) */
		interruptStatus:32;
	unsigned int			/* Interrupt Status Register  (INT_MASK) */
		interruptMask:32;
	int reg108:8,	/* (?) */
		:24;
	unsigned int			/* GPIO and DMA Control Register  (GPIO_DMA_CTL) */
		fifo_enable:1,
		risc_enable:1,
		pktp:2,		/* unused */
		pltp1:2,	/* unused */
		pltp23:2,	/* unused */
		:2,			/* (00) res */
		gpclkmode:1,/* unused */
		gpiomode:2,	/* unused */
		gpwec:1,	/* unused */
		gpuinti:1,	/* unused */
		gpuintc:1,	/* unused */
		:16;
	I2C_t I2C;				/* 2C Data/Control Register */
	unsigned int			/* RISC Program Start Address Register  (RISC_STRT_ADD) */
		risc_ipc:32;
	unsigned int			/* GPIO Output Enable Control Register  (GPIO_OUT_EN) */
		gpoe:24, :8;
	unsigned int			/* GPIO Registered Input Control  (GPIO_REG_INP) */
		GPIE:24,	/* unused */
		:8;
	unsigned int			/* RISC Program Counter Register */
		risc_pc:32;

/* -----------------------------------------------// */
	int :32, :32, :32, :32, :32, :32, :32;
	int :32, :32, :32, :32, :32, :32, :32, :32;
	int :32, :32, :32, :32, :32, :32, :32, :32;
	int :32, :32, :32, :32, :32, :32, :32, :32;
	int :32, :32, :32, :32, :32, :32, :32, :32;
	int :32, :32, :32, :32, :32, :32, :32, :32;
	int :32, :32, :32, :32, :32, :32, :32, :32;
/* -----------------------------------------------// */

	unsigned int			/* GPIO Data I/O  (GPIO_DATA) */
		GPData:32;
} *BTmm;

#define fatal(s)	fatalexit(s, __FILE__, __LINE__)

void fatalexit( char *message, char *file, int line );


/*
 * "risc.h"
 * RISC DMA processor functions for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
#define IRQ		0x1000000	/* trigger IRQ */
#define SOL		0x4000000	/* start of scanline */
#define EOL		0x8000000	/* end of scanline */
#define RESYNC		0x0008000	/* resync */
#define SC_RESYNC	1
#define SC_VRE		4		/* start of even field */
#define SC_VRO		0xC		/* start of odd field */
#define SC_FM1		6		/* packed data */
#define SC_FM3		0xE		/* planar data */
#define WRITE(flags, count)		(0x10000000 | flags | count)
#define SKIP(flags, count)		(0x20000000 | flags | count)
#define WRITEC(flags, count)		(0x50000000 | flags | count)
#define JUMP(flags)			(0x70000000 | flags)
#define SYNC(flags, code)		(0x80000000 | flags | code)
#define WRITE123(flags, count1)		(0x90000000 | flags | count1)
#define SKIP123(flags, count1)		(0xA0000000 | flags | count1)
#define WRITE1S23(flags, count)		(0xB0000000 | flags | count)
#define COUNT23(count2, count3)		((count3 << 12) | count2)


int getRiscIPC();				/* RISC program start address */
int getRiscPC();				/* RISC program counter */

void setRiscIPC(int);
void setRiscPC(int);


/*
 * "scaler.h"
 * Scaler functions for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
typedef enum { AutoFormat=0, CIF, QCIF, ICON } HorizFilter;


void scalerInit();			/* initialize scaler, read HW settings */
void doScale();				/* update all scaler registers */

void getAnalogWindow(int *, int *, int *, int *);	/* analog : H/V active, delay window */
void getFieldSize(int *, int *);	/* digital/analog scale */
HorizFilter getHFilter();		/* horizontal low-pass filter type */
int  getVFilter();			/* filter to remove junk lines from top of video */

void setAnalogWindow(int, int, int, int );
void setFieldSize(int, int);
void setHFilter(HorizFilter);
void setVFilter(int);


/*
 * "tuner.h"
 * Philips FL1216 MK2 tuner interface, connected to BT878 I2C
 * Author :  Ewald Snel, 1998 - 1999.
 */
typedef enum { TunerBusy = -1, Lower, Correct, Raise, NoCarrier } PllOffset;


PllOffset getPllOffset();		/* carier offset function to detect correct freq. */
int getChannel();			/* last set TV channel in Hz */

void setChannel(int);			/* set carier to TV channel (Hz) */


/*
 * "tv.h"
 * BTCAP capture device API for BT848 - BT879A
 * Author :  Ewald Snel, 1998 - 1999.
 */
void initBTCAP();			/* initialize device memory base only */
void finitBTCAP();			/* fully initialize hardware, including scaler & I2C */

#endif
