/*
 *	PAL software decoder
 *	includes Y/C separation and YCrCb to RGB conversion
 *
 *	Copyright (C) 1999, Ewald Snel
 *
 *WWW		http://esnel.op.het.net/
 *e-mail	esnel@cistron.nl
 */

#include <stdio.h>
#include <math.h>
#include <time.h>
#include <stdlib.h>
#include "ccd.h"
#include "ronnyvideo.h"
#include "esnelDecode.h"


unsigned char scanLuma[500];
char scanCbU[300], scanCrV[300];
int cosint[1024];

unsigned char clip8[1024];
int PvU[decWidth / 2], PvV[decWidth / 2];


/*
 * Decode next field of PAL data
 */
void decodeFrameEsnelColorC(unsigned char *src, int offset, int pitch)
{
    static unsigned int cbPhase, cbHF, cbVF;
    static int i, y, vsaturation, cbOdd;
    static int pitch_local;

    pitch_local = pitch;

    if (!isInitialized) {
	InitializeDecoder();
    }

    src += offset;
    GetColorBurstData(src, &cbPhase, &cbHF, &cbVF, &cbOdd);

    src += getColorBurstWidth();

    cbHF *= 10;
    vsaturation = cbOdd ? -saturation : saturation;

    for (y = 10; y < decHeight; y += 2) {
	GetChromaSamples(src, cbPhase, cbHF, cbVF, saturation, vsaturation);

	for (i = 0; i < 2; i++) {
	  GetLumaSamples(src, adjust_luma_level, brightness, contrast);
	  SetPixels(y + i, pitch_local);
	  src += rawWidth;
	}

	cbPhase	+= 2 * cbVF;
    }
}


/*
 * Initialize decoder
 */
void InitializeDecoder()
{
    int i;

    for (i = 0; i < 1024; i++) {
	cosint[i] = (int) (0x7fffffff * cos(i * M_PI / 512.0)) >> 16;
	clip8[i]  = (i < 320) ? 0 : ((i > 320 + 255) ? 255 : i - 320);
    }
    
    isInitialized = true;
}


void SetPixels(int y, int pitch)
{
    int i, y0, y1, u, v, cr, cg, cb;
    RGBTRIPLE pixels[decWidth];

    for (i = 0; i < decWidth/2; i++) {
	u  = scanCbU[i];
	v  = scanCrV[i];
	y0 = scanLuma[i+i];
	y1 = scanLuma[i+i+1];
	cr = (64*320+32 +         73*v) >> 6;
	cg = (64*320+32 -  25*u - 37*v) >> 6;
	cb = (64*320+32 + 130*u       ) >> 6;

	pixels[i*2].rgbtRed = clip8[y0 + cr];
	pixels[i*2].rgbtGreen = clip8[y0 + cg];
	pixels[i*2].rgbtBlue = clip8[y0 + cb];

	pixels[i*2+1].rgbtRed = clip8[y1 + cr];
	pixels[i*2+1].rgbtGreen = clip8[y1 + cg];
	pixels[i*2+1].rgbtBlue = clip8[y1 + cb];
    }

    drawPixels(12, y, pixels+30, decWidth-58,pitch);
}


/*
 * Decode single scanline of Raw 8bit samples in PAL format to grayscale
 * luma (Y) samples at half PAL (50%) resolution.
 */
void GetLumaSamples(unsigned char *src, int lladjust, int brightness, int contrast )
{
    int x, l;

    /* adjust luma level to color burst */
    if (lladjust) {
	int cbluma = 0;

	for (x = 0; x < 32; x++) {
	    cbluma += src[x - 192];
	}
	brightness += ((384 - cbluma) * contrast) >> 11;
    }

    /* luma notchfilter, remove color frequency, downsample */
    for (x = 0; x < decWidth; x++) {
	l  = brightness;
	l += ((src[-3]+src[-2]+src[-1]+src[ 0] + src[1]+src[2]+src[3]+src[4]) * contrast) >> 9;
	scanLuma[x] = (l < 0) ? 0 : ((l > 255) ? 255 : l);
	src += 5;
    }
}


/*
 * Decode single scanline of Raw 8bit samples in PAL format to chroma (U,V)
 * samples at half resolution.
 */
void GetChromaSamples(unsigned char *src, unsigned int cbPhase, unsigned int cbHF, unsigned int cbVF, int saturation, int vsaturation)
{
    int x, i, c, s, u, v, lu, lv, fc, fs;

    for (x = 0; x < decWidth / 2; x++) {
	u = 0;
	v = 0;

	for (i = 0; i < 2; i++) {
	    /* cosine filter samples, extract color frequency, downsample */
	    c  =   704 * (src[ 0] - src[1]);
	    c +=  1559 * (src[-1] - src[2]);
	    c +=  1078 * (src[-2] - src[3]);
	    c +=  -457 * (src[-3] - src[4]);
	    c += -1020 * (src[-4] - src[5]);
	    c +=  -282 * (src[-5] - src[6]);
	    c +=   199 * (src[-6] - src[7]);
	    c +=   141 * (src[-7] - src[8]);
	    c >>= 4;

	    /* sine filter samples, extract color frequency, downsample */
	    s  =  1700 * (src[ 0] + src[1]);
	    s +=   364 * (src[-1] + src[2]);
	    s += -1525 * (src[-2] + src[3]);
	    s += -2182 * (src[-3] + src[4]);
	    s +=   821 * (src[-4] + src[5]);
	    s +=   962 * (src[-5] + src[6]);
	    s +=   199 * (src[-6] + src[7]);
	    s +=  -340 * (src[-7] + src[8]);
	    s >>= 4;

	    /* combine sine and cosine data at color frequency phase */
	    fs = cosint[(cbPhase - 0x40000000) >> 22];
	    fc = cosint[ cbPhase               >> 22];
	    u += (fs*c + fc*s) >> 9;
	    v += (fc*c - fs*s) >> 9;

	    v = -v;
	    src += rawWidth;
	    cbPhase += cbVF;
	}
	src     += 10 - 2*rawWidth;
	cbPhase	+= cbHF - 2*cbVF;

	u *= saturation;
	v *= vsaturation;
	lu = PvU[x];
	lv = PvV[x];
	PvU[x] = u;
	PvV[x] = v;

	/* combine 2 lines to correct phase variance */
	u = (u + lu) >> 22;
	v = (v + lv) >> 22;
	scanCbU[x] = (u < -128) ? -128 : ((u > 127) ? 127 : u);
	scanCrV[x] = (v < -128) ? -128 : ((v > 127) ? 127 : v);
    }
}


/*
 * Get color burst information
 */
void GetColorBurstData(unsigned char *src, unsigned int *cbPhase, unsigned int *cbHF, unsigned int *cbVF, int *cbOdd)
{
    static int cbPhases[decHeight];
    double f_amp;
    unsigned int c0, c1, s0, s1, p0, p1, p2, p3, f_cb;
    int i, v, x, y, f_cos, f_sin;
    double div;

    /* determine phase offsets of colorburst */
    for (y = 0; y < decHeight; y++) {
	for (x = 0, f_cos = 0, f_sin = 0, f_cb = 0; x < 72; x++) {
	    i      = *src++;
	    f_cos += cosint[ f_cb               >> 22] * i;
	    f_sin += cosint[(f_cb - 0x40000000) >> 22] * i;
	    f_cb  += FcHF0;
	}
	src  += (rawWidth - 72);
	div = sqrt( (double) f_cos*f_cos + (double) f_sin*f_sin );
	if (div != 0)
	    f_amp = 1 / div;
	else
	    f_amp = 10000;

	c1 = (unsigned int) (acos( f_cos * f_amp ) * divPi);
	s0 = (unsigned int) (asin( f_sin * f_amp ) * divPi) + 0x80000000;
	c0 = ~c1;
	s1 = ~s0 + 0x80000000;
	p0 = abs(c0 - s0);  p1 = abs(c0 - s1);
	p2 = abs(c1 - s0);  p3 = abs(c1 - s1);

	if ((p0 < p2 && p0 < p3) || (p1 < p2 && p1 < p3))  c1 = c0;
	if ((p0 < p1 && p0 < p3) || (p2 < p1 && p2 < p3))  s1 = s0;

	cbPhases[y] = (s1 >> 1) + (c1 >> 1);
    }

    /* Vc polarity detection  (even/odd frame) */
    for (i=2, p0=0, p1=0; i < (decHeight - 2); i += 2) {
	p0 += abs( cbPhases[i] - cbPhases[i - 1] ) >> 8;
	p1 += abs( cbPhases[i + 1] - cbPhases[i] ) >> 8;
    }
    *cbOdd = (p0 < p1);

    /* sample rate correction (color burst) */
    for (i=8, v=0; i < (decHeight /2); i++) {
	v += (cbPhases[(decHeight /2) + i] - cbPhases[i]) >> 8;
    }
    v         = ((v /(decHeight /2)) << 8) /((decHeight /2) - 8);
    *cbHF     = v /rawWidth +  FcHF0;
    *cbVF     = v           +  FcVF0;
    i         = cbPhases[0] +  200 * (*cbHF) - 0x60000000L;
    i        -= (*cbOdd ? 0 : -0x40000000);
    *cbPhase  = i - *cbHF - (*cbHF >> 1);
}
