/*
 * interface to the v4l driver
 *
 *   (c) 1997,98 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <X11/Intrinsic.h>

#include "grab.h"

#include <asm/types.h>		/* XXX glibc */
#include "bttv/videodev.h"
#include "bttv/bttv.h"
#include "cablecrypt.h"

#define SYNC_TIMEOUT 3

unsigned short y_to_rgb16[256];

#define BURSTOFFSET 76



/* ---------------------------------------------------------------------- */

/* prototypes */
extern void putlinesw(void* src, void* dest, int size);

static int   grab_open(char *filename, int sw, int sh,
		       int format, int pixmap, void *base, int width);
static int   grab_close();
/* static int   grab_overlay(int x, int y, int width, int height, int format, */
/* 		  struct OVERLAY_CLIP *oc, int count); */
static void* grab_scr(void *dest, int width, int height, int single, int dest_xsize);
static void* grab_one(int width, int height);
static int   grab_tune(unsigned long freq);
static int   grab_tuned();
static int   grab_input(int input, int norm);
static int   grab_picture(int color, int bright, int hue, int contrast);
static int   grab_audio(int mute, int volume, int *mode);

/* ---------------------------------------------------------------------- */

static char *device_cap[] = {
    "capture", "tuner", "teletext", "overlay", "chromakey", "clipping",
    "frameram", "scales", "monochrome", NULL
};

static char *device_pal[] = {
    "-", "grey", "hi240", "rgb16", "rgb24", "rgb32", "rgb15",
    "yuv422", "yuyv", "uyvy", "yuv420", "yuv411", "raw",
    "yuv422p", "yuv411p"
};
#define PALETTE(x) ((x < sizeof(device_pal)/sizeof(char*)) ? device_pal[x] : "UNKNOWN")

static struct STRTAB norms[] = {
    {  0, "PAL" },
    {  1, "NTSC" },
    {  2, "SECAM" },
    {  3, "AUTO" },
    { -1, NULL }
};
static struct STRTAB *inputs;

static int    fd = -1;

#ifdef STRANGE_SYNC_PROBLEMS
static int    vbifd = -1;
#endif

extern int decode_routine;

/* generic informations */
static struct video_capability  capability;
static struct video_channel     *channels;
static struct video_audio       *audios;
static struct video_tuner       tuner;
static struct video_picture     pict;
static int                      cur_input;
static int                      cur_norm;

/* overlay */
/* static struct video_window      ov_win; */
/* static struct video_clip        ov_clips[32]; */
static struct video_buffer      ov_fbuf;

/* screen grab */
static struct video_mmap        gb_even;
static struct video_mmap        gb_odd;
static int                      even;
static int                      gb_grab,gb_sync;
static struct video_mbuf        gb_buffers = { 2*0x151000, 0, {0,0x151000 }};

static char *map = NULL;

/* state */
/* static int                      overlay, swidth, sheight; */
static int swidth, sheight;

/* pass 0/1 by reference */
/* static int                      one = 1, zero = 0; */

struct GRABBER grab_v4l = {
    "video4linux",
    VIDEO_RGB16 | VIDEO_RGB24 | VIDEO_RGB32,
    0,
    norms,NULL,

    grab_open,
    grab_close,

    grab_scr,
    grab_one,
    grab_tune,
    grab_tuned,
    grab_input,
    grab_picture,
    grab_audio
};


/* ---------------------------------------------------------------------- */
static int alarms;
static void sigalarm(int signal)
{
    alarms++;
    fprintf(stderr,"v4l: timeout (got SIGALRM), hardware/driver problems?\n");
}


static void siginit(void)
{
    struct sigaction act, old;
    
    memset(&act, 0, sizeof(act));
    act.sa_handler = sigalarm;
    sigemptyset(&act.sa_mask);
    sigaction(SIGALRM, &act, &old);
}

static int grab_open(char *filename, int sw, int sh, int format, int pixmap, void *base, int bpl)
{
    int i;

    if (-1 != fd)
	goto err;
    
    if (-1 == (fd = open(filename ? filename : "/dev/video0",O_RDWR))) {
	fprintf(stderr,"open %s: %s\n",
		filename ? filename : "/dev/video0",strerror(errno));
	goto err;
    }

    if (-1 == ioctl(fd,VIDIOCGCAP,&capability))
	goto err;

    if (debug)
	fprintf(stderr, "v4l: open\n");

    siginit();

    swidth  = sw;
    sheight = sh;

    if (debug) {
	fprintf(stderr,"%s:",capability.name);
	for (i = 0; device_cap[i] != NULL; i++)
	    if (capability.type & (1 << i))
		fprintf(stderr," %s",device_cap[i]);
	fprintf(stderr,"\n");
    }
    
#ifdef STRANGE_SYNC_PROBLEMS
    /* for some reason if we don't open the vbi we can't SYNC later on
       some systems */
    if (-1 == (vbifd = open ("/dev/vbi", O_RDONLY))) {
      fprintf (stderr, "can't open /dev/vbi for reading, %s\n",
	       strerror (errno));
      goto err;
    }
#endif

    /* input sources */
    if (debug)
	fprintf(stderr,"  channels: %d\n",capability.channels);

    channels = malloc(sizeof(struct video_channel)*capability.channels);
    memset(channels,0,sizeof(struct video_channel)*capability.channels);
    inputs = malloc(sizeof(struct STRTAB)*(capability.channels+1));
    memset(channels,0,sizeof(struct STRTAB)*(capability.channels+1));

    for (i = 0; i < capability.channels; i++) {
	channels[i].channel = i;

	if (-1 == ioctl(fd,VIDIOCGCHAN,&channels[i]))
	    perror("ioctl VIDIOCGCHAN"), exit(0);

	inputs[i].nr  = i;
	inputs[i].str = channels[i].name;

	if (debug)
	    fprintf(stderr,"    %s: %d %s%s %s%s\n",
		    channels[i].name,
		    channels[i].tuners,
		    (channels[i].flags & VIDEO_VC_TUNER)   ? "tuner "  : "",
		    (channels[i].flags & VIDEO_VC_AUDIO)   ? "audio "  : "",
		    (channels[i].type & VIDEO_TYPE_TV)     ? "tv "     : "",
		    (channels[i].type & VIDEO_TYPE_CAMERA) ? "camera " : "");
    }

    inputs[i].nr  = -1;
    inputs[i].str = NULL;
    grab_v4l.inputs =inputs;

    /* ioctl probe, switch to input 0 */
    if (-1 == ioctl(fd,VIDIOCSCHAN,&channels[0])) {
	fprintf(stderr,"v4l: you need a newer bttv version (>= 0.5.14)\n");
	goto err;
    }

    /* audios */
    if (debug)
	fprintf(stderr,"  audios  : %d\n",capability.audios);
    audios = malloc(sizeof(struct video_audio)*capability.audios);
    memset(audios,0,sizeof(struct video_audio)*capability.audios);
    for (i = 0; i < capability.audios; i++) {
	audios[i].audio = i;
	if (-1 == ioctl(fd, VIDIOCGAUDIO, &audios[i]))
	    perror("ioctl VIDIOCGCAUDIO") /* , exit(0) */ ;
	if (debug) {
	    fprintf(stderr,"    %d (%s): ",i,audios[i].name);
	    if (audios[i].flags & VIDEO_AUDIO_MUTABLE)
		fprintf(stderr, "muted=%s ",
			(audios[i].flags&VIDEO_AUDIO_MUTE) ? "yes":"no");
	    if (audios[i].flags & VIDEO_AUDIO_VOLUME)
		fprintf(stderr, "volume=%d ",audios[i].volume);
	    if (audios[i].flags & VIDEO_AUDIO_BASS)
		fprintf(stderr, "bass=%d ",audios[i].bass);
	    if (audios[i].flags & VIDEO_AUDIO_TREBLE)
		fprintf(stderr, "treble=%d ",audios[i].treble);
	    fprintf(stderr, "\n");
	}
    }
    if (audios[0].flags & VIDEO_AUDIO_VOLUME)
	grab_v4l.flags |= CAN_AUDIO_VOLUME;

    if (debug)
	fprintf(stderr,"  size    : %dx%d => %dx%d\n",
		capability.minwidth, capability.minheight,
		capability.maxwidth, capability.maxheight);

    /* tuner (more than one???) */
    if (ioctl(fd, VIDIOCGTUNER, &tuner) == -1)
	perror("ioctl VIDIOCGTUNER");
    if (debug)
	fprintf(stderr,"  tuner   : %s %lu-%lu",
		tuner.name,tuner.rangelow,tuner.rangehigh);
    for (i = 0; norms[i].str != NULL; i++) {
	if (tuner.flags & (1 << i)) {
	    if (debug)
		fprintf(stderr, " %s", norms[i].str);
	} else
	    norms[i].nr = -1;
    }
    if (debug)
	fprintf(stderr,"\n");
    
    /* frame buffer */
    if (-1 == ioctl(fd,VIDIOCGFBUF,&ov_fbuf))
	perror("ioctl VIDIOCGFBUF");
    if (debug)
	fprintf(stderr,"  fbuffer : base=0x%p size=%dx%d depth=%d bpl=%d\n",
		ov_fbuf.base, ov_fbuf.width, ov_fbuf.height,
		ov_fbuf.depth, ov_fbuf.bytesperline);

    /* picture parameters */
    if (-1 == ioctl(fd,VIDIOCGPICT,&pict))
	perror("ioctl VIDIOCGPICT");

    if (debug) {
	fprintf(stderr,
		"  picture : brightness=%d hue=%d colour=%d contrast=%d\n",
		pict.brightness, pict.hue, pict.colour, pict.contrast);
	fprintf(stderr,
		"  picture : whiteness=%d depth=%d palette=%s\n",
		pict.whiteness, pict.depth, PALETTE(pict.palette));
    }

    /* double-check settings */
    if (sw && sh) {
	if (debug) fprintf(stderr,"v4l: %dx%d, %d bit/pixel, %d byte/scanline\n",
		ov_fbuf.width,ov_fbuf.height,
		ov_fbuf.depth,ov_fbuf.bytesperline);
	if ((ov_fbuf.bytesperline != bpl) ||
	    (ov_fbuf.width  != sw) ||
	    (ov_fbuf.height != sh)) {
	    fprintf(stderr,"v4l and dga disagree about the screen size\n");
	    fprintf(stderr,"v4l: %dx%d, %d byte/scanline\n", ov_fbuf.width, ov_fbuf.height, ov_fbuf.bytesperline);
	    fprintf(stderr,"dga: %dx%d, %d byte/scanline\n", sw, sh, bpl);
	    fprintf(stderr,"Is v4l-conf installed correctly?\n");
	    fprintf(stderr,"Continue anyway...\n");
	    /* exit(1); */
	}
    }
    if (format &&
	((format == VIDEO_GRAY  && ov_fbuf.depth !=  8) ||
	 (format == VIDEO_RGB08 && ov_fbuf.depth !=  8) ||
	 (format == VIDEO_RGB15 && ov_fbuf.depth != 15) ||
	 (format == VIDEO_RGB16 && ov_fbuf.depth != 16) ||
	 (format == VIDEO_RGB24 && ov_fbuf.depth != 24) ||
	 (format == VIDEO_RGB32 && ov_fbuf.depth != 32))) {
        fprintf(stderr,"v4l and dga disagree about the color depth\n");
        fprintf(stderr,"Is v4l-conf installed correctly?\n");
	fprintf(stderr,"Continue anyway...\n");
	/* exit(1); */
    }
    if (sw && sh && have_dga) {
	if (debug) fprintf(stderr,"v4l: framebuffer base=%p\n", ov_fbuf.base);
    }

    /* map grab buffer */
    if (-1 == ioctl(fd,VIDIOCGMBUF,&gb_buffers)) {
	perror("ioctl VIDIOCGMBUF");
    }
    map = mmap(0,gb_buffers.size,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);
    if ((char*)-1 == map) {
	perror("mmap");
    } else {
	if (debug) fprintf(stderr,"v4l: mmap()'ed buffer size = 0x%x\n", gb_buffers.size);
    }

    gb_even.format = gb_odd.format = VIDEO_PALETTE_RAW;

    gb_even.frame = 0;
    gb_odd.frame  = 1;

    return 0;

err:
    if (fd != -1) {
	close(fd);
	fd = -1;
    }
    
#ifdef STRANGE_SYNC_PROBLEMS
    if (vbifd != -1) {
      close(vbifd);
      vbifd = -1;
    }
#endif
    
    return -1;
}


static int grab_close()
{
    ioctl(fd, BTTV_BURST_OFF, NULL);
    if (gb_grab > gb_sync) {
	if (-1 == ioctl(fd, VIDIOCSYNC, 0))
	    perror("ioctl VIDIOCSYNC");
	else
	    gb_sync++;
    }

    if ((char*) -1 != map)
	munmap(map, gb_buffers.size);

    if (-1 == fd)
	return 0;

    if (debug)
	fprintf(stderr, "v4l: close\n");

    close(fd);

    fd = -1;

#ifdef STRANGE_SYNC_PROBLEMS
    close(vbifd);

    vbifd = -1;
#endif
    return 0;
}


/* ---------------------------------------------------------------------- */

static int grab_queue(struct video_mmap *gb)
{
    gb->format = 12;

    if (-1 == ioctl(fd,VIDIOCMCAPTURE,gb)) {
	if (errno == EAGAIN)
	    fprintf(stderr,"grabber chip can't sync (no station tuned in?)\n");
	else
	    fprintf(stderr,"ioctl VIDIOCMCAPTURE(%d,%s,%dx%d): %s\n",
		    gb->frame,PALETTE(gb->format),gb->width,gb->height,
		    strerror(errno));
	return -1;
    }
    if (debug)
	fprintf(stderr,"* ");
    gb_grab++;
    return 0;
}


static void grab_wait(struct video_mmap *gb)
{
    if (debug) {
	fprintf(stderr,"s%d",gb->frame);
    }

    gb->format = 12;
    alarms = 0;
    alarm(SYNC_TIMEOUT);
    
 retry:
    if (-1 == ioctl(fd, VIDIOCSYNC, &(gb->frame))) {
	if (errno == EINTR && !alarms) {
	    goto retry;
	}
	perror("ioctl VIDIOCSYNC");
    } else {
	gb_sync++;
    }
    
    alarm(0);
    
    if (debug) {
	fprintf(stderr, "* ");
    }
}


static void* grab_scr(void *dest, int width, int height, int single, int dest_xsize)
{
    void *buf;
    int height_orig;
    int width_orig;
    int width_start;

/*     if (debug) */
/* 	fprintf(stderr, "grab_scr(%x, %d, %d, %d, %d)\n", dest, width, height, single, dest_xsize); */

    if ((char*)-1 == map)
	return NULL;

    if (!gb_even.format)
	return NULL;

    if (width > 768)
        width = 768;

    height_orig = height;
    width_orig = width;
    width_start = width;
    ioctl(fd, BTTV_BURST_ON, NULL);

/*     height = 288 * 2; */
/*     width_orig = 720; */
/*     width = 720; */
    width += width * BURSTOFFSET / 922;
    width &= ~1;

    gb_even.format = 12;
    gb_odd.format = 12;

/*     width = 567; */

    gb_even.width  = width;
    gb_even.height = height;
    gb_odd.width  = width;
    gb_odd.height = height;

    if (single) {
        if (gb_grab > gb_sync)
            grab_wait(even ? &gb_even : &gb_odd);
    } else {
        if (gb_grab == gb_sync)
            if (-1 == grab_queue(even ? &gb_even : &gb_odd))
                return NULL;
    }

    if (-1 == grab_queue(even ? &gb_odd : &gb_even))
        return NULL;

    if (gb_grab > gb_sync+1) {
        grab_wait(even ? &gb_even : &gb_odd);
        buf = map + gb_buffers.offsets[even ? 0 : 1];
    } else {
        grab_wait(even ? &gb_odd : &gb_even);
        buf = map + gb_buffers.offsets[even ? 1 : 0];
    }

    even = !even;

    switch (decode_routine) {
    case 0:
        decodeFrameCableCryptGrayScaleAsm((unsigned char *) dest, (unsigned char *) buf, width_orig, width, height_orig, dest_xsize);
	break;

    case 1:
        decodeFrameCableCryptColorC((unsigned char *) dest, (unsigned char *) buf, width_orig, width, height_orig, dest_xsize);
	break;

    case 2:
        decodeFrameCableCryptColorDoubleC((unsigned char *) dest, (unsigned char *) buf, width_orig, width, height_orig, dest_xsize);
	break;

    case 3:
        decodeFrameCableCryptColorMMX((unsigned char *) dest, (unsigned char *) buf, width_orig, width, height_orig, dest_xsize);
	break;

    case 4:
        decodeFrameCableCryptColorDoubleMMX((unsigned char *) dest, (unsigned char *) buf, width_orig, width, height_orig, dest_xsize);
	break;

    default:
	printf("Routine not supported! Exiting...\n");
	exit(-1);
    }

    return dest;
}

void rgb16to24(unsigned char *to, unsigned char *from, int pixels)
{
    int i;
	
    for (i = 0; i < pixels; i++) {
	to[0] = (*(from + 1)) & 0xf8;
	to[1] = (((*(from + 1)) & 0x07) << 5) + (((*from) & 0xe0) >> 3);
	to[2] = ((*from) & 0x1f) << 3;
	to += 3;
	from += 2;
    }
}

static void *grab_one(int width, int height)
{
    struct video_mmap gb;
    char dest[width * 288 * 2 * 2];

    printf("grab_one(%d, %d)\n", width, height);

    if ((char *) -1 == map)
	return NULL;
    
    if (gb_grab > gb_sync)
	grab_wait(even ? &gb_even : &gb_odd);
    
/*    gb.format = VIDEO_PALETTE_RGB24; */
/*    gb.frame  = 0; */
/*    gb.width  = width; */
/*    gb.height = height; */

    gb.format = VIDEO_PALETTE_RAW;
    gb.frame  = 0;
    gb.width  = width;
    gb.height = 288 * 2;

    memset(map, 0, width * height * 3);
    if (-1 == grab_queue(&gb))
	return NULL;
    grab_wait(&gb);

    rgb16to24((unsigned char*) map, (unsigned char*) dest, width * height);

    return map;
}

/* ---------------------------------------------------------------------- */

static int grab_tune(unsigned long freq)
{
/*    printf("grab_tune(%d)\n",freq); */

    if (debug)
	fprintf(stderr,"v4l: freq: %.3f\n",(float)freq/16);
    if (-1 == ioctl(fd, VIDIOCSFREQ, &freq))
	perror("ioctl VIDIOCSFREQ");
    return 0;
}

static int grab_tuned()
{
#if 1
    /* (quick & dirty -- grabbing works with hsync only) */
    return grab_one(64, 48) ? 1 : 0;
#else
    if (-1 == ioctl(fd, VIDIOCGTUNER, &tuner)) {
	perror("ioctl VIDIOCGTUNER");
	return 0;
    }
    return tuner.signal ? 1 : 0;
#endif
}

static int grab_input(int input, int norm)
{
    if (-1 != input) {
	if (debug)
	    fprintf(stderr,"v4l: input: %d\n",input);
	cur_input = input;
    }
    if (-1 != norm) {
	if (debug)
	    fprintf(stderr,"v4l: norm : %d\n",norm);
	cur_norm = norm;
    }

    channels[cur_input].norm = cur_norm;
    if (-1 == ioctl(fd, VIDIOCSCHAN, &channels[cur_input]))
	perror("ioctl VIDIOCSCHAN");
    
    return 0;
}

int grab_picture(int color, int bright, int hue, int contrast)
{
/*    printf("grab_picture(%d,%d,%d,%d)\n",color,bright,hue,contrast); */

    if (color != -1)
	pict.colour = color;
    if (contrast != -1)
	pict.contrast = contrast;
    if (bright != -1)
	pict.brightness = bright;
    if (hue != -1)
	pict.hue = hue;

    if (-1 == ioctl(fd, VIDIOCSPICT, &pict))
	perror("ioctl VIDIOCSPICT");

    return 0;
}

int grab_audio(int unmute, int volume, int *mode)
{
/*    printf("grab_audio(%d,%d,%x)\n",mute,volume,mode); */
	
/*    if (mute != -1) { */
	if (unmute) {
	    audios[0].flags &= ~VIDEO_AUDIO_MUTE;
	} else {
	    audios[0].flags |= VIDEO_AUDIO_MUTE;
	}
/*    } */
    if (volume != -1)
	audios[0].volume = volume;

    audios[0].audio = cur_input;
    audios[0].mode = mode ? *mode : 0;
    if (-1 == ioctl(fd, VIDIOCSAUDIO, &audios[0]))
	perror("ioctl VIDIOCSAUDIO");

    if (mode) {
	if (-1 == ioctl(fd, VIDIOCGAUDIO, &audios[0]))
	    perror("ioctl VIDIOCGAUDIO");
	*mode = audios[0].mode;
    }
    
    return 0;
}
