/*
 * main.c for CableTV -- a CableCrypt decoder TV application
 * based on xawtv by
 *   (c) 1997 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/io.h>
#include <sys/mman.h>

#include <getopt.h>

#include "config.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/Xmd.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Box.h>
#ifdef HAVE_LIBXXF86DGA
# include <X11/extensions/xf86dga.h>
# include <X11/extensions/xf86dgastr.h>
#endif
#ifdef HAVE_LIBXXF86VM
# include <X11/extensions/xf86vmode.h>
# include <X11/extensions/xf86vmstr.h>
#endif
#ifdef HAVE_MITSHM
# include <X11/extensions/XShm.h>
#endif
#ifdef HAVE_LIBXDPMS
# include <X11/extensions/dpms.h>
/* XFree 3.3.x hasn't prototypes for this ... */
Bool   DPMSQueryExtension(Display*, int*, int*);
Bool   DPMSCapable(Display*);
Status DPMSInfo(Display*, CARD16*, BOOL*);
Status DPMSEnable(Display*);
Status DPMSDisable(Display*);
#endif
/*
#ifdef HAVE_LIBXV
# include <X11/extensions/Xv.h>
# include <X11/extensions/Xvlib.h>
#endif
*/
#ifdef HAVE_LIBPCI
#include <pci/pci.h>
#endif


#include "mixer.h"
#include "channel.h"
#include "channels.h"
#include "grab.h"
#include "x11.h"
#include "toolbox.h"
#include "esnelDecode.h"
#include "lirc-support.h"
#include "ccd.h"


#define ONSCREEN_TIME       6000
#define TITLE_TIME          6000
#define SCAN_TIME            100

#define WIDTH_INC             64
#define HEIGHT_INC            48
#define LABEL_WIDTH         "16"
#define VIDMODE_DELAY        100   /* 0.1 sec */

/*--- public variables ----------------------------------------------------*/

char *video_base;
int direct_video = 0;
int direct_xsize;
int noDGA = 0;

XtAppContext      app_context;
Widget            app_shell, tv;
Widget            opt_shell, opt_paned, chan_shell, conf_shell;
Widget            on_shell, on_label;
Widget            c_norm,c_input,c_freq,c_audio,c_cap,c_decode;
Widget            s_bright,s_color,s_hue,s_contrast,s_volume;
Widget            chan_viewport, chan_box;
Display           *dpy;
XtWorkProcId      idle_id;

int               have_config = 0;
Atom              wm_protocols[2],xawtv_remote,xawtv_station;
XtIntervalId      title_timer, audio_timer, scan_timer, on_timer;
int               pointer_on = 1, on_skip = 0;
int               debug = 0;
int               fs = 0;
int               fs_width,fs_height,fs_xoff,fs_yoff;
int               pix_width=128,pix_height=96;
int               bpp = 0;

char              modename[64];
char              *progname;
int               xfree_ext = 1;
int               have_dga = 0;
int               have_vm = 0;
#ifdef HAVE_LIBXXF86VM
int               vm_count;
XF86VidModeModeInfo **vm_modelines;
#endif

/*                            PAL  NTSC SECAM */
static int    maxwidth[]  = { 768, 640, 768 };
static int    maxheight[] = { 576, 480, 576 };

/* variabled to save working state of bt848 */
int pllfrequency,
    dropevenfield,
    droprate,
    verticaltotal,
    colorformat,
    testregister,
    extframe,
    fifoenabled,
    captureevenenabled,
    captureoddenabled;

/* is locking on? */
int locking = 0;
int decode_routine;
int fps = 0;
int picoffset = 0;

char *routine_string[] =  { "pal grayscale"  , "pal color", "pal color MMX" };

static unsigned long BTType;


void set_timer_title();

/*--- drivers -------------------------------------------------------------*/

char *device = "/dev/video0";

extern struct GRABBER grab_v4l;
struct GRABBER *grabbers[] = {
    &grab_v4l,
};

int grabber;

/*--- channels ------------------------------------------------------------*/

struct STRTAB    *cmenu    = NULL;
char title[512];

int cur_color;
int cur_bright;
int cur_hue;
int cur_contrast;
int cur_capture;

int cur_mute   = 0;
int cur_volume = 65535;

/*--- actions -------------------------------------------------------------*/

void CloseMainAction(Widget, XEvent*, String*, Cardinal*);
void SetChannelAction(Widget, XEvent*, String*, Cardinal*);
void TuneAction(Widget, XEvent*, String*, Cardinal*);
/* void ScanAction(Widget, XEvent*, String*, Cardinal*); */
void ChannelAction(Widget, XEvent*, String*, Cardinal*);
void VolumeAction(Widget, XEvent*, String*, Cardinal*);
void PointerAction(Widget, XEvent*, String*, Cardinal*);
void FullScreenAction(Widget, XEvent*, String*, Cardinal*);
/* void RemoteAction(Widget, XEvent*, String*, Cardinal*); */

void LockingAction (Widget, XEvent*, String*, Cardinal*);
void BrightnessUpAction (Widget, XEvent*, String*, Cardinal*);
void BrightnessDownAction (Widget, XEvent*, String*, Cardinal*);
void ContrastUpAction (Widget, XEvent*, String*, Cardinal*);
void ContrastDownAction (Widget, XEvent*, String*, Cardinal*);
void SaturationUpAction (Widget, XEvent*, String*, Cardinal*);
void SaturationDownAction (Widget, XEvent*, String*, Cardinal*);
void AdjustLumaLevelAction (Widget, XEvent*, String*, Cardinal*);
void InvertPolarityAction (Widget, XEvent*, String*, Cardinal*);
void InvertChromaAction (Widget, XEvent*, String*, Cardinal*);

void VerticalTotalUpAction (Widget, XEvent*, String*, Cardinal*);
void VerticalTotalDownAction (Widget, XEvent*, String*, Cardinal*);
void PllFrequencyUpAction (Widget, XEvent*, String*, Cardinal*);
void PllFrequencyDownAction (Widget, XEvent*, String*, Cardinal*);
void TestRegisterAction (Widget, XEvent*, String*, Cardinal*);

void ShowFPSAction (Widget, XEvent*, String*, Cardinal*);
void VersionInfoAction (Widget, XEvent*, String*, Cardinal*);

void RoutineChangeAction (Widget, XEvent*, String*, Cardinal*);

static XtActionsRec actionTable[] = {
    { "CloseMain",   CloseMainAction  },
    { "SetChannel",  SetChannelAction },
    { "Tune",        TuneAction },
/*     { "Scan",        ScanAction }, */
    { "Channel",     ChannelAction },
    { "Volume",      VolumeAction },
    { "Pointer",     PointerAction },
    { "FullScreen",  FullScreenAction },
/*     { "Remote",      RemoteAction }, */

    { "Locking", LockingAction },
    { "BrightnessUp", BrightnessUpAction },
    { "BrightnessDown", BrightnessDownAction },
    { "ContrastUp", ContrastUpAction },
    { "ContrastDown", ContrastDownAction },
    { "SaturationUp", SaturationUpAction },
    { "SaturationDown", SaturationDownAction },
    { "AdjustLumaLevel", AdjustLumaLevelAction },
    { "InvertPolarity", InvertPolarityAction },
    { "InvertChroma", InvertChromaAction },

    { "VerticalTotalUp", VerticalTotalUpAction },
    { "VerticalTotalDown", VerticalTotalDownAction },
    { "PllFrequencyUp", PllFrequencyUpAction },
    { "PllFrequencyDown", PllFrequencyDownAction },
    { "TestRegister", TestRegisterAction },

    { "ShowFPS", ShowFPSAction },
    { "VersionInfo", VersionInfoAction },

    { "RoutineChange", RoutineChangeAction },
};

static struct STRTAB stereo[] = {
    {  0, "auto"    },
    {  1, "mono"    },
    {  2, "stereo"  },
    {  3, "lang1"   },
    {  4, "lang2"   },
    { -1, NULL,     },
};


/*--- exit ----------------------------------------------------------------*/

Boolean ExitWP(XtPointer client_data)
{
    /* exit if the application is idle,
     * i.e. all the DestroyCallback's are called.
     */

    /* restore default state of bt848 */
    setPllFrequency(pllfrequency);
    setDropEvenField(dropevenfield);
    setDropRate(droprate);
    setVerticalTotal(verticaltotal);
    setColorFormat(colorformat);
    setTest(testregister);
    setExtFrame(extframe);
    setFifoEnabled(fifoenabled);
    setCaptureEvenEnabled(captureevenenabled);
    setCaptureOddEnabled(captureoddenabled);

    exit(0);
}

void ExitCB(Widget widget, XtPointer client_data, XtPointer calldata)
{
/*    audio_off(); */
    video_overlay(NULL);
    video_close();

    if (have_mixer)
	mixer_close();

    grabbers[grabber]->grab_audio(0, -1, null);
    
    if (fs)
	XtCallActionProc(widget, "FullScreen", NULL, NULL, 0);

    XSync(dpy, False);

    if (grabbers[grabber]->grab_close) {
        grabbers[grabber]->grab_close();
    }

    XtAppAddWorkProc(app_context, ExitWP, NULL);
    XtDestroyWidget(app_shell);
}

void CloseMainAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    static Dimension x,y,w,h;
    char *argv[32];
    int   argc = 0;

    if (fs) {
	XtCallActionProc(widget, "FullScreen", NULL, NULL, 0);
    }

    if (event->type == ClientMessage) {
	if (event->xclient.data.l[0] == wm_protocols[1]) {
	    if (debug)
		fprintf(stderr,"Main: wm_save_yourself\n");

	    argv[argc++] = progname;

	    if (fs) {
		argv[argc++] = (char *) strdup("-f");
	    } else {
		XtVaGetValues(app_shell, XtNx, &x, XtNy, &y, XtNwidth, &w, XtNheight, &h, NULL);
		argv[argc++] = (char *) strdup("-geometry");
		sprintf(argv[argc++] = malloc(32), "%dx%d+%d+%d", w, h, x, y);
	    }
	    argv[argc++] = (char *) strdup("-c");
	    argv[argc++] = (char *) strdup(device);
	    
	    if (!xfree_ext)
		argv[argc++] = (char *) strdup("-x");
	    if (!pointer_on)
		argv[argc++] = (char *) strdup("-m");

	    argv[argc++] = channels[cur_sender]->name;

	    XSetCommand(XtDisplay(app_shell), XtWindow(app_shell), argv, argc);
	    return;
	}
    }

    ExitCB(widget,NULL,NULL);
}

/*--- onscreen display (fullscreen) --------------------------------------*/

void create_onscreen()
{
    on_shell = XtVaCreateWidget("onscreen",transientShellWidgetClass,
				app_shell,
				XtNoverrideRedirect,True,
				NULL);    
    on_label = XtVaCreateManagedWidget("label", labelWidgetClass, on_shell,
					NULL);
}

void popdown_onscreen(XtPointer client_data, XtIntervalId *id)
{
    XtPopdown(on_shell);
    on_timer = 0;
}

void display_onscreen()
{
    static int first = 1;
    Dimension x, y;

    if (on_skip) {
	on_skip = 0;
	return;
    }
    if (!fs)
	return;

    XtVaGetValues(app_shell, XtNx, &x, XtNy, &y, NULL);
    XtVaSetValues(on_shell, XtNx, x+20, XtNy, y+10, NULL);
    XtVaSetValues(on_label, XtNlabel, title, NULL);
    XtPopup(on_shell, XtGrabNone);
    
    if (on_timer)
	XtRemoveTimeOut(on_timer);
    
    on_timer = XtAppAddTimeOut(app_context, ONSCREEN_TIME, popdown_onscreen, NULL);

    if (first) {
	first = 0;
	XDefineCursor(dpy, XtWindow(on_shell), no_ptr);
	XDefineCursor(dpy, XtWindow(on_label), no_ptr);
    }
}

/*--- tv -----------------------------------------------------------------*/

static Boolean idle_grabdisplay(XtPointer data)
{
    if (debug) 
	fprintf(stderr, "idle loop enter\n");
    
    if ((grabbers[grabber]->grab_scr == NULL) || (video_displayframe(grabbers[grabber]->grab_scr) == -1)) {
  	idle_id = 0;
  	usleep(300000);
        printf("GRABBING ERROR\n");
    }

    if (!locking) {
	sprintf(title, "CableTV: start position: [%4d]", picoffset);
        set_timer_title();
    }

    if (debug)
	fprintf(stderr, "idle loop leave\n");
    
    return FALSE;
}

void set_title()
{
    if (cur_sender != -1) {
	sprintf(title, "CableTV: %s", channels[cur_sender]->name);
    } else {
	sprintf(title, "CableTV: channel %s", tvtuner[cur_channel].name);
	if (cur_fine != 0)
	    sprintf(title + strlen(title), " (%d)", cur_fine);

	sprintf(title + strlen (title), " (%s/%s)",
		grabbers[grabber]->norms[cur_norm].str,
		chan_names[chan_tab].str);
    }
    XtVaSetValues(app_shell, XtNtitle, title, NULL);
    display_onscreen();

    if (title_timer) {
	XtRemoveTimeOut(title_timer);
	title_timer = 0;
    }
}

void set_property(int freq, char *channel, char *name)
{
    int len;
    char line[80];

    len  = sprintf(line, "%.3f", (float) freq / 16) + 1;
    len += sprintf(line + len, "%s", channel) + 1;
    if (name != NULL)
	len += sprintf(line + len, "%s", name) + 1;
    
    XChangeProperty(dpy, XtWindow(app_shell), xawtv_station, XA_STRING, 8, PropModeReplace, (unsigned char *) line, len);
}

void set_title_timeout(XtPointer client_data, XtIntervalId *id)
{
    on_skip = 1;
    set_title();
}

void set_timer_title()
{
    XtVaSetValues(app_shell, XtNtitle, title, NULL);
    display_onscreen();
    if (title_timer)
	XtRemoveTimeOut(title_timer);
    
    title_timer = XtAppAddTimeOut(app_context, TITLE_TIME, set_title_timeout,NULL);
}

void change_audio(int mode)
{
    if (grabbers[grabber]->grab_audio) {
	grabbers[grabber]->grab_audio(1, -1, &mode);

	if (mode > 5) mode = 0;
    }
    sprintf(title, "%-" LABEL_WIDTH "s: %s", "Audio", stereo[mode].str);
    if (c_audio)
	XtVaSetValues(c_audio, XtNlabel, title, NULL);
    set_title();
    XtVaSetValues(app_shell, XtNtitle, title, NULL);
}

void watch_audio(XtPointer data, XtIntervalId *id)
{
    on_skip = 1;
    change_audio(0);
    audio_timer = 0;
}

void set_norm(int j)
{
    sprintf(title,"%-" LABEL_WIDTH "s: %s", "TV Norm", grabbers[grabber]->norms[j].str);
    if (c_norm)
	XtVaSetValues(c_norm,XtNlabel,title,NULL);
    cur_norm = j;
    grabbers[grabber]->grab_input(-1,cur_norm);
    video_setmax(maxwidth[cur_norm],maxheight[cur_norm]);
}

void set_source(int j)
{
    sprintf(title, "%-" LABEL_WIDTH "s: %s", "Video Source", grabbers[grabber]->inputs[j].str);
    if (c_input)
	XtVaSetValues(c_input, XtNlabel, title, NULL);
    cur_input = j;
    grabbers[grabber]->grab_input(cur_input, -1);
}

void set_freqtab(int j)
{
    sprintf(title, "%-" LABEL_WIDTH "s: %s", "Frequency table", chan_names[j].str);
    if (c_freq)
	XtVaSetValues(c_freq, XtNlabel, title, NULL);
    chan_tab = j;
}

void set_capture(int capture)
{
    static int niced = 0;

    if (debug)
	fprintf(stderr, "set_capture\n");
	
    idle_id = XtAppAddWorkProc(app_context, idle_grabdisplay, NULL);
    
    if (idle_id) {
	if (debug)
	    fprintf(stderr, "Removed idle proc!\n");
	XtRemoveWorkProc(idle_id);
    }
    
    idle_id = 0;
    XClearArea(XtDisplay(tv), XtWindow(tv), 0,0,0,0, True);

    cur_capture = capture;

    sprintf(title, "%-" LABEL_WIDTH "s: %s", "Capture", "grabdisplay");
    if (!niced)
	nice(niced = 10);
    
    idle_id = XtAppAddWorkProc(app_context, idle_grabdisplay, NULL);
    if (idle_id) {
	if (debug)
	    fprintf(stderr, "No idle proc set!\n");
    }
    if (c_cap)
	XtVaSetValues(c_cap, XtNlabel, title, NULL);
}


/* the RightWay[tm] to set float resources (copyed from Xaw specs) */
void set_float(Widget widget, char *name, float value)
{
    Arg args[1];

    if (sizeof(float) > sizeof(XtArgVal)) {
	/*
	 * If a float is larger than an XtArgVal then pass this 
	 * resource value by reference.
	 */
	XtSetArg(args[0], name, &value);
    } else {
        /*
	 * Convince C not to perform an automatic conversion, which
	 * would truncate 0.5 to 0. 
	 */
	XtArgVal * l_top = (XtArgVal *) &value;
	XtSetArg(args[0], name, *l_top);
    }
    XtSetValues(widget, args, 1);
}


void set_picparams(int color, int bright, int hue, int contrast)
{
    if (color != -1) {
	cur_color = color;
	if (s_color)
	    set_float(s_color, XtNtopOfThumb, (float) cur_color / 65536);
    }
    if (bright != -1) {
	cur_bright = bright;
	if (s_bright)
	    set_float(s_bright, XtNtopOfThumb, (float) cur_bright / 65536);
    }
    if (hue != -1) {
	cur_hue = hue;
	if (s_hue)
	    set_float(s_hue, XtNtopOfThumb, (float) cur_hue / 65536);
    }
    if (contrast != -1) {
	cur_contrast = contrast;
	if (s_contrast)
	    set_float(s_contrast, XtNtopOfThumb, (float) cur_contrast / 65536);
    }
    grabbers[grabber]->grab_picture(cur_color, cur_bright, cur_hue, cur_contrast);
}


void pixit()
{
    Pixmap pix;
    unsigned char *data;

    if (0 == pix_width || 0 == pix_height)
	return;

    if (cur_sender != -1 && grabbers[grabber]->grab_scr) {
	strcpy(title, channels[cur_sender]->name);
/* 	fprintf(stderr, " malloc(%d * %d * 16);\n", pix_width, pix_height); */
	data = malloc(pix_width * pix_height * 16);    /* FUX */
	
	if (NULL != grabbers[grabber]->grab_scr(data, pix_width, pix_height, 1, pix_width) &&
	    0 != (pix = x11_create_pixmap(channels[cur_sender]->button, data, pix_width, pix_height, title))) {
	    
	    if (channels[cur_sender]->pixmap)
		XFreePixmap(dpy,channels[cur_sender]->pixmap);
	    
	    channels[cur_sender]->pixmap = pix;
	    XtVaSetValues(channels[cur_sender]->button,
			  XtNbackgroundPixmap, pix,
			  XtNlabel, "",
			  XtNwidth, pix_width,
			  XtNheight, pix_height,
			  NULL);
	}
	
	free(data);
    }
}

    
void set_channel(struct CHANNEL *channel)
{
    if (debug)
	fprintf(stderr, "set_channel\n");
    
    /* image parameters */
    set_picparams(channel->color, channel->bright,
		  channel->hue, channel->contrast);
    set_capture(channel->capture);

    /* input source */
    if (cur_input   != channel->source)
	set_source(channel->source);
    if (cur_norm    != channel->norm)
	set_norm(channel->norm);

    /* station */
    cur_channel  = channel->channel;
    cur_fine     = channel->fine;
    grabbers[grabber]->grab_tune(channel->freq);
    set_title();
    set_property(channel->freq,channel->cname,channel->name);

    if (scan_timer) {
	XtRemoveTimeOut(scan_timer);
	scan_timer = 0;
    }
    if (audio_timer) {
	XtRemoveTimeOut(audio_timer);
	audio_timer = 0;
    }

    /*
     * set the timeout for audio on to 1000 msec.
     */
    audio_timer = XtAppAddTimeOut(app_context, 1000, watch_audio, NULL);
}


void set_volume()
{
    int vol;
    
    if (have_mixer) {
	/* sound card */
	vol = cur_volume * 100 / 65536;
	mixer_set_volume(vol);
	cur_mute ? mixer_mute() : mixer_unmute();
    } else {
	/* v4l */
	if (grabbers[grabber]->grab_audio)
	    grabbers[grabber]->grab_audio(cur_mute, cur_volume,NULL);
    }

    if (s_volume)
	set_float(s_volume, XtNtopOfThumb, (float) cur_volume / 65536);
}


void VolumeAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    if (*num_params < 1)
	return;

    if (strcasecmp(params[0], "mute") == 0)
	cur_mute = !cur_mute;
    else if (strcasecmp(params[0],"inc") == 0)
	cur_volume += 512;
    else if (strcasecmp(params[0],"dec") == 0)
	cur_volume -= 512;
    else
	cur_volume = atoi(params[0]);

    if (cur_volume < 0)     cur_volume = 0;
    if (cur_volume > 65535) cur_volume = 65535;

    set_volume();

    if (cur_mute)
	sprintf(title, "CableTV: Volume (%s): muted", have_mixer ? "mixer" : "v4l");
    else
	sprintf(title, "CableTV: Volume (%s): %d%%", have_mixer ? "mixer" : "v4l",
		cur_volume * 100 / 65535);
    set_timer_title();
}


void SetChannelAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    int i;
    
    if (*num_params > 1)
	return;

    if (count && 0 == strcmp(params[0], "next")) {
	i = (cur_sender + 1) % count;
    } else if (count && 0 == strcmp(params[0], "prev")) {
	i = (cur_sender + count - 1) % count;
    } else {
	/* search the configured channels first... */
	for (i = 0; i < count; i++)
	    if (0 == strcasecmp(channels[i]->name, params[0]))
		break;
	/* ... if it failes, take the argument as index */
	if (i == count)
	    i = atoi(params[0]);
    }
    if (i >= 0 && i < count) {
	pixit();
	cur_sender = i;
	set_channel(channels[i]);
    }
}

#if 0
void
RemoteAction(Widget widget, XEvent * event,
	     String * params, Cardinal * num_params)
{
    Atom            type;
    int             format, argc, i;
    char            *argv[32];
    unsigned long   nitems, bytesafter;
    unsigned char   *args = NULL;

    if (event->type == PropertyNotify) {
#if 0
	fprintf(stderr,"PropertyNotify %s\n",
		XGetAtomName(dpy,event->xproperty.atom));
#endif
	if (event->xproperty.atom == xawtv_remote &&
	    Success == XGetWindowProperty(dpy,
					  event->xproperty.window,
					  event->xproperty.atom,
					  0, (65536 / sizeof(long)),
					  True, XA_STRING,
					  &type, &format, &nitems, &bytesafter,
					  &args) &&
	    nitems != 0) {
	    if (debug)
		fprintf(stderr, "remote control: ");
	    for (i = 0, argc = 0; i < nitems; i += strlen(args + i) + 1) {
		if(debug)
		    fprintf(stderr, "%s ", args+i);
		argv[argc++] = args+i;
	    }
	    if (debug)
		fprintf(stderr, "\n");

	    if (0 == strcasecmp(argv[0],"setstation") && argc > 1) {
		XtCallActionProc(tv,"SetChannel",NULL,argv+1,1);
	    } else if (0 == strcasecmp(argv[0],"setchannel") && argc > 1) {
		XtCallActionProc(tv,"Tune",NULL,argv+1,1);
	    } else if (0 == strcasecmp(argv[0],"volume") && argc > 1) {
		XtCallActionProc(tv,"Volume",NULL,argv+1,1);
	    }
	    XFree(args);
	}
    }
}
#endif

void scan_timeout(XtPointer client_data, XtIntervalId *id)
{
    static String argv[] = { "next", NULL };

    scan_timer = 0;
    
    /* check */
    if (!grabbers[grabber]->grab_tuned)
	return;
    if (grabbers[grabber]->grab_tuned())
	return;

    XtCallActionProc(tv,"Tune",NULL,argv,1);
    scan_timer = XtAppAddTimeOut
	(app_context, SCAN_TIME, scan_timeout, NULL);
}

#if 0
void ScanAction (Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    static String argv[] = { "next", NULL };

    pixit ();
    XtCallActionProc (tv, "Tune", NULL, argv, 1);
    scan_timer = XtAppAddTimeOut (app_context, SCAN_TIME, scan_timeout, NULL);
}
#endif

void TuneAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    int c, freq;

    if (*num_params < 1)
	return;
    
    if (strcasecmp(params[0], "next") == 0) {
	do {
	    cur_channel = (cur_channel + 1) % CHAN_ENTRIES;
	} while (!tvtuner[cur_channel].freq[chan_tab]);
	
	cur_fine = defaults.fine;
    } else if (strcasecmp(params[0], "prev") == 0) {
	do {
	    cur_channel = (cur_channel + CHAN_ENTRIES - 1) % CHAN_ENTRIES;
	} while (!tvtuner[cur_channel].freq[chan_tab]);
	
	cur_fine = defaults.fine;
    } else if (strcasecmp(params[0], "fine_up") == 0) {
        cur_fine++;
    } else if (strcasecmp(params[0], "fine_down") == 0) {
	cur_fine--;
    } else {
	if (-1 != (c = lookup_channel(params[0]))) {
	    cur_channel = c;
	    cur_fine = defaults.fine;
	}
    }

    pixit();
    cur_sender = -1;
    set_capture(defaults.capture);

    freq = get_freq(cur_channel) + cur_fine;
    grabbers[grabber]->grab_tune(freq);
    set_title();
    set_property(freq, tvtuner[cur_channel].name, NULL);

    if (scan_timer) {
	XtRemoveTimeOut(scan_timer);
	scan_timer = 0;
    }
    if (audio_timer) {
	XtRemoveTimeOut(audio_timer);
	audio_timer = 0;
    }
    audio_timer = XtAppAddTimeOut(app_context, 10000, watch_audio, NULL);
}

void ChannelAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    int i;

    if (0 == count)
	return;
    i = popup_menu(widget,"Stations",cmenu);

    if (i != -1) {
	pixit();
	cur_sender = i-1;
	set_channel(channels[cur_sender]);
    }
}

void PointerAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    pointer_on = !pointer_on;
    XDefineCursor(dpy, XtWindow(tv), pointer_on ? left_ptr : no_ptr);
}

Boolean MyResize(XtPointer client_data)
{
    video_new_size();
    return TRUE;
}

#ifdef HAVE_LIBXXF86VM
static void vidmode_timer(XtPointer clientData, XtIntervalId *id)
{
    set_capture(CAPTURE_OVERLAY);
}

static void set_vidmode (int nr)
{
    if (CAPTURE_OVERLAY == cur_capture)
	video_overlay(NULL);
    usleep(VIDMODE_DELAY*1000);
    if (debug)
	fprintf(stderr,"switching video mode to %dx%d now\n",
		vm_modelines[nr]->hdisplay,
		vm_modelines[nr]->vdisplay);
    XF86VidModeSwitchToMode(dpy, XDefaultScreen(dpy), vm_modelines[nr]);

    if (CAPTURE_OVERLAY == cur_capture) {
	cur_capture = 0;
	XtAppAddTimeOut(app_context, VIDMODE_DELAY, vidmode_timer, NULL);
    }
}
#endif

void FullScreenAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    static Dimension x, y, w, h;
    static int timeout, interval, prefer_blanking, allow_exposures, rpx, rpy, mouse;
#ifdef HAVE_LIBXXF86VM
    static int vm_switched;
#endif
#ifdef HAVE_LIBXDPMS
    static BOOL dpms_on;
    CARD16 dpms_state;
    int dpms_dummy;
#endif

    Window root, child;
    int wpx, wpy;
    unsigned int mask;
    
    if (fs) {
#ifdef HAVE_LIBXXF86VM
	if (have_vm && vm_switched) {
	    set_vidmode(0);
	    vm_switched = 0;
	}
/* if (direct_video) */
/*     XF86DGADirectVideo(dpy,XDefaultScreen(dpy),0); */
#endif

	if (on_timer) {
	    XtPopdown(on_shell);
	    XtRemoveTimeOut(on_timer);
	    on_timer = 0;
	}
	    
	XtVaSetValues(app_shell, XtNwidthInc, WIDTH_INC, XtNheightInc,HEIGHT_INC,
		      XtNx, x + fs_xoff,
		      XtNy, y + fs_yoff,
		      XtNwidth, w,
		      XtNheight, h,
		      NULL);

	XSetScreenSaver(dpy,timeout,interval,prefer_blanking,allow_exposures);
#ifdef HAVE_LIBXDPMS
        if ((DPMSQueryExtension(dpy, &dpms_dummy, &dpms_dummy)) &&
            (DPMSCapable(dpy)) && (dpms_on)) {
                DPMSEnable(dpy);
        }
#endif

	XWarpPointer(dpy, None, RootWindowOfScreen(XtScreen(tv)),
		     0, 0, 0, 0, rpx, rpy);
	fs = 0;
	direct_video = 0;
    } else {
	int vp_x, vp_y, vp_width, vp_height;

	vp_x = 0;
	vp_y = 0;
	vp_width  = swidth;
	vp_height = sheight;
	XQueryPointer(dpy, RootWindowOfScreen(XtScreen(tv)),
		      &root, &child, &rpx, &rpy, &wpx, &wpy, &mask);

#ifdef HAVE_LIBXXF86VM
	if (have_vm) {
	    int i;
	    XF86VidModeGetAllModeLines(dpy,XDefaultScreen(dpy),
				       &vm_count,&vm_modelines);
	    for (i = 0; i < vm_count; i++)
		if (fs_width  == vm_modelines[i]->hdisplay &&
		    fs_height == vm_modelines[i]->vdisplay)
		    break;
	    if (i != 0 && i != vm_count) {
		set_vidmode(i);
		vm_switched = 1;
		vp_width = vm_modelines[i]->hdisplay;
		vp_height = vm_modelines[i]->vdisplay;
	    } else {
		vm_switched = 0;
		vp_width = vm_modelines[0]->hdisplay;
		vp_height = vm_modelines[0]->vdisplay;
	    }
#if 0
	    XF86VidModeGetViewPort(dpy,XDefaultScreen(dpy),&vp_x,&vp_y);
#else
	    XWarpPointer(dpy, None, RootWindowOfScreen(XtScreen(tv)),
			 0, 0, 0, 0, vp_width/2, vp_height/2);
	    XF86VidModeSetViewPort(dpy,XDefaultScreen(dpy),0,0);
#endif

	    if (debug)
		fprintf(stderr,"viewport: %dx%d+%d+%d\n",
			vp_width,vp_height,vp_x,vp_y);
	}
#endif
	XtVaGetValues(app_shell,
		      XtNx,          &x,
		      XtNy,          &y,
		      XtNwidth,      &w,
		      XtNheight,     &h,
		      NULL);

	XtVaSetValues(app_shell,
		      XtNwidthInc,   1,
		      XtNheightInc,  1,
		      NULL);
	XtVaSetValues(app_shell,
		      XtNx,          (vp_x & 0xfffc) + fs_xoff,
		      XtNy,          vp_y            + fs_yoff,
		      XtNwidth,      vp_width,
		      XtNheight,     vp_height,
		      NULL);

        XRaiseWindow(dpy, XtWindow(app_shell));	

	XGetScreenSaver(dpy, &timeout, &interval, &prefer_blanking, &allow_exposures);
	XSetScreenSaver(dpy, 0, 0, DefaultBlanking, DefaultExposures);
#ifdef HAVE_LIBXDPMS
        if (DPMSQueryExtension(dpy, &dpms_dummy, &dpms_dummy) && DPMSCapable(dpy)) {
            DPMSInfo(dpy, &dpms_state, &dpms_on);
            DPMSDisable(dpy);
        }
#endif

	XWarpPointer(dpy, None, XtWindow(tv), 0, 0, 0, 0, 30, 15);
	mouse = pointer_on;
	fs = 1;
    }
    if (mouse)
	XtCallActionProc(tv, "Pointer", NULL, NULL, 0);
    XtAppAddWorkProc(app_context, MyResize, NULL);
}

void button_cb(Widget widget, XtPointer clientdata, XtPointer call_data)
{
    struct CHANNEL *channel = clientdata;
    int i;

    pixit();
    for (i = 0; i < count; i++)
	if (channels[i] == channel)
	    break;
    cur_sender = i;
    set_channel(channels[i]);
}

void create_chanwin()
{
    chan_shell = XtVaAppCreateShell("Channels", "Cabletv",
				    topLevelShellWidgetClass,
				    dpy,
				    XtNclientLeader,app_shell,
				    NULL);
    XtOverrideTranslations(chan_shell, XtParseTranslationTable
			   ("<Message>WM_PROTOCOLS: Channels()"));
    chan_viewport = XtVaCreateManagedWidget("viewport",
					    viewportWidgetClass, chan_shell,
					    XtNallowHoriz, False,
					    XtNallowVert, True,
					    NULL);
    chan_box = XtVaCreateManagedWidget("channelbox",
				       boxWidgetClass, chan_viewport,
				       XtNsensitive, True,
				       NULL);
}

void
channel_menu()
{
    int  i,max,len;
    char str[100];

    cmenu = malloc((count+1)*sizeof(struct STRTAB));
    memset(cmenu,0,(count+1)*sizeof(struct STRTAB));
    for (i = 0, max = 0; i < count; i++) {
	len = strlen(channels[i]->name);
	if (max < len)
	    max = len;
    }
    for (i = 0; i < count; i++) {
	cmenu[i].nr      = i+1;
	cmenu[i].str     = channels[i]->name;
	if (channels[i]->key) {
	    sprintf(str,"%-*s %s",max+2,channels[i]->name,channels[i]->key);
	    cmenu[i].str=strdup(str);
	}
    }
}

/*--- main ---------------------------------------------------------------*/

static void xfree_init()
{
  int  flags,foo,bar,i,ma,mi;
  
#ifdef HAVE_LIBXXF86DGA

    if (XF86DGAQueryExtension(dpy, &foo, &bar)) {
	XF86DGAQueryDirectVideo(dpy, XDefaultScreen(dpy), &flags);
	if (flags & XF86DGADirectPresent) {
	    XF86DGAQueryVersion(dpy, &ma, &mi);
	    if (debug) fprintf(stderr, "DGA version %d.%d", ma, mi);
	    have_dga = 1;
	}
    }
#endif
#ifdef HAVE_LIBXXF86VM
    if (XF86VidModeQueryExtension(dpy, &foo, &bar)) {
	XF86VidModeQueryVersion(dpy, &ma, &mi);
	if (debug) 
            fprintf(stderr, "VidMode version %d.%d\n", ma, mi);
        have_vm = 1;
        XF86VidModeGetAllModeLines(dpy,XDefaultScreen(dpy), &vm_count,&vm_modelines);

        if (debug) {
            fprintf(stderr,"  available video mode(s):");
            for (i = 0; i < vm_count; i++) {
                fprintf(stderr, " %dx%d", vm_modelines[i]->hdisplay, vm_modelines[i]->vdisplay);
            }	    
	    fprintf(stderr,"\n");
	}
    }
#endif
}

static void grabber_init()
{
    int sw,sh;
    void *base = NULL;
    int  width = 0, depth = 0;
#ifdef HAVE_LIBXXF86DGA
    int bar,fred;

    if (have_dga) {
	XF86DGAGetVideoLL(dpy,XDefaultScreen(dpy),(int*)&base,
			  &width,&bar,&fred);
    }
#endif
    sw = XtScreen(app_shell)->width;
    sh = XtScreen(app_shell)->height;

    x11_pixmap_format = VIDEO_YUY2;

    if (0 == width)
	width = sw;
    switch (x11_native_format) {
        case VIDEO_GRAY:  
        case VIDEO_RGB08: depth =  8; break;
        case VIDEO_RGB15: depth = 15; break;
        case VIDEO_RGB16: depth = 16; break;
        case VIDEO_RGB24: depth = 24; break;
        case VIDEO_RGB32: depth = 32; break;
    }
    width *= (depth+7)/8;
    if (debug) fprintf(stderr,"x11: %dx%d, %d bit/pixel, %d byte/scanline\n", sw,sh,depth, width);
    for (grabber = 0; grabber < sizeof(grabbers)/sizeof(struct GRABBERS*);
	 grabber++) {
	if (-1 != grabbers[grabber]->grab_open
	    (device,sw,sh,x11_native_format,x11_pixmap_format,base,
	     width ? width : sw))
	    break;
    }
    if (grabber == sizeof(grabbers)/sizeof(struct GRABBERS*)) {
	fprintf(stderr,"no video grabber device available\n");
	exit(1);
    }
}


void VerticalTotalUpAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    if (BTType == BT878 || BTType == BT879) {
        setVerticalTotal(getVerticalTotal() + 1);
    }
    else {
        setPllFrequency(getPllFrequency() - 56700);
    }
}


void VerticalTotalDownAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    if (BTType == BT878 || BTType == BT879) {
        setVerticalTotal(getVerticalTotal() - 1);
    }
    else {
        setPllFrequency(getPllFrequency() + 56700);
    }
}


void PllFrequencyUpAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    setPllFrequency(getPllFrequency() + 36);
}


void PllFrequencyDownAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    setPllFrequency(getPllFrequency() - 36);
}


void LockingAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    if (locking) {
      locking = 0;
    } else {
      locking = 1;
    }

    sprintf(title, "CableTV: Locking: %s", (locking) ? "on" : "off");
    set_timer_title();
}


void TestRegisterAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    if (testregister) {
        setTest(1);
        testregister = 0;
    } else {
        setTest(239);
        testregister = 1;
    }

    sprintf(title, "CableTV: Test Register: %s", (testregister) ? "on" : "off");
    set_timer_title();
}


void BrightnessUpAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Brightness: %d", BrightnessUp());
    set_timer_title();
}

void BrightnessDownAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Brightness: %d", BrightnessDown());
    set_timer_title();
}


void ContrastUpAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Contrast: %d", ContrastUp());
    set_timer_title ();
}

void ContrastDownAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Contrast: %d", ContrastDown());
    set_timer_title();
}


void SaturationUpAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Saturation: %d", SaturationUp());
    set_timer_title();
}

void SaturationDownAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Saturation: %d", SaturationDown());
    set_timer_title();
}


void AdjustLumaLevelAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Adjust Luma Level: %s", (AdjustLumaLevel()) ? "on" : "off");
    set_timer_title();
}


void InvertPolarityAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Invert Polarity: %s", (InvertPolarity()) ? "on" : "off");
    set_timer_title();
}


void InvertChromaAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: Invert Chroma: %s", (InvertChroma()) ? "on" : "off");
    set_timer_title();
}


void ShowFPSAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: FPS: %d", fps);
    set_timer_title();
}


void VersionInfoAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    sprintf(title, "CableTV: %s", PACKAGE_VERSION);
    set_timer_title();
}


void RoutineChangeAction(Widget widget, XEvent *event, String *params, Cardinal *num_params)
{
    if (decode_routine == 0) {
        decode_routine = 1;
    } else if (decode_routine == 1) {
        decode_routine = 2;
    } else if (decode_routine == 2) {
        decode_routine = 0;
    }

    sprintf(title, "CableTV: Decode Routine: %s", routine_string[decode_routine]);
    set_timer_title();
}

unsigned long int get_bt848_address(void)
{
#ifndef HAVE_LIBPCI
    int found = 0;
    unsigned long int i, device, bt848_adr = 0, pcireg;

    if (iopl(3) < 0) {
        printf("error iopl: %s (must be root)\n", strerror (errno));
        exit(-1);
    }

    for (i = 0; i < 0x10000 && found == 0; i++) {
        pcireg = 0x80000000 | (i << 8);

        outl(pcireg, 0xcf8);
        device = inl(0xcf8 + 4);

        switch (device) {
            case BT848:
            case BT849:
            case BT878:
            case BT879:
                BTType = device;
                outl(pcireg + 0x10, 0xcf8);
                bt848_adr = inl(0xcf8 + 4) & 0xfffffff0;
                found = 1;
        }

    }

    if (iopl(0) < 0) {
        printf("error iopl: %s (could not drop IO privilege)\n",
	    strerror (errno));
        exit(-1);
    }
#else
    struct pci_dev *dev;
    struct pci_access *pacc;
    unsigned long int bt848_adr = 0;

    pacc = pci_alloc();
    pci_init(pacc);
    pci_scan_bus(pacc);
    for (dev = pacc->devices; dev && bt848_adr == 0; dev = dev->next)
        if (dev->vendor_id == PCI_VENDOR_ID_BROOKTREE)
            switch (dev->device_id){
                case PCI_DEVICE_ID_BROOKTREE_848:
                case PCI_DEVICE_ID_BROOKTREE_849A:
                case PCI_DEVICE_ID_BROOKTREE_878_1:
                case PCI_DEVICE_ID_BROOKTREE_879:
                    BTType = (dev->device_id << 16 | dev->vendor_id);
		    bt848_adr = pci_read_long(dev, PCI_BASE_ADDRESS_0) & 0xfffffff0;
	    }
    pci_cleanup(pacc);
#endif
    
    return bt848_adr;
}


void mmapBTTV()
{
    int fd;
    unsigned long int bt848_adr = 0;

    bt848_adr = get_bt848_address();

    if (debug) fprintf(stderr, "BT848 base address: %x\n", (int) bt848_adr);

    if ((fd = open("/dev/mem", O_RDWR)) < 0) {
        fprintf(stderr, "error open `/dev/mem': %s\n", strerror (errno));
        exit(-1);
    }

    BTmm = mmap(0, 0x1000, PROT_READ | PROT_WRITE, MAP_SHARED, fd, bt848_adr);
    if (BTmm == (void *) -1) {
        fprintf(stderr, "error mmap: %s\n", strerror (errno));
        exit(-1);
    }
}

void initBTTV()
{
    /* save state of bt848 chip */
    pllfrequency = getPllFrequency();
    dropevenfield = isDropEvenField();
    droprate = getDropRate();
    verticaltotal = getVerticalTotal();
    colorformat = getColorFormat();
    testregister = getTest();
    extframe = isExtFrame();
    fifoenabled = isFifoEnabled();
    captureevenenabled = isCaptureEvenEnabled();
    captureoddenabled = isCaptureOddEnabled();

    /* set bt848 chip */
    scalerInit();
    setCrystal(XT0);
    setPllFrequency(35436933);
    setPllEnabled(1);
    setDropEvenField(true);		/* align fields */
    setDropRate(0);			/* don't drop any frames */
    setVerticalTotal(625);		/* enable vertical total */
    setVideoFormat(1);
    setColorFormat(Raw8X);		/* output format is RAW samples */
    setTest(239);
    setExtFrame(true);			/* extend capture size */

    setFifoEnabled(true);		/* enable FIFO processor */
    setCaptureEvenEnabled(true);	/* enable capture of even and odd fields */
}



int main(int argc, char *argv[])
{
    int i, c, fullscreen, nomouse;
    char v4l_conf[128] = "v4l-conf -q";
    Dimension w;

    int option_index = 0;
    static struct option long_options[] = {
	{"verbose", 0, 0, 'v'},
	{"nodga", 0, 0, 'd'},
	{"nomouse", 0, 0, 'm'},
	{"fullscreen", 0, 0, 'f'},
	{"noxext", 0, 0, 'x'},
	{"device", 1, 0, 'c'},
	{"routine", 1, 0, 'r'},
	{"help", 0, 0, 'h'},
	{0, 0, 0, 0}
    };
   
    /* only the euid needs to be 0 to access /dev/mem */
    if (0 != geteuid()) {
        fprintf(stderr, "To run CableTV you must be root!  You need to have access to /dev/mem!\n");
        return 1;
    }

    /* This should be called the soonest posible and root privilege be droped */
    mmapBTTV();
    /* drop root privilege */
    setuid(getuid());
    seteuid(getuid());

    progname = strdup(argv[0]);

    fprintf(stderr,
	    "!! WARNING NOTE !!\n\n" 
	    "This program is intended for educational and research use only.\n" 
	    "It might not be legal to decode and view an encrypted channel.\n" 
	    "All use of this program is at your own risk!!\n" 
	    "\n\n\nCableTV - CableCrypt Decoder for Linux - %s\n"
	    "by TWISTI & FUX (10/2002)\n\n", PACKAGE_VERSION);

    /* set default options */
    debug = 0;
    fullscreen = 0;
    nomouse = 0;
    decode_routine = 4;
    xfree_ext = 1;
    
    while ((c = getopt_long(argc, argv, "dhxmfvc:r:", long_options, &option_index)) != -1) {
	switch (c) {
	case 'v':
	    debug = 1;
	    break;
	case 'd':
	    noDGA = 1;
	    break;
	case 'm':
	    nomouse = 1;
	    break;
	case 'f':
	    fullscreen = 1;
	    break;
	case 'x':
	    xfree_ext = 0;
	    break;
	case 'c':
	    device = optarg;
	    /* v4l-conf needs this too */
	    strcat(v4l_conf, " -c ");
	    strcat(v4l_conf, device);
	    break;
	case 'r':
            decode_routine = atoi(optarg);
            break;
	case 'h':
	default:
	    fprintf(stderr, "Usage: %s [options]\n", argv[0]);
	    fprintf(stderr, "Options:\n");
	    fprintf(stderr, "  -v, --verbose        Debug output.\n");
	    fprintf(stderr, "  -r n, --routine=n    n specifies the decoding routine:\n" );
	    fprintf(stderr, "                       0 ... grayscale in assembler (no MMX required!)\n");
	    fprintf(stderr, "                       1 ... pal color in C\n");
	    fprintf(stderr, "                       2 ... interpolated pal color in C\n");
	    fprintf(stderr, "                       3 ... MMX optimized pal color\n");
	    fprintf(stderr, "                       4 ... MMX optimized interpolated pal color (default)\n\n");
	    fprintf(stderr, "  -f, --fullscreen     Start up in fullscreen.\n");
	    fprintf(stderr, "  -m, --nomouse        Start up with mouse pointer disabled.\n");
	    fprintf(stderr, "  -d, --nodga          Start without DGA.\n");
	    fprintf(stderr, "  -c dev, --device=dev Specify other device than /dev/video.\n");
	    fprintf(stderr, "  -h, --help           Print this message and exit.\n");
	    return 1;
	}
    }

    if (debug) fprintf(stderr,"----------------------------------------------------\n");

    switch (system(v4l_conf)) {
    case -1: /* can't run */
	fprintf(stderr, "Couldn't start v4l-conf!\n");
	break;
    case 0: /* ok */
	break;
    default: /* non-zero return */
	fprintf(stderr, "v4l-conf had some trouble, trying to continue anyway...\n");
	break;
    }

    if (decode_routine == 2 || decode_routine == 4) {
	argv[argc++] = "-display";
	argv[argc++] = ":0.0";
	argv[argc++] = "-geometry";
	argv[argc++] = "768x576";
    }
    
    app_shell = XtAppInitialize(&app_context,
				"CableTV",
				NULL, 0, /* opt_desc, 7, */
				&argc, argv,
				NULL /* fallback_res */,
				NULL, 0);

    XtAppAddActions(app_context, actionTable, sizeof(actionTable) / sizeof(XtActionsRec));

    wm_protocols[0] = XInternAtom(XtDisplay (app_shell), "WM_DELETE_WINDOW", False);
    wm_protocols[1] = XInternAtom(XtDisplay (app_shell), "WM_SAVE_YOURSELF", False);

    xawtv_station = XInternAtom(XtDisplay (app_shell), "_XAWTV_STATION", False);
/*     xawtv_remote = XInternAtom(XtDisplay (app_shell), "_XAWTV_REMOTE", False); */

    dpy = XtDisplay(app_shell);

    if (xfree_ext)
	xfree_init();

    create_onscreen();
    create_chanwin();
    
    tv = video_init(app_shell);
    XtVaGetValues(tv, XtNwidth, &w, NULL);
    if (!w) {
	fprintf(stderr, "Can't find app-defaults file!  Install it in `/etc/X11/app-defaults/'!\n");
	exit(1);
    }

    switch(bpp) {
        case  8: x11_native_format = VIDEO_RGB08; break;
        case 15: x11_native_format = VIDEO_RGB15; break;
        case 16: x11_native_format = VIDEO_RGB16; break;
        case 24: x11_native_format = VIDEO_RGB24; break;
        case 32: x11_native_format = VIDEO_RGB32; break;
        default: bpp = 0;
    }

    grabber_init();

    read_config();

    set_freqtab(chan_tab);
    channel_menu();
    if (have_mixer)
	cur_volume = mixer_get_volume() * 65535 / 100;
    set_volume();
    cur_capture = 0;

    sprintf(modename, "%dx%d", XtScreen(app_shell)->width, XtScreen(app_shell)->height);

    switch (x11_native_format) {
        case VIDEO_RGB08: strcat(modename, ", 8 bit"); break;
        case VIDEO_RGB15: strcat(modename, ", 15 bit"); break;
        case VIDEO_RGB16: strcat(modename, ", 16 bit"); break;
        case VIDEO_RGB24: strcat(modename, ", 24 bit"); break;
        case VIDEO_RGB32: strcat(modename, ", 32 bit"); break;
    }

    if (x11_native_format == VIDEO_RGB16) {
        bpp = 16;
    } else {	    
        printf ("\nDecoding runs only in 16-Bit Mode!\n");
        return 1;
    }

    XtRealizeWidget(app_shell);
    create_pointers(app_shell);
    create_bitmaps(app_shell);
    XDefineCursor(dpy, XtWindow(app_shell), left_ptr);
    XSetWMProtocols(XtDisplay(app_shell), XtWindow(app_shell), wm_protocols, 2);

    XtVaSetValues(app_shell,
		  XtNwidthInc,  WIDTH_INC,
		  XtNheightInc, HEIGHT_INC,
		  XtNminWidth,  WIDTH_INC,
		  XtNminHeight, HEIGHT_INC,
		  NULL);

    if (optind + 1 == argc) {
	for (i = 0; i < count; i++)
	    if (0 == strcasecmp(channels[i]->name, argv[optind]))
	        cur_sender = i;
    }

    if (count) {
	if ((cur_sender < 0) || (cur_sender >= count))
	    cur_sender = 0;
	set_channel(channels[cur_sender]);
    } else {
	set_channel(&defaults);
    }
    
    if (fullscreen) {
	FullScreenAction(NULL, NULL, NULL, NULL);
    } else {
	XtAppAddWorkProc(app_context, MyResize, NULL);
    }
    
    if (nomouse)
	PointerAction(NULL, NULL, NULL, NULL);

/*    strcpy (title, "CableTV 1.1: "); */
/*    strcat (title, modename); */
/*    set_timer_title (); */

    initBTTV();

#ifdef HAVE_LIBLIRC_CLIENT
    lirc_supp_init(app_context, tv);
#endif
    
    if (debug) fprintf(stderr,"----------------------------------------------------\n\n");
    XtAppMainLoop(app_context);

    /* keep compiler happy */
    return 0;
}
