#--------------------------------------------------------
# This software is put into public domain by the author. 
#--------------------------------------------------------
#         Sang Cho - ChongJu University S. Korea
#                            sangcho@cju.ac.kr
#                            2005-01-23
#--------------------------------------------------------
# modification history       2005-01-24  
# GUI added                  2005-02-07, 08, 09, 10, 11, 12, 13, 14, 15, 16, 17
#                            2005-02-18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28
#                            2005-03-01, 02
# minor bug corrected        2007-05-31

import sys, time
import string
import threading
import os
import wx                    # you need wxpython package to run this program
#------------------------------------------------------
keyboard=threading.Condition()
gosign=threading.Condition()
fontsize=12

#------------------------------------------------------
EVT_RESULT_ID = wx.NewId()

def EVT_RESULT(win, func):
    win.Connect(-1, -1, EVT_RESULT_ID, func)

class ResultEvent(wx.PyEvent):
    """Simple event to carry arbitrary result data"""

    def __init__(self, data):
        wx.PyEvent.__init__(self)
        self.SetEventType(EVT_RESULT_ID)
        self.data = data

EVT_KEYBOARD_ID = wx.NewId()

def EVT_KEYBOARD(win, func):
    win.Connect(-1, -1, EVT_KEYBOARD_ID, func)

class KeyboardEvent(wx.PyEvent):
    """Simple event to carry arbitrary result data"""

    def __init__(self, data):
        wx.PyEvent.__init__(self)
        self.SetEventType(EVT_KEYBOARD_ID)
        self.data = data
#------------------------------------------------------------------------------------------

class SBMCore:
    "Simple Byte Machine Core"
    
    def __init__(self, win):
    
        self.memP=[]               # holds 2K instructions
        self.memD=[]               # holds 2K words
        self.memS=[]               # holds 2K words
        self.PC=0                  # PC program counter
        self.ACC=[0,0]             # AUX and ACC auxiliary accumulator and accumulator
        self.inst=[0,0,0]          # opcode and address plus line number
        self.SP=2048               # bottom of stack
        self.XP=32                 # special stack pointer for call and return     
        self.SBMexception=False    # exception
        self.SBMreason=""          # reason of exception
        self.opTable=[]            # opcode method table
        self.win = win             # SBM screen and keyboard
        self.log=win.log           # SBM log window

        for x in range(2048):      # initialize memP, memD, memS
            self.memP.append(self.inst)
            self.memD.append(0)
            self.memS.append(0)

        self.memS[self.XP]=self.SP # main is first called function and entitled       2005-01-24
        self.XP = self.XP - 1      # to use stack area as well as other functions     2005-01-24

        self.opTable.append(self.NOP)     # opcode= 0
        self.opTable.append(self.GET)     # opcode= 1
        self.opTable.append(self.PUT)     # opcode= 2
        self.opTable.append(self.DCG)     # opcode= 3
        self.opTable.append(self.BIN)     # opcode= 4
        self.opTable.append(self.SETV)    # opcode= 5
        self.opTable.append(self.SWAP)    # opcode= 6
        self.opTable.append(self.LDA)     # opcode= 7
        self.opTable.append(self.STA)     # opcode= 8
        self.opTable.append(self.LDAI)    # opcode= 9
        self.opTable.append(self.STAI)    # opcode=10
        self.opTable.append(self.ADD)     # opcode=11
        self.opTable.append(self.SUB)     # opcode=12
        self.opTable.append(self.MUL)     # opcode=13
        self.opTable.append(self.DIV)     # opcode=14
        self.opTable.append(self.INC)     # opcode=15
        self.opTable.append(self.DEC)     # opcode=16
        self.opTable.append(self.AND)     # opcode=17
        self.opTable.append(self.OR)      # opcode=18
        self.opTable.append(self.XOR)     # opcode=19
        self.opTable.append(self.NOT)     # opcode=20
        self.opTable.append(self.JMP)     # opcode=21
        self.opTable.append(self.JMPZ)    # opcode=22
        self.opTable.append(self.JMPP)    # opcode=23
        self.opTable.append(self.JMPM)    # opcode=24
        self.opTable.append(self.CALL)    # opcode=25
        self.opTable.append(self.RET)     # opcode=26
        self.opTable.append(self.PUSH)    # opcode=27
        self.opTable.append(self.POP)     # opcode=28
        self.opTable.append(self.XLDA)    # opcode=29
        self.opTable.append(self.XSTA)    # opcode=30
        self.opTable.append(self.HALT)    # opcode=31

    def NOP(self):
        "NOP do nothing"
        self.PC=self.PC+1

    def GET(self):
        "GET read 1 byte from keyboard to ACC"
        # synchronize
        wx.PostEvent(self.win, KeyboardEvent(1))
        keyboard.acquire()
        keyboard.wait()
        keyboard.notify()
        keyboard.release()
        wx.PostEvent(self.win, KeyboardEvent(0))

        byte=self.win.get()
        self.ACC[0]=byte
        self.PC=self.PC+1

    def PUT(self):
        "PUT write 1 byte from ACC to screen"
        byte=self.ACC[0]
        self.win.put(byte)
        self.PC=self.PC+1

    def DCG(self):
        "DCG call DCA or DCLR depending on value"
        if self.inst[1]==1:
            self.DCA()
        else:
            self.DCLR()

    def DCA(self): 
        "DCA convert ACC to decimal and put the result into BFR"
        l='     ' + str(self.ACC[0])                 # 2005-02-11
        line=l[-6:]                                  #
        for x in range(6):                           #
            self.memD[x]=ord(line[x])                #
        self.PC=self.PC+1

    def DCLR(self):
        "DCLR clear BFR with spaces"
        for x in range(6):
            self.memD[x]=32
        self.PC=self.PC+1

    def BIN(self):
        "BIN convert decimal in BFR to binary and put into ACC"
        line=''
        for x in range(6):
            v=self.memD[x]
            if v==0:
                line=line+' '
            else:
                line=line+chr(self.memD[x])
        if line=='      ':
            num=0
        else:
            num=int(line)
        self.ACC[0]=num
        self.PC=self.PC+1

    def SETV(self):
        "SETV set ACC with immediate value"
        self.ACC[0]=self.inst[1]
        self.PC=self.PC+1

    def SWAP(self):
        "SWAP swap ACC and AUX"
        self.ACC[0],self.ACC[1]=self.ACC[1],self.ACC[0]
        self.PC=self.PC+1

    def LDA(self):
        "LDA load ACC from memD"
        self.ACC[0]=self.memD[self.inst[1]]
        self.PC=self.PC+1

    def STA(self):
        "STA store ACC into memD"
        self.memD[self.inst[1]]=self.ACC[0]
        self.PC=self.PC+1

    def LDAI(self):
        "LDAI load ACC indirect"
        addr=self.memD[self.inst[1]]
        self.ACC[0]=self.memD[addr]
        self.PC=self.PC+1

    def STAI(self):
        "STAI store ACC indirect"
        addr=self.memD[self.inst[1]]
        self.memD[addr]=self.ACC[0]
        self.PC=self.PC+1

    def ADD(self):
        "ADD add memD to ACC"
        value=self.ACC[0]+self.memD[self.inst[1]]
        self.ACC[0]=value
        self.PC=self.PC+1

    def SUB(self):
        "SUB subtract memD from ACC"
        value=self.ACC[0]-self.memD[self.inst[1]]
        self.ACC[0]=value
        self.PC=self.PC+1    

    def MUL(self):
        "MUL multiply memD to ACC"
        value=self.ACC[0]*self.memD[self.inst[1]]
        self.ACC[0]=value
        self.PC=self.PC+1

    def DIV(self):
        "DIV divide ACC by memD"
        dval=self.memD[self.inst[1]]
        if dval==0:
            self.SBMreason= "Divide by zero exception... system halted."
            self.SBMexception=True
            return
        self.ACC[0],self.ACC[1]=divmod(self.ACC[0],dval)
        self.PC=self.PC+1

    def INC(self):
        "INC increment memD by 1"
        value=self.memD[self.inst[1]]
        value=value+1
        self.memD[self.inst[1]]=value
        self.PC=self.PC+1

    def DEC(self):
        "DEC decrement memD by 1"
        value=self.memD[self.inst[1]]
        value=value-1
        self.memD[self.inst[1]]=value
        self.PC=self.PC+1

    def AND(self):
        "AND and memD to ACC"
        value=self.ACC[0] & self.memD[self.inst[1]]
        self.ACC[0]=value
        self.PC=self.PC+1

    def OR(self):
        "OR or memD to ACC"
        value=self.ACC[0] | self.memD[self.inst[1]]
        self.ACC[0]=value
        self.PC=self.PC+1
 
    def XOR(self):
        "XOR xor memD to ACC"
        value=self.ACC[0] ^ self.memD[self.inst[1]]
        self.ACC[0]=value
        self.PC=self.PC+1    

    def NOT(self):
        "NOT not ACC"
        self.ACC[0]=~self.ACC[0]
        self.PC=self.PC+1 

    def JMP(self):
        "JMP unconditional jump"
        self.PC=self.inst[1]

    def JMPZ(self):
        "JMPZ jump if ACC==0"
        if self.ACC[0]==0: 
            self.PC=self.inst[1]
        else:
            self.PC=self.PC+1

    def JMPP(self):
        "JMPP jump if ACC>0"
        if self.ACC[0]>0: 
            self.PC=self.inst[1]
        else:
            self.PC=self.PC+1        

    def JMPM(self):
        "JMPM jump if ACC<0"
        if self.ACC[0]<0: 
            self.PC=self.inst[1]
        else:
            self.PC=self.PC+1   

    def CALL(self):
        "CALL function call"
        #print 'CALL', self.SP, self.PC, self.inst
        self.SP=self.SP-1
        self.memS[self.SP]=self.PC+1
        self.XP=self.XP-1              # hods return address SP
        self.memS[self.XP]=self.SP
        self.SP=self.SP-31             # reserve 31 words for local storage automatically
        self.PC=self.inst[1]

    def RET(self):
        "RET return fo caller"
        #print 'RET'
        self.SP=self.memS[self.XP]     # here you can see why XP is needed
        self.XP=self.XP+1
        self.PC=self.memS[self.SP]
        self.SP=self.SP+1

    def PUSH(self):
        "PUSH push ACC into memS"
        self.SP=self.SP-1
        self.memS[self.SP]=self.ACC[0]
        self.PC=self.PC+1

    def POP(self):
        "POP pop ACC from memS"
        self.ACC[0]=self.memS[self.SP]
        self.SP=self.SP+1
        self.PC=self.PC+1

    def XLDA(self):
        "XLDA load ACC from memS"
        base=self.memS[self.XP]        # here again XP is needed
        addr=base+self.inst[1]
        self.ACC[0]=self.memS[addr]
        self.PC=self.PC+1

    def XSTA(self):
        "XSTA store ACC to memS"
        base=self.memS[self.XP]        # here again XP is needed
        addr=base+self.inst[1]
        self.memS[addr]=self.ACC[0]
        self.PC=self.PC+1

    def HALT(self):
        "HALT stops machine"
        self.SBMreason= "System stoped by HALT instruction."
        self.SBMexception=True

#------------------------------------------------------

class SBMFrame (SBMCore):
    "Simple Byte Machine Framework"
    
    def __init__(self, win):
        SBMCore.__init__(self, win)
        self.assembled=False
        self.tics=0
        self.onestep=False
        self.terminate=False
        self.resetting=False

    def init(self):
        self.PC=0                  # PC program counter
        self.ACC=[0,0]             # AUX and ACC auxiliary accumulator and accumulator
        self.inst=self.memP[0]     # opcode and address plus line number
        self.SP=2048               # bottom of stack
        self.XP=32                 # special stack pointer for call and return     
        self.SBMexception=False    # exception
        self.SBMreason=""          # reason of exception
        self.tics=0
        self.onestep=False
        
    def reset(self):
        self.resetting=True
        time.sleep(0.01)
        if self.onestep:
            gosign.acquire()
            gosign.notify()
            gosign.release()
        self.init()
        for x in range(2048):
            self.memD[x]=self.memDSave[x]
            self.memS[x]=self.memSSave[x]

        
    def assemble(self, file):
        self.resetting=False
        self.init()
        for x in range(2048):
            self.memP[x]=[0,0,0]
            self.memD[x]=0
            self.memS[x]=0
        
        self.asm = Assembler(self)

        f=open(file,'r')
        self.lines=f.readlines()
        f.close

        self.asm.as(self.lines, self)
        self.assembled=True
        self.memDSave=self.memD[:]
        self.memSSave=self.memS[:]

                      
    def SBMLoop(self):
 
        while True:
            #print '..place zero..'
            gosign.acquire()
            gosign.wait()
            self.resetting=False
            #print '..place one.. PC=', self.PC
            gosign.notify()
            gosign.release()
            if self.terminate: break
            
            while not self.SBMexception:
                #print '............... PC=', self.PC
                self.inst=self.memP[self.PC]
                self.opTable[self.inst[0]]()
                self.tics=self.tics+1    
                
                if self.resetting or self.terminate: break
                if self.onestep:
                    wx.PostEvent(self.win.parent, ResultEvent(None))
                    gosign.acquire()
                    gosign.wait()
                    #print '......place two......'
                    gosign.notify()
                    gosign.release()
                    if self.resetting or self.terminate: break

            # End while not self.SBMexception:
            #print '........... END of inner while.....'
            if self.terminate: break
            wx.PostEvent(self.win.parent, ResultEvent(None))
        # End while True:

#------------------------------------------------------

class Scanner:
    
    def __init__(self):
        pass
    
    def getChar(self):
        if self.pos < self.length:
            self.c = self.line[self.pos]
            self.pos = self.pos + 1
        
    def removeSpace(self):
        while (self.pos < self.length) and (self.c == ' '):
            self.getChar()
    
    def getId(self):
        id = ''
        while (self.pos < self.length) and self.c.isalnum():
            id = id + self.c
            self.getChar()
        self.token.append([1,id])
    
    def getNum(self):
        num = self.c
        self.getChar()
        while (self.pos < self.length) and self.c.isdigit():
            num = num+self.c
            self.getChar()
        self.token.append([2,int(num)])
    
    def getString(self):
        str = ''
        self.getChar()
        while 1:
            while (self.pos < self.length) and (self.c != '"'):
                str = str + self.c
                self.getChar()
            if self.c == '"':
                if (self.pos + 1 < self.length) and (self.line[self.pos-1:self.pos+2] == '"""'):
                    str = str + '"'
                    self.pos = self.pos + 2
                    self.getChar()
                else:
                    self.token.append([3,str])
                    self.getChar()
                    return
            else:
                self.token.append([9,str])
                return
    
    def getComment(self):
        while (self.pos < self.length):
            self.getChar()
     
    def getSpecial(self):
        self.token.append([4, self.c])
        self.getChar()
            
    def parse(self, line):
        self.line = line
        self.pos = 0
        self.token = []
        self.length = len(self.line)    
        if self.length == 0: 
            return self.token
        self.getChar()
        while 1:
            if self.c == ' ': 
                self.removeSpace()
            elif self.c.isalpha():
                self.getId()
            elif self.c.isdigit() or self.c== '-':
                self.getNum()
            elif self.c == '"':
                self.getString()
            elif self.c == '#':
                self.getComment()
            else:
                self.getSpecial()
            if self.pos >= self.length:
                break;
        return self.token

#------------------------------------------------------

class Assembler:

    def __init__(self, parent):
        self.scan = Scanner()
        self.opcodeDic = {
            'NOP' : [ 0, 0], 'GET' : [ 1, 0], 'PUT' : [ 2, 0], 'DCA' : [ 3, 3], 'DCLR': [ 3, 3], 
            'BIN' : [ 4, 0], 'SETV': [ 5, 2], 'SWAP': [ 6, 0], 'LDA' : [ 7, 1], 'STA' : [ 8, 1],
            'LDAI': [ 9, 1], 'STAI': [10, 1], 'ADD' : [11, 1], 'SUB' : [12, 1], 'MUL' : [13, 1], 
            'DIV' : [14, 1], 'INC' : [15, 1], 'DEC' : [16, 1], 'AND' : [17, 1], 'OR'  : [18, 1],   
            'XOR' : [19, 1], 'NOT' : [20, 0], 'JMP' : [21, 4], 'JMPZ': [22, 4], 'JMPP': [23, 4], 
            'JMPM': [24, 4], 'CALL': [25, 5], 'RET' : [26, 0], 'PUSH': [27, 0], 'POP' : [28, 0], 
            'XLDA': [29, 2], 'XSTA': [30, 2], 'HALT': [31, 0] }

        self.labelDic = {}
        self.fixupList = []
        self.variableDic = {}
        self.progBegin = 0
        self.log=parent.log
        
    def getLine(self):
        while self.lnum < self.totlines:
            self.line = self.lines[self.lnum]
            if self.line[-1] != '\n':
                self.line = self.line + '\n'
            self.cnum = self.lnum
            self.lnum = self.lnum + 1
            self.token = self.scan.parse(self.line)
            if len(self.token)>0:
                break;
            
    def addVariable(self, var, loc, type, length):
        if self.variableDic.has_key(var):
            self.log.WriteText('There is already the same variable ... error in line %d\n' % self.cnum)
            return
        self.variableDic.update({var : [loc, type, length]})
        
    def getDataAddress(self, var):
 
        if not self.variableDic.has_key(var):
            self.log.WriteText('There is no variable %s ... error in line %d.\n' % (id, self.cnum))
            return 0

        if len(self.token) == 2:
            return self.variableDic[var][0]
        elif len(self.token) == 4:
            p = self.token[2]
            n = self.token[3]
            if (p[1] != '+') or (n[0] != 2):
                self.log.WriteText('There is some error in line %d.\n' % self.cnum)
                return 0
            return self.variableDic[var][0]+n[1]

    def addProgLabel(self, label):
        if self.labelDic.has_key(label):
            self.log.WriteText('There is already the same label ... error in line %d\n' % self.cnum)
            return
        self.labelDic.update({label : self.PC})
        
    def getProgLabelAddress(self, label):
        if not self.labelDic.has_key(label):
            self.fixupList.append([self.PC, label, self.cnum])
            return 0
        return self.labelDic[label]
    
    
    def fixup(self):
        #print self.fixupList
        for x in self.fixupList:
            if not self.labelDic.has_key(x[1]):
                self.log.WriteText('Fixup failed .. at the line %d.\n' % x[2])
            else:
                inst = self.memP[x[0]]
                inst[1] = self.labelDic[x[1]]
                self.memP[x[0]] = inst
    
    def setVarValues(self):
        t = self.token[2]
        l = len(self.token)
        if (t[0] != 1) or (t[1] not in ['CHAR', 'WORD']):
            self.log.WriteText('Variable type error in line %d.\n' % self.cnum)
            return 
        if t[1]=='CHAR':
            self.type = 2
        else:
            self.type = 1
        if l == 3:
            self.Location = self.Location + 1
            self.length = 1
            return
        elif l == 4:
            self.log.WriteText('Variable declare malformed[01] in line %d.\n' % self.cnum)
            return
        if t[1] == 'WORD':
            u = self.token[3]
            v = self.token[4]
            if l == 5:
                if (u[1] == '=') and (v[0] == 2):
                    self.memD[self.Location] = v[1]
                    self.Location = self.Location + 1
                    self.length = 1
                    return
                if (u[1] == '*') and (v[0] == 2):
                    self.Location = self.Location + v[1]
                    self.length = v[1]
                    return
            if (u[1] == '*') and (v[1] == '='):
                loc = self.Location
                n = self.set4Values(loc)
                self.Location = self.Location + n
                self.log.WriteText('Variable declare malformed[02] in line %d.\n' % self.cnum)
                return
            if (u[1] == '*') and (v[0] == 2):
                loc = self.Location
	        self.Location = self.Location + v[1]
	        self.length = v[1]
	        self.set4Values(loc)
                return
            else:
                self.log.WriteText('Variable declare malformed[03] in line %d.\n' % self.cnum)
                return
        else:
            u = self.token[3]
            v = self.token[4]
            if l == 5:
                if (u[1] == '=') and (v[0] == 3):
                    self.memD[self.Location] = ord(v[1])
                    self.Location = self.Location + 1
                    self.length = 1
                    return
                if (u[1] == '*') and (v[0] == 2):
                    self.Location = self.Location + v[1]
                    self.length = v[1]
                    return
            if (u[1] == '*') and (v[1] == '='):
                loc = self.Location
	        n = self.set4String(loc)
	        self.Location = self.Location + n
	        self.length = n
                return            
                self.log.WriteText('Variable declare malformed[04] in line %d.\n' % self.cnum)
                return
            if (u[1] == '*') and (v[0] == 2):
                loc = self.Location
	        self.Location = self.Location + v[1]
	        self.length = v[1]
	        self.set4String(loc)
                return
            else:
                self.log.WriteText('Variable declare malformed[05] in line %d.\n' % self.cnum)
                return
                
    def set4Values(self, loc):
        tot = 0
        u = self.token[4]
        v = self.token[5]
        if u[1] == '=': 
            c = 5
        elif v[1] == '=':
            c = 6
        else:
            self.log.WriteText('Variable declare malformed[06] in line %d.\n' % self.cnum)
            return tot
        n = len(self.token)
        t = self.token[c]
        if t[0] != 2:
            self.log.WriteText('Variable declare malformed[07] in line %d.\n' % self.cnum)
            return tot
        self.memD[loc] = t[1]
        tot = tot + 1    
        loc = loc + 1
        c = c + 2
        while c < n:
            u = self.token[c-1]
            v = self.token[c]
            if (u[1] != ',') or (v[0] != 2):
                self.log.WriteText('Variable declare malformed[08] in line %d.\n' % self.cnum)
                return tot
            self.memD[loc] = v[1]
            loc = loc + 1
            tot = tot + 1
            c = c + 2
        return tot

    def set4String(self, loc):
        tot = 0
        u = self.token[4]
        v = self.token[5]
        if u[1] == '=': 
            c = 5
        elif v[1] == '=':
            c = 6
        else:
            self.log.WriteText('Variable declare malformed[09] in line %d.\n' % self.cnum)
            return tot
        n = len(self.token)
        t = self.token[c]
        if t[0] != 3:
            self.log.WriteText('Variable declare malformed[10] in line %d.\n' % self.cnum)
            return tot
        str = t[1]
        for x in str:
            self.memD[loc] = ord(x)
            loc = loc + 1
            tot = tot + 1
        c = c + 2
        while c < n:
            u = self.token[c-1]
            v = self.token[c]
            if (u[1] != ',') or (v[0] not in [2,3]):
                self.log.WriteText('Variable declare malformed[11] in line %d.\n' % self.cnum)
                return tot
            if v[0] == 3:
                str = v[1]
                for x in str:
                    self.memD[loc] = ord(x)
                    loc = loc + 1
                    tot = tot + 1
            else:
                self.memD[loc] = v[1]
                loc = loc + 1
                tot = tot + 1
            c = c + 2        
        return tot

    def getDataLines(self):

        self.getLine()
        #print self.token
        t = self.token[0]
        type = t[0]
        id = t[1]
        if type != 1:
            self.log.WriteText('Data line does not start with id in line %d.\n' % self.cnum)
            return
        if id == 'END':
            t = self.token[1]
            type = t[0]
            id = t[1]
            if type != 1:
                self.log.WriteText('Data name does not start with id in line %d.\n' % self.cnum)
                return
            if id != self.dataName:
                self.log.WriteText('Data name does not match with the first one in line %d.\n' \
                    % self.cnum)
                return
            self.EndDataFound = 1
            if len(self.token) >2:
                self.log.WriteText('There is some extra item in line %d.\n' % self.cnum)
                return    
            return
        elif self.variableDic.has_key(id): 
            self.log.WriteText('Variable already defined above in line %d.\n' % self.cnum)
            return

        t = self.token[1]
        if (t[0] != 4) or (t[1] != ':'):
            self.log.WriteText('Wrong variable in line %d.\n' % self.cnum)
            return
        #self.addVariable(id)               #modified 2005-02-15
        loc = self.Location
        self.setVarValues()
        self.addVariable(id, loc, self.type, self.length)            #modified 2005-02-15 
        
    def getProgLines(self):

        self.getLine()
        # print self.token
        
        u = self.token[0]
        if len(self.token)>1:
            v = self.token[1]
        t  = u[0]
        id = u[1]
        if t != 1:
            self.log.WriteText('Program line does not start with id in line %d.\n' % self.cnum)
            return
        if id == 'END':
            t  = v[0]
            id = v[1]
            if t != 1:
                self.log.WriteText('Program name does not start with id in line %d.\n' % self.cnum)
                return
            if id != self.progName:
                self.log.WriteText('Program name does not match with the first one in line %d.\n' \
                    % self.cnum)
                return
            self.EndProgFound = 1
            if len(self.token) >2:
                self.log.WriteText('There is some extra item in line %d.\n' % self.cnum)
                return
            return
        elif not self.opcodeDic.has_key(id):
            if (v[0] != 4) or (v[1] != ':'):
                self.log.WriteText('Wrong opcode in line %d.\n' % self.cnum)
                return
            self.addProgLabel(id)
            return
            
        opcode = id
        i = self.opcodeDic[opcode]
        i = i[:]           # very important - cut off from self.opcodeDic
        i.append(self.cnum)
        t = i[1]
        if t in [1,2]:
            if v[0] == 2:
                i[1] = v[1]
            else:
                i[1] = self.getDataAddress(v[1])
        elif t == 3:
            if opcode == 'DCA':
                i[1] = 1
            else:
                i[1] = 2
        elif t == 4:
            i[1] = self.getProgLabelAddress(v[1])
        elif t == 5:
            i[1] = self.getProgLabelAddress(v[1])
        self.memP[self.PC] = i
        self.PC = self.PC + 1

    def getFuncLines(self):                                    # add 2005-01-24

        self.getLine()
        #print self.token
        
        u = self.token[0]
        if len(self.token) > 1:
            v = self.token[1]
        t  = u[0]
        id = u[1]
        if t != 1:
            self.log.WriteText('Function line does not start with id in line %d.\n' % self.cnum)
            return
        if id == 'END':
            t  = v[0]
            id = v[1]
            if t != 1:
                self.log.WriteText('Function name does not start with id in line %d.\n' % self.cnum)
                return
            if id != self.funcName:
                self.log.WriteText('Function name does not match with the first one in line %d.\n' \
                    % self.cnum)
                return
            self.EndFuncFound = 1
            if len(self.token) >2:
                self.log.WriteText('There is some extra item in line %d.\n' % self.cnum)
                return
            return
        elif not self.opcodeDic.has_key(id):
            if (v[0] != 4) or (v[1] != ':'):
                self.log.WriteText('Wrong opcode in line %d.\n' % self.cnum)
                return
            self.addProgLabel(self.funcName+':'+id)
            return
            
        opcode = id
        i = self.opcodeDic[opcode]
        i = i[:]           # very important - cut off from self.opcodeDic
        i.append(self.cnum)
        #print i
        t = i[1]
        if t in [1,2]:
            if v[0] == 2:
                i[1] = v[1]
            else:
                i[1] = self.getDataAddress(v[1])
        elif t == 3:
            if opcode == 'DCA':
                i[1] = 1
            else:
                i[1] = 2
        elif t == 4:
            i[1] = self.getProgLabelAddress(self.funcName+':'+v[1])
        elif t == 5:
            i[1] = self.getProgLabelAddress(v[1])
        self.memP[self.PC] = i
        self.PC = self.PC + 1        
               

    def getData(self):
        t = self.token[1]
        if t[0]==1:
            self.dataName = t[1]
        while 1:
            self.getDataLines()
            if self.EndDataFound:
                break
    
    def getProgram(self):
         t = self.token[1]
         if t[0]==1:
             self.progName = t[1]
             self.progBegin=self.cnum
         while 1:
             self.getProgLines()
             if self.EndProgFound:
                break
 
 
    def getFunction(self):
        t = self.token[1]
        if t[0]==1:
            self.funcName = t[1]
            self.addProgLabel(t[1])
        while 1:
            self.getFuncLines()
            if self.EndFuncFound:
                break

    def errorOne(self):
        t = self.token[0]
        if t[1]=='\n':
            return
        self.log.WriteText('errorOne ')
        self.log.WriteText(self.token[0])
        
    def passOne(self):
        while 1:
            self.getLine()
            #print self.token
            if len(self.token)==0:
                pass
            else:
	        t=self.token[0]
	        if (t[0]==1) and (t[1]=='DATA'):
	            self.getData()
	        elif (t[0]==1) and (t[1]=='PROGRAM'):
	            self.getProgram()
	        elif (t[0]==1) and (t[1]=='FUNCTION'):
	            self.getFunction()
	        else:
                    self.errorOne()
            if self.lnum >= self.totlines:
                break;
            
    def passTwo(self):
        self.fixup()
        
    def as(self, lines, sbm):
    
        self.lines = lines
        self.totlines = len(lines)
        self.lnum = 0
        self.cnum = 0
        self.EndProgFound = 0
        self.EndDataFound = 0
        self.EndFuncFound = 0

        self.memD = sbm.memD
        self.memP = sbm.memP
        
        self.PC = 0
        self.Location = 32
               
        self.passOne()
        self.passTwo()

#----------------------------------------------------------------------

#----------------------------------------------------------------------
# This shows how to catch the Modified event from the wx.StyledTextCtrl

class MyText(wx.TextCtrl):
    def __init__(self, parent, ID):
        wx.TextCtrl.__init__(self, parent, ID,  
            style=wx.TE_MULTILINE|wx.TE_READONLY|wx.TE_NOHIDESEL)

#----------------------------------------------------------------------

class MyButton(wx.Button):
    def __init__(self, parent, id, pos, asc):
        if asc == ord('&'):
            st = '&&'
        else:
            st = chr(asc)
        wx.Button.__init__(self, parent, id, st, pos, (16,20))
        self.value=asc
#----------------------------------------------------------------------


class SBMPopup(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent, -1, "Screen", 
            pos=(828, 65),
            size=wx.Size(450, 540), 
            style = wx.FRAME_NO_TASKBAR | wx.CAPTION | wx.SYSTEM_MENU )


        self.panel = wx.Panel(self, -1)
        self.parent=parent
        self.log=parent.log
        wx.EVT_PAINT(self.panel, self.OnPaint)
        EVT_KEYBOARD(self,self.OnKeyboard)

        self.xpos = 2
        self.ypos = 4
        self.key  = 0
        self.keywait=0

        self.buffer = wx.EmptyBitmap(450, 400)
        mdc = wx.BufferedDC(None, self.buffer)
        mdc.SetBackground(wx.Brush('BLACK'))
        mdc.Clear()
        del mdc

        buttonNum=0
        asc=65
        for y in range(4):
            for x in range(26):
                if( buttonNum == 26) : asc =97
                elif( buttonNum == 52) : asc = 33
                elif( buttonNum == 84) : break
                buttonNum = buttonNum + 1
                b = MyButton(self.panel, buttonNum, (16*x+12, 20*y+420), asc)
                asc = asc + 1
                self.Bind(wx.EVT_BUTTON, self.LeftDown,b)
        asc = 91
        for z in range(5):
             buttonNum = buttonNum + 1
             b = MyButton(self.panel, buttonNum, (16*x+12, 20*y+420), asc)
             asc = asc + 1
             self.Bind(wx.EVT_BUTTON, self.LeftDown, b)
             x = x + 1
        asc = 123
        for z in range(4):
             buttonNum = buttonNum + 1
             b = MyButton(self.panel, buttonNum, (16*x+12, 20*y+420), asc)
             asc = asc + 1
             self.Bind(wx.EVT_BUTTON, self.LeftDown, b)
             x = x + 1
                  
        b = wx.Button(self.panel, 214, "Space",  (252, 480), (88,20))
        b.value=32
        self.Bind(wx.EVT_BUTTON, self.LeftDown, b)
              
        b = wx.Button(self.panel, 215, "Enter",  (340, 480), (88,20))
        b.value=13
        self.Bind(wx.EVT_BUTTON, self.LeftDown, b)
        b.SetFocus()

    def clear(self):
        self.xpos = 2
        self.ypos = 4

        mdc = wx.BufferedDC(None, self.buffer)
        mdc.SetBackground(wx.Brush('BLACK'))
        mdc.Clear()
        del mdc
        if self.keywait:
            keyboard.acquire()
            keyboard.notify()
            keyboard.release()
            self.keywait=0
        self.Refresh()

    def OnPaint(self, event):
        wx.BufferedPaintDC(self.panel, self.buffer)

        dc  = wx.ClientDC(self.panel)        
        dc.SetBrush(wx.GREY_BRUSH)
        dc.SetPen(wx.Pen('GREY', 1))
        dc.DrawRectangle(self.xpos, self.ypos+12, 11, 4)
        if self.keywait:
            dc.SetPen(wx.Pen('BLACK', 1))
            dc.SetBrush(wx.RED_BRUSH)
            dc.DrawRectangle(202, 408, 36, 4)
        else:
            dc.SetPen(wx.Pen('BLACK', 1))
            dc.SetBrush(wx.GREEN_BRUSH)
            dc.DrawRectangle(202, 408, 36, 4)
        
        del dc

    def OnKeyboard(self, evt):
        dc  = wx.ClientDC(self.panel)    
        if evt.data==1:
            dc.SetPen(wx.Pen('BLACK', 1))
            dc.SetBrush(wx.RED_BRUSH)
            dc.DrawRectangle(202, 408, 36, 4)
            self.keywait=1
        else:
            dc.SetPen(wx.Pen('BLACK', 1))
            dc.SetBrush(wx.GREEN_BRUSH)
            dc.DrawRectangle(202, 408, 36, 4)
            self.keywait=0
        del dc
    
    def LeftDown(self, evt):
        hObject=evt.GetEventObject()
        val = hObject.value
        self.put(val)
        keyboard.acquire()
        keyboard.notify()
        keyboard.release()
        self.key=val

    def get(self):     # wait for LeftDown of keyboard and return ascii value 
        return self.key
    
    def put(self, val):

        mdc = wx.BufferedDC(None, self.buffer)
        dc  = wx.ClientDC(self.panel)        

        if self.ypos < 400:
            dc.SetBrush(wx.BLACK_BRUSH)
            dc.SetPen(wx.Pen('BLACK', 1))
            dc.DrawRectangle(self.xpos, self.ypos+12, 11, 4)

        mdc.SetTextForeground('WHITE')
        mdc.SetTextBackground('BLACK')
        mdc.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))

        dc.SetTextForeground('WHITE')
        dc.SetTextBackground('BLACK')
        dc.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        
        if(val == 13):
            self.xpos = 2
            self.ypos = self.ypos + 18
        else:        
            if self.ypos < 400:
                mdc.DrawText(chr(val), self.xpos, self.ypos)
                dc.DrawText(chr(val), self.xpos, self.ypos)
            self.xpos = self.xpos + 11
            
        if self.xpos > 431:
            self.xpos = 2
            self.ypos = self.ypos + 18
            
        if self.ypos < 400:
            dc.SetBrush(wx.GREY_BRUSH)
            dc.SetPen(wx.Pen('GREY', 1))
            dc.DrawRectangle(self.xpos, self.ypos+12, 11, 4)
            
        del mdc
        del dc


#------------------------------------------------------

class DATAPopup(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent, -1, "DATA", 
            pos=(12,25),
            size=wx.Size(550,400), 
            style = wx.FRAME_NO_TASKBAR|wx.CAPTION|wx.SYSTEM_MENU )

        self.panel = wx.Panel(self, -1)
        wx.EVT_PAINT(self.panel, self.OnPaint)

        address = wx.StaticText(self.panel, -1, 'Adress',  ( 40,0))
        address.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        variable= wx.StaticText(self.panel, -1, 'Variable',(150,0))
        variable.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        value   = wx.StaticText(self.panel, -1, 'Value',   (300,0))
        value.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        
        self.Type=[0,0,0,0,0,0,0,1,2,5,6,1,1,1,1,3,3,1,1,1,0,0,0,0,0,0,0,0,0,1,2,0]
        self.dataTable = parent.dataTable
        self.memD      = parent.s.memD
        self.peek      = parent.s

    def new(self, parent):
        self.dataTable = parent.dataTable
        self.memD      = parent.s.memD
        self.Refresh() 

    def show(self, dc):
        def showPTR(x,y):
            dc.SetBrush(wx.RED_BRUSH)
            dc.DrawRectangle(x,y,6,6)  
            dc.SetBrush(wx.WHITE_BRUSH)
        def showCHAR(x,y):
            v=x[1][2]
            u=x[1][0]
            h=250
            for a in range(v):
                h=h+30
                if h>490:
                    h=40
                    y=y+24
                b=self.memD[u]
                u=u+1
                dc.DrawRectangle(h,20+y,26,22)  
                if 31 < b < 127 :             
                    dc.DrawText(chr(b),h+10,22+y)
                else:
                    if b==13: s='\\n'
                    else: s='\\0'
                    dc.DrawText(s,h+4,22+y)
                if var==u-1: showPTR(h,20+y)

            return y
                    
        def showAddr(x, y):
            dc.DrawRectangle( 30,20+y, 74,22)
            v=x[1]
            if v[2]==1:
                a='      '+str(v[0])
                dc.DrawText(a[-4:], 38,22+y)
            else:
                a=' '+str(v[0])+':'+str(v[0]+v[2])
                dc.DrawText(a[-7:], 32,22+y)
        
        t=self.Type[self.peek.inst[0]]
        if t>0 and self.peek.tics>0:
            if t<4:
                ptr=-1
                var=self.peek.inst[1]
            else:
                ptr=self.peek.inst[1]
                var=self.memD[ptr]
        else:
            ptr=-1
            var=-1
        #print self.peek.inst[0],t,ptr,var
        dc.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        dc.SetBrush(wx.WHITE_BRUSH)
        dc.DrawRectangle( 30,20, 74,22)
        dc.DrawRectangle(140,20,100,22)
        dc.DrawText('0:6',    56,22)
        dc.DrawText('Buffer',160,22)
        h=250
        for x in range(6):
            v=self.memD[x]
            h=h+30
            if 31 < v < 127 :             # isprint
                dc.SetBrush(wx.WHITE_BRUSH)
                dc.DrawRectangle(h,20,26,22)  
                dc.DrawText(chr(v),h+10,22)
            else:
                dc.SetBrush(wx.GREY_BRUSH)
                dc.DrawRectangle(h,20,26,22)
            if var==x: showPTR(h,20)
 
        dc.SetBrush(wx.WHITE_BRUSH)
    
        d = self.dataTable
        y = 0
        for x in d:
            y=y+24
            showAddr(x,y)
            dc.DrawRectangle(140,20+y,100,22)
            dc.DrawText(x[0],  172,22+y)
            if x[1][1]==1:
                dc.DrawRectangle(280,20+y, 80,22)
                v='      '+str(self.memD[x[1][0]])
                dc.DrawText(v[-4:],292,22+y)
                if var==x[1][0]: showPTR(280,20+y)
            else:
                y=showCHAR(x,y)


    def OnPaint(self, evt):
        dc = wx.PaintDC(self.panel)
        self.show(dc)
        del dc
                
    def LeftDown(self, evt):
        pass

    def update(self) :
        dc = wx.ClientDC(self.panel)
        self.show(dc)
        del dc
#------------------------------------------------------
            
class PROGPopup(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent, -1, "PROG", 
            pos=(12, 435),
            size=wx.Size(550, 550), 
            style = wx.FRAME_NO_TASKBAR | wx.CAPTION | wx.SYSTEM_MENU )

        self.panel = wx.Panel(self, -1)
        self.text = MyText(self.panel, -1)
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.Add(self.text, 1, wx.EXPAND)
        self.panel.SetSizer(sizer)
        self.panel.SetAutoLayout(True)
        self.s         = parent.s
        self.dataTable = parent.dataTable
        self.memD      = parent.s.memD
        self.memP      = parent.s.memP
        self.labDic    = parent.labDic
        self.lines     = parent.s.asm.lines
        self.begin     = parent.s.asm.progBegin
        self.totlines  = parent.s.asm.totlines
        self.select    = [0,12]

        wx.EVT_PAINT(self.panel, self.OnPaint)

        b = self.begin
        t = self.totlines  
        s,e=0,0
        self.position=[]
        y=0
        i=self.memP[y]
        n=i[2]
        for x in range(t)[b:t]:
            aa='      '+str(x-b+1)
            if x!=n:
                bb='          '
            else:
                bb='      '+str(y)
                bb=bb[-6:]+'    '
                y=y+1
                i=self.memP[y]
                n=i[2]
            l=aa[-6:]+bb+self.lines[x][:49]
            if l[-1]!='\n':l=l+'\n'
            s=e
            e=e+len(l)+1
            self.position.append([s+16,e])
            self.text.AppendText(l)
        #print self.position    
        font = wx.Font(fontsize-2, wx.SWISS, wx.NORMAL, wx.NORMAL, False,'Courier New')
        self.text.SetFont(font)


    def OnPaint(self, evt):
        wx.PaintDC(self.panel)
        i = self.memP[self.s.PC]
        s,e=self.position[i[2]-self.begin]
        self.text.SetSelection(s,e-1)
        
        
    def LeftDown(self, evt):
        pass

    def new(self, parent) :
        self.s         = parent.s
        self.dataTable = parent.dataTable
        self.memD      = parent.s.memD
        self.memP      = parent.s.memP
        self.labDic    = parent.labDic
        self.lines     = parent.s.asm.lines
        self.begin     = parent.s.asm.progBegin
        self.totlines  = parent.s.asm.totlines
        self.text.Clear()
        b = self.begin
        t = self.totlines  
        s,e=0,0
        self.position=[]
        y=0
        i=self.memP[y]
        n=i[2]
        for x in range(t)[b:t]:
            aa='      '+str(x-b+1)
            if x!=n:
                bb='          '
            else:
                bb='      '+str(y)
                bb=bb[-6:]+'    '
                y=y+1
                i=self.memP[y]
                n=i[2]
            l=aa[-6:]+bb+self.lines[x][:49]
            if l[-1]!='\n':l=l+'\n'
            s=e
            e=e+len(l)+1
            self.position.append([s+16,e])
            self.text.AppendText(l)
        #print self.position    
        #print self.memP[0:50]
        font = wx.Font(fontsize-2, wx.SWISS, wx.NORMAL, wx.NORMAL, False,'Courier New')
        self.text.SetFont(font)
        i = self.memP[self.s.PC]
        s,e=self.position[i[2]-self.begin]
        self.text.SetSelection(s,e-1)
    
    def update(self) :
        i = self.memP[self.s.PC]
        s,e=self.position[i[2]-self.begin]
        self.text.SetSelection(s,e-1)
        
#------------------------------------------------------------------------------------

#------------------------------------------------------------------------------------

class MyFrame(wx.Frame) :
    def __init__(
            self, parent, ID, title, pos = wx.DefaultPosition,
            size = wx.DefaultSize, style = wx.DEFAULT_FRAME_STYLE,
            Num=0,List=[]
            ) :
        wx.Frame.__init__(self, parent, ID, title, pos, size, style);

        splitter = wx.SplitterWindow(self, -1, style=wx.CLIP_CHILDREN | wx.SP_LIVE_UPDATE | wx.SP_3D)
        self.panel = wx.Panel(splitter, -1, style=wx.CLIP_CHILDREN)
        self.log = wx.TextCtrl(splitter, -1,
                              style = wx.TE_MULTILINE|wx.TE_READONLY|wx.HSCROLL)

        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow);
        wx.EVT_PAINT(self.panel, self.OnPaint)
        EVT_RESULT(self,self.OnFinish)

        acc = wx.StaticText(self.panel, -1, 'ACC',  (20, 12))
        acc.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        aux = wx.StaticText(self.panel, -1, 'AUX',  (20, 36))
        aux.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        pct = wx.StaticText(self.panel, -1, 'PC ',  (20, 60))
        pct.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        code = wx.StaticText(self.panel, -1, 'CODE',(20, 84))
        code.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        tics = wx.StaticText(self.panel, -1, 'Tics',(20, 108))
        tics.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        #text.SetSize(text.GetBestSize())
 
        
        self.binT = ['0000', '0001', '0010', '0011', '0100', '0101', '0110', '0111',
                     '1000', '1001', '1010', '1011', '1100', '1101', '1110', '1111']

        b = wx.Button(self.panel, 202, 'Screen', (25, 170))
        self.Bind(wx.EVT_BUTTON, self.clearScreen, b)

        b = wx.Button(self.panel, 203, 'Load', (125, 170))
        self.Bind(wx.EVT_BUTTON, self.load, b)

        b = wx.Button(self.panel, 204, 'Assemble', (225, 170))
        self.Bind(wx.EVT_BUTTON, self.assemble, b)

        b = wx.Button(self.panel, 205, 'Run', (325, 170))
        self.Bind(wx.EVT_BUTTON, self.run, b)

        b = wx.Button(self.panel, 206, 'Show', (25, 140))
        self.Bind(wx.EVT_BUTTON, self.showAll, b)

        b = wx.Button(self.panel, 207, 'Options', (125, 140))
        self.Bind(wx.EVT_BUTTON, self.showOptions, b)

        b = wx.Button(self.panel, 208, 'Reset', (225, 140))
        self.Bind(wx.EVT_BUTTON, self.reset, b)

        b = wx.Button(self.panel, 209, 'Go', (325, 140))
        self.Bind(wx.EVT_BUTTON, self.go, b)

        self.dataPopped = False
        self.sbmCreated = False
        self.assembled = False
    
        self.win = SBMPopup(self)
        self.win.Show(True)

        splitter.SplitHorizontally(self.panel, self.log, -80)
        splitter.SetMinimumPaneSize(80)

        self.runthread=None
        
        image = wx.Image('image/GreenEye32.bmp', wx.BITMAP_TYPE_ANY)
	image = image.ConvertToBitmap()
	
	icon = wx.EmptyIcon()
	icon.CopyFromBitmap(image)
	
	self.SetIcon(icon)


    def toBinStr(self, v):
        s=''
        t = []
        for x in [0,1,2,3]:
            v,r=divmod(v,16)
            t.append(r)
        t.reverse()
        for x in t:
            s=s+self.binT[x]
        return s

    def showControl(self, dc) :
        if self.sbmCreated:
            acc=self.toBinStr(self.s.ACC[0])
            aux=self.toBinStr(self.s.ACC[1])
            sss=self.toBinStr(self.s.PC)
            pc=sss[-11:]
        else:
            acc = '0000000000000000'
            aux = '0000000000000000'
            pc  = '00000000000'
            
        dc.SetBrush(wx.WHITE_BRUSH)
        dc.DrawRectangle( 70, 12,164,22)
        dc.DrawRectangle(250, 12, 64,22)
        #dc.DrawRectangle(330, 12, 26,22)
        dc.DrawRectangle( 70, 36,164,22)
        dc.DrawRectangle(250, 36, 64,22)
        dc.DrawRectangle(120, 60,114,22)
        dc.DrawRectangle(270, 60, 44,22)  
        dc.DrawRectangle( 70, 84,330,22)  
        dc.DrawRectangle(250,108, 64,22)  

        dc.SetFont(wx.Font(fontsize, wx.SWISS, wx.NORMAL, wx.NORMAL, False, 'Courier New'))
        dc.DrawText(acc,72,13)
        dc.DrawText(aux,72,37)
        dc.DrawText(pc,122,61)
        
        if self.sbmCreated:
            v   = self.s.ACC[0]
            acc = '     ' + str(v)
            aux = '     ' + str(self.s.ACC[1])
            pc  = '   '   + str(self.s.PC)
            tics= '     ' + str(self.s.tics)
        else:
            v   = 0
            acc = '     0'
            aux = '     0'
            pc  = '   0'
            tics= '     0'
            
        dc.DrawText(acc[-6:],252,13)
        dc.DrawText(aux[-6:],252,37)
        dc.DrawText(pc[-4:], 272,61)
        dc.DrawText(tics[-6:],252,109)
        
        if 31 < v < 127 :             # isprint
            dc.DrawRectangle(330, 12, 26, 22)  
            dc.DrawText(chr(v), 338, 13)
        else:
            dc.SetBrush(wx.GREY_BRUSH)
            dc.DrawRectangle(330, 12, 26, 22)
        
        if self.assembled:
            l = self.s.asm.lines
            c = self.s.memP[self.s.PC]
            dc.DrawText(l[c[2]][0:31], 82, 85)

        
    def OnPaint(self, evt):
        dc  = wx.PaintDC(self.panel)        
        self.showControl(dc) 
        del dc
  
    def clearScreen(self, evt):
        self.win.clear()

    def load(self, evt) :

        wildcard = 'SBM assemly source (*.s)|*.s|'     \
            'All files (*.*)|*.*'

        if not self.sbmCreated:
            self.s=SBMFrame(self.win)
            self.sbmCreated = True
        self.assembled = False

        # Create the dialog. 
        dlg = wx.FileDialog(
            self, message='Choose a file', defaultDir=os.getcwd(), 
            defaultFile='', wildcard=wildcard, style=wx.OPEN | wx.CHANGE_DIR
            )

        # Show the dialog and retrieve the user response. 
        # If it is the OK response, process the data.
        if dlg.ShowModal() == wx.ID_OK:
            # This returns a Python list of files that were selected.
            files = dlg.GetFilenames()
            for file in files:
                self.file = file

        # Destroy the dialog. Don't do this until you are done with it!
        # BAD things can happen otherwise!
        dlg.Destroy()

    def assemble(self, evt) :
        self.s.assemble(self.file)
        self.log.WriteText('assemble...\n')
        if self.runthread==None:
            self.runthread=threading.Thread(target=self.s.SBMLoop, args=())
            self.runthread.start()
        #print threading.enumerate()
        if self.dataPopped:
            self.showAll(ResultEvent(None))
        self.assembled = True
 
    def run(self, evt) :
        self.s.onestep=False
        # sync
        gosign.acquire()
        gosign.notify()
        gosign.release()
            
    def go(self, evt) :
        self.s.onestep=True
        # sync
        gosign.acquire()
        gosign.notify()
        gosign.release()
            
    def OnFinish(self, evt) :
        self.controlUpdate()
        self.dataUpdate()
        self.progUpdate()
        
    def showAll(self, evt) :
        t = []
        d = self.s.asm.variableDic
        self.varDic = {}
        for x in d.items():
            y=x[1][:]
            y[0]=x[0]
            t.append([x[1][0], y])
            self.varDic.update({x[1][0]:y})
        t.sort()
        self.dataTable = []
        for x in t:
            y=x[1][:]
            y[0]=x[0]
            self.dataTable.append([x[1][0], y])

        t = []
        d = self.s.asm.labelDic
        self.labDic = {}
        for x in d.items():
            t.append([x[1], x[0]])
            self.labDic.update({x[1]:x[0]})
        t.sort()
        self.labTable = []
        for x in t:
            self.labTable.append([x[1], x[0]])

        if self.dataPopped:  
            self.data.new(self)
            self.prog.new(self)
        else:
            self.dataPopped = True

            self.data = DATAPopup(self)
            self.data.Show(True)

            self.prog = PROGPopup(self)
            self.prog.Show(True)
            self.SetFocus()

        self.controlUpdate()
        self.dataUpdate()
        self.progUpdate()
        

    def dataUpdate(self) :
        if not self.dataPopped: return
        self.data.update()
        

    def showOptions(self, evt) :
        pass
        
    def progUpdate(self) :
        if not self.dataPopped: return
        self.prog.update()
    
            
    def controlUpdate(self) :
        dc=wx.ClientDC(self.panel)
        self.showControl(dc)
        del dc

    def reset(self, evt) :
        self.win.clear()
        self.s.reset()   
        self.controlUpdate()
        self.dataUpdate()
        self.progUpdate()
        
    def OnCloseWindow(self, evt) :
        if self.runthread:
            self.s.terminate=True
            gosign.acquire()
            gosign.notify()
            gosign.release()
        self.win.Destroy()
        if self.dataPopped:
            self.data.Destroy()
            self.prog.Destroy()
        self.Destroy()

#------------------------------------------------------

if __name__ == '__main__':
    class MyApp(wx.App) :
        def __init__(self, redirect = 0) :
            wx.App.__init__(self, redirect)
        
        def OnInit(self) :
            frame = MyFrame(
                None, -1, 'SBM', pos = (828, 610), size = (450, 330))
            frame.Show(True)
            self.SetTopWindow(frame)
            return True
        
    app = MyApp()
    app.MainLoop()

