/****************************************************************************
 * Elf32 loader plugin
 *
 * Default IDA ELF loader plugin has a bug, as it's able only to process
 * properly file if section header in ELF is present. 
 * For ELF loader, these sections are useless, as all data must be present
 * if Program Header (where to load, from where to load, what libraries are
 * required, path to interp - if not static build, symbols, names of symbols
 * etc.)
 * It's possible to mess up IDA thus it's not able to properly load file, and
 * even when it loads it doesn't handle PLTGOT properly, thus all calls will
 * look ugly. Purpose of this loader plugin, is to overcome this problem, and
 * load file only, and only based on program header, in similar way as it's 
 * loaded by ELF loader itself.
 *
 *
 * There are some things to fix, like checking it there is no read out of bounds
 * etc... but for now code seems stable, and can load any elf32 for intel32, and
 * parse properly DT_GNU_HASH
 *
 *                                                  (c) 2013 deroko of ARTeam                              
 *****************************************************************************/ 
#include        "defs.h"

#define DT_GNU_HASH 0x6ffffef5

unsigned char elfmag[] = {0x7f, 0x45, 0x4c, 0x46};

Elf32_Ehdr	*pelf;
Elf32_Phdr	*pphdr;
Elf32_Phdr	*pphdr_dyn;
Elf32_Sym	*psym;
Elf32_Rel	*ppltrel;
Elf32_Rel	*prel;
unsigned int    rel_size;
unsigned int	pltrel_size;
unsigned char 	*buff;
char		*stringtable;
unsigned int	stringtable_size;
unsigned long	got;
bool		b_shared;
unsigned        int size;

static bool    verify_access(void *data, unsigned int sz){
        unsigned long ptr;
        unsigned long ptr_end;
        
        ptr     = (unsigned long)data;
        ptr_end = ptr + sz;
        
        if (!(ptr >= (unsigned long)buff && ptr < (unsigned long)buff + size))
                return false;
        if (!(ptr_end >= (unsigned long)buff && ptr_end < (unsigned long)buff + size))
                return false;
        return true;
}

static unsigned int rva2raw(unsigned int va){
        unsigned int index;
        if (!verify_access(pphdr, sizeof(Elf32_Phdr) * pelf->e_phnum)) return 0;
                
        for (index = 0; index < pelf->e_phnum; index++){
                if (va >= pphdr[index].p_vaddr && va < (pphdr[index].p_vaddr + pphdr[index].p_memsz)){
                        va -= pphdr[index].p_vaddr;
                        va += pphdr[index].p_offset;
                        return va;
                }
        }
        return 0;
}

static unsigned int raw2rva(unsigned int raw){
	unsigned int index;
	
	//check if all Program header is allocated...
	if (!verify_access(pphdr, sizeof(Elf32_Phdr) * pelf->e_phnum)) return 0;
	        
	for (index = 0; index < pelf->e_phnum; index++){
		if (raw >= pphdr[index].p_offset && raw < (pphdr[index].p_offset + pphdr[index].p_filesz)){
			raw -= pphdr[index].p_offset;
			raw += pphdr[index].p_vaddr;
			return raw;
		}
	}
	return 0;
}

static unsigned long find_dyn_val(unsigned int dyn_tag){
        Elf32_Dyn *pdyn;
        
        pdyn = (Elf32_Dyn *)(buff + pphdr_dyn->p_offset);
        if (!verify_access(pdyn, sizeof(Elf32_Dyn))) return 0;
        while (pdyn->d_tag != DT_NULL){
                if (pdyn->d_tag == dyn_tag){
                        return pdyn->d_un.d_val;
                }
                pdyn++;
                if (!verify_access(pdyn, sizeof(Elf32_Dyn))) return 0;
        }
        return 0;
}

static unsigned long find_dyn_ptr(unsigned int dyn_tag){
        Elf32_Dyn *pdyn;

        pdyn = (Elf32_Dyn *)(buff + pphdr_dyn->p_offset);
        if (!verify_access(pdyn, sizeof(Elf32_Dyn))) return 0;
                
        while (pdyn->d_tag != DT_NULL){
                if (pdyn->d_tag == dyn_tag){
                        return pdyn->d_un.d_ptr;
                }
                pdyn++;
                if (!verify_access(pdyn, sizeof(Elf32_Dyn))) return 0;
        }
        return 0;
}


static int create_xseg(ea_t base,
		       ea_t start,
		       ea_t end)
{
	segment_t s;
	s.sel = setup_selector(base);
	s.startEA = start;
	s.endEA	  = end;
	s.align   = saRelByte;
	s.comb    = scPub;
	s.bitness = 0;
	return add_segm_ex(&s, "extern", "XTRN",ADDSEG_NOSREG); 		
}

void idaapi load_file(linput_t *li, ushort neflag, const char *)
{
	unsigned int index;
	unsigned char pattern[6];
	char	segname[100];
	char	*str;
	unsigned int entry_index = 0;
	Elf32_Dyn *pdyn;
	size = qlsize(li);
	buff = (unsigned char *)malloc(size);

	unsigned char *gnuhash;
        unsigned int  ngnubuckets;// = *(unsigned int *)&gnuhash[0];
        unsigned int  symidx     ;// = *(unsigned int *)&gnuhash[4];
        unsigned int  gnusymidx;
	unsigned int  bitmaskwords;//= *(unsigned int *)&gnuhash[8];
        unsigned int  *gnubuckets;// = (unsigned int *)(gnuhash + 16 + bitmaskwords * 4);
        unsigned int  *gnuchains;//  = (unsigned int *)((unsigned char *)gnubuckets + ngnubuckets * 4);
        unsigned int  maxsym;// = 0;
	unsigned int  idx;

	lreadbytes(li, buff, size, 0);
	pelf = (Elf32_Ehdr *)buff;
	
	if (pelf->e_type == ET_EXEC){
		b_shared = false;
	}else if (pelf->e_type == ET_DYN){
		b_shared = true;
	}else{
		msg("[X] Unknown e_type : %d\n", pelf->e_type);
		msg("[X] Aborting loader... nottify author about this...\n");
		return;
	}
	create_filename_cmt();

	pphdr = (Elf32_Phdr *)(buff + pelf->e_phoff);	
        if (!verify_access(pphdr, sizeof(Elf32_Phdr) * pelf->e_phnum)){
                msg("[X] Program header is getting out of application scope...");
                msg("[X] can't load this binary...");
                goto __exit0;
        }
        
	//find all PT_LOAD and load them into memory by creating segments
	for (index = 0; index < pelf->e_phnum; index++){
		if (pphdr[index].p_type != PT_LOAD) continue;
		file2base(li, 
		          pphdr[index].p_offset,
			  pphdr[index].p_vaddr,
			  pphdr[index].p_vaddr + pphdr[index].p_filesz,
			  FILEREG_PATCHABLE);
		memset(segname, 0, sizeof(segname));
		qsnprintf(segname, sizeof(segname), "PT_LOAD%d", index);
		if (pphdr[index].p_flags & PF_X)
			add_segm(0,
				 pphdr[index].p_vaddr,
				 pphdr[index].p_vaddr + pphdr[index].p_memsz,
				 segname,
				 "CODE");
		else
			add_segm(0,
                                 pphdr[index].p_vaddr,
                                 pphdr[index].p_vaddr + pphdr[index].p_memsz,
                                 segname,
                                 "DATA");

		segment_t *s = getseg(pphdr[index].p_vaddr);
		if (pphdr[index].p_flags & PF_X)
			s->perm |= SEGPERM_EXEC;
		if (pphdr[index].p_flags & PF_W)
			s->perm |= SEGPERM_WRITE;
		if (pphdr[index].p_flags & PF_R)
			s->perm |= SEGPERM_READ;
		set_segm_addressing(s, 1);
		for (unsigned int i = 0; i < SREG_NUM; i++){
			s->defsr[i] = 0;
		}
	}
	add_entry(entry_index, pelf->e_entry, "_start", true);
	add_long_cmt(pelf->e_entry, 1, "Exported entry %d. _start", entry_index);
	entry_index++;
	add_pgm_cmt("Entrypoint : %.08X", pelf->e_entry);

	for (index = 0; index < pelf->e_phnum; index++){
		if (pphdr[index].p_type == PT_DYNAMIC){
			pphdr_dyn = &pphdr[index];
			break;
		}
	}
	if (pphdr_dyn == NULL) goto __exit0;
	stringtable_size = find_dyn_val(DT_STRSZ);
	if (stringtable_size != 0){
		if (find_dyn_ptr(DT_SYMTAB)){
			stringtable = (char *)malloc(stringtable_size + 1);
			memset(stringtable, 0, stringtable_size + 1);
			memcpy(stringtable, buff + rva2raw(find_dyn_ptr(DT_STRTAB)), stringtable_size);
		}
	}

	pdyn = (Elf32_Dyn *)(buff + pphdr_dyn->p_offset);
	while (pdyn->d_tag != DT_NULL){
		unsigned int stridx;
		if (pdyn->d_tag == DT_NEEDED){
			stridx = pdyn->d_un.d_val;
			msg("needed %s\n", &stringtable[stridx]); 	
			add_pgm_cmt("Requires : %s", &stringtable[stridx]);
		}
		pdyn++;

	}	
	
	got = find_dyn_ptr(DT_PLTGOT);
	//check if there is DT_JMPREL, and if not just bail out...
	//we don't have "imports"
	if (find_dyn_ptr(DT_SYMTAB))
		psym = (Elf32_Sym *)(buff + rva2raw(find_dyn_ptr(DT_SYMTAB)));
	if (!psym) goto __exit0;

	if (find_dyn_val(DT_PLTREL) != DT_REL || !find_dyn_ptr(DT_JMPREL)) goto __dorel;
	ppltrel = (Elf32_Rel *)(buff + rva2raw(find_dyn_ptr(DT_JMPREL)));
	pltrel_size = find_dyn_val(DT_PLTRELSZ);
	for (index = 0; index < pltrel_size/sizeof(Elf32_Rel); index++){
		symidx = ELF32_R_SYM(ppltrel[index].r_info);
		//unsigned int type   = ELF32_R_TYPE(ppltrel[index].r_info);
		if (symidx == 0) continue;	//this is empty symbol entry
		//we do one nasty hack here... combine all externs into XTRN segment
		//thus all these will be part of segment
		if (psym[symidx].st_name == 0) continue;
		create_xseg(0, ppltrel[index].r_offset, ppltrel[index].r_offset+4);
		do_name_anyway(ppltrel[index].r_offset, &stringtable[psym[symidx].st_name]);		
		//search for byte patern which leads to this symbol...
		//crease serach patern for jmp dword offset to symbol
		if (b_shared == false){
			pattern[0] = 0xFF;
			pattern[1] = 0x25;
			*(unsigned int *)&pattern[2] = ppltrel[index].r_offset;
		}else{
			//for shared files we try to locate jmp [ebx+xx] which indicated
			//GOT + offset to our stored symbol... thus we need to handle this
			//case as well...
			pattern[0] = 0xFF;
			pattern[1] = 0xA3;
			*(unsigned int *)&pattern[2] = ppltrel[index].r_offset - got;
		}
		for (unsigned int i = 0; i < size-6; i++){
			if (!memcmp(&buff[i], pattern, 6)){
				size_t slen;
				slen = strlen(&stringtable[psym[symidx].st_name]);

				str = (char *)malloc(slen+2);
				memset(str, 0, slen+2);
				str[0] = '_';
				memcpy(&str[1], &stringtable[psym[symidx].st_name], slen);
				//msg("Found pattern at : %.08X\n", raw2rva((unsigned long)&buff[i] - (unsigned long)buff));
				do_name_anyway(raw2rva((unsigned long)&buff[i] - (unsigned long)buff), str);
				free(str);
			}
		}
	}
__dorel:
	if (!find_dyn_ptr(DT_REL)) goto __no_rel;
        prel = (Elf32_Rel *)(buff + rva2raw(find_dyn_ptr(DT_REL)));
        rel_size = find_dyn_val(DT_RELSZ);
        for (index = 0; index < rel_size/sizeof(Elf32_Rel); index++){
                symidx = ELF32_R_SYM(prel[index].r_info);
                if (symidx == 0) continue;
		if (psym[symidx].st_name == 0) continue;
		//unsigned int type   = ELF32_R_TYPE(prel[index].r_info);
                //if (symidx == 0) continue;    //it's NULL sym entry...
                //msg("Exernal symbol %s\n", &stringtable[psym[symidx].st_name]);
                //msg("Symbold address : %.08X\n", prel[index].r_offset);
                if (false == do_name_anyway(prel[index].r_offset, &stringtable[psym[symidx].st_name])){
                        msg("[X] Failed to add name\n");
                }
        }
__no_rel:
	//Add name for _GLOBAL_OFFSET_TABLE_ if it exists...
	if (!got) goto __check_init;
	do_name_anyway(find_dyn_ptr(DT_PLTGOT), "_GLOBAL_OFFSET_TABLE_");

__check_init:
	if (find_dyn_ptr(DT_INIT)){
		//do_name_anyway(find_dyn_ptr(DT_INIT, "_init_proc");
		add_entry(entry_index, find_dyn_ptr(DT_INIT), "_init_proc", true);
		entry_index++;
	}
	if (find_dyn_ptr(DT_FINI)){
		//do_name_anyway(find_dyn_ptr(DT_FINI, "_term_proc");
		add_entry(entry_index, find_dyn_ptr(DT_FINI), "_term_proc", true);
		entry_index++;
	}
	
	unsigned int initsz;
	unsigned int *pinit;
	if (find_dyn_ptr(DT_INIT_ARRAY)){
		char	szinitproc[100];

		initsz = find_dyn_val(DT_INIT_ARRAYSZ);
		pinit = (unsigned int *)(buff + rva2raw(find_dyn_ptr(DT_INIT_ARRAY)));
		do_name_anyway(find_dyn_ptr(DT_INIT_ARRAY), "_init_proc_array");
		for (index = 0; index < initsz/sizeof(unsigned int); index++){
			memset(szinitproc, 0, sizeof(szinitproc));
			qsnprintf(szinitproc, sizeof(szinitproc), "_init_proc_array_%d", index);
			add_entry(entry_index, pinit[index], szinitproc, true);
			entry_index++;
		}
	}	
	unsigned int finisz;
	unsigned int *pfini;
	if (find_dyn_ptr(DT_FINI_ARRAY)){
		char szfiniproc[100];
		finisz = find_dyn_val(DT_FINI_ARRAYSZ);
		pfini = (unsigned int *)(buff + rva2raw(find_dyn_ptr(DT_FINI_ARRAY)));
		do_name_anyway(find_dyn_ptr(DT_FINI_ARRAY), "_term_proc_array");
		for (index = 0; index < finisz/sizeof(unsigned int); index++){
			memset(szfiniproc, 0, sizeof(szfiniproc));
			qsnprintf(szfiniproc, sizeof(szfiniproc), "_term_proc_array_%d", index);
			add_entry(entry_index, pfini[index], szfiniproc, true);
			entry_index++;
		}
	}

	if (!find_dyn_ptr(DT_GNU_HASH)) goto __exit0;
	//we will process only DT_GNU_HASH which is a mess...
	//layout is really weird, and I really don't understand most here
	//as there is no public documentation, and I really don't wanna spend
	//my life time to go over binutils...
	//[number of buckets] [DWORD]  <---- number of buckets
	//[symbol index     ] [DWORD]  <---- symbolx of index where exports start
	//[bitmaskwords     ] [DWORD]  <---- multiply by 4 or 8 (x64) to get to gnubuckets
	//[unknown          ] [DWORD]  <---- have no idea, it's called shift2 in binutils (and used only on mips)
        //[variable array of bitmaskwords] <------ bitmaskwords * 4 or 8(x64) to get to gnubuckets
        //[gnubuckets]
        //[gnuhashbuckets]
        //so lets do it...
	gnuhash = (unsigned char *)(rva2raw(find_dyn_ptr(DT_GNU_HASH)) + buff);
	ngnubuckets = *(unsigned int *)&gnuhash[0];
	gnusymidx   = *(unsigned int *)&gnuhash[4];
	bitmaskwords= *(unsigned int *)&gnuhash[8];
	gnubuckets = (unsigned int *)(gnuhash + 16 + bitmaskwords * 4);
	gnuchains  = (unsigned int *)((unsigned char *)gnubuckets + ngnubuckets * 4);
	maxsym = 0;
	for (unsigned int i = 0; i < ngnubuckets; i++){
		if (gnubuckets[i] > maxsym)
			maxsym = gnubuckets[i];
	}
	//search for last gnuchain entry...
	maxsym -= gnusymidx;
	index = maxsym;
	maxsym++;
	
	while ((gnuchains[index] & 1) == 0){
		maxsym++;
		index++;
	}
	msg("Total number of symbols : %d\n", maxsym);
	//now we know how many symbols are there... so we can display all symbols 
	//starting at index symidx for maxsym++;
	//we will do stuff as is done by gnu loader...
	for (index = 0; index < ngnubuckets; index++){
		if (gnubuckets[index] == 0) continue;
		symidx = gnubuckets[index];
		idx = gnubuckets[index] - gnusymidx;
		while (1){
			msg("exported symbol name : %s\n",  &stringtable[psym[symidx].st_name]);
			if (ELF32_ST_BIND(psym[symidx].st_info) != STB_GLOBAL && ELF32_ST_BIND(psym[symidx].st_info) != STB_WEAK) continue;
			if (ELF32_ST_TYPE(psym[symidx].st_info) == STT_FUNC){
				add_long_cmt(psym[symidx].st_value, 1, "Exported entry %d. %s", entry_index, &stringtable[psym[symidx].st_name]);
				add_entry(entry_index, psym[symidx].st_value, &stringtable[psym[symidx].st_name], true);
                		entry_index++;
			}else if (ELF32_ST_TYPE(psym[symidx].st_info) == STT_OBJECT){
				add_long_cmt(psym[symidx].st_value, 1, "Exported entry %d. %s", entry_index, &stringtable[psym[symidx].st_name]);
				add_entry(entry_index, psym[symidx].st_value, &stringtable[psym[symidx].st_name], false);
				entry_index++;
			}
			if (gnuchains[idx] & 1) break;
			idx++;
			symidx++;
		}
	}

__exit0:
	if (stringtable != NULL)
		free(stringtable);
	jumpto(pelf->e_entry);
	free(buff);
}

int idaapi accept_file(linput_t *li, char fileformatname[MAX_FILE_FORMAT_NAME], int n)
{
	Elf32_Ehdr	elf;
	int	ret;
	if (n) return 0;

	ret = lreadbytes(li, &elf, sizeof(elf), false);
	if (ret == -1)
		return 0;
	if (memcmp(&elf, elfmag, 4))
		return 0;
	//ingore ELF64 in this file...
	if (elf.e_ident[EI_CLASS] != ELFCLASS32) return 0;
	if (elf.e_machine != EM_386) return 0;
	qsnprintf(fileformatname, MAX_FILE_FORMAT_NAME, "Elf32");
	return 1;
	
}

loader_t LDSC =
{
  IDP_INTERFACE_VERSION,
  LDRF_RELOAD,               // loader flags
//
//      check input file format. if recognized, then return 1
//      and fill 'fileformatname'.
//      otherwise return 0
//
  accept_file,
//
//      load file into the database.
//
  load_file,
//
//      create output file from the database.
//      this function may be absent.
//
  NULL,
  NULL
};

