#ifndef __DEBUGGER__
#define __DEBUGGER__

#include        <windows.h>
#include        <winioctl.h>

#define MAX_BREAK       260

/*************************************************************************
 * IOCTL_INIT_PROCESS - tells tracer which process to use, and what event will
 *                      be used to sync communication between r0/r3
 *                      retrives struct : INIT_PROCESS
 * IOCTL_STOP_PROCESS - tells tracer to stop tracing. If there is no traced process
 *                      nothing happens
 * IOCTL_SET_REGISTERS- updates registers with data passed in TRACER_DATA,
 *                      state must be s_ready, s_softice to continue tracing
 *                      if regs are not modified, then just state should be changed
 * IOCTL_GET_REIGSTERS- gets registers and state. State can be s_except or s_exit
 *                      where on s_exit there is no more valid process
 * IOCTL_INIT_RDSTC   - not implemented. Was supposed to hook rdtsc so it can be used
 *                      as tick counter and break at the same time
 * IOCTL_STOP_RDTSC   - not implemented. Same as above, but never implemented
 **************************************************************************/
#define IOCTL_INIT_PROCESS    CTL_CODE(FILE_DEVICE_UNKNOWN, 0x800, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_STOP_PROCESS    CTL_CODE(FILE_DEVICE_UNKNOWN, 0x810, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_INIT_RDTSC      CTL_CODE(FILE_DEVICE_UNKNOWN, 0x820, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_STOP_RDTSC      CTL_CODE(FILE_DEVICE_UNKNOWN, 0x830, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_SET_REGISTERS   CTL_CODE(FILE_DEVICE_UNKNOWN, 0x840, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_GET_REGISTERS   CTL_CODE(FILE_DEVICE_UNKNOWN, 0x850, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_GET_VERSION     CTL_CODE(FILE_DEVICE_UNKNOWN, 0x860, METHOD_BUFFERED, FILE_ANY_ACCESS)
/*******************************************************
 * Use this struct to pass data via INIT_TRACER
 * pid   - handle of a process which should be logged
 * event - auto-reset ring3 created event
 *******************************************************/                  
typedef struct _INIT_PROCESS{
        HANDLE  dwProcessId;
        HANDLE  hEvent;
}INIT_PROCESS, *PINIT_PROCESS;

/*******************************************************
 * define states used by debugger
 * StatusException     - defines that exception occured
 * StatusContinue      - send only by user mode program to tell
 *                       that debugging can continue
 * StatusDefault       - call original handler, sent by user mode
 *                      program to tell that debugging can continue
 * StatusExecuteSoftice    - send only  by user mode program to tell
 *                      tracer to stop, and generates int 3
 * StatusExit          - state is set when traced process exits
 *******************************************************/
typedef enum{
        StatusException,                         
        StatusContinue,                
        StatusDefault,
        StatusExecuteSoftice,
        StatusExit
}STATUS_ENUM;

/*******************************************************
 * This struct is used to control registers of a current
 * thread
 *******************************************************/

typedef struct{
        ULONG   reg_eax;
        ULONG   reg_ecx;
        ULONG   reg_edx;
        ULONG   reg_ebx;
        ULONG   reg_esp;
        ULONG   reg_ebp;
        ULONG   reg_esi;
        ULONG   reg_edi;
        ULONG   reg_eip;
        ULONG   reg_eflags;
        ULONG   reg_dr0;
        ULONG   reg_dr1;
        ULONG   reg_dr2;
        ULONG   reg_dr3;
        ULONG   reg_dr6;
        ULONG   reg_dr7;
        ULONG   handler;                //handler index which caused exception
}EXCEPTION, *PEXCEPTION;

 
typedef struct _TRACER_DATA{
        HANDLE  dwProcessId;
        HANDLE  dwThreadId;
        ULONG   dwStatus;
        EXCEPTION  x86_regs; 
}TRACER_DATA, *PTRACER_DATA;


VOID    DebugStop();
BOOL    DebugContinue(PTRACER_DATA pdata);
BOOL    DebugWait(PTRACER_DATA pdata, DWORD timeout);
BOOL    DebugInitProcess(__in DWORD dwProcessId);
VOID    BreakRemove(__in HANDLE hProcess, __in PVOID lpMemory);
VOID    BreakSet(__in HANDLE hProcess, __in PVOID lpMemory);

                        
// Dr6
#define BPM0_DETECTED                    0x00000001
#define BPM1_DETECTED                    0x00000002
#define BPM2_DETECTED                    0x00000004
#define BPM3_DETECTED                    0x00000008
#define TF_DETECTED                      0x00004000

// Dr7
#define BPM0_LOCAL_ENABLED               0x00000001
#define BPM0_W                           0x00010000
#define BPM0_RW                          0x00030000

#define BPM1_LOCAL_ENABLED               0x00000004
#define BPM1_W                           0x00100000
#define BPM1_RW                          0x00300000

#define BPM2_LOCAL_ENABLED               0x00000010
#define BPM2_W                           0x01000000
#define BPM2_RW                          0x03000000

#define BPM3_LOCAL_ENABLED               0x00000040
#define BPM3_W                           0x10000000
#define BPM3_RW                          0x30000000

#define BPM_LOCAL_EXACT                  0x00000100

#endif
