'   
'   VGA REGISTER I/O ADDRESSES AND BASE MEMORY ADDRESS
'   
'   The VGA's memory is exposed into the DOS memory area, starting at address
'   A000:0000 and extending to address A000:FFFF, when it is operating in the
'   color graphics mode.  The constant segment, A000, is defined here.  Also,
'   the I/O addresses for most of the useful register banks and features are
'   included here.

CONST VGASegment% = &HA000

CONST VREGMiscellaneousWR% = &H3C2
CONST VREGMiscellaneousRD% = &H3CC
CONST VREGFeatureControlWR% = &H3DA
CONST VREGFeatureControlRD% = &H3CA
CONST VREGInputStatus0RD% = &H3C2
CONST VREGInputStatus1RD% = &H3DA
CONST VREGSequencerIDXRW% = &H3C4
CONST VREGSequencerDATRW% = VREGSequencerIDXRW + 1
CONST VREGCrtcIDXRW% = &H3D4
CONST VREGCrtcDATRW% = VREGCrtcIDXRW + 1
CONST VREGGraphicsIDXRW% = &H3CE
CONST VREGGraphicsDATRW% = VREGGraphicsIDXRW + 1
CONST VREGAttributeIDXRW% = &H3C0
CONST VREGAttributeDATWR% = VREGAttributeIDXRW
CONST VREGAttributeDATRD% = VREGAttributeIDXRW + 1
CONST VREGColorGetIDXWR% = &H3C7
CONST VREGColorSetIDXWR% = &H3C8
CONST VREGColorIDXRD% = &H3C8
CONST VREGColorDATRW% = &H3C9
CONST VREGColorDACStateRD% = &H3C7
CONST VREGColorMaskRW% = &H3C6


'   
'   VGA CONTROL ROUTINES
'   
'   These routines are designed to set up specific VGA register values.
'   They do not depend on or modify any shared global variables for their
'   operation, so they may be used at any time.

DECLARE SUB SetPlaneMask (mask AS INTEGER)
DECLARE SUB SetWriteFunction (func AS INTEGER)
DECLARE SUB SetWriteMode (mode AS INTEGER)
DECLARE SUB SetDataRotation (rot AS INTEGER)
DECLARE SUB SetBitMask (mask AS INTEGER)
DECLARE SUB SetDisplayPitch (byteskew AS INTEGER)
DECLARE SUB SetGraphicsMode (p1 AS LONG, p2 AS LONG, p3 AS LONG, p4 AS LONG)


'   
'   VGA BASIC GRAPHICS SUPPORT ROUTINES
'   
'   These routines are designed for more general support of the VGA graphics.

DECLARE SUB SetResolution (xres AS INTEGER, yres AS INTEGER)
DECLARE SUB SetVirtualWidth (vwidth AS INTEGER)
DECLARE SUB InitVGAMemory (c AS INTEGER)
DECLARE SUB PositionWindowPA (plane AS INTEGER, addr AS INTEGER)
DECLARE SUB WritePixelPA (plane AS INTEGER, addr AS INTEGER, c AS INTEGER)
DECLARE SUB PXYtoPA (page AS INTEGER, x AS INTEGER, y AS INTEGER, plane AS INTEGER, addr AS INTEGER)

    DIM SHARED vgaWindowWidth AS INTEGER
    DIM SHARED vgaWindowHeight AS INTEGER
    DIM SHARED vgaVirtualWidth AS INTEGER
    DIM SHARED vgaPlaneSelect(0 TO 3) AS INTEGER


'   
'   KEYBOARD KEYS AND SUPPORT
'   

CONST keyNULL% = &H1FF
CONST keyENTER% = 13
CONST keyESCAPE% = 27
CONST keyUPARROW% = &H4800
CONST keyLEFTARROW% = &H4B00
CONST keyRIGHTARROW% = &H4D00
CONST keyDOWNARROW% = &H5000
CONST ctrlUPARROW% = &H8D00
CONST ctrlLEFTARROW% = &H7300
CONST ctrlRIGHTARROW% = &H7400
CONST ctrlDOWNARROW% = &H9100
CONST keyHOME% = &H4700

DECLARE FUNCTION ReadKey% ()
DECLARE FUNCTION WaitKey% ()

    DIM i AS INTEGER, ckey AS INTEGER, count AS INTEGER
    DIM x AS INTEGER, y AS INTEGER, plane AS INTEGER, offset AS INTEGER

        SetResolution 320, 240
        SetVirtualWidth 400
        InitVGAMemory 0
        PositionWindowPA 0, 0
        DEF SEG = VGASegment
        RANDOMIZE TIMER
        FOR i = 0 TO 2500
            LET x = RND * 400
            LET y = RND * 400
            PXYtoPA 0, x, y, plane, offset
            WritePixelPA plane, offset, RND * 15 + 1
        NEXT i
        DEF SEG

        LET x = 0
        LET y = 0
        LET ckey = WaitKey
        DO WHILE ckey <> keyESCAPE
            LET count = 1
            LET i = ReadKey
            DO WHILE i = ckey
                LET count = count + 1
                LET i = ReadKey
            LOOP
            SELECT CASE ckey
            CASE keyLEFTARROW
                IF x > 0 THEN
                    LET x = x - count
                    IF x < 0 THEN
                        LET x = 0
                    END IF
                    PXYtoPA 0, x, y, plane, offset
                    PositionWindowPA plane, offset
                END IF
            CASE keyRIGHTARROW
                IF x < 80 THEN
                    LET x = x + count
                    IF x > 80 THEN
                        LET x = 80
                    END IF
                    PXYtoPA 0, x, y, plane, offset
                    PositionWindowPA plane, offset
                END IF
            CASE keyUPARROW
                IF y > 0 THEN
                    LET y = y - count
                    IF y < 0 THEN
                        LET y = 0
                    END IF
                    PXYtoPA 0, x, y, plane, offset
                    PositionWindowPA plane, offset
                END IF
            CASE keyDOWNARROW
                IF y < 160 THEN
                    LET y = y + count
                    IF y > 160 THEN
                        LET y = 160
                    END IF
                    PXYtoPA 0, x, y, plane, offset
                    PositionWindowPA plane, offset
                END IF
            CASE keyHOME
                IF x <> 0 OR y <> 0 THEN
                    LET x = 0
                    LET y = 0
                    PXYtoPA 0, x, y, plane, offset
                    PositionWindowPA plane, offset
                END IF
            CASE ELSE
            END SELECT
            IF i = keyNULL THEN
                LET ckey = WaitKey
            END IF
        LOOP

        SCREEN 0, 0, 0, 0
        WIDTH 80, 50

        END

SUB InitVGAMemory (c AS INTEGER)
'   
'   Set all VGA memory to a particular color value.
'   
'   This routine accepts a color value and sets all of VGA RAM to it.  This
'   is normally only done once, after setting up the VGA display resolution.

    DIM i AS INTEGER

        IF vgaWindowWidth > 0 THEN
            SetWriteMode 0
            SetWriteFunction 0
            SetPlaneMask &HF
            DEF SEG = VGASegment
            FOR i = 0 TO &H7FFE
                POKE i, c
            NEXT i
            POKE &H7FFF, c
            FOR i = &H8000 TO &HFFFF
                POKE i, c
            NEXT i
            DEF SEG
        END IF

END SUB

SUB PositionWindowPA (plane AS INTEGER, addr AS INTEGER)
'   
'   Position the screen's display window, given the plane and offset.
'   
'   The screen window's upper left corner is positioned at the memory byte
'   specified by the given plane and address offset.  Only the lower two
'   bits of 'plane' are used.

    DIM dummy AS INTEGER

        IF vgaWindowWidth > 0 THEN
            WAIT VREGInputStatus1RD, &H8
            WAIT VREGInputStatus1RD, &H8, &H8
            OUT VREGCrtcIDXRW, &HC
            OUT VREGCrtcDATRW, (addr AND &HFF00) \ &H100
            OUT VREGCrtcIDXRW, &HD
            OUT VREGCrtcDATRW, addr AND &HFF
            LET dummy = INP(VREGInputStatus1RD) ' Reset attribute index.
            OUT VREGAttributeIDXRW, &H33
            OUT VREGAttributeDATWR, 2 * CINT(plane AND 3)
        END IF

END SUB

SUB PXYtoPA (page AS INTEGER, x AS INTEGER, y AS INTEGER, plane AS INTEGER, addr AS INTEGER)
'   
'   Compute the (plane, offset) from (page, x, y, virtualwidth).
'   
'   The actual plane and offset are computed, given the page, x, and y --
'   assuming the virtual width value already initialized.  (This calculation
'   would be much simpler if the BASIC editing environment supported unsigned
'   integer calculations.)
'
'   It may be better, in many cases, to replicate this code rather than
'   calling it as a function.  Or use this logic as part of other code that
'   is better optimized.

    DIM r AS LONG

        LET plane = x AND 3
        LET r = CLNG(page) + (x - plane) \ 4 + y * CLNG(vgaVirtualWidth \ 4)
        IF (r AND &H8000&) <> 0 THEN
            LET addr = r OR &HFFFF0000
        ELSE
            LET addr = r AND &H7FFF
        END IF

END SUB

FUNCTION ReadKey%

    DIM k AS STRING

        LET k = INKEY$
        SELECT CASE LEN(k)
        CASE 0
            LET ReadKey = keyNULL
        CASE 1
            LET ReadKey = ASC(k)
        CASE 2
            LET ReadKey = CVI(k)
        END SELECT

END FUNCTION

SUB SetBitMask (mask AS INTEGER)
'   
'   Set the VGA bit mask control register.
'   
'   The VGA uses a bit mask control register to help it decide which bits in
'   each pixel will be modified when asked to write new values into memory.
'   Every write operation to VGA memory can be limited, so that only certain
'   bits are changed.  Normally, when writing bytes into VGA RAM, this mask
'   is set to all 1s (&HFF) so that all bits can be changed.
'
'   The VGA bit mask control register is register 8 of the VGA GRAPHICS
'   section.  All 8 bits of the register are modified.

        OUT VREGGraphicsIDXRW, &H8
        OUT VREGGraphicsDATRW, mask

END SUB

SUB SetDataRotation (rot AS INTEGER)
'   
'   Set the VGA data rotation control.
'   
'   The VGA can automatically rotate the data before using it during write
'   operations.  Any number of bit positions can be rotated, from 0 to 7.
'   (This value is ignored in write MODE 1, for example.)  Normally, this is
'   simply set to 0 so that the data remains unrotated before being used.
'
'   The VGA data rotation control bits are part of register 3 of the VGA
'   GRAPHICS section.  Only the appropriate bits of the register are
'   modified.

        OUT VREGGraphicsIDXRW, &H3
        OUT VREGGraphicsDATRW, (INP(VREGGraphicsDATRW) AND &HF8) OR (rot AND &H7)

END SUB

SUB SetDisplayPitch (byteskew AS INTEGER)
'   
'   Set the VGA horizontal byte skew.
'   
'   The VGA uses an "offset" register to hold the value that it multiplies
'   by two and then adds to the starting address used for the previous
'   display row in order to get the new starting address for the next
'   display row.
'
'   In 256 color modes, this value should be set to the desired virtual page
'   width divided by 8.  (The value is multiplied by 2 and there are 4 memory
'   planes, so 2*4 = 8.)  In 256-color resolutions of 320 pixels wide, this
'   value is set to 40.  Note that virtual page resolutions are limited to
'   multiples of 8, in 256 color modes.
'
'   The VGA horizontal skew is register 19 (&H13) of the VGA CRTC section.
'   All 8 bits of the register are modified.

        OUT VREGCrtcIDXRW, &H13
        OUT VREGCrtcDATRW, byteskew

END SUB

SUB SetGraphicsMode (p1 AS LONG, p2 AS LONG, p3 AS LONG, p4 AS LONG)
'   
'   Set up principal VGA register values.
'   
'   A common group of VGA register values is sufficient to initialize the
'   VGA into a variety of screen resolutions.  These are primarily designed
'   for setting up the VGA into unchained modes, though.
'
'   The values in p1, p2, p3, and p4 hold packed byte values for the various
'   VGA registers.  The order can be important, so don't change the way it
'   is done below unless you know what you are doing.

        SCREEN 13, 0, 0, 0

        OUT VREGSequencerIDXRW, &H0
        OUT VREGSequencerDATRW, &H1
        OUT VREGMiscellaneousWR, p4 AND &HFF&
        OUT VREGSequencerIDXRW, &H0
        OUT VREGSequencerDATRW, &H3
        OUT VREGCrtcIDXRW, &H11
        OUT VREGCrtcDATRW, (p3 AND &H7F00&) \ &H100&
        OUT VREGCrtcIDXRW, &H0
        OUT VREGCrtcDATRW, (p1 AND &H7F000000) \ &H1000000
        OUT VREGCrtcIDXRW, &H1
        OUT VREGCrtcDATRW, (p1 AND &HFF0000) \ &H10000
        OUT VREGCrtcIDXRW, &H2
        OUT VREGCrtcDATRW, (p1 AND &HFF00&) \ &H100&
        OUT VREGCrtcIDXRW, &H3
        OUT VREGCrtcDATRW, p1 AND &HFF&
        OUT VREGCrtcIDXRW, &H4
        OUT VREGCrtcDATRW, (p2 AND &H7F000000) \ &H1000000
        OUT VREGCrtcIDXRW, &H5
        OUT VREGCrtcDATRW, (p2 AND &HFF0000) \ &H10000
        OUT VREGCrtcIDXRW, &H6
        OUT VREGCrtcDATRW, (p2 AND &HFF00&) \ &H100&
        OUT VREGCrtcIDXRW, &H7
        OUT VREGCrtcDATRW, p2 AND &HFF&
        OUT VREGCrtcIDXRW, &H8
        OUT VREGCrtcDATRW, &H0
        OUT VREGCrtcIDXRW, &H9
        OUT VREGCrtcDATRW, (p3 AND &H7F000000) \ &H1000000
        OUT VREGCrtcIDXRW, &H10
        OUT VREGCrtcDATRW, (p3 AND &HFF0000) \ &H10000
        OUT VREGCrtcIDXRW, &H11
        OUT VREGCrtcDATRW, (p3 AND &HFF00&) \ &H100&
        OUT VREGCrtcIDXRW, &H12
        OUT VREGCrtcDATRW, p3 AND &HFF&
        OUT VREGCrtcIDXRW, &H13
        OUT VREGCrtcDATRW, (p4 AND &H7F000000) \ &H1000000
        OUT VREGCrtcIDXRW, &H14
        OUT VREGCrtcDATRW, &H0
        OUT VREGCrtcIDXRW, &H15
        OUT VREGCrtcDATRW, (p4 AND &HFF0000) \ &H10000
        OUT VREGCrtcIDXRW, &H16
        OUT VREGCrtcDATRW, (p4 AND &HFF00&) \ &H100&
        OUT VREGCrtcIDXRW, &H17
        OUT VREGCrtcDATRW, &HE3
        OUT VREGCrtcIDXRW, &H11
        OUT VREGCrtcDATRW, INP(VREGCrtcDATRW) OR &H80
        OUT VREGSequencerIDXRW, &H4
        OUT VREGSequencerDATRW, &H6

END SUB

SUB SetPlaneMask (mask AS INTEGER)
'   
'   Set the VGA plane select mask.
'   
'   The upper 12 bits of 'mask' are ignored and the lower four bits are
'   associated with the VGA's memory planes and either enable or disable
'   the ability to write to specific planes.  The bits in 'mask' are mapped
'   as X X X X X X X X X X X X 3 2 1 0, with X meaning "ignored" and digits
'   refering to the plane they control/affect.  A bit value of 1 means
'   "enabled" and a bit value of 0 means "disabled."
'
'   The VGA plane select register is register 2 of the VGA SEQUENCER
'   section.  The upper four bits are reserved, so this routine attempts
'   to preserve their value.  Only the low-order 4 bits are modified.

        OUT VREGSequencerIDXRW, &H2
        OUT VREGSequencerDATRW, (INP(VREGSequencerDATRW) AND &HF0) OR (mask AND &HF)

END SUB

SUB SetResolution (xres AS INTEGER, yres AS INTEGER)
'   
'   Set the VGA screen resolution (in pixels).
'   
'   This routine accepts the desired screen resolution and sets up the VGA
'   registers accordingly.  In addition, it initializes the shared variables
'   used by other VGA support routines.
'
'   If the specified resolution cannot be achieved here, the shared global
'   values of vgaWindowWidth, vgaWindowHeight, and vgaVirtualWidth will all
'   be set to 0.  It is recommended to check only vgaWindowWidth, though,
'   as a sufficient test.

    DIM xw AS INTEGER, yw AS INTEGER

        LET xw = xres
        LET yw = yres
        IF xres = 320 AND yres = 200 THEN
            SetGraphicsMode &H5F4F5082, &H5480BF1F, &H419C8E8F, &H2896B963
        ELSEIF xres = 320 AND yres = 240 THEN
            SetGraphicsMode &H5F4F5082, &H55800D3E, &H41EAACDF, &H28E706E3
        ELSEIF xres = 320 AND yres = 400 THEN
            SetGraphicsMode &H5F4F5082, &H5480BF1F, &H409C8E8F, &H2896B963
        ELSEIF xres = 360 AND yres = 256 THEN
            SetGraphicsMode &H6B595A8E, &H5F8A30F0, &H6107A900, &H2D1F2FE7
        ELSEIF xres = 360 AND yres = 360 THEN
            SetGraphicsMode &H6B595A8E, &H5E8ABF1F, &H40888567, &H2D6DBA67
        ELSEIF xres = 360 AND yres = 400 THEN
            SetGraphicsMode &H6B595A8E, &H5E8ABF1F, &H409C8E8F, &H2D96B967
        ELSEIF xres = 360 AND yres = 480 THEN
            SetGraphicsMode &H6B595A8E, &H5E8A0D3E, &H40EAACDF, &H2DE706E7
        ELSEIF xres = 376 AND yres = 564 THEN
            SetGraphicsMode &H6E5D5E91, &H628F62F0, &H60378933, &H2F3C5CE7
        ELSE
            LET xw = 0
            LET yw = 0
            EXIT SUB
        END IF

        LET vgaWindowWidth = xw
        LET vgaWindowHeight = yw
        LET vgaVirtualWidth = xw

        LET vgaPlaneSelect(0) = 1
        LET vgaPlaneSelect(1) = 2
        LET vgaPlaneSelect(2) = 4
        LET vgaPlaneSelect(3) = 8

END SUB

SUB SetVirtualWidth (vwidth AS INTEGER)
'   
'   Set the VGA virtual horizontal width (in pixels).
'   
'   Establishes the virtual horizontal width of the VGA display, in pixels.
'
'   This routine does NOT affect the actual screen resolution seen by the
'   operator.  Instead, it only affects the apparent size of the virtual
'   memory page.  The value set here affects the calculation of the pixel
'   addresses, given their X and Y coordinates, and affects the addresses
'   used by the VGA when displaying them.
'
'   The virtual width should not be set to a value smaller than the actual
'   width of the screen resolution.  But this routine doesn't forbid it, so
'   you may attempt such oddities, if you want.  A value of 0 is disallowed.

    DIM vw AS INTEGER

        LET vw = vwidth AND &H7F8
        IF vw > 0 AND vw = vwidth AND vgaWindowWidth > 0 THEN
            SetDisplayPitch vw \ 8
            LET vgaVirtualWidth = vw
        END IF

END SUB

SUB SetWriteFunction (func AS INTEGER)
'   
'   Set the VGA's logical function for memory write operations.
'   
'   The VGA permits four different logical functions for writing values
'   into memory.  This is sometimes known as the VGA's "ALU operation."
'   Only the lower two bits of 'func' are used, the other bits are ignored.
'
'   Normally, the written value is simply placed into the designated memory
'   location, but there are more options offered.  The other logical
'   operations operate with the previous value already located at the
'   indicated address.
'
'   The logical functions are:  (0) normal mode, (1) AND mode, (2) OR mode,
'   and (3) XOR mode -- requiring two bits to represent all four functions.
'
'   The VGA logical function bits are part of register 3 of the VGA GRAPHICS
'   section.  Only the appropriate bits of the register are modified.

        OUT VREGGraphicsIDXRW, &H3
        OUT VREGGraphicsDATRW, (INP(VREGGraphicsDATRW) AND &HE7) OR (8 * (func AND &H3))

END SUB

SUB SetWriteMode (mode AS INTEGER)
'   
'   Set the VGA write mode.
'   
'   The VGA offers four write modes and you'll need to take some care in
'   thinking about which one is appropriate.  Only the lower two bits of
'   'mode' are used, the other bits are ignored.
'
'   MODE 0  This mode listens to the data value specified by POKE.
'           This mode obeys other settings, as well, such as the
'           data rotation and ALU operations that have been specified.
'
'           This mode is usually used for transferring bytes from
'           system memory into VGA memory, one byte at a time.
'
'   MODE 1  This mode ignores the data value specified by POKE, using
'           only the address instead.  This mode reads its data from
'           the VGA's internal data latches, instead and does NOT obey
'           settings designed for data sent by POKE, such as data
'           rotation and ALU operations.
'
'           This mode is usually used for transferring bytes from
'           one part of VGA memory to another part of VGA memory,
'           four bytes at a time.
'
'   MODE 2  This mode is really designed for 4-bit color modes.  It is
'           similar to MODE 0, except that only 4 bits of the data
'           value are used and the bits are split up and sent to their
'           respective planes.  Read up on the documentation for this
'           mode before using it.
'
'           This mode can be used to modify the same bit or group of
'           bits in a span of VGA memory.  The four bits in each data
'           value are separated and directed to their respective memory
'           planes and used to modify the specified bit or group of bits.
'
'   MODE 3  This mode uses the data value specified by POKE as one
'           part of a bit mask that controls which bits in the memory
'           location addressed are to be modified.  It's an unusual
'           mode, like MODE 2 in some ways, and should only be used
'           in conjunction with documentation at hand.  You can modify
'           individual bits of each byte with this mode, though.
'
'           This mode can be used to modify bits of a span of VGA memory,
'           where the bits to be changed vary from location to location.
'
'   The VGA write mode bits are part of register 5 of the VGA GRAPHICS
'   section.  Only the appropriate bits of the register are modified.

        OUT VREGGraphicsIDXRW, &H5
        OUT VREGGraphicsDATRW, (INP(VREGGraphicsDATRW) AND &HFC) OR (mode AND &H3)

END SUB

FUNCTION WaitKey%

    DIM ky AS INTEGER

        DO
            LET ky = ReadKey
        LOOP WHILE ky = keyNULL
        LET WaitKey = ky

END FUNCTION

SUB WritePixelPA (plane AS INTEGER, addr AS INTEGER, c AS INTEGER)
'   
'   Writes the given pixel color, given the plane and offset.
'   
'   The given 8-bit color value is written at the given VGA memory plane and
'   offset.  Since this routine may be called frequently, the DEF SEG needed
'   to specify the VGA memory area is not performed here.
'
'   It may be better, in many cases, to replicate this code rather than
'   calling it as a function.  Much faster, that way.

        OUT VREGSequencerIDXRW, &H2
        OUT VREGSequencerDATRW, vgaPlaneSelect(plane AND 3)
        POKE addr, c

END SUB

