'
'   File:   eq_07.bas
'   Creation Date:  Thu 15-Jul-2004 15:36:22        Jonathan D. Kirwan
'   Last Modified:  Fri 16-Jul-2004 13:32:34        Initial version.
'
'   Copyright (C) 2004 Jonathan Dale Kirwan, All Rights Reserved
'
'
'   DESCRIPTION
'
'   This is a program demonstrating parsing of mathematical expressions.
'   This version handles algebraic equations of the form:
'
'       unary := SIN | COS | TAN | SEC | CSC | CTN | ATAN | ASIN | ACOS |
'                   ASEC | ACSC | ACTN | ABS | SGN | INT | SQRT | LOG | EXP
'       moreitems := ^ item moreitems | <null>
'       item := number | ( expression ) | unary ( expression )
'       morefactors := * factor morefactors | / factor morefactors | <null>
'       factor := item moreitems
'       moreterms := + term moreterms | - term moreterms | <null>
'       term := factor morefactors
'       expression := term moreterms
'
'   Where a number is:
'
'       digit := 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9
'       digits := digit digits | <null>
'       mantissa := . digit digits | digit . digits | digit digits
'       scaleid := e | E
'       scale := scaleid sign digit digits | <null>
'       sign := + | - | <null>
'       number := sign mantissa scale
'
'   And where a variable name or function name is:
'
'       alpha := A | B | C | D | E | F | G | H | I | J | K | L | M | N |
'               O | P | Q | R | S | T | U | V | W | X | Y | Z | a | b |
'               c | d | e | f | g | h | i | j | k | l | m | n | o | p |
'               q | r | s | t | u | v | w | x | y | z
'       alphanumeric := alpha | digit
'       alphanumerics := alphanumeric alphanumerics | <null>
'       name := alpha alphanumerics
'
'   It also calculates the resulting value and displays it, as well as
'   handling some standard unary functions.
'
'   See,
'
'       http://users.easystreet.com/jkirwan/new/parsing.html
'
'   for more detailed information on the design.
'
'
'   MODIFICATIONS
'
'   No modifications.
'
'
'   COPYRIGHT NOTICE
'
'   Jonathan Dale Kirwan grants you a non-transferable, non-exclusive,
'   royalty-free worldwide license to use, copy, modify, prepare deriva-
'   tive works of and distribute this software, subject to your agreement
'   that you acquire no ownership right, title, or interest in this soft-
'   ware and your agreement that this software is research work which is
'   provided 'as is', where Jonathan Dale Kirwan disclaims all warranties
'   with regard to this software, including all implied warranties of
'   merchantability and fitness of purpose.  In no event shall Jonathan
'   Dale Kirwan be liable for any direct, indirect, consequential or
'   special damages or any damages whatsoever resulting from loss of use,
'   data or profits, whether in an action of contract, negligence or
'   other tortious action, arising out of or in connection with the use
'   or performance of this software.

DECLARE FUNCTION Number% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION IsOverflow% (stk AS STRING)
DECLARE FUNCTION Pop# (stk AS STRING)
DECLARE FUNCTION Expression% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

DECLARE FUNCTION Item% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Factor% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Term% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

DECLARE FUNCTION Mantissa% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION ScaleID% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Digit% (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Digits (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Scale (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Sign (eqpos AS INTEGER, eq AS STRING)

DECLARE FUNCTION Alpha% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION AlphaNumeric (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB AlphaNumerics (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Unary% (eqpos AS INTEGER, eq AS STRING, token AS STRING)
DECLARE SUB UnaryFunction (token AS STRING, stk AS STRING)

DECLARE SUB Push (value AS DOUBLE, stk AS STRING)

DECLARE SUB Add (stk AS STRING)
DECLARE SUB Subtract (stk AS STRING)
DECLARE SUB Multiply (stk AS STRING)
DECLARE SUB Divide (stk AS STRING)
DECLARE SUB Power (stk AS STRING)

DECLARE SUB SkipSpaces (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Match% (charlist AS STRING, eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Overflow# ()

    DIM eq AS STRING, eqpos AS INTEGER, status AS INTEGER, stk AS STRING

        CLS
        INPUT "Enter the equation: ", eq
        DO WHILE RTRIM$(eq) <> ""
            LET eqpos = 1
            LET status = Expression(eqpos, eq, stk)
            IF LEN(LTRIM$(eq)) + eqpos = LEN(eq) + 1 THEN
                PRINT "  The entire equation fails!"
            ELSEIF status AND eqpos > LEN(eq) THEN
                PRINT "  The entire equation succeeds!  Value is ";
                IF LEN(stk) = 0 THEN
                    PRINT "<null>"
                ELSEIF IsOverflow(stk) THEN
                    PRINT "Overflow!"
                ELSE
                    PRINT Pop(stk)
                END IF
            ELSE
                PRINT "  "; eq
                PRINT TAB(eqpos + 2); "^-- error from this point."
                PRINT "  ";
                IF LEN(stk) = 0 THEN
                    PRINT "<null>"
                ELSEIF IsOverflow(stk) THEN
                    PRINT "Overflow!"
                ELSE
                    PRINT Pop(stk)
                END IF
            END IF
            PRINT
            INPUT "Enter the equation: ", eq
        LOOP
        END

ErrorHandler:

    DIM SHARED ErrNumber AS INTEGER

        LET ErrNumber = ERR
        RESUME NEXT

SUB Add (stk AS STRING)

    DIM sum AS DOUBLE, addend1 AS DOUBLE, addend2 AS DOUBLE

        LET addend2 = Pop(stk)
        LET addend1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET sum = addend1 + addend2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push sum, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Alpha% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", eqpos, eq)

    LET Alpha = status

END FUNCTION

FUNCTION AlphaNumeric (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", eqpos, eq)

    LET AlphaNumeric = status

END FUNCTION

SUB AlphaNumerics (eqpos AS INTEGER, eq AS STRING)

        DO WHILE AlphaNumeric(eqpos, eq)
        LOOP

END SUB

FUNCTION Digit% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789", eqpos, eq)

    LET Digit = status

END FUNCTION

SUB Digits (eqpos AS INTEGER, eq AS STRING)

        DO WHILE Digit(eqpos, eq)
        LOOP

END SUB

SUB Divide (stk AS STRING)

    DIM ratio AS DOUBLE, dividend AS DOUBLE, divisor AS DOUBLE

        LET divisor = Pop(stk)
        LET dividend = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET ratio = dividend / divisor
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push ratio, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Expression% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Term(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("+", eqpos, eq) THEN
                LET status = Term(eqpos, eq, stk)
                IF status THEN
                    Add stk
                END IF
            ELSEIF Match("-", eqpos, eq) THEN
                LET status = Term(eqpos, eq, stk)
                IF status THEN
                    Subtract stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Expression = status

END FUNCTION

FUNCTION Factor% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Item(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("^", eqpos, eq) THEN
                LET status = Item(eqpos, eq, stk)
                IF status THEN
                    Power stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Factor = status

END FUNCTION

FUNCTION IsOverflow% (stk AS STRING)

    DIM status AS INTEGER

        LET status = (MID$(stk, LEN(stk) - 7) = MKD$(Overflow))

    LET IsOverflow = status

END FUNCTION

FUNCTION Item% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER, value AS DOUBLE
    DIM token AS STRING

        SkipSpaces eqpos, eq
        LET savepos = eqpos
        IF Match("(", eqpos, eq) THEN
            LET status = Expression(eqpos, eq, stk)
            IF status THEN
                SkipSpaces eqpos, eq
                LET status = Match(")", eqpos, eq)
            END IF
        ELSEIF Unary(eqpos, eq, token) THEN
            LET status = Expression(eqpos, eq, stk)
            IF status THEN
                SkipSpaces eqpos, eq
                LET status = Match(")", eqpos, eq)
                IF status THEN
                    UnaryFunction token, stk
                END IF
            END IF
        ELSE
            LET status = Number(eqpos, eq)
            IF status THEN
                LET ErrNumber = 0
                ON ERROR GOTO ErrorHandler
                LET value = VAL(MID$(eq, savepos, eqpos - savepos))
                ON ERROR GOTO 0
                IF ErrNumber = 0 THEN
                    Push value, stk
                ELSE
                    Push Overflow, stk
                END IF
            END IF
        END IF

    LET Item = status

END FUNCTION

FUNCTION Mantissa% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        IF Match(".", eqpos, eq) THEN
            LET status = Digit(eqpos, eq)
            IF status THEN
                Digits eqpos, eq
            END IF
        ELSEIF Digit(eqpos, eq) THEN
            Digits eqpos, eq
            LET status = -1 OR Match(".", eqpos, eq)
            Digits eqpos, eq
        ELSE
            LET status = 0
        END IF

    LET Mantissa = status

END FUNCTION

FUNCTION Match% (charlist AS STRING, eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        IF eqpos <= LEN(eq) THEN
            IF INSTR(charlist, MID$(eq, eqpos, 1)) <> 0 THEN
                LET eqpos = eqpos + 1
                LET status = -1
            END IF
        ELSE
            LET status = 0
        END IF

    LET Match = status

END FUNCTION

SUB Multiply (stk AS STRING)

    DIM product AS DOUBLE, multiplicand1 AS DOUBLE, multiplicand2 AS DOUBLE

        LET multiplicand2 = Pop(stk)
        LET multiplicand1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET product = multiplicand1 * multiplicand2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push product, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Number% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER, dummy AS INTEGER

        Sign eqpos, eq
        LET status = Mantissa(eqpos, eq)
        IF status THEN
            Scale eqpos, eq
        END IF

    LET Number = status

END FUNCTION

FUNCTION Overflow#

    DIM value AS DOUBLE

        LET value = CVD(STRING$(8, 255))

    LET Overflow = value

END FUNCTION

FUNCTION Pop# (stk AS STRING)

    DIM slen AS INTEGER, value AS DOUBLE

        LET slen = LEN(stk)
        LET value = CVD(MID$(stk, slen - 7))
        LET stk = LEFT$(stk, slen - 8)

    LET Pop = value

END FUNCTION

SUB Power (stk AS STRING)

    DIM result AS DOUBLE, pow1 AS DOUBLE, pow2 AS DOUBLE

        LET pow2 = Pop(stk)
        LET pow1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET result = pow1 ^ pow2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push result, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

SUB Push (value AS DOUBLE, stk AS STRING)

        LET stk = stk + MKD$(value)

END SUB

SUB Scale (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER

        LET savepos = eqpos
        IF ScaleID(eqpos, eq) THEN
            Sign eqpos, eq
            LET status = Digit(eqpos, eq)
            IF status THEN
                Digits eqpos, eq
            ELSE
                LET eqpos = savepos
            END IF
        END IF

END SUB

FUNCTION ScaleID% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("eE", eqpos, eq)

    LET ScaleID = status

END FUNCTION

SUB Sign (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("+-", eqpos, eq)

END SUB

SUB SkipSpaces (eqpos AS INTEGER, eq AS STRING)

        DO WHILE eqpos <= LEN(eq)
            IF MID$(eq, eqpos, 1) <> " " THEN
                EXIT DO
            END IF
            LET eqpos = eqpos + 1
        LOOP

END SUB

SUB Subtract (stk AS STRING)

    DIM difference AS DOUBLE, minuend AS DOUBLE, subtrahend AS DOUBLE

        LET subtrahend = Pop(stk)
        LET minuend = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET difference = minuend - subtrahend
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push difference, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Term% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Factor(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("*", eqpos, eq) THEN
                LET status = Factor(eqpos, eq, stk)
                IF status THEN
                    Multiply stk
                END IF
            ELSEIF Match("/", eqpos, eq) THEN
                LET status = Factor(eqpos, eq, stk)
                IF status THEN
                    Divide stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Term = status

END FUNCTION

FUNCTION Unary% (eqpos AS INTEGER, eq AS STRING, token AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER

        LET savepos = eqpos
        IF Alpha(eqpos, eq) THEN
            AlphaNumerics eqpos, eq
            LET token = MID$(eq, savepos, eqpos - savepos)
            SkipSpaces eqpos, eq
            LET status = Match("(", eqpos, eq)
            IF NOT status THEN
                LET eqpos = savepos
            END IF
        ELSE
            LET status = 0
        END IF

    LET Unary = status

END FUNCTION

SUB UnaryFunction (token AS STRING, stk AS STRING)

    DIM value AS DOUBLE, result AS DOUBLE, status AS INTEGER

        LET value = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        SELECT CASE UCASE$(token)
        CASE "SIN"
            LET result = SIN(value)
        CASE "COS"
            LET result = COS(value)
        CASE "TAN"
            LET result = TAN(value)
        CASE "SEC"
            LET result = 1# / SIN(value)
        CASE "CSC"
            LET result = 1# / COS(value)
        CASE "CTN"
            LET result = 1# / TAN(value)
        CASE "ATAN"
            LET result = ATN(value)
        CASE "ASIN"
            LET result = ATN(value / SQR(1# - value * value))
        CASE "ACOS"
            LET result = ATN(SQR(1# / (value * value) - 1#))
        CASE "ASEC"
            LET result = ATN(1# / SQR(value * value - 1#))
        CASE "ACSC"
            LET result = ATN(SQR(value * value - 1#))
        CASE "ACTN"
            LET result = ATN(1# / value)
        CASE "ABS"
            LET result = ABS(value)
        CASE "SGN"
            LET result = SGN(value)
        CASE "INT"
            LET result = INT(value)
        CASE "SQRT", "SQR"
            LET result = SQR(value)
        CASE "LOG", "LN"
            LET result = LOG(value)
        CASE "EXP"
            LET result = EXP(value)
        CASE "SINH"
            LET result = (EXP(value) - EXP(-value)) / 2#
        CASE "COSH"
            LET result = (EXP(value) + EXP(-value)) / 2#
        CASE "TANH"
            LET result = (EXP(value) - EXP(-value)) / (EXP(value) + EXP(-value))
        CASE "SECH"
            LET result = 2# / (EXP(value) - EXP(-value))
        CASE "CSCH"
            LET result = 2# / (EXP(value) + EXP(-value))
        CASE "COTH"
            LET result = (EXP(value) + EXP(-value)) / (EXP(value) - EXP(-value))
        CASE "RND"
            LET result = RND(value)
        CASE ELSE
            LET result = value
        END SELECT
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push result, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB
