/*
    File:   except.h
    Author: Jonathan Dale Kirwan
 
    Creation Date:  Fri 08-Apr-2005 01:31:37
    Last Modified:  Mon 16-Jan-2006 00:08:54

    Copyright 2005, 2006, Jonathan Dale Kirwan, All Rights Reserved.

    URL:    http://users.easystreet.com/jkirwan/new/c_expr.html
    email:  jkirwan@easystreet.com


    DESCRIPTION

    The exception handler provided here includes a few key macros that may
    be used to support exception handling in general programs.  The design
    of this system supports easy porting for multi-threaded programs.

    To declare an exception item, simply:

        except_t myexcept= { "My error message." };

    This declares an exception which may be raised or handled by an
    exception handler.  The except_t initializer may also be a pointer to
    another string -- for example, one prepared using sprintf().

    To raise an exception,

        RAISE( e );

    This raises exception 'e', which must be an except_t type.  If there is
    no exception handler currently in force, a default message is written
    and the program is terminated.  Otherwise, the appropriate handler code
    is invoked.

    An exception try-block might look like:

        TRY
            try-block of code
        EXCEPT( e1 )
            code for the 'e1' type of exception
        EXCEPT( e2 )
            code for the 'e2' type of exception
        ELSE
            code for all other exceptions
        ENDTRY;

    Or,

        TRY
            try-block of code
        EXCEPT( e1 )
            code for the 'e1' type of exception
        EXCEPT( e2 )
            code for the 'e2' type of exception
        ELSE
            RERAISE;
        ENDTRY;

    If you don't want to handle exceptions other than the specific ones
    listed and want to propogate the remaining exceptions up to the next
    outer level handler.

    An ENDTRY without an ELSE to handle unspecified exceptions is the same
    as using the ELSE RERAISE, as shown above.  Any unhandled exceptions are
    then re-thrown.

    The macro FINALLY is used like this:

        TRY
            try-block of code
        EXCEPT( e1 )
            code for the 'e1' type of exception
        EXCEPT( e2 )
            code for the 'e2' type of exception
        FINALLY
            clean-up code
        ENDTRY;

    and is exactly equivalent to:

        TRY
            try-block of code
        EXCEPT( e1 )
            code for the 'e1' type of exception
        EXCEPT( e2 )
            code for the 'e2' type of exception
        ELSE
            clean-up code
            RERAISE;
        ENDTRY;
        clean-up code

    Note that the clean-up code (could be something else, of course) is
    used if there are no exceptions and the try-block succeeds or if any
    unhandled exceptions take place.

    Sometimes, it is desirable to perform a function return within the
    try-block handling section.  If so, use the RETURN macro,

        TRY
            try-block of code
        EXCEPT( e1 )
            some code for the 'e1' type of exception
            RETURN 0;
        ELSE
            RERAISE;
        ENDTRY;

    Using RETURN simply guarantees that the handling structures are cleaned
    up properly before returning from the function.  Outside of the try-block
    you should use the normal c return statement.

    If, for some reason in an EXCEPT() or ELSE clause, you want to access
    the error message string, it can be accessed as shown here:

        printf( "error: %s\n", REASON );

    To allow special message handling of unhandled messages, UNHANLED(h) is
    provided and can specify a subroutine that will be passed the default,
    pre-formatted message for such exceptions.  In this way, it's possible
    to display any desired message other than (or in addition to) the message
    provided below.


    CREDITS

    The approach taken here was heavily drawn from David Hanson's book, "C
    Interfaces and Implementations."  In particular, Chapter 4.  I've used
    a different version of these routines in a multi-threaded application,
    so some of what I learned from that has been folded back into this code.


    TARGET COMPILER

    Microsoft 16-bit Visual C/C++, IDE version 1.52C (DOS version 8.00c)
 
 
    MODIFICATIONS
 
    Original source.
*/

#ifndef EXCEPT_H
#define EXCEPT_H

#include <setjmp.h>


/*  ----------------------------------------------------------------------  */
#define EXCEPT_DETAILS  0   /* 0 = no details, 1 = include file and line */
/*  ----------------------------------------------------------------------
    This macro controls some of the detail that is kept for an exception
    event.  Currently, two values are supported:
        0   no unnecessary details are kept
        1   file name and line number are included
*/


/*  ----------------------------------------------------------------------  */
#define T except_t
#define U() exceptframe_t
/*  ----------------------------------------------------------------------
    I'm defining these two names to make it easier to change the symbols,
    in case there is a conflict when trying to include these files with
    existing programs.  The busier looking U() form is used here in order
    to force immediate expansion in the TRY #define below.
*/


/*  ----------------------------------------------------------------------  */
typedef struct T {
/*  ----------------------------------------------------------------------
    This structure just holds exception-specific information.  Each unique
    exception, for now, is just an ASCII message.  This could be changed
    or expanded so as to include an error code or number, etc.  It's up to
    you.
*/
    char *reason;       /* user-defined exception message */
} T;


/*  ----------------------------------------------------------------------  */
typedef struct U() {
/*  ----------------------------------------------------------------------
    This structure must be able to link to other structures of the same
    type (linked into a stack), must include a jump buffer for longjump,
    and must point to an exception structure.  The rest is optional and
    is, for now, set to provide the file name and line number of the
    triggering exception event.
*/
    struct U() *prev;       /* links to the next outer exception trap */
    jmp_buf env;            /* provided for setjump/longjump */
    const T *exception;     /* current exception event */
#if ( EXCEPT_DETAILS == 1 )
    const char *file;       /* file name of current exception event */
    int line;               /* line number of current exception event */
#endif
} U();


/*  ----------------------------------------------------------------------  */
enum {
/*  ----------------------------------------------------------------------
    Simple list of the current state of an exception, during handling in
    TRY..ENDTRY blocks.
*/
    EXCEPTENTERED= 0,
    EXCEPTRAISED,
    EXCEPTHANDLED,
    EXCEPTFINALIZED
};


/*  ----------------------------------------------------------------------  */
extern void psetexceptstack( U() *stack );
extern U() *pexceptstack( void );
extern void psetunhandled( void (*handler)( const char *msg ) );
extern void (* punhandled( void ))( const char *msg );
/*  ----------------------------------------------------------------------
    The above functions shouldn't be directly referenced in user code.
    But they are necessary support for TRY..ENDTRY blocks.  All of these
    are currently defined in except.c, but they may be moved to the thread
    handler in multi-threaded systems.
*/


/*  ----------------------------------------------------------------------  */
#if ( EXCEPT_DETAILS == 1 )
#define RAISE(e)    exceptraise( &(e), __FILE__, __LINE__ )
extern void exceptraise( const T *e, const char *file, int line );
#else
#define RAISE(e)    exceptraise( &(e) )
extern void exceptraise( const T *e );
#endif
#define RERAISE     exceptraise( exceptframe.exception, exceptframe.file, exceptframe.line )
#define RETURN      switch( (psetexceptstack( pexceptstack( )->prev ), 0 ) default: return
#define TRY         do { volatile int exceptflag; U() exceptframe; \
                        exceptframe.prev= pexceptstack( ); psetexceptstack( & exceptframe ); \
                        exceptflag= setjmp( exceptframe.env ); \
                        if ( exceptflag == EXCEPTENTERED ) {
#define EXCEPT(e)   if ( exceptflag == EXCEPTENTERED ) psetexceptstack( pexceptstack( )->prev ); \
                        } else if (exceptframe.exception == &(e) ) { exceptflag= EXCEPTHANDLED;
#define ELSE        if ( exceptflag == EXCEPTENTERED ) psetexceptstack( pexceptstack( )->prev ); \
                        } else { exceptflag= EXCEPTHANDLED;
#define FINALLY     if ( exceptflag == EXCEPTENTERED ) psetexceptstack( pexceptstack( )->prev ); \
                        } { if ( exceptflag == EXCEPTENTERED ) exceptflag= EXCEPTFINALIZED;
#define ENDTRY      if ( exceptflag == EXCEPTENTERED ) psetexceptstack( pexceptstack( )->prev ); \
                        } if ( exceptflag == EXCEPTRAISED ) RERAISE; } while (0)
#define REASON      (exceptframe.exception->reason)
#define UNHANDLED(h) psetunhandled( (h) )
/*  ----------------------------------------------------------------------
    Well, that's the meat of the matter for the TRY..ENDTRY blocks and
    invoking exceptions.
*/


/*  ----------------------------------------------------------------------  */
#undef T
#undef U
/*  ----------------------------------------------------------------------
    Removing them, now.
*/

#endif
