/*
    File:   gaussrnd.c
    Author: Jonathan Dale Kirwan

    Creation Date:  Fri 30-Dec-2005 14:34:49
    Last Modified:  Fri 30-Dec-2005 15:21:31

    Copyright 2005, 2006, Jonathan Dale Kirwan, All Rights Reserved.


    DESCRIPTION

    This module provides a Gaussian distributed random number, with a mean of
    zero and a standard deviation of 1.  It requires a uniform random number
    generator, providing uniform deviates from 0.0 to 1.0 (not including 1.0.)


    TARGET COMPILER

    Microsoft 16-bit Visual C/C++, IDE version 1.52C (DOS version 8.00c)
 
 
    MODIFICATIONS
 
    Original source.
*/

#include <math.h>
#include "rnd.h"

static char f= 0;


/*  ----------------------------------------------------------------------  */
double rndgauss( void ) {
/*  ----------------------------------------------------------------------
    This routine uses a technique documented well in any of the several
    editions of Numerical Recipes, under the heading of Normal Deviates.
    The method is a modified form of Box-Muller, using the unit circle
    instead of a unit square to avoid trigonometric functions.

    This routine alternates between two different states because the method
    develops two normal deviates simultaneously.  One state in which there
    is no currently saved normal deviate in v1; so that two new deviates
    should be produced, saving one for the next time and returning the other.
    Another state in which there is a saved value that should be returned and
    simply then marked 'empty'.
*/
    static double v1;
    auto double fac, rsq, v2;

        /*  Compute two normal deviates and return the first one. */
        if ( f == 0 ) {
            f= 1;
            do {
                v1= 2.0 * rnd( ) - 1.0;
                v2= 2.0 * rnd( ) - 1.0;
                rsq= v1 * v1 + v2 * v2;
            } while ( rsq >= 1.0 || rsq <= 0.0 );
            fac= sqrt( -2.0 * log( rsq ) / rsq );
            v1= v1 * fac;
            return v2 * fac;
        }

        /*  Or else return the second one. */
        f= 0;

    return v1;
}


/*  ----------------------------------------------------------------------  */
void rndgauss_init( void ) {
/*  ----------------------------------------------------------------------
    This routine initializes the Gaussian random number generator -- in
    other words, flushes any prior held value in v1.
*/

        f= 0;

    return;
}
