'   FLOATING POINT DEMONSTRATION PROGRAM
'
'   File:   float.bas
'   Author: Jonathan Dale Kirwan
'
'   Creation Date:  Sun 26-Oct-1997 17:20:40
'   Last Modified:  Wed 06-Jul-2005 02:09:10
'        Modified:  Thu 22-Apr-1999 15:45:19
'
'   Copyright (C) 1997, 1998, 1999, 2004 Jonathan Dale Kirwan
'   All Rights Reserved:  See COPYRIGHT NOTICE below.
'
'
'   DESCRIPTION
'
'   This program allows you to experiment with floating point formats.
'   It supports two different format types.  IEEE and Microsoft's early
'   binary format, in both 32-bit and 64-bit versions.  To keep it simple,
'   these formats are not available for exploration at the same time.
'   Instead, you run the QuickBASIC or QBASIC environment with the /MBF
'   option, if you want the Microsoft Binary Format and without it for
'   the IEEE format.  The program will detect which mode was used and
'   adjust itself, accordingly.
'
'   You may use algebriac expressions to compute a value, as well.  Just
'   enter the expression as you would in BASIC.  Only +, -, *, /, and
'   parentheses are supported, though.
'
'
'   DESIGN
'
'   The BASIC functions CVL(), CVS(), CVD(), MKL$(), MKS$(), and MKD$()
'   are used to figure out the internal format information.  The newer
'   QuickBASIC programs do not actually store floating point values in
'   the /MBF format, even when the command option is specified.  Instead,
'   variables are stored in the IEEE format.  The only difference caused
'   by using /MBF is to change the way MKS$(), MKD$(), CVL(), and CVD()
'   do their conversion work.  This can lead to problems, since the IEEE
'   DOUBLE values have 11-bit exponents while the format supported by
'   MKD$() [when the /MBR option is used] only supports 8 bit exponents.
'
'
'   TARGET COMPILER
'
'   This module is designed to be used with Microsoft's QBASIC v1.1 and
'   the QuickBASIC 4.5 compiler and later compilers.  They both support
'   the /MBF switch option on the command line to allow the user to
'   switch operation between the two different floating point formats.
'
'
'   MODIFICATIONS
'
'   08/20/98    jk  Added support for DOUBLE precision floating point
'   values and changed the organization of the code.
'
'   20-Apr-1999 jk  Added support for entering the values as expressions.
'
'   22-Apr-1999 jk  Modified to use the EVALUATE module.
'
'   12-Aug-2004 jk  Removed old expression code and replaced it with the
'   newer code I've documented on the web site.
'
'
'   COPYRIGHT NOTICE
'
'   Jonathan Dale Kirwan grants you a non-transferable, non-exclusive,
'   royalty-free worldwide license to use, copy, modify, prepare deriva-
'   tive works of and distribute this software, subject to your agreement
'   that you acquire no ownership right, title, or interest in this soft-
'   ware and your agreement that this software is research work which is
'   provided 'as is', where Jonathan Dale Kirwan disclaims all warranties
'   with regard to this software, including all implied warranties of
'   merchantability and fitness of purpose.  In no event shall Jonathan
'   Dale Kirwan be liable for any direct, indirect, consequential or
'   special damages or any damages whatsoever resulting from loss of use,
'   data or profits, whether in an action of contract, negligence or
'   other tortious action, arising out of or in connection with the use
'   or performance of this software.

'$INCLUDE: 'VARIABLE.BI'
'$INCLUDE: 'PARSE.BI'

CONST PROGTITLE$ = "FLOATING POINT FORMATS"
CONST PROGVERSION% = 5

CONST MAXVARS% = 1000
CONST MAXHEAD% = 210

CONST colorNORMAL% = 7
CONST colorFIGURES% = 12
CONST colorTITLE% = 13
CONST colorDATA% = 14
CONST colorHITANYKEY% = 14

CONST TRUE% = -1
CONST FALSE% = 0

TYPE FPEXPOSED
    Sign AS INTEGER             ' (0) is positive, (-1) is negative
    Exponent AS INTEGER         ' all exponent bits
    Mantissa1 AS LONG           ' most significant mantissa bits
    Mantissa2 AS LONG           ' least significant mantissa bits
END TYPE

DECLARE SUB DisplayTitle (IEEEMode AS INTEGER, TitleName AS STRING, Version AS INTEGER)
DECLARE SUB DisplayEraseLine ()
DECLARE FUNCTION GetUserInput% (Prompt AS STRING, FP32Mode AS INTEGER, FP32Value AS SINGLE, FP64Value AS DOUBLE)
DECLARE FUNCTION GetInput$ (Prompt AS STRING)
DECLARE SUB ParseFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB ParseFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB ParseFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB ParseFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP32Common (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayMBFCommon (Parts AS FPEXPOSED)
DECLARE SUB DisplayBits (Pattern AS LONG, BitCount AS INTEGER)
DECLARE FUNCTION HexValue& (Value AS STRING)

    DIM SHARED ScreenRows AS INTEGER

        LET ScreenRows = 50
        ON ERROR GOTO RowCheck
        LOCATE ScreenRows, 1
        ON ERROR GOTO 0

    DIM SHARED IEEEMode AS INTEGER

        LET IEEEMode = (CVL(MKS$(1!)) = &H3F800000)

    DIM SHARED RedrawScreen AS INTEGER

        LET RedrawScreen = -1

    DIM Parts AS FPEXPOSED
    DIM FP32Mode AS INTEGER, FP32Value AS SINGLE, FP64Value AS DOUBLE

        InitValues
        SetValue "PI", 4# * ATN(1#)
        SetValue "E", EXP(1#)

        LET FP32Mode = TRUE
        LET FP32Value = 0!
        DO
            IF RedrawScreen THEN
                DisplayTitle IEEEMode, PROGTITLE, PROGVERSION
                PRINT "  This program displays the binary storage format for single and double pre-"
                PRINT "  cision floating point values.  You can enter floating point values and see"
                PRINT "  the resulting format or you can enter a hexadecimal value and see the type"
                PRINT "  of floating point number that results from it.  Use ! to switch to single"
                PRINT "  precision, or a # to switch to double precision, and prefix with &H to use"
                PRINT "  a hexadecimal float value.  Just ENTER exits the program.  Or ? for help."
                LET RedrawScreen = 0
            END IF
            LOCATE 11
            IF FP32Mode AND IEEEMode THEN
                ParseFP32IEEE FP32Value, Parts
                DisplayFP32IEEE FP32Value, Parts
            ELSEIF FP32Mode THEN
                ParseFP32MBF FP32Value, Parts
                DisplayFP32MBF FP32Value, Parts
            ELSEIF IEEEMode THEN
                ParseFP64IEEE FP64Value, Parts
                DisplayFP64IEEE FP64Value, Parts
            ELSE
                LET FP64Value = CVD(MKD$(FP64Value))
                ParseFP64MBF FP64Value, Parts
                DisplayFP64MBF FP64Value, Parts
            END IF
            LOCATE 23
        LOOP WHILE GetUserInput("  Enter the value: ", FP32Mode, FP32Value, FP64Value)

        END

RowCheck:
        IF ScreenRows = 50 THEN
            LET ScreenRows = 43
            RESUME
        END IF
        LET ScreenRows = 25
        RESUME NEXT

SUB DisplayBits (Pattern AS LONG, BitCount AS INTEGER)

    DIM i AS INTEGER
    DIM mask AS LONG
    DIM bc AS INTEGER

        LET bc = BitCount
        IF bc = 32 THEN
            LET bc = 31
            PRINT USING "#"; -((Pattern AND &H80000000) <> 0);
            LOCATE , POS(0) + 1
        END IF

        LET mask = 2 ^ (bc - 1)
        FOR i = 1 TO bc
            PRINT USING "#"; -((Pattern AND mask) <> 0);
            LOCATE , POS(0) + 1
            LET mask = mask \ 2
        NEXT i

END SUB

SUB DisplayEraseLine

        LOCATE , 1
        PRINT SPACE$(80);
        LOCATE , 1

END SUB

SUB DisplayFP32Common (FP32Value AS SINGLE, Parts AS FPEXPOSED)

        LOCATE CSRLIN + 4, 1
        DisplayEraseLine
        PRINT
        DisplayEraseLine
        PRINT
        DisplayEraseLine

        LOCATE CSRLIN + 2, 1
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Hex: ";
        COLOR colorDATA
        PRINT RIGHT$("0000000" + HEX$(CVL(MKS$(FP32Value))), 8)
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Val: ";
        COLOR colorDATA
        PRINT FP32Value

END SUB

SUB DisplayFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)

        LOCATE , 1
        COLOR colorFIGURES
        PRINT "   Sgn       Exponent        (hidden bit)                        Mantissa"
        PRINT "   Ŀ   Ŀ   Ŀ"
        PRINT "                                         "
        PRINT "         "
        COLOR colorDATA
        LOCATE CSRLIN - 2, 5
        DisplayBits CLNG(Parts.Sign), 1
        LOCATE , 11
        DisplayBits CLNG(Parts.Exponent), 8
        LOCATE , 31
        DisplayBits Parts.Mantissa1, 23

        LOCATE CSRLIN - 2, 29
        IF Parts.Exponent = 0 THEN
            DisplayBits 0, 1
        ELSE
            DisplayBits 1, 1
        END IF

        DisplayFP32Common FP32Value, Parts

        COLOR colorNORMAL

END SUB

SUB DisplayFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)

        DisplayMBFCommon Parts
        DisplayFP32Common FP32Value, Parts
        COLOR colorNORMAL

END SUB

SUB DisplayFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING

        LOCATE , 1

        COLOR colorFIGURES
        PRINT "   Sgn           Exponent          (hidden bit)            Upper Mantissa"
        PRINT "   Ŀ   Ŀ   Ŀ"
        PRINT "                                         "
        PRINT "         "
        COLOR colorDATA

        LOCATE CSRLIN - 2, 5
        DisplayBits CLNG(Parts.Sign), 1
        LOCATE , 11
        DisplayBits CLNG(Parts.Exponent), 11
        LOCATE , 37
        DisplayBits Parts.Mantissa1 \ &H8, 20

        LOCATE CSRLIN - 2, 35
        IF Parts.Exponent = 0 THEN
            DisplayBits 0, 1
        ELSE
            DisplayBits 1, 1
        END IF

        LOCATE CSRLIN + 4, 1
        COLOR colorFIGURES
        PRINT "           Ŀ"
        PRINT "    Lower                                  "
        PRINT "   Mantissa"
        COLOR colorDATA

        LOCATE CSRLIN - 2, 13
        DisplayBits Parts.Mantissa1 AND &H7, 3
        LOCATE , 19
        DisplayBits Parts.Mantissa2, 29

        LET dbl = MKD$(FP64Value)
        LOCATE CSRLIN + 3, 1
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Hex: ";
        COLOR colorDATA
        PRINT RIGHT$("0000000" + HEX$(CVL(MID$(dbl, 5))), 8); RIGHT$("0000000" + HEX$(CVL(dbl)), 8)
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Val: ";
        COLOR colorDATA
        PRINT FP64Value

        COLOR colorNORMAL

END SUB

SUB DisplayFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING

        DisplayMBFCommon Parts
        COLOR colorFIGURES
        LOCATE , 63: PRINT "Upper";

        LOCATE CSRLIN + 4, 1
        PRINT "           Ŀ"
        PRINT "    Lower                                  "
        PRINT "   Mantissa"
        COLOR colorDATA

        LOCATE CSRLIN - 2, 13
        DisplayBits Parts.Mantissa2, 32

        LET dbl = MKD$(FP64Value)
        LOCATE CSRLIN + 3, 1
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Hex: ";
        COLOR colorDATA
        PRINT RIGHT$("0000000" + HEX$(CVL(MID$(dbl, 5))), 8); RIGHT$("0000000" + HEX$(CVL(dbl)), 8)
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Val: ";
        COLOR colorDATA
        PRINT FP64Value

        COLOR colorNORMAL

END SUB

SUB DisplayMBFCommon (Parts AS FPEXPOSED)

        LOCATE , 1
        COLOR colorFIGURES
        PRINT "       Exponent        Sgn   (hidden bit)                        Mantissa"
        PRINT "   Ŀ   Ŀ   Ŀ"
        PRINT "                                         "
        PRINT "         "
        COLOR colorDATA
        LOCATE CSRLIN - 2, 25
        DisplayBits CLNG(Parts.Sign), 1
        LOCATE , 5
        DisplayBits CLNG(Parts.Exponent), 8
        LOCATE , 31
        DisplayBits Parts.Mantissa1, 23

        LOCATE CSRLIN - 2, 29
        IF Parts.Exponent = 0 AND Parts.Mantissa1 = 0 AND Parts.Mantissa2 = 0 THEN
            DisplayBits 0, 1
        ELSE
            DisplayBits 1, 1
        END IF

END SUB

SUB DisplayTitle (IEEEMode AS INTEGER, TitleName AS STRING, Version AS INTEGER)

        CLS
        COLOR colorTITLE
        PRINT " "; STRING$(78, "")
        IF IEEEMode THEN
            PRINT " IEEE";
        ELSE
            PRINT " MBF ";
        END IF
        PRINT SPACE$(36 - LEN(TitleName) \ 2); TitleName;
        LOCATE , 71 + (Version > 9)
        PRINT "Version"; Version
        PRINT " "; STRING$(78, "")
        COLOR colorNORMAL

END SUB

FUNCTION GetInput$ (Prompt AS STRING)

    DIM Value AS STRING

        DisplayEraseLine
        PRINT Prompt;
        INPUT "", Value

        LET GetInput = LTRIM$(RTRIM$(UCASE$(Value)))

END FUNCTION

FUNCTION GetUserInput% (Prompt AS STRING, FP32Mode AS INTEGER, FP32Value AS SINGLE, FP64Value AS DOUBLE)

    DIM Value AS STRING, v AS DOUBLE, status AS INTEGER
    DIM p AS INTEGER, expstack AS STRING

        LET Value = GetInput(Prompt)
        IF Value = "" THEN
            LET GetUserInput = FALSE
            EXIT FUNCTION
        END IF

        IF Value = "#" THEN
            LET FP32Mode = FALSE
            LET GetUserInput = TRUE
            EXIT FUNCTION
        ELSEIF Value = "!" THEN
            LET FP32Mode = TRUE
            LET GetUserInput = TRUE
            EXIT FUNCTION
        ELSEIF Value = "?" THEN
            IF ScreenRows = 25 THEN
                DisplayTitle IEEEMode, PROGTITLE, PROGVERSION
            ELSE
                PRINT
            END IF
            PRINT
            PRINT "  Built-in Variables:"
            PRINT "    PI, E, ANS"
            PRINT
            PRINT "  Built-in Functions:"
            PRINT "    SIN, COS, TAN, SEC, CSC, CTN,"
            PRINT "    ASIN, ACOS, ATAN, ASEC, ACSC, ACTN,"
            PRINT "    SINH, COSH, TANH, SECH, CSCH, COTH,"
            PRINT "    DEG, RAD, EXP, LOG, LN, LOG10, LOG2,"
            PRINT "    SQRT, SQR, ABS, SGN, INT, RND"
            PRINT
            PRINT "  Examples:"
            PRINT "    a= SIN(PI/3); b= SIN(PI/4); DEG(ASIN(a*SQRT(1-b^2)+b*SQRT(1-a^2)))"
            PRINT "    -(4+13)*8; ANS*2"
            PRINT "    e^2+e^-2"
            PRINT "    4+&H41C80000"
            PRINT "    INT(RND(1)*5)"
            IF ScreenRows = 25 THEN
                LOCATE 22, 1
                COLOR colorHITANYKEY
                PRINT TAB(24); "<hit any key to return to main page>"
                COLOR colorNORMAL
                DO WHILE INKEY$ = ""
                LOOP
                LET RedrawScreen = -1
            END IF
            LET GetUserInput = TRUE
            EXIT FUNCTION
        END IF

        LET p = 1
        LET expstack = ""
        LET status = Statement(p, Value, expstack)
        IF NOT status THEN
            LET v = VAL(Value)
        ELSE
            DO
                IF LEN(expstack) = 0 THEN
                    LET v = VAL(Value)
                ELSEIF IsOverflow(expstack) THEN
                    LET v = CVD(MKL$(0) + MKL$(&H7FF00000))
                ELSEIF p > LEN(Value) THEN
                    LET v = Pop(expstack)
                    SetValue "ANS", v
                ELSE
                    LET v = VAL(Value)
                END IF
                SkipSpaces p, Value
                IF NOT Match(";", p, Value) THEN
                    EXIT DO
                END IF
                LET status = Statement(p, Value, expstack)
            LOOP WHILE status
        END IF
        IF FP32Mode THEN
            LET FP32Value = CSNG(v)
        ELSE
            LET FP64Value = v
        END IF

        LET GetUserInput = TRUE

END FUNCTION

FUNCTION HexValue& (Value AS STRING)

    DIM i AS INTEGER
    DIM eval AS STRING

        LET eval = Value
        FOR i = LEN(eval) TO 1 STEP -1
            IF INSTR("0123456789ABCDEF", MID$(eval, i, 1)) = 0 THEN
                LET eval = LEFT$(eval, i - 1) + MID$(eval, i + 1)
            END IF
        NEXT i
        DO
            FOR i = 1 TO LEN(eval)
                IF MID$(eval, i, 1) <> "0" THEN
                    EXIT FOR
                END IF
            NEXT i
            LET eval = MID$(eval, i)
            IF LEN(eval) > 8 THEN
                LET eval = RIGHT$(eval, 8)
            END IF
        LOOP WHILE LEFT$(eval, 1) = "0"

        IF LEN(eval) = 8 AND INSTR("89ABCDEF", LEFT$(eval, 1)) <> 0 THEN
            LET HexValue = CLNG(VAL("&H" + eval))
        ELSE
            LET HexValue = CLNG(VAL("&H" + eval + "&"))
        END IF

END FUNCTION

SUB ParseFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)

    DIM Pattern AS LONG

        LET Pattern = CVL(MKS$(FP32Value))
        LET Parts.Sign = (Pattern AND &H80000000) = &H80000000
        LET Parts.Exponent = (Pattern AND &H7F800000) \ &H800000
        IF (Parts.Exponent AND &H80&) = &H80& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFFF00
        END IF
        LET Parts.Mantissa1 = Pattern AND &H7FFFFF
        LET Parts.Mantissa2 = 0&

END SUB

SUB ParseFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)

    DIM Pattern AS LONG

        LET Pattern = CVL(MKS$(FP32Value))
        LET Parts.Sign = (Pattern AND &H800000) = &H800000
        LET Parts.Exponent = ((Pattern AND &HFF000000) \ &H1000000) AND &HFF&
        IF (Parts.Exponent AND &H80&) = &H80& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFFF00
        END IF
        LET Parts.Mantissa1 = Pattern AND &H7FFFFF
        LET Parts.Mantissa2 = 0&

END SUB

SUB ParseFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING, pattern1 AS LONG, pattern2 AS LONG

        LET dbl = MKD$(FP64Value)
        LET pattern1 = CVL(RIGHT$(dbl, 4))
        LET pattern2 = CVL(LEFT$(dbl, 4))
        LET Parts.Sign = (pattern1 AND &H80000000) = &H80000000
        LET Parts.Exponent = (pattern1 AND &H7FF00000) \ &H100000
        IF (Parts.Exponent AND &H400&) = &H400& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFF800
        END IF
        LET Parts.Mantissa1 = ((pattern2 AND &HE0000000) \ &H20000000) AND &H7&
        LET Parts.Mantissa1 = Parts.Mantissa1 OR ((pattern1 AND &HFFFFF) * &H8&)
        LET Parts.Mantissa2 = pattern2 AND &H1FFFFFFF

END SUB

SUB ParseFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING, pattern1 AS LONG, pattern2 AS LONG

        LET dbl = MKD$(FP64Value)
        LET pattern1 = CVL(RIGHT$(dbl, 4))
        LET pattern2 = CVL(LEFT$(dbl, 4))
        LET Parts.Sign = (pattern1 AND &H800000) = &H800000
        LET Parts.Exponent = ((pattern1 AND &HFF000000) \ &H1000000) AND &HFF&
        IF (Parts.Exponent AND &H80&) = &H80& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFFF00
        END IF
        LET Parts.Mantissa1 = pattern1 AND &H7FFFFF
        LET Parts.Mantissa2 = pattern2

END SUB
