'   FLOATING POINT DEMONSTRATION PROGRAM
'
'   File:   floatqb.bas
'   Author: Jonathan Dale Kirwan
'
'   Creation Date:  Sun 26-Oct-1997 17:20:40
'   Last Modified:  Thu 12-Aug-2004 18:08:29
'        Modified:  Thu 22-Apr-1999 15:45:19
'
'   Copyright (C) 1997, 1998, 1999, 2004 Jonathan Dale Kirwan
'   All Rights Reserved:  See COPYRIGHT NOTICE below.
'
'
'   DESCRIPTION
'
'   This program allows you to experiment with floating point formats.
'   It supports two different format types.  IEEE and Microsoft's early
'   binary format, in both 32-bit and 64-bit versions.  To keep it simple,
'   these formats are not available for exploration at the same time.
'   Instead, you run the QuickBASIC or QBASIC environment with the /MBF
'   option, if you want the Microsoft Binary Format and without it for
'   the IEEE format.  The program will detect which mode was used and
'   adjust itself, accordingly.
'
'   You may use algebriac expressions to compute a value, as well.  Just
'   enter the expression as you would in BASIC.  Only +, -, *, /, and
'   parentheses are supported, though.
'
'
'   DESIGN
'
'   The BASIC functions CVL(), CVS(), CVD(), MKL$(), MKS$(), and MKD$()
'   are used to figure out the internal format information.  The newer
'   QuickBASIC programs do not actually store floating point values in
'   the /MBF format, even when the command option is specified.  Instead,
'   variables are stored in the IEEE format.  The only difference caused
'   by using /MBF is to change the way MKS$(), MKD$(), CVL(), and CVD()
'   do their conversion work.  This can lead to problems, since the IEEE
'   DOUBLE values have 11-bit exponents while the format supported by
'   MKD$() [when the /MBR option is used] only supports 8 bit exponents.
'
'
'   TARGET COMPILER
'
'   This module is designed to be used with Microsoft's QBASIC v1.1 and
'   the QuickBASIC 4.5 compiler and later compilers.  They both support
'   the /MBF switch option on the command line to allow the user to
'   switch operation between the two different floating point formats.
'
'
'   MODIFICATIONS
'
'   08/20/98    jk  Added support for DOUBLE precision floating point
'   values and changed the organization of the code.
'
'   20-Apr-1999 jk  Added support for entering the values as expressions.
'
'   22-Apr-1999 jk  Modified to incorporate the newer EVALUATE module.
'
'   12-Aug-2004 jk  Removed old expression code and replaced it with the
'   newer code I've documented on the web site.
'
'
'   COPYRIGHT NOTICE
'
'   Jonathan Dale Kirwan grants you a non-transferable, non-exclusive,
'   royalty-free worldwide license to use, copy, modify, prepare deriva-
'   tive works of and distribute this software, subject to your agreement
'   that you acquire no ownership right, title, or interest in this soft-
'   ware and your agreement that this software is research work which is
'   provided 'as is', where Jonathan Dale Kirwan disclaims all warranties
'   with regard to this software, including all implied warranties of
'   merchantability and fitness of purpose.  In no event shall Jonathan
'   Dale Kirwan be liable for any direct, indirect, consequential or
'   special damages or any damages whatsoever resulting from loss of use,
'   data or profits, whether in an action of contract, negligence or
'   other tortious action, arising out of or in connection with the use
'   or performance of this software.

CONST PROGTITLE$ = "FLOATING POINT FORMATS"
CONST PROGVERSION% = 3

CONST MAXVARS% = 1000
CONST MAXHEAD% = 210

CONST colorNORMAL% = 7
CONST colorFIGURES% = 12
CONST colorTITLE% = 13
CONST colorDATA% = 14

CONST TRUE% = -1
CONST FALSE% = 0

TYPE FPEXPOSED
    Sign AS INTEGER             ' (0) is positive, (-1) is negative
    Exponent AS INTEGER         ' all exponent bits
    Mantissa1 AS LONG           ' most significant mantissa bits
    Mantissa2 AS LONG           ' least significant mantissa bits
END TYPE

DECLARE SUB DisplayTitle (IEEEMode AS INTEGER, TitleName AS STRING, Version AS INTEGER)
DECLARE SUB DisplayEraseLine ()
DECLARE FUNCTION GetUserInput% (Prompt AS STRING, FP32Mode AS INTEGER, FP32Value AS SINGLE, FP64Value AS DOUBLE)
DECLARE FUNCTION GetInput$ (Prompt AS STRING)
DECLARE SUB ParseFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB ParseFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB ParseFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB ParseFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayFP32Common (FP32Value AS SINGLE, Parts AS FPEXPOSED)
DECLARE SUB DisplayMBFCommon (Parts AS FPEXPOSED)
DECLARE SUB DisplayBits (Pattern AS LONG, BitCount AS INTEGER)
DECLARE FUNCTION HexValue& (Value AS STRING)

DECLARE FUNCTION Number% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION IsOverflow% (stk AS STRING)
DECLARE FUNCTION Pop# (stk AS STRING)
DECLARE FUNCTION Expression% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Statement% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE SUB SList (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

DECLARE FUNCTION Item% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Factor% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Term% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

DECLARE FUNCTION Mantissa% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION ScaleID% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Digit% (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Digits (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Scale (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Sign (eqpos AS INTEGER, eq AS STRING)

DECLARE FUNCTION Alpha% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION AlphaNumeric (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB AlphaNumerics (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Symbol% (eqpos AS INTEGER, eq AS STRING, token AS STRING)
DECLARE SUB UnaryFunction (token AS STRING, stk AS STRING)

DECLARE SUB Push (value AS DOUBLE, stk AS STRING)

DECLARE SUB Add (stk AS STRING)
DECLARE SUB Subtract (stk AS STRING)
DECLARE SUB Multiply (stk AS STRING)
DECLARE SUB Divide (stk AS STRING)
DECLARE SUB Power (stk AS STRING)

DECLARE SUB SkipSpaces (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Match% (charlist AS STRING, eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Overflow# ()

DECLARE SUB InitValues ()
DECLARE SUB SetValue (text AS STRING, value AS DOUBLE)
DECLARE FUNCTION GetValue# (text AS STRING, idx AS INTEGER)

DECLARE FUNCTION Hash& (msg AS STRING)
DECLARE FUNCTION ReHash& (priorhash AS LONG, msg AS STRING)

TYPE SYMBOLENTRY
    hashcode AS LONG
    nextptr AS INTEGER
END TYPE

DECLARE FUNCTION FindEntry% (vname AS STRING, hashtable() AS SYMBOLENTRY)
DECLARE FUNCTION AddEntry% (vname AS STRING, hashtable() AS SYMBOLENTRY)
DECLARE SUB InitEntries (vlimit AS INTEGER, hlimit AS INTEGER, hashtable() AS SYMBOLENTRY)


    DIM Parts AS FPEXPOSED, IEEEMode AS INTEGER
    DIM FP32Mode AS INTEGER, FP32Value AS SINGLE, FP64Value AS DOUBLE

        LET IEEEMode = (CVL(MKS$(1!)) = &H3F800000)

        DisplayTitle IEEEMode, PROGTITLE, PROGVERSION

        PRINT "  This program displays the binary storage format for single and double pre-"
        PRINT "  cision floating point values.  You can enter floating point values and see"
        PRINT "  the resulting format or you can enter a hexadecimal value and see the type"
        PRINT "  of floating point number that results from it.  Append ! to switch to single"
        PRINT "  precision, append # to switch to double precision, and prefix with &H to use"
        PRINT "  a hexadecimal value.  A blank entry exits the program."

    REDIM SHARED vartbl(0 TO 0) AS SYMBOLENTRY
    REDIM SHARED varval(1 TO 1) AS DOUBLE

        InitValues
        SetValue "PI", 4# * ATN(1#)
        SetValue "E", EXP(1#)

        LET FP32Mode = TRUE
        LET FP32Value = 0!
        DO
            LOCATE 11
            IF FP32Mode AND IEEEMode THEN
                ParseFP32IEEE FP32Value, Parts
                DisplayFP32IEEE FP32Value, Parts
            ELSEIF FP32Mode THEN
                ParseFP32MBF FP32Value, Parts
                DisplayFP32MBF FP32Value, Parts
            ELSEIF IEEEMode THEN
                ParseFP64IEEE FP64Value, Parts
                DisplayFP64IEEE FP64Value, Parts
            ELSE
                LET FP64Value = CVD(MKD$(FP64Value))
                ParseFP64MBF FP64Value, Parts
                DisplayFP64MBF FP64Value, Parts
            END IF
            LOCATE 23
        LOOP WHILE GetUserInput("  Enter the value: ", FP32Mode, FP32Value, FP64Value)

        END

ErrorHandler:

    DIM SHARED ErrNumber AS INTEGER

        LET ErrNumber = ERR
        RESUME NEXT

DATA 2078917053,  143302914, 1027100827, 1953210302,  755253631, 2002600785
DATA 1405390230,   45248011, 1099951567,  433832350, 2018585307,  438263339
DATA  813528929, 1703199216,  618906479,  573714703,  766270699,  275680090
DATA 1510320440, 1583583926, 1723401032, 1965443329, 1098183682, 1636505764
DATA  980071615, 1011597961,  643279273, 1315461275,  157584038, 1069844923
DATA  471560540,   89017443, 1213147837, 1498661368, 2042227746, 1968401469
DATA 1353778505, 1300134328, 2013649480,  306246424, 1733966678, 1884751139
DATA  744509763,  400011959, 1440466707, 1363416242,  973726663,   59253759
DATA 1639096332,  336563455, 1642837685, 1215013716,  154523136,  593537720
DATA  704035832, 1134594751, 1605135681, 1347315106,  302572379, 1762719719
DATA  269676381,  774132919, 1851737163, 1482824219,  125310639, 1746481261
DATA 1303742040, 1479089144,  899131941, 1169907872, 1785335569,  485614972
DATA  907175364,  382361684,  885626931,  200158423, 1745777927, 1859353594
DATA  259412182, 1237390611,   48433401, 1902249868,  304920680,  202956538
DATA  348303940, 1008956512, 1337551289, 1953439621,  208787970, 1640123668
DATA 1568675693,  478464352,  266772940, 1272929208, 1961288571,  392083579
DATA  871926821, 1117546963, 1871172724, 1771058762,  139971187, 1509024645
DATA  109190086, 1047146551, 1891386329,  994817018, 1247304975, 1489680608
DATA  706686964, 1506717157,  579587572,  755120366, 1261483377,  884508252
DATA  958076904, 1609787317, 1893464764,  148144545, 1415743291, 2102252735
DATA 1788268214,  836935336,  433233439, 2055041154, 2109864544,  247038362
DATA  299641085,  834307717, 1364585325,   23330161,  457882831, 1504556512
DATA 1532354806,  567072918,  404219416, 1276257488, 1561889936, 1651524391
DATA  618454448,  121093252, 1010757900, 1198042020,  876213618,  124757630
DATA 2082550272, 1834290522, 1734544947, 1828531389, 1982435068, 1002804590
DATA 1783300476, 1623219634, 1839739926,   69050267, 1530777140, 1802120822
DATA  316088629, 1830418225,  488944891, 1680673954, 1853748387,  946827723
DATA 1037746818, 1238619545, 1513900641, 1441966234,  367393385,  928306929
DATA  946006977,  985847834, 1049400181, 1956764878,   36406206, 1925613800
DATA 2081522508, 2118956479, 1612420674, 1668583807, 1800004220, 1447372094
DATA  523904750, 1435821048,  923108080,  216161028, 1504871315,  306401572
DATA 2018281851, 1820959944, 2136819798,  359743094, 1354150250, 1843084537
DATA 1306570817,  244413420,  934220434,  672987810, 1686379655, 1301613820
DATA 1601294739,  484902984,  139978006,  503211273,  294184214,  176384212
DATA  281341425,  228223074,  147857043, 1893762099, 1896806882, 1947861263
DATA 1193650546,  273227984, 1236198663, 2116758626,  489389012,  593586330
DATA  275676551,  360187215,  267062626,  265012701,  719930310, 1621212876
DATA 2108097238, 2026501127, 1865626297,  894834024,  552005290, 1404522304
DATA   48964196,    5816381, 1889425288,  188942202,  509027654,   36125855
DATA  365326415,  790369079,  264348929,  513183458,  536647531,   13672163
DATA  313561074, 1730298077,  286900147, 1549759737, 1699573055,  776289160
DATA 2143346068, 1975249606, 1136476375,  262925046,   92778659, 1856406685
DATA 1884137923,   53392249, 1735424165, 1602280572,          0

SUB DisplayBits (Pattern AS LONG, BitCount AS INTEGER)

    DIM i AS INTEGER
    DIM mask AS LONG
    DIM bc AS INTEGER

        LET bc = BitCount
        IF bc = 32 THEN
            LET bc = 31
            PRINT USING "#"; -((Pattern AND &H80000000) <> 0);
            LOCATE , POS(0) + 1
        END IF

        LET mask = 2 ^ (bc - 1)
        FOR i = 1 TO bc
            PRINT USING "#"; -((Pattern AND mask) <> 0);
            LOCATE , POS(0) + 1
            LET mask = mask \ 2
        NEXT i

END SUB

SUB DisplayEraseLine

        LOCATE , 1
        PRINT SPACE$(80);
        LOCATE , 1

END SUB

SUB DisplayFP32Common (FP32Value AS SINGLE, Parts AS FPEXPOSED)

        LOCATE CSRLIN + 4, 1
        DisplayEraseLine
        PRINT
        DisplayEraseLine
        PRINT
        DisplayEraseLine

        LOCATE CSRLIN + 2, 1
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Hex: ";
        COLOR colorDATA
        PRINT RIGHT$("0000000" + HEX$(CVL(MKS$(FP32Value))), 8)
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Val: ";
        COLOR colorDATA
        PRINT FP32Value

END SUB

SUB DisplayFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)

        LOCATE , 1
        COLOR colorFIGURES
        PRINT "   Sgn       Exponent        (hidden bit)                        Mantissa"
        PRINT "   Ŀ   Ŀ   Ŀ"
        PRINT "                                         "
        PRINT "         "
        COLOR colorDATA
        LOCATE CSRLIN - 2, 5
        DisplayBits CLNG(Parts.Sign), 1
        LOCATE , 11
        DisplayBits CLNG(Parts.Exponent), 8
        LOCATE , 31
        DisplayBits Parts.Mantissa1, 23

        LOCATE CSRLIN - 2, 29
        IF Parts.Exponent = 0 THEN
            DisplayBits 0, 1
        ELSE
            DisplayBits 1, 1
        END IF

        DisplayFP32Common FP32Value, Parts

        COLOR colorNORMAL

END SUB

SUB DisplayFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)

        DisplayMBFCommon Parts
        DisplayFP32Common FP32Value, Parts
        COLOR colorNORMAL

END SUB

SUB DisplayFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING

        LOCATE , 1

        COLOR colorFIGURES
        PRINT "   Sgn           Exponent          (hidden bit)            Upper Mantissa"
        PRINT "   Ŀ   Ŀ   Ŀ"
        PRINT "                                         "
        PRINT "         "
        COLOR colorDATA

        LOCATE CSRLIN - 2, 5
        DisplayBits CLNG(Parts.Sign), 1
        LOCATE , 11
        DisplayBits CLNG(Parts.Exponent), 11
        LOCATE , 37
        DisplayBits Parts.Mantissa1 \ &H8, 20

        LOCATE CSRLIN - 2, 35
        IF Parts.Exponent = 0 THEN
            DisplayBits 0, 1
        ELSE
            DisplayBits 1, 1
        END IF

        LOCATE CSRLIN + 4, 1
        COLOR colorFIGURES
        PRINT "           Ŀ"
        PRINT "    Lower                                  "
        PRINT "   Mantissa"
        COLOR colorDATA

        LOCATE CSRLIN - 2, 13
        DisplayBits Parts.Mantissa1 AND &H7, 3
        LOCATE , 19
        DisplayBits Parts.Mantissa2, 29

        LET dbl = MKD$(FP64Value)
        LOCATE CSRLIN + 3, 1
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Hex: ";
        COLOR colorDATA
        PRINT RIGHT$("0000000" + HEX$(CVL(MID$(dbl, 5))), 8); RIGHT$("0000000" + HEX$(CVL(dbl)), 8)
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Val: ";
        COLOR colorDATA
        PRINT FP64Value

        COLOR colorNORMAL

END SUB

SUB DisplayFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING

        DisplayMBFCommon Parts
        COLOR colorFIGURES
        LOCATE , 63: PRINT "Upper";

        LOCATE CSRLIN + 4, 1
        PRINT "           Ŀ"
        PRINT "    Lower                                  "
        PRINT "   Mantissa"
        COLOR colorDATA

        LOCATE CSRLIN - 2, 13
        DisplayBits Parts.Mantissa2, 32

        LET dbl = MKD$(FP64Value)
        LOCATE CSRLIN + 3, 1
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Hex: ";
        COLOR colorDATA
        PRINT RIGHT$("0000000" + HEX$(CVL(MID$(dbl, 5))), 8); RIGHT$("0000000" + HEX$(CVL(dbl)), 8)
        COLOR colorFIGURES
        DisplayEraseLine
        PRINT "   Val: ";
        COLOR colorDATA
        PRINT FP64Value

        COLOR colorNORMAL

END SUB

SUB DisplayMBFCommon (Parts AS FPEXPOSED)

        LOCATE , 1
        COLOR colorFIGURES
        PRINT "       Exponent        Sgn   (hidden bit)                        Mantissa"
        PRINT "   Ŀ   Ŀ   Ŀ"
        PRINT "                                         "
        PRINT "         "
        COLOR colorDATA
        LOCATE CSRLIN - 2, 25
        DisplayBits CLNG(Parts.Sign), 1
        LOCATE , 5
        DisplayBits CLNG(Parts.Exponent), 8
        LOCATE , 31
        DisplayBits Parts.Mantissa1, 23

        LOCATE CSRLIN - 2, 29
        IF Parts.Exponent = 0 AND Parts.Mantissa1 = 0 AND Parts.Mantissa2 = 0 THEN
            DisplayBits 0, 1
        ELSE
            DisplayBits 1, 1
        END IF

END SUB

SUB DisplayTitle (IEEEMode AS INTEGER, TitleName AS STRING, Version AS INTEGER)

        CLS
        COLOR colorTITLE
        PRINT " "; STRING$(78, "")
        IF IEEEMode THEN
            PRINT " IEEE";
        ELSE
            PRINT " MBF ";
        END IF
        PRINT SPACE$(36 - LEN(TitleName) \ 2); TitleName;
        LOCATE , 71 + (Version > 9)
        PRINT "Version"; Version
        PRINT " "; STRING$(78, "")
        COLOR colorNORMAL

END SUB

FUNCTION GetInput$ (Prompt AS STRING)

    DIM Value AS STRING

        DisplayEraseLine
        PRINT Prompt;
        INPUT "", Value

        LET GetInput = LTRIM$(RTRIM$(UCASE$(Value)))

END FUNCTION

FUNCTION GetUserInput% (Prompt AS STRING, FP32Mode AS INTEGER, FP32Value AS SINGLE, FP64Value AS DOUBLE)

    DIM Value AS STRING, v AS DOUBLE
    DIM p AS INTEGER, expstack AS STRING

        LET Value = GetInput(Prompt)
        IF Value = "" THEN
            LET GetUserInput = FALSE
            EXIT FUNCTION
        END IF

        IF RIGHT$(Value, 1) = "#" THEN
            LET Value = LEFT$(Value, LEN(Value) - 1)
            LET FP32Mode = FALSE
        ELSEIF RIGHT$(Value, 1) = "!" THEN
            LET Value = LEFT$(Value, LEN(Value) - 1)
            LET FP32Mode = TRUE
        ELSEIF RIGHT$(Value, 1) = "&" THEN
            LET Value = LEFT$(Value, LEN(Value) - 1)
        END IF

        IF INSTR(Value, "&H") = 1 THEN
            IF FP32Mode THEN
                LET FP32Value = CVS(MKL$(HexValue(MID$(Value, 3))))
            ELSE
                LET Value = RIGHT$(STRING$(16, "0") + MID$(Value, 3), 16)
                LET FP64Value = CVD(MKL$(HexValue(RIGHT$(Value, 8))) + MKL$(HexValue(LEFT$(Value, 8))))
            END IF
        ELSE
            LET p = 1
            LET expstack = ""
            IF Statement(p, Value, expstack) THEN
                IF LEN(expstack) = 0 THEN
                    LET v = VAL(Value)
                ELSEIF IsOverflow(expstack) THEN
                    LET v = CVD(MKL$(0) + MKL$(&H7FF00000))
                ELSEIF p > LEN(Value) THEN
                    LET v = Pop(expstack)
                    SetValue "ANS", v
                ELSE
                    LET v = VAL(Value)
                END IF
            ELSE
                LET v = VAL(Value)
            END IF
            IF FP32Mode THEN
                LET FP32Value = v
            ELSE
                LET FP64Value = v
            END IF
        END IF

        LET GetUserInput = TRUE

END FUNCTION

FUNCTION HexValue& (Value AS STRING)

    DIM i AS INTEGER
    DIM eval AS STRING

        LET eval = Value
        FOR i = LEN(eval) TO 1 STEP -1
            IF INSTR("0123456789ABCDEF", MID$(eval, i, 1)) = 0 THEN
                LET eval = LEFT$(eval, i - 1) + MID$(eval, i + 1)
            END IF
        NEXT i
        DO
            FOR i = 1 TO LEN(eval)
                IF MID$(eval, i, 1) <> "0" THEN
                    EXIT FOR
                END IF
            NEXT i
            LET eval = MID$(eval, i)
            IF LEN(eval) > 8 THEN
                LET eval = RIGHT$(eval, 8)
            END IF
        LOOP WHILE LEFT$(eval, 1) = "0"

        IF LEN(eval) = 8 AND INSTR("89ABCDEF", LEFT$(eval, 1)) <> 0 THEN
            LET HexValue = CLNG(VAL("&H" + eval))
        ELSE
            LET HexValue = CLNG(VAL("&H" + eval + "&"))
        END IF

END FUNCTION

SUB ParseFP32IEEE (FP32Value AS SINGLE, Parts AS FPEXPOSED)

    DIM Pattern AS LONG

        LET Pattern = CVL(MKS$(FP32Value))
        LET Parts.Sign = (Pattern AND &H80000000) = &H80000000
        LET Parts.Exponent = (Pattern AND &H7F800000) \ &H800000
        IF (Parts.Exponent AND &H80&) = &H80& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFFF00
        END IF
        LET Parts.Mantissa1 = Pattern AND &H7FFFFF
        LET Parts.Mantissa2 = 0&

END SUB

SUB ParseFP32MBF (FP32Value AS SINGLE, Parts AS FPEXPOSED)

    DIM Pattern AS LONG

        LET Pattern = CVL(MKS$(FP32Value))
        LET Parts.Sign = (Pattern AND &H800000) = &H800000
        LET Parts.Exponent = ((Pattern AND &HFF000000) \ &H1000000) AND &HFF&
        IF (Parts.Exponent AND &H80&) = &H80& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFFF00
        END IF
        LET Parts.Mantissa1 = Pattern AND &H7FFFFF
        LET Parts.Mantissa2 = 0&

END SUB

SUB ParseFP64IEEE (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING, pattern1 AS LONG, pattern2 AS LONG

        LET dbl = MKD$(FP64Value)
        LET pattern1 = CVL(RIGHT$(dbl, 4))
        LET pattern2 = CVL(LEFT$(dbl, 4))
        LET Parts.Sign = (pattern1 AND &H80000000) = &H80000000
        LET Parts.Exponent = (pattern1 AND &H7FF00000) \ &H100000
        IF (Parts.Exponent AND &H400&) = &H400& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFF800
        END IF
        LET Parts.Mantissa1 = ((pattern2 AND &HE0000000) \ &H20000000) AND &H7&
        LET Parts.Mantissa1 = Parts.Mantissa1 OR ((pattern1 AND &HFFFFF) * &H8&)
        LET Parts.Mantissa2 = pattern2 AND &H1FFFFFFF

END SUB

SUB ParseFP64MBF (FP64Value AS DOUBLE, Parts AS FPEXPOSED)

    DIM dbl AS STRING, pattern1 AS LONG, pattern2 AS LONG

        LET dbl = MKD$(FP64Value)
        LET pattern1 = CVL(RIGHT$(dbl, 4))
        LET pattern2 = CVL(LEFT$(dbl, 4))
        LET Parts.Sign = (pattern1 AND &H800000) = &H800000
        LET Parts.Exponent = ((pattern1 AND &HFF000000) \ &H1000000) AND &HFF&
        IF (Parts.Exponent AND &H80&) = &H80& THEN
            LET Parts.Exponent = Parts.Exponent OR &HFFFFFF00
        END IF
        LET Parts.Mantissa1 = pattern1 AND &H7FFFFF
        LET Parts.Mantissa2 = pattern2

END SUB

SUB Add (stk AS STRING)

    DIM sum AS DOUBLE, addend1 AS DOUBLE, addend2 AS DOUBLE

        LET addend2 = Pop(stk)
        LET addend1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET sum = addend1 + addend2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push sum, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION AddEntry% (vname AS STRING, hashtable() AS SYMBOLENTRY)

    DIM hashcode AS LONG, idx AS INTEGER, pidx AS INTEGER, hlimit AS INTEGER

        LET hashcode = ReHash(0&, vname)
        LET hlimit = CINT(hashtable(0).hashcode)
        LET idx = hashcode MOD hlimit
        IF idx < 0 THEN
            LET idx = idx + hlimit
        END IF
        LET idx = idx + 1

        IF hashtable(idx).nextptr >= 0 THEN
            DO
                IF hashtable(idx).hashcode = hashcode THEN
                    LET AddEntry = idx
                    EXIT FUNCTION
                END IF
                LET pidx = idx
                LET idx = hashtable(idx).nextptr
            LOOP WHILE idx > 0
            IF hashtable(0).nextptr <= UBOUND(hashtable) THEN
                LET idx = hashtable(0).nextptr
                LET hashtable(0).nextptr = idx + 1
                LET hashtable(pidx).nextptr = idx
            ELSE
                LET AddEntry = 0
                EXIT FUNCTION
            END IF
        END IF

        LET hashtable(idx).hashcode = hashcode
        LET hashtable(idx).nextptr = 0

        LET AddEntry = idx

END FUNCTION

FUNCTION Alpha% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", eqpos, eq)

    LET Alpha = status

END FUNCTION

FUNCTION AlphaNumeric (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", eqpos, eq)

    LET AlphaNumeric = status

END FUNCTION

SUB AlphaNumerics (eqpos AS INTEGER, eq AS STRING)

        DO WHILE AlphaNumeric(eqpos, eq)
        LOOP

END SUB

FUNCTION Digit% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789", eqpos, eq)

    LET Digit = status

END FUNCTION

SUB Digits (eqpos AS INTEGER, eq AS STRING)

        DO WHILE Digit(eqpos, eq)
        LOOP

END SUB

SUB Divide (stk AS STRING)

    DIM ratio AS DOUBLE, dividend AS DOUBLE, divisor AS DOUBLE

        LET divisor = Pop(stk)
        LET dividend = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET ratio = dividend / divisor
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push ratio, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Expression% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        IF Match("-", eqpos, eq) THEN
            LET status = Term(eqpos, eq, stk)
            IF status THEN
                Push -1#, stk
                Multiply stk
            END IF
        ELSE
            LET status = Match("+", eqpos, eq)
            LET status = Term(eqpos, eq, stk)
        END IF

        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("+", eqpos, eq) THEN
                LET status = Term(eqpos, eq, stk)
                IF status THEN
                    Add stk
                END IF
            ELSEIF Match("-", eqpos, eq) THEN
                LET status = Term(eqpos, eq, stk)
                IF status THEN
                    Subtract stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Expression = status

END FUNCTION

FUNCTION Factor% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Item(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("^", eqpos, eq) THEN
                LET status = Item(eqpos, eq, stk)
                IF status THEN
                    Power stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Factor = status

END FUNCTION

FUNCTION FindEntry% (vname AS STRING, hashtable() AS SYMBOLENTRY)

    DIM hashcode AS LONG, idx AS INTEGER, hlimit AS INTEGER

        LET hashcode = ReHash(0&, vname)
        LET hlimit = CINT(hashtable(0).hashcode)
        LET idx = hashcode MOD hlimit
        IF idx < 0 THEN
            LET idx = idx + hlimit
        END IF
        LET idx = idx + 1

        IF hashtable(idx).nextptr >= 0 THEN
            DO
                IF hashtable(idx).hashcode = hashcode THEN
                    EXIT DO
                END IF
                LET idx = hashtable(idx).nextptr
            LOOP WHILE idx > 0
        ELSE
            LET idx = 0
        END IF

        LET FindEntry = idx

END FUNCTION

FUNCTION GetValue# (text AS STRING, idx AS INTEGER)

    DIM value AS DOUBLE

        LET idx = FindEntry(text, vartbl())
        IF idx > 0 THEN
            LET value = varval(idx)
        ELSE
            LET value = 0#
        END IF

    LET GetValue = value

END FUNCTION

FUNCTION Hash& (msg AS STRING)

        LET Hash = ReHash(0&, msg)

END FUNCTION

SUB InitEntries (vlimit AS INTEGER, hlimit AS INTEGER, hashtable() AS SYMBOLENTRY)

    DIM i AS INTEGER

        IF hlimit <= vlimit THEN
            IF LBOUND(hashtable) <> 0 OR UBOUND(hashtable) <> vlimit THEN
                REDIM hashtable(0 TO vlimit) AS SYMBOLENTRY
            END IF
            LET hashtable(0).hashcode = CLNG(hlimit)    ' Size of the header.
            LET hashtable(0).nextptr = hlimit + 1       ' Next available entry.
            FOR i = 1 TO hlimit
                LET hashtable(i).nextptr = -1
            NEXT i
        END IF

END SUB

SUB InitValues

        REDIM vartbl(0 TO MAXVARS) AS SYMBOLENTRY
        InitEntries MAXVARS, MAXHEAD, vartbl()
        REDIM varval(1 TO MAXVARS) AS DOUBLE

END SUB

FUNCTION IsOverflow% (stk AS STRING)

    DIM status AS INTEGER

        LET status = (MID$(stk, LEN(stk) - 7) = MKD$(Overflow))

    LET IsOverflow = status

END FUNCTION

FUNCTION Item% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER, value AS DOUBLE
    DIM token AS STRING, idx AS INTEGER

        SkipSpaces eqpos, eq
        LET savepos = eqpos
        IF Match("(", eqpos, eq) THEN
            LET status = Expression(eqpos, eq, stk)
            IF status THEN
                SkipSpaces eqpos, eq
                LET status = Match(")", eqpos, eq)
            END IF
        ELSEIF Symbol(eqpos, eq, token) THEN
            SkipSpaces eqpos, eq
            IF NOT Match("(", eqpos, eq) THEN
                LET value = GetValue(UCASE$(token), idx)
                IF idx > 0 THEN
                    Push value, stk
                    LET status = -1
                ELSE
                    LET status = 0
                END IF
            ELSE
                LET status = Expression(eqpos, eq, stk)
                IF status THEN
                    SkipSpaces eqpos, eq
                    LET status = Match(")", eqpos, eq)
                    IF status THEN
                        UnaryFunction token, stk
                    END IF
                END IF
            END IF
        ELSE
            LET status = Number(eqpos, eq)
            IF status THEN
                LET ErrNumber = 0
                ON ERROR GOTO ErrorHandler
                LET value = VAL(MID$(eq, savepos, eqpos - savepos))
                ON ERROR GOTO 0
                IF ErrNumber = 0 THEN
                    Push value, stk
                ELSE
                    Push Overflow, stk
                END IF
            END IF
        END IF

    LET Item = status

END FUNCTION

FUNCTION Mantissa% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        IF Match(".", eqpos, eq) THEN
            LET status = Digit(eqpos, eq)
            IF status THEN
                Digits eqpos, eq
            END IF
        ELSEIF Digit(eqpos, eq) THEN
            Digits eqpos, eq
            LET status = -1 OR Match(".", eqpos, eq)
            Digits eqpos, eq
        ELSE
            LET status = 0
        END IF

    LET Mantissa = status

END FUNCTION

FUNCTION Match% (charlist AS STRING, eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        IF eqpos <= LEN(eq) THEN
            IF INSTR(charlist, MID$(eq, eqpos, 1)) <> 0 THEN
                LET eqpos = eqpos + 1
                LET status = -1
            END IF
        ELSE
            LET status = 0
        END IF

    LET Match = status

END FUNCTION

SUB Multiply (stk AS STRING)

    DIM product AS DOUBLE, multiplicand1 AS DOUBLE, multiplicand2 AS DOUBLE

        LET multiplicand2 = Pop(stk)
        LET multiplicand1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET product = multiplicand1 * multiplicand2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push product, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Number% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER, dummy AS INTEGER

        Sign eqpos, eq
        LET status = Mantissa(eqpos, eq)
        IF status THEN
            Scale eqpos, eq
        END IF

    LET Number = status

END FUNCTION

FUNCTION Overflow#

    DIM value AS DOUBLE

        LET value = CVD(STRING$(8, 255))

    LET Overflow = value

END FUNCTION

FUNCTION Pop# (stk AS STRING)

    DIM slen AS INTEGER, value AS DOUBLE

        LET slen = LEN(stk)
        LET value = CVD(MID$(stk, slen - 7))
        LET stk = LEFT$(stk, slen - 8)

    LET Pop = value

END FUNCTION

SUB Power (stk AS STRING)

    DIM result AS DOUBLE, pow1 AS DOUBLE, pow2 AS DOUBLE

        LET pow2 = Pop(stk)
        LET pow1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET result = pow1 ^ pow2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push result, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

SUB Push (value AS DOUBLE, stk AS STRING)

        LET stk = stk + MKD$(value)

END SUB

FUNCTION ReHash& (priorhash AS LONG, msg AS STRING)

    STATIC IsLoaded AS INTEGER
    STATIC scatter() AS LONG

    DIM i AS INTEGER

        IF NOT IsLoaded THEN
            RESTORE
            DIM scatter(0 TO 255) AS LONG
            FOR i = 0 TO 255
                READ scatter(i)
            NEXT i
            LET IsLoaded = -1
        END IF

    DIM s AS LONG, current AS LONG

        LET current = priorhash
        FOR i = 1 TO LEN(msg)
            SELECT CASE (current AND &HC0000000)
            CASE &H0&
                LET current = current * 2
            CASE &H40000000
                LET current = (current OR &H80000000) * 2
            CASE &H80000000
                LET current = (current AND &H7FFFFFFF) * 2
            CASE &HC0000000
                LET current = current * 2
            END SELECT
            LET s = scatter(ASC(MID$(msg, i, 1)))
            IF current < 0& THEN
                LET current = current + s
            ELSEIF current > &H7FFFFFFF - s THEN
                LET current = (&H7FFFFFFF - current) + (&H7FFFFFFF - s) + 2
            ELSE
                LET current = current + s
            END IF
        NEXT i

        LET ReHash = current

END FUNCTION

SUB Scale (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER

        LET savepos = eqpos
        IF ScaleID(eqpos, eq) THEN
            Sign eqpos, eq
            LET status = Digit(eqpos, eq)
            IF status THEN
                Digits eqpos, eq
            ELSE
                LET eqpos = savepos
            END IF
        END IF

END SUB

FUNCTION ScaleID% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("eE", eqpos, eq)

    LET ScaleID = status

END FUNCTION

SUB SetValue (text AS STRING, value AS DOUBLE)

    DIM idx AS INTEGER

        LET idx = AddEntry(text, vartbl())
        IF idx > 0 THEN
            LET varval(idx) = value
        END IF

END SUB

SUB Sign (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("+-", eqpos, eq)

END SUB

SUB SkipSpaces (eqpos AS INTEGER, eq AS STRING)

        DO WHILE eqpos <= LEN(eq)
            IF MID$(eq, eqpos, 1) <> " " THEN
                EXIT DO
            END IF
            LET eqpos = eqpos + 1
        LOOP

END SUB

SUB SList (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, value AS DOUBLE

        LET status = Statement(eqpos, eq, stk)
        DO WHILE status
            IF LEN(stk) = 0 THEN
                PRINT "<null>"
            ELSEIF IsOverflow(stk) THEN
                PRINT "Overflow!"
            ELSE
                LET value = Pop(stk)
                PRINT value
                SetValue "ANS", value
            END IF
            SkipSpaces eqpos, eq
            IF NOT Match(";", eqpos, eq) THEN
                EXIT DO
            END IF
            LET status = Statement(eqpos, eq, stk)
        LOOP

END SUB

FUNCTION Statement% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER, token AS STRING
    DIM value AS DOUBLE

        SkipSpaces eqpos, eq
        LET savepos = eqpos
        IF Symbol(eqpos, eq, token) THEN
            SkipSpaces eqpos, eq
            IF Match("=", eqpos, eq) THEN
                LET status = Expression(eqpos, eq, stk)
                IF status THEN
                    LET value = Pop(stk)
                    SetValue UCASE$(token), value
                    Push value, stk
                END IF
            ELSE
                LET eqpos = savepos
                LET status = Expression(eqpos, eq, stk)
            END IF
        ELSE
            LET status = Expression(eqpos, eq, stk)
        END IF

    LET Statement = status

END FUNCTION

SUB Subtract (stk AS STRING)

    DIM difference AS DOUBLE, minuend AS DOUBLE, subtrahend AS DOUBLE

        LET subtrahend = Pop(stk)
        LET minuend = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET difference = minuend - subtrahend
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push difference, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Symbol% (eqpos AS INTEGER, eq AS STRING, token AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER

        LET savepos = eqpos
        IF Alpha(eqpos, eq) THEN
            AlphaNumerics eqpos, eq
            LET token = MID$(eq, savepos, eqpos - savepos)
            LET status = -1
        ELSE
            LET status = 0
        END IF

    LET Symbol = status

END FUNCTION

FUNCTION Term% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Factor(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("*", eqpos, eq) THEN
                LET status = Factor(eqpos, eq, stk)
                IF status THEN
                    Multiply stk
                END IF
            ELSEIF Match("/", eqpos, eq) THEN
                LET status = Factor(eqpos, eq, stk)
                IF status THEN
                    Divide stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Term = status

END FUNCTION

SUB UnaryFunction (token AS STRING, stk AS STRING)

    DIM value AS DOUBLE, result AS DOUBLE, status AS INTEGER

        LET value = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        SELECT CASE UCASE$(token)
        CASE "SIN"
            LET result = SIN(value)
        CASE "COS"
            LET result = COS(value)
        CASE "TAN"
            LET result = TAN(value)
        CASE "SEC"
            LET result = 1# / SIN(value)
        CASE "CSC"
            LET result = 1# / COS(value)
        CASE "CTN"
            LET result = 1# / TAN(value)
        CASE "ATAN"
            LET result = ATN(value)
        CASE "ASIN"
            LET result = ATN(value / SQR(1# - value * value))
        CASE "ACOS"
            LET result = ATN(SQR(1# / (value * value) - 1#))
        CASE "ASEC"
            LET result = ATN(1# / SQR(value * value - 1#))
        CASE "ACSC"
            LET result = ATN(SQR(value * value - 1#))
        CASE "ACTN"
            LET result = ATN(1# / value)
        CASE "ABS"
            LET result = ABS(value)
        CASE "SGN"
            LET result = SGN(value)
        CASE "INT"
            LET result = INT(value)
        CASE "SQRT", "SQR"
            LET result = SQR(value)
        CASE "LOG", "LN"
            LET result = LOG(value)
        CASE "LOG2"
            LET result = LOG(value) / LOG(2#)
        CASE "LOG10"
            LET result = LOG(value) / LOG(10#)
        CASE "DEG"
            LET result = value * 45# / ATN(1#)
        CASE "RAD"
            LET result = value * ATN(1#) / 45#
        CASE "EXP"
            LET result = EXP(value)
        CASE "SINH"
            LET result = (EXP(value) - EXP(-value)) / 2#
        CASE "COSH"
            LET result = (EXP(value) + EXP(-value)) / 2#
        CASE "TANH"
            LET result = (EXP(value) - EXP(-value)) / (EXP(value) + EXP(-value))
        CASE "SECH"
            LET result = 2# / (EXP(value) - EXP(-value))
        CASE "CSCH"
            LET result = 2# / (EXP(value) + EXP(-value))
        CASE "COTH"
            LET result = (EXP(value) + EXP(-value)) / (EXP(value) - EXP(-value))
        CASE "RND"
            LET result = RND(value)
        CASE ELSE
            LET result = value
        END SELECT
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push result, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB
