'
'   File:   parse.bas
'   Creation Date:  Thu 15-Jul-2004 15:36:22        Jonathan D. Kirwan
'   Last Modified:  Sun 18-Jul-2004 11:53:20        Initial version.
'
'   Copyright (C) 2004 Jonathan Dale Kirwan, All Rights Reserved
'
'
'   DESCRIPTION
'
'   This is a program demonstrating parsing of mathematical expressions.
'   This version handles algebraic equations of the form:
'
'       unary := SIN | COS | TAN | SEC | CSC | CTN | ATAN | ASIN | ACOS |
'                   ASEC | ACSC | ACTN | ABS | SGN | INT | SQRT | LOG | EXP
'       moreitems := ^ item moreitems | <null>
'       item := name | number | ( expression ) | unary ( expression )
'       morefactors := * factor morefactors | / factor morefactors | <null>
'       factor := item moreitems
'       moreterms := + term moreterms | - term moreterms | <null>
'       term := factor morefactors
'       expression := sign term moreterms
'       statement := expression | name = expression
'       moreexpr := ; expression moreexpr | <null>
'       list := expression moreexpr
'
'   Where a number is:
'
'       digit := 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9
'       digits := digit digits | <null>
'       mantissa := . digit digits | digit . digits | digit digits
'       scaleid := e | E
'       scale := scaleid sign digit digits | <null>
'       sign := + | - | <null>
'       number := sign mantissa scale
'
'   And where a variable name or function name is:
'
'       alpha := A | B | C | D | E | F | G | H | I | J | K | L | M |
'                N | O | P | Q | R | S | T | U | V | W | X | Y | Z |
'                a | b | c | d | e | f | g | h | i | j | k | l | m |
'                n | o | p | q | r | s | t | u | v | w | x | y | z
'       alphanumeric := alpha | digit
'       alphanumerics := alphanumeric alphanumerics | <null>
'       name := alpha alphanumerics
'
'   It also calculates the resulting value and displays it, as well as
'   handling some standard unary functions. It also supports two special
'   values, PI and E, for use in calculating values and allows statements
'   to define new variable values.  Statements can be repeated, if
'   separated by semi-colons, as well.
'
'   See,
'
'       http://users.easystreet.com/jkirwan/new/parsing.html
'
'   for more detailed information on the design.
'
'
'   MODIFICATIONS
'
'   No modifications.
'
'
'   COPYRIGHT NOTICE
'
'   Jonathan Dale Kirwan grants you a non-transferable, non-exclusive,
'   royalty-free worldwide license to use, copy, modify, prepare deriva-
'   tive works of and distribute this software, subject to your agreement
'   that you acquire no ownership right, title, or interest in this soft-
'   ware and your agreement that this software is research work which is
'   provided 'as is', where Jonathan Dale Kirwan disclaims all warranties
'   with regard to this software, including all implied warranties of
'   merchantability and fitness of purpose.  In no event shall Jonathan
'   Dale Kirwan be liable for any direct, indirect, consequential or
'   special damages or any damages whatsoever resulting from loss of use,
'   data or profits, whether in an action of contract, negligence or
'   other tortious action, arising out of or in connection with the use
'   or performance of this software.

'$INCLUDE: 'VARIABLE.BI'
'$INCLUDE: 'PARSE.BI'

DECLARE FUNCTION Item% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Factor% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)
DECLARE FUNCTION Term% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

DECLARE FUNCTION Mantissa% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION ScaleID% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Digit% (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Digits (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION HexDigit% (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB HexDigits (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Scale (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB Sign (eqpos AS INTEGER, eq AS STRING)

DECLARE FUNCTION Alpha% (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION AlphaNumeric (eqpos AS INTEGER, eq AS STRING)
DECLARE SUB AlphaNumerics (eqpos AS INTEGER, eq AS STRING)
DECLARE FUNCTION Symbol% (eqpos AS INTEGER, eq AS STRING, token AS STRING)
DECLARE SUB UnaryFunction (token AS STRING, stk AS STRING)

DECLARE SUB Push (value AS DOUBLE, stk AS STRING)

DECLARE SUB Add (stk AS STRING)
DECLARE SUB Subtract (stk AS STRING)
DECLARE SUB Multiply (stk AS STRING)
DECLARE SUB Divide (stk AS STRING)
DECLARE SUB Power (stk AS STRING)

DECLARE FUNCTION Overflow# ()

        END

ErrorHandler:

    DIM SHARED ErrNumber AS INTEGER

        LET ErrNumber = ERR
        RESUME NEXT

SUB Add (stk AS STRING)

    DIM sum AS DOUBLE, addend1 AS DOUBLE, addend2 AS DOUBLE

        LET addend2 = Pop(stk)
        LET addend1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET sum = addend1 + addend2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push sum, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Alpha% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", eqpos, eq)

    LET Alpha = status

END FUNCTION

FUNCTION AlphaNumeric (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", eqpos, eq)

    LET AlphaNumeric = status

END FUNCTION

SUB AlphaNumerics (eqpos AS INTEGER, eq AS STRING)

        DO WHILE AlphaNumeric(eqpos, eq)
        LOOP

END SUB

FUNCTION Digit% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789", eqpos, eq)

    LET Digit = status

END FUNCTION

SUB Digits (eqpos AS INTEGER, eq AS STRING)

        DO WHILE Digit(eqpos, eq)
        LOOP

END SUB

SUB Divide (stk AS STRING)

    DIM ratio AS DOUBLE, dividend AS DOUBLE, divisor AS DOUBLE

        LET divisor = Pop(stk)
        LET dividend = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET ratio = dividend / divisor
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push ratio, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Expression% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        IF Match("-", eqpos, eq) THEN
            LET status = Term(eqpos, eq, stk)
            IF status THEN
                Push -1#, stk
                Multiply stk
            END IF
        ELSE
            LET status = Match("+", eqpos, eq)
            LET status = Term(eqpos, eq, stk)
        END IF

        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("+", eqpos, eq) THEN
                LET status = Term(eqpos, eq, stk)
                IF status THEN
                    Add stk
                END IF
            ELSEIF Match("-", eqpos, eq) THEN
                LET status = Term(eqpos, eq, stk)
                IF status THEN
                    Subtract stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Expression = status

END FUNCTION

FUNCTION Factor% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Item(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("^", eqpos, eq) THEN
                LET status = Item(eqpos, eq, stk)
                IF status THEN
                    Power stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Factor = status

END FUNCTION

FUNCTION HexDigit% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("0123456789ABCDEF", eqpos, eq)

    LET HexDigit = status

END FUNCTION

SUB HexDigits (eqpos AS INTEGER, eq AS STRING)

        DO WHILE HexDigit(eqpos, eq)
        LOOP

END SUB

FUNCTION IsOverflow% (stk AS STRING)

    DIM status AS INTEGER

        LET status = (MID$(stk, LEN(stk) - 7) = MKD$(Overflow))

    LET IsOverflow = status

END FUNCTION

FUNCTION Item% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER, value AS DOUBLE
    DIM token AS STRING, idx AS INTEGER

        SkipSpaces eqpos, eq
        LET savepos = eqpos
        IF Match("(", eqpos, eq) THEN
            LET status = Expression(eqpos, eq, stk)
            IF status THEN
                SkipSpaces eqpos, eq
                LET status = Match(")", eqpos, eq)
            END IF
        ELSEIF Symbol(eqpos, eq, token) THEN
            SkipSpaces eqpos, eq
            IF NOT Match("(", eqpos, eq) THEN
                LET value = GetValue(UCASE$(token), idx)
                IF idx > 0 THEN
                    Push value, stk
                    LET status = -1
                ELSE
                    LET status = 0
                END IF
            ELSE
                LET status = Expression(eqpos, eq, stk)
                IF status THEN
                    SkipSpaces eqpos, eq
                    LET status = Match(")", eqpos, eq)
                    IF status THEN
                        UnaryFunction token, stk
                    END IF
                END IF
            END IF
        ELSEIF Match("&", eqpos, eq) THEN
            LET status = 0
            IF Match("Hh", eqpos, eq) THEN
                IF HexDigit(eqpos, eq) THEN
                    HexDigits eqpos, eq
                    LET token = MID$(eq, savepos + 2, eqpos - savepos - 2)
                    DO WHILE LEN(token) > 0
                        IF LEFT$(token, 1) <> "0" THEN
                            EXIT DO
                        END IF
                        LET token = MID$(token, 2)
                    LOOP
                    LET ErrNumber = 0
                    ON ERROR GOTO ErrorHandler
                    IF LEN(token) <= 8 THEN
                        LET token = LEFT$(token + STRING$(8, "0"), 8)
                        LET value = CDBL(CVS(MKL$(VAL("&H" + token))))
                    ELSE
                        LET token = LEFT$(token + STRING$(16, "0"), 16)
                        LET value = CVD(MKL$(VAL(RIGHT$(token, 8))) + MKL$(VAL(LEFT$(token, 8))))
                    END IF
                    ON ERROR GOTO 0
                    IF ErrNumber = 0 THEN
                        Push value, stk
                    ELSE
                        Push Overflow, stk
                    END IF
                    LET status = -1
                END IF
            END IF
        ELSE
            LET status = Number(eqpos, eq)
            IF status THEN
                LET ErrNumber = 0
                ON ERROR GOTO ErrorHandler
                LET value = VAL(MID$(eq, savepos, eqpos - savepos))
                ON ERROR GOTO 0
                IF ErrNumber = 0 THEN
                    Push value, stk
                ELSE
                    Push Overflow, stk
                END IF
            END IF
        END IF

    LET Item = status

END FUNCTION

FUNCTION Mantissa% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        IF Match(".", eqpos, eq) THEN
            LET status = Digit(eqpos, eq)
            IF status THEN
                Digits eqpos, eq
            END IF
        ELSEIF Digit(eqpos, eq) THEN
            Digits eqpos, eq
            LET status = -1 OR Match(".", eqpos, eq)
            Digits eqpos, eq
        ELSE
            LET status = 0
        END IF

    LET Mantissa = status

END FUNCTION

FUNCTION Match% (charlist AS STRING, eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        IF eqpos <= LEN(eq) THEN
            IF INSTR(charlist, MID$(eq, eqpos, 1)) <> 0 THEN
                LET eqpos = eqpos + 1
                LET status = -1
            END IF
        ELSE
            LET status = 0
        END IF

    LET Match = status

END FUNCTION

SUB Multiply (stk AS STRING)

    DIM product AS DOUBLE, multiplicand1 AS DOUBLE, multiplicand2 AS DOUBLE

        LET multiplicand2 = Pop(stk)
        LET multiplicand1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET product = multiplicand1 * multiplicand2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push product, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Number% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER, dummy AS INTEGER

        Sign eqpos, eq
        LET status = Mantissa(eqpos, eq)
        IF status THEN
            Scale eqpos, eq
        END IF

    LET Number = status

END FUNCTION

FUNCTION Overflow#

    DIM value AS DOUBLE

        LET value = CVD(STRING$(8, 255))

    LET Overflow = value

END FUNCTION

FUNCTION Pop# (stk AS STRING)

    DIM slen AS INTEGER, value AS DOUBLE

        LET slen = LEN(stk)
        LET value = CVD(MID$(stk, slen - 7))
        LET stk = LEFT$(stk, slen - 8)

    LET Pop = value

END FUNCTION

SUB Power (stk AS STRING)

    DIM result AS DOUBLE, pow1 AS DOUBLE, pow2 AS DOUBLE

        LET pow2 = Pop(stk)
        LET pow1 = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET result = pow1 ^ pow2
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push result, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

SUB Push (value AS DOUBLE, stk AS STRING)

        LET stk = stk + MKD$(value)

END SUB

SUB Scale (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER

        LET savepos = eqpos
        IF ScaleID(eqpos, eq) THEN
            Sign eqpos, eq
            LET status = Digit(eqpos, eq)
            IF status THEN
                Digits eqpos, eq
            ELSE
                LET eqpos = savepos
            END IF
        END IF

END SUB

FUNCTION ScaleID% (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("eE", eqpos, eq)

    LET ScaleID = status

END FUNCTION

SUB Sign (eqpos AS INTEGER, eq AS STRING)

    DIM status AS INTEGER

        LET status = Match("+-", eqpos, eq)

END SUB

SUB SkipSpaces (eqpos AS INTEGER, eq AS STRING)

        DO WHILE eqpos <= LEN(eq)
            IF MID$(eq, eqpos, 1) <> " " THEN
                EXIT DO
            END IF
            LET eqpos = eqpos + 1
        LOOP

END SUB

SUB SList (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, value AS DOUBLE

        LET status = Statement(eqpos, eq, stk)
        DO WHILE status
            IF LEN(stk) = 0 THEN
                PRINT "<null>"
            ELSEIF IsOverflow(stk) THEN
                PRINT "Overflow!"
            ELSE
                LET value = Pop(stk)
                PRINT value
                SetValue "ANS", value
            END IF
            SkipSpaces eqpos, eq
            IF NOT Match(";", eqpos, eq) THEN
                EXIT DO
            END IF
            LET status = Statement(eqpos, eq, stk)
        LOOP

END SUB

FUNCTION Statement% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER, token AS STRING
    DIM value AS DOUBLE

        SkipSpaces eqpos, eq
        LET savepos = eqpos
        IF Symbol(eqpos, eq, token) THEN
            SkipSpaces eqpos, eq
            IF Match("=", eqpos, eq) THEN
                LET status = Expression(eqpos, eq, stk)
                IF status THEN
                    LET value = Pop(stk)
                    SetValue UCASE$(token), value
                    Push value, stk
                END IF
            ELSE
                LET eqpos = savepos
                LET status = Expression(eqpos, eq, stk)
            END IF
        ELSE
            LET status = Expression(eqpos, eq, stk)
        END IF

    LET Statement = status

END FUNCTION

SUB Subtract (stk AS STRING)

    DIM difference AS DOUBLE, minuend AS DOUBLE, subtrahend AS DOUBLE

        LET subtrahend = Pop(stk)
        LET minuend = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        LET difference = minuend - subtrahend
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push difference, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB

FUNCTION Symbol% (eqpos AS INTEGER, eq AS STRING, token AS STRING)

    DIM status AS INTEGER, savepos AS INTEGER

        LET savepos = eqpos
        IF Alpha(eqpos, eq) THEN
            AlphaNumerics eqpos, eq
            LET token = MID$(eq, savepos, eqpos - savepos)
            LET status = -1
        ELSE
            LET status = 0
        END IF

    LET Symbol = status

END FUNCTION

FUNCTION Term% (eqpos AS INTEGER, eq AS STRING, stk AS STRING)

    DIM status AS INTEGER

        LET status = Factor(eqpos, eq, stk)
        DO WHILE status
            SkipSpaces eqpos, eq
            IF Match("*", eqpos, eq) THEN
                LET status = Factor(eqpos, eq, stk)
                IF status THEN
                    Multiply stk
                END IF
            ELSEIF Match("/", eqpos, eq) THEN
                LET status = Factor(eqpos, eq, stk)
                IF status THEN
                    Divide stk
                END IF
            ELSE
                EXIT DO
            END IF
        LOOP

    LET Term = status

END FUNCTION

SUB UnaryFunction (token AS STRING, stk AS STRING)

    DIM value AS DOUBLE, result AS DOUBLE, status AS INTEGER

        LET value = Pop(stk)

        LET ErrNumber = 0
        ON ERROR GOTO ErrorHandler
        SELECT CASE UCASE$(token)
        CASE "SIN"
            LET result = SIN(value)
        CASE "COS"
            LET result = COS(value)
        CASE "TAN"
            LET result = TAN(value)
        CASE "SEC"
            LET result = 1# / SIN(value)
        CASE "CSC"
            LET result = 1# / COS(value)
        CASE "CTN"
            LET result = 1# / TAN(value)
        CASE "ATAN"
            LET result = ATN(value)
        CASE "ASIN"
            LET result = ATN(value / SQR(1# - value * value))
        CASE "ACOS"
            LET result = ATN(SQR(1# / (value * value) - 1#))
        CASE "ASEC"
            LET result = ATN(1# / SQR(value * value - 1#))
        CASE "ACSC"
            LET result = ATN(SQR(value * value - 1#))
        CASE "ACTN"
            LET result = ATN(1# / value)
        CASE "ABS"
            LET result = ABS(value)
        CASE "SGN"
            LET result = SGN(value)
        CASE "INT"
            LET result = INT(value)
        CASE "SQRT", "SQR"
            LET result = SQR(value)
        CASE "LOG", "LN"
            LET result = LOG(value)
        CASE "LOG2"
            LET result = LOG(value) / LOG(2#)
        CASE "LOG10"
            LET result = LOG(value) / LOG(10#)
        CASE "DEG"
            LET result = value * 45# / ATN(1#)
        CASE "RAD"
            LET result = value * ATN(1#) / 45#
        CASE "EXP"
            LET result = EXP(value)
        CASE "SINH"
            LET result = (EXP(value) - EXP(-value)) / 2#
        CASE "COSH"
            LET result = (EXP(value) + EXP(-value)) / 2#
        CASE "TANH"
            LET result = (EXP(value) - EXP(-value)) / (EXP(value) + EXP(-value))
        CASE "SECH"
            LET result = 2# / (EXP(value) - EXP(-value))
        CASE "CSCH"
            LET result = 2# / (EXP(value) + EXP(-value))
        CASE "COTH"
            LET result = (EXP(value) + EXP(-value)) / (EXP(value) - EXP(-value))
        CASE "RND"
            LET result = RND(value)
        CASE ELSE
            LET result = value
        END SELECT
        ON ERROR GOTO 0

        IF ErrNumber = 0 THEN
            Push result, stk
        ELSE
            Push Overflow, stk
        END IF

END SUB
