
////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//              CONDITIONAL BRANCH LOGGING PLUGIN FOR OLLYDBG                 //
//			   		by Blabberer / dELTA / Kayaker 	                          //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <commctrl.h>
#include "plugin.h"


//////////////////////////////////////////////////////////////////
//
//				General Logging function Declarations
//
//////////////////////////////////////////////////////////////////

typedef struct _LOGGING_RANGE
{
	DWORD	r_start;
	DWORD	r_end;
	char	*desc;
	BYTE	enabled;
} LOGGING_RANGE,*PLOGGING_RANGE;


typedef struct _BRANCHES
{
//									Mnemonic        1-byte Opcode       2-byte Opcode
	BYTE	op_ja_jnbe;         // JA JNBE					77                  0F87
	BYTE	op_jae_jnb_jnc;     // JAE JNB JNC      	    73                  0F83
	BYTE	op_jb_jc_jnae;      // JB JC JNAE			    72                  0F82
	BYTE	op_jbe_jna;         // JBE JNA	 			    76                  0F86
	BYTE	op_jcxz_jecxz;      // JCXZ JECXZ               E3                  E367
	BYTE	op_je_jz;           // JE JZ 				    74                  0F84
	BYTE	op_jg_jnle;         // JG JNLE				    7F                  0F8F
	BYTE	op_jge_jnl;         // JGE JNL				    7D                  0F8D
	BYTE	op_jl_jnge;         // JL JNGE					7C                  0F8C
	BYTE	op_jle_jng;         // JLE JNG	 			    7E                  0F8E
	BYTE	op_jne_jnz;         // JNE JNZ 				    75                  0F85
	BYTE	op_jno;             // JNO					    71                  0F81
	BYTE	op_jnp_jpo;         // JNP JPO				    7B                  0F8B
	BYTE	op_jns;             // JNS					    79                  0F89
	BYTE	op_jo;              // JO					    70                  0F80
	BYTE	op_jp_jpe;          // JP JPE				    7A                  0F8A
	BYTE	op_js;              // JS					    78                  0F88
	BYTE    op_loopx;			// LOOPxx					E0, E1, E2		
}BRANCHES, *PBRANCHES;


typedef struct _CONFIG_DATA
{
        DWORD           noof_logged_ranges;
        PLOGGING_RANGE  *logged_ranges;
        DWORD           reserved_1;
        DWORD           noof_excluded_ranges;
        PLOGGING_RANGE  *excluded_ranges;
        DWORD           reserved_2;
        char            *log_file_path;
        BYTE            log_all_branch;
        BYTE            delphidummy1;
        BYTE            delphidummy2;
        BYTE            delphidummy3;
        PBRANCHES       log_spec_branch;
} CONFIG_DATA, *PCONFIG_DATA;


HINSTANCE			hinst;
HWND				hwmain;
HMODULE				guidll;
HCURSOR				hCursorWait;


int		Checkjump(char *answer, ulong parm, t_reg *reg);
void	Setasmmodel_JCC(t_asmmodel model[NMODELS]);
void	Setasmmodel_Jxx(t_asmmodel model[NMODELS], PCONFIG_DATA ccdata);
void	SetLoggingBreakpoints(t_asmmodel model[NMODELS], PCONFIG_DATA ccdata);


// cbl_gui.dll load check
BOOLEAN				GUIDLLLOADED = FALSE;
void	BroadcastDllLoadFailure(void);

// Toolbar button
HWND				hToolbarButton;
HBITMAP				hToolbarBitmap;	
static WNDPROC		OldWndProc;
void	CreateTooltip (HWND hwnd);
LRESULT CALLBACK ToolbarButtonWinproc(HWND hw,UINT msg,WPARAM wp,LPARAM lp);

// High resolution timer
LARGE_INTEGER		start_time;
LARGE_INTEGER		stop_time;
LARGE_INTEGER		freq;
double				time_diff;

void	START_CODE_EXECUTION_TIMER(void);
void	STOP_CODE_EXECUTION_TIMER(void);


HWND	hwndHelpWindow;
BOOL	HelpWindow(void);
LRESULT CALLBACK HelpWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);


//////////////////////////////////////////////////////////////////
//
//			Guidll Export Declarations - General
//
//////////////////////////////////////////////////////////////////

typedef DWORD (__stdcall *PSHOW_PLUGIN_SETTINGS_MODAL)	(DWORD AppHandle,
														 PCONFIG_DATA *used_config_data);
typedef BOOL  (__stdcall *PWRITE_STRINGTOLOG)		    (char *string_to_write);
typedef void  (__stdcall *PSHOWWAITMESSAGEDIALOG)       (HWND current_app_handle);
typedef void  (__stdcall *PHIDEWAITMESSAGEDIALOG)       (void); 


PSHOW_PLUGIN_SETTINGS_MODAL		pshowpsm;
PWRITE_STRINGTOLOG				pwritestrtolog;
PSHOWWAITMESSAGEDIALOG			pshowdialog;
PHIDEWAITMESSAGEDIALOG			pclosedialog;


//////////////////////////////////////////////////////////////////
//
//			Guidll Export Declarations - Range Settings 
//
//////////////////////////////////////////////////////////////////

//Adds a range to the given config-struct
typedef void  (__stdcall *PADDRANGE) (PCONFIG_DATA target_config_data_ptr,
									  DWORD range_start,
									  DWORD range_end,
									  char *range_description,
									  BYTE is_exclusion_range,
									  BYTE is_enabled); 

//Removes a range from the given config-struct
typedef void  (__stdcall *PREMOVERANGE) (PCONFIG_DATA target_config_data_ptr,
									  DWORD range_start,
									  DWORD range_end,
									  BYTE is_exclusion_range); 
									  
//Returns a pointer to the current config struct
//(is mostly useful before the first call to the settings dialog)
typedef PCONFIG_DATA  (__stdcall *PGETCURRENTRAWSETTINGSPTR) (void); 
									  
//Sets the given current config struct as the "current" one
//(is mostly useful if you manipulate the raw structure yourself)
typedef void  (__stdcall *PSETCURRENTRAWSETTINGSPTR) (PCONFIG_DATA new_config_data_ptr); 


PADDRANGE					paddrange;
PREMOVERANGE				premoverange;
PGETCURRENTRAWSETTINGSPTR	pgetcurrawsetptr;
PSETCURRENTRAWSETTINGSPTR	psetcurrawsetptr;


//////////////////////////////////////////////////////////////////
//
//			Guidll Export Declarations - UDD save/restore functions
//			
//////////////////////////////////////////////////////////////////

//Returns a linear buffer containing the entire current config
//(useful for saving the entire config in e.g. a .udd file)
typedef PULONG (__stdcall *PGETSERIALIZEDSETTINGS) (PULONG serialized_settings_size); 


//Updates the config with the given linear buffer packed config
//(useful for restoring an entire config from e.g. a .udd file)
typedef void (__stdcall *PSETSERIALIZEDSETTINGS) (PULONG serialized_settings_buf,
												  ULONG serialized_settings_size); 

PGETSERIALIZEDSETTINGS		GetSerializedSettings;
PSETSERIALIZEDSETTINGS		SetSerializedSettings;

PULONG		g_serialized_settings_buf;
ULONG		g_serialized_settings_size;



//////////////////////////////////////////////////////////////////
//
//			Assembly model Declarations - Conditonal Jumps
//			
//////////////////////////////////////////////////////////////////

// use an enum to reference the *_opcode[] tables
enum JMPTYPE_T{JA,JAE,JB,JBE,
			   JCXZ,JE,JG,JGE,
			   JL,JLE,JNE,JNO,
			   JNP,JNS,JO,JP,
			   JS, LOOPxx} jmptype;


// matching t_asmmodel values for JCC
UCHAR	jcccode1 = 0x70;
UCHAR	jccmask1 = 0xF0;
USHORT	jcccode2 = 0x800F;
USHORT	jccmask2 = 0xF0FF;

// matching t_asmmodel values for Jxx
UCHAR	onebyte_mask = 0xFF;
USHORT	twobyte_mask = 0xFFFF;

unsigned char onebyte_opcode[17] =
{
	0x77, 0x73, 0x72, 0x76,			// JA,JAE,JB,JBE
	0xE3, 0x74, 0x7F, 0x7D,			// JCXZ,JE,JG,JGE
	0x7C, 0x7E,	0x75, 0x71,			// JL,JLE,JNE,JNO
	0x7B, 0x79, 0x70, 0x7A,			// JNP,JNS,JO,JP
	0x78							// JS
};

unsigned char twobyte_opcode[34] =
{
	0x0F, 0x87, 0x0F, 0x83, 0x0F, 0x82, 0x0F, 0x86,			// JA,JAE,JB,JBE
	0x67, 0xE3, 0x0F, 0x84, 0x0F, 0x8F, 0x0F, 0x8D,			// JCXZ,JE,JG,JGE
	0x0F, 0x8C, 0x0F, 0x8E,	0x0F, 0x85, 0x0F, 0x81, 		// JL,JLE,JNE,JNO
	0x0F, 0x8B, 0x0F, 0x89, 0x0F, 0x80, 0x0F, 0x8A, 		// JNP,JNS,JO,JP
	0x0F, 0x78												// JS
};

unsigned char loopxx_opcode[3] =
{
    0xE0,		// LOOPNZ / LOOPNE
    0xE1,		// LOOPZ / LOOPE
    0xE2		// LOOP
};


//////////////////////////////////////////////////////////////////
//
//			Custom Jump Table Window function Declarations
//
//////////////////////////////////////////////////////////////////

// A sorted data structure composed of a t_sortheader
// followed by the data element(s) we want to maintain
typedef struct t_jmplog {

	// t_sortheader	
	ulong	addr;					// Address of conditional jump we are logging
	ulong	size;	    			// Size of addr element			(1)
	ulong	type;					// Type of data element, TY_xxx	(0)

	// Data elements
	int		index;					// Item count
	int		count_jump_taken;		// Count of 'Jump Taken' results for address
	int		count_jump_not_taken;	// Count of 'Jump Not Taken' results for address
	char	last_jump_result[32];	// Final jump result (taken / not taken)

} t_jmplog;

t_table		jmptbl;					// jmptbl table
char		jmptblwinclass[32];		// Name of jmptbl window class


// Context menu items
#define DISABLE_BP				       		1
#define REMOVE_BP				        	2
#define ENABLE_AS_CONDITIONAL_LOGGED_BP 	3
#define ENABLE_AS_ACTIVE_BP			      	4
#define DISABLE_ALL_BPS						5
#define ENABLE_ALL_BPS						6
#define REMOVE_ALL_BPS						7
#define DELETE_SELECTED_ENTRY		  		8
#define DELETE_ALL_ENTRIES			    	9
#define DELETE_ENTRIES_NOT_EXECUTED		    10
#define DELETE_ENTRIES_EXECUTED			    11
#define FOLLOW_IN_DISASM		   			12
#define RESET_JUMP_RESULTS		   			13

// Sort flags
#define SORT_UP				            	0
#define SORT_DOWN				       		1
#define COLUMN1_SORT_UP			            10
#define COLUMN1_SORT_DOWN		            11
#define COLUMN2_SORT_UP			            20
#define COLUMN2_SORT_DOWN		            21
#define COLUMN6_SORT_UP			            60
#define COLUMN6_SORT_DOWN		            61

int	SortFlag_Jump_Column1;
int	SortFlag_Jump_Column2;
int	SortFlag_Jump_Column6;

void	JmptblCreatewindow(void);
int		JmptblSortFunction(t_jmplog *b1,t_jmplog *b2,int sort);
int		JmptblDrawFunction(char *s,char *mask,int *select,t_sortheader *ph,int column);
LRESULT CALLBACK JmptblWinproc(HWND hw,UINT msg,WPARAM wp,LPARAM lp);



//////////////////////////////////////////////////////////////////
//
//			Custom Range Table Window function Declarations
//
// Will hold start/end addresses for every analyzed procedure function
// so they can be selected as Included or Excluded ranges
//
//////////////////////////////////////////////////////////////////

// A sorted data structure composed of a t_sortheader
// followed by the data element(s) we want to maintain
typedef struct t_range {

	// t_sortheader	
	ulong	procstart;				// Address of proc start for range
	ulong	size;	    			// Size of addr element			(1)
	ulong	type;					// Type of data element, TY_xxx	(0)

	// Data elements
	int		index;					// Item count
	ulong	procend;				// Address of proc end for range
	char	symbolicname[512];		// Procedure symbolic name
} t_range;

t_table		rangetbl;				// rangetbl table
char		rangetblwinclass[32];	// Name of rangetbl window class


// Context menu items
#define ADD_AS_INCLUDED			       		1
#define ADD_AS_EXCLUDED			        	2
#define CLEAR_RANGE_TABLE			        3

// Sort flags
int	SortFlag_Range_Column1;
int	SortFlag_Range_Column2;


void	RangetblCreatewindow(void);
int		RangetblSortFunction(t_range *b1,t_range *b2,int sort);
int		RangetblDrawFunction(char *s,char *mask,int *select,t_sortheader *ph,int column);
void	UpdateRangeTable(ULONG baseaddress);
LRESULT CALLBACK RangetblWinproc(HWND hw,UINT msg,WPARAM wp,LPARAM lp);


//////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////

#pragma argsused

BOOL WINAPI DllMain(HINSTANCE hi,DWORD reason,LPVOID reserved) 
{
	if (reason==DLL_PROCESS_ATTACH)

		hinst=hi;
  	return 1;
};


extc int _export cdecl ODBG_Plugindata(char shortname[32])
{
	strcpy(shortname,"Conditional Branch Logger");
	return PLUGIN_VERSION;
};


extc int _export cdecl ODBG_Plugininit(int ollydbgversion, HWND hw, ulong *features)
{

	char szOllyIniPath[MAX_PATH];    
	char szOllyPluginPath[TEXTLEN];    
	
	//////////////////////////////////////
	
	if (ollydbgversion<PLUGIN_VERSION)
		return -1;

	hwmain = hw;

	Addtolist(0,0,"Conditional Branch Logger plugin ver 1.0 ");
	Addtolist(0,-1," by Blabberer, dELTA and Kayaker");


	///////////////////////////////////////////////////////////////////
	// Try to load the gui dll from the Ollydbg Plugin directory first.
	// If that fails then load from the default locations of
	// Ollydbg main directory, %PATH%, etc.
	///////////////////////////////////////////////////////////////////

	GetCurrentDirectory(sizeof(szOllyIniPath), szOllyIniPath);
	lstrcat(szOllyIniPath, "\\ollydbg.ini");
	GetPrivateProfileString("History", "Plugin path", NULL, szOllyPluginPath,
							sizeof(szOllyPluginPath), szOllyIniPath);

	lstrcat(szOllyPluginPath, "\\cbl_gui.dll");

	guidll = LoadLibrary(szOllyPluginPath);

	if(!guidll)
	{	
		guidll = LoadLibrary("cbl_gui.dll");

		if(!guidll)
		{
			BroadcastDllLoadFailure();
			return 0;
		}
	}


	// Get Export addresses

	pshowpsm = 			(PSHOW_PLUGIN_SETTINGS_MODAL)	GetProcAddress(guidll,"ShowPluginSettingsModal");
	pshowdialog =		(PSHOWWAITMESSAGEDIALOG)	    GetProcAddress(guidll,"ShowWaitMessageDialog");
	pwritestrtolog =	(PWRITE_STRINGTOLOG)		    GetProcAddress(guidll,"WriteStringToLog");
	pclosedialog = 		(PHIDEWAITMESSAGEDIALOG)	    GetProcAddress(guidll,"HideWaitMessageDialog");

	paddrange =			(PADDRANGE)						GetProcAddress(guidll,"AddRange");
	premoverange =		(PREMOVERANGE)					GetProcAddress(guidll,"RemoveRange");
	pgetcurrawsetptr =	(PGETCURRENTRAWSETTINGSPTR)		GetProcAddress(guidll,"GetCurrentRawSettingsPtr");
	psetcurrawsetptr =	(PSETCURRENTRAWSETTINGSPTR)		GetProcAddress(guidll,"SetCurrentRawSettingsPtr");

	GetSerializedSettings = (PGETSERIALIZEDSETTINGS)	GetProcAddress(guidll,"GetSerializedSettings");
	SetSerializedSettings = (PSETSERIALIZEDSETTINGS)	GetProcAddress(guidll,"SetSerializedSettings");
			
			
	if((pshowpsm) &&  (pwritestrtolog) && (pshowdialog) && (pclosedialog)
		&& paddrange && premoverange && pgetcurrawsetptr && psetcurrawsetptr
		&& GetSerializedSettings && SetSerializedSettings)
	{
		Addtolist(0,0,"cbl_gui.dll loaded and all addresses are retrieved");
		GUIDLLLOADED = TRUE;

	} else {

		Addtolist(0,1,"cbl_gui.dll loaded but addresses not retrieved");
		BroadcastDllLoadFailure();
		return 0;
	}


	//============================================================
	// Jump Table

	// Initialize descriptor of sorted data for our plugin custom window
	if (Createsorteddata(&jmptbl.data, "JumpTable",
		sizeof(t_jmplog), 10, (SORTFUNC *)JmptblSortFunction, NULL) != 0)
		return -1;
		
	// Register class of our plugin custom window
	if (Registerpluginclass(jmptblwinclass, "CBL_ICO", hinst, JmptblWinproc) < 0)
	{
		// If error, destroy sorted data and exit.
		Destroysorteddata(&jmptbl.data);
		return -1; 
	};

    // Restore custom window if it was visible when Ollydbg last shut down
	if (Plugingetvalue(VAL_RESTOREWINDOWPOS)!=0 &&
		Pluginreadintfromini(hinst,"Restore Conditional Branch Logger window",0)!=0)
		JmptblCreatewindow();


	//============================================================
	// Range Table
		
	// Initialize descriptor of sorted data for our plugin custom window
	if (Createsorteddata(&rangetbl.data, "RangeTable",
		sizeof(t_range), 10, (SORTFUNC *)RangetblSortFunction, NULL) != 0)
		return -1;
		
	// Register class of our plugin custom window
	if (Registerpluginclass(rangetblwinclass, "CBL_ICO", hinst, RangetblWinproc) < 0)
	{
		// If error, destroy sorted data and exit.
		Destroysorteddata(&rangetbl.data);
		return -1; 
	};

	// Create Range Table window.
	RangetblCreatewindow();
	

	//============================================================
	
	// Create a standard hourglass cursor to be enabled on long routines
	hCursorWait = LoadCursor(NULL, IDC_WAIT); 

 
	///////////////////////////////////////////////////////////////////
    // Create a toolbar button to reenable custom window if it has been hidden/closed
	///////////////////////////////////////////////////////////////////
    
    // Divert original Ollydbg window handler in order to handle our toolbar button    
    OldWndProc = (WNDPROC)SetWindowLong(hwmain, GWL_WNDPROC, (long)ToolbarButtonWinproc);

    // Create the button and add a bitmap image to it
    // Another plugin, Olly ToolBar Manager, also adds a toolbar button at offset 0x2AE,
    // we'll park next to it to avoid conflict
    
	hToolbarButton = CreateWindowEx(0, "Button", NULL,
	    WS_CHILD | WS_VISIBLE | BS_BITMAP | BS_PUSHBUTTON,
        0x2AE + 20, // int x,
        2,	        // int y,
        18,         // int nWidth,
        18,         // int nHeight,
        hwmain, NULL, hinst, NULL);
    
   
	hToolbarBitmap = (HBITMAP)LoadImage(hinst, "CBL_BMP", IMAGE_BITMAP, 16, 16, LR_DEFAULTCOLOR);
	SendMessage(hToolbarButton, BM_SETIMAGE, IMAGE_BITMAP, (LPARAM)hToolbarBitmap);

	CreateTooltip (hToolbarButton);	// Create a tooltip message for the button
	
	
	///////////////////////////////////////////////////////////////////

	return 0;
};


extc void _export cdecl ODBG_Pluginaction(int origin,int action,void *item) 
{
	t_asmmodel		model[NMODELS];
	PCONFIG_DATA	ccdata = 0;
	PCONFIG_DATA	returnedfromdlg;
	t_table			valref;
	ULONG			currentCPUaddress;
	char			NameBuffer[32] = {0};
	char			RangeDescriptionBuffer[64] = {0}; 
	int				i;
	
	// Jump Table items
	t_jmplog *		jmplog; 	// pointer to sorted data structure defining the jumps we are logging

	t_dump *		cpudasm;
	t_memory *		memorydescriptor;
	t_module *		selectedmodule;
	t_table *		valmodules;
	t_sorted *		modsorteddata;


	//////////////////////////////////////

	if (origin==PM_MAIN)
	{
   		switch (action)
		{
			case 0:				// Configuration

			if(Getstatus() == STAT_NONE)
			{
				Flash("Start some process first");
				break;
			}

			if (!GUIDLLLOADED)
			{
				BroadcastDllLoadFailure();
				break;
			}
			
			
			// Get current address displayed in CPU window
			cpudasm = (t_dump *)Plugingetvalue(VAL_CPUDASM);
			currentCPUaddress = cpudasm->addr;
		
			
			///////////////////////////////////////////////////////////////////
			// If there are no previous ranges set then initialize Included range
			// with default range of entire code section of main module.
			///////////////////////////////////////////////////////////////////

			// Get current range settings as stored in the gui dll
			returnedfromdlg = pgetcurrawsetptr();

			if(((returnedfromdlg->noof_logged_ranges) == 0) && 
			   ((returnedfromdlg->noof_excluded_ranges) ==0))
			{
				
				// Find the name of the module
				selectedmodule = Findmodule(cpudasm->base);

				if(selectedmodule)
				{
					_snprintf(RangeDescriptionBuffer, SHORTLEN, selectedmodule->name);
					lstrcat(RangeDescriptionBuffer, "  .TEXT section");

				}

				// Add to Included range
				paddrange(returnedfromdlg,
					cpudasm->base, (cpudasm->base+cpudasm->size),
					RangeDescriptionBuffer, FALSE, TRUE);
			}	

			
			// Get current range settings as stored in the gui dll
			returnedfromdlg = pgetcurrawsetptr();

			// Update internal PCONFIG_DATA structure
			ccdata = returnedfromdlg;
			

			///////////////////////////////////////////////////////////////////
			// Display gui dll which manages current range settings,
			// jump types, log file, saved settings, etc.
			// When this gui window is closed, processing will begin.
			///////////////////////////////////////////////////////////////////

			if((pshowpsm(0, &ccdata)) != 1)
			{
				break;
			}

	
			// Display "Please Wait" dialog
			pshowdialog(hwmain);
			SetCursor(hCursorWait);

			// Create Jump Table window.
			JmptblCreatewindow();
			
			
			START_CODE_EXECUTION_TIMER();
			

			///////////////////////////////////////////////////////////////////
			// Begin main initialization routines
			///////////////////////////////////////////////////////////////////

			memset(&model, 0, sizeof(model));

			// If user selected "Log all conditional branch types"			
			if(ccdata->log_all_branch)
			{
				// Set up model for "JCC CONST" conditional jump types
				// plus the addition of JCXZ / JECXZ
				Setasmmodel_JCC(model);

			
			// If user selected "Log only selected branch types"			
			} else {

				// Set up model for individual Jxx conditional jump types
				Setasmmodel_Jxx(model, ccdata);			
			}


			// Activate the conditional logging breakpoints for the ranges defined
			SetLoggingBreakpoints(model, ccdata);	


			///////////////////////////////////////////////////////////////////
			// All the breakpoint ranges have now been initialized.
			// Now we just do a little bit of housekeeping with regards to
			// setting up the Jump Table window which will allow further control
			// of the breakpoints and display the logging results.
			///////////////////////////////////////////////////////////////////

        	// Close the default OllyDbg Reference window which was
			// created by the Findallcommands function
        	valref = *(t_table *)Plugingetvalue(VAL_REFERENCES);
	       	SendMessage(valref.hw, WM_CLOSE, 0, 0); 	


			// Set/Reset sort flags
			SortFlag_Jump_Column1 = SORT_UP;
			SortFlag_Jump_Column2 = SORT_UP;
			SortFlag_Jump_Column6 = SORT_UP;	
			
			// Sort table by address before displaying
			Sortsorteddata(&jmptbl.data, COLUMN2_SORT_UP);
		
            /////////////////////////////////////////////

			jmplog = (t_jmplog*)jmptbl.data.data;
			
			for(i=0; i < jmptbl.data.n; i++)
			{
				jmplog->index = i+1;	// Initialize an Item number for each entry
				jmplog++;
			}

            /////////////////////////////////////////////

			STOP_CODE_EXECUTION_TIMER();

			
			// Restore CPU window to where it was before
			Setdisasm(currentCPUaddress,0,CPU_ASMCENTER|CPU_ASMFOCUS);

			// Close "Please Wait" dialog
			pclosedialog();
			

			break;
						
            /////////////////////////////////////////////


			case 1:				// Set Ranges By Procedure
				
			if(Getstatus() == STAT_NONE)
			{
				Flash("Start some process first");
				break;
			}

			if (!GUIDLLLOADED)
			{
				BroadcastDllLoadFailure();
				break;
			}


			///////////////////////////////////////////////////////////////////
			// Add to the Range Table all functions analyzed by OllyDbg
			// for the current module section in the CPU window.
			///////////////////////////////////////////////////////////////////

			cpudasm = (t_dump *)Plugingetvalue(VAL_CPUDASM);

			UpdateRangeTable(cpudasm->base);
			// Display Range Table window
			Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");

			break;
			
            /////////////////////////////////////////////

			case 2:

				HelpWindow();
				
				break;
			
			default:
			break;

		};	// end switch(action)
	}


	//====================================================================


	if (origin == PM_DISASM)		// CPU window menu items
	{

		// ODBG_Pluginaction passes a pointer to selected
		// t_dump structure in *item parameter
		cpudasm = (t_dump*) item;
		
		// Get current range settings as stored in the gui dll
		returnedfromdlg =  pgetcurrawsetptr();

		// Find the name of the module
		selectedmodule = Findmodule(cpudasm->base);

		if(selectedmodule)
		{
			// Analyze module if it hasn't already been marked as such
			if(!selectedmodule->codedec)
			{
				Analysecode(selectedmodule);
			}

			_snprintf(RangeDescriptionBuffer, SHORTLEN, selectedmodule->name);
			lstrcat(RangeDescriptionBuffer, "  CPU Window Selection");

		}		


		switch(action)
		{
		
			case 0:		// Add selection as Included range
			
				paddrange(returnedfromdlg,
					cpudasm->sel0, cpudasm->sel1,
					RangeDescriptionBuffer, FALSE, TRUE);
				
				break;

			case 1:		// Add selection as Excluded range

				paddrange(returnedfromdlg,
					cpudasm->sel0, cpudasm->sel1,
					RangeDescriptionBuffer, TRUE, TRUE);
				
				break;


			default:
				break;

		};		// end switch(action)
	}


	//====================================================================
	//====================================================================

			
	if (origin == PM_MEMORY)		// Memory window menu items
	{

		// ODBG_Pluginaction passes a pointer to selected
		// t_memory structure in *item parameter
		memorydescriptor = (t_memory*) item;

		// Find the name of the associated module, if defined
		selectedmodule = Findmodule(memorydescriptor->base);


		switch(action)
		{

			case 0:		// Add section as Included range


				// =============================================================
				// CAUTION: There are no restrictions as to what can be added as an
				// Included range, so you could set breakpoints in the PE header, etc.
				// However this does allow setting breakpoints in non-standard
				// sections such as memory mapped executable regions.
				// =============================================================

	
				// Get current range settings as stored in the gui dll
				returnedfromdlg = pgetcurrawsetptr();

				if(selectedmodule)
				{

					// Analyze module if it hasn't already been marked as such
					if(!selectedmodule->codedec)
					{
						Analysecode(selectedmodule);
					}					
					
					_snprintf(RangeDescriptionBuffer, SHORTLEN, selectedmodule->name);
					lstrcat(RangeDescriptionBuffer, "  ");

				}

				// Check if section is a regular .code section
				if (memorydescriptor->type & TY_CODE)
				{
					// Add the section name
					_snprintf(NameBuffer, SHORTLEN, memorydescriptor->sect);
					CharUpper(NameBuffer);
					lstrcat(NameBuffer, " section");

				} else {
					
					if(MessageBox(hwmain,
						"This appears to be a non-standard code section.\n"
		  				"Are you sure you want to add it as an Included range?\n",
		  				"Conditional Branch Logger",
		  				MB_OKCANCEL) == IDOK)
					{

						// If there is a defined section name
						if( *(BYTE*)memorydescriptor->sect)
						{
							// Add the section name
							_snprintf(NameBuffer, SHORTLEN, memorydescriptor->sect);
							CharUpper(NameBuffer);
							lstrcat(NameBuffer, " section");
						
						} else {

							lstrcat(NameBuffer, "Undefined section");
						}

					} else {

						break;
					}
				
				}

				lstrcat(RangeDescriptionBuffer, NameBuffer);
				paddrange(returnedfromdlg,
					memorydescriptor->base,
					(memorydescriptor->base+memorydescriptor->size),
					RangeDescriptionBuffer, FALSE, TRUE);
				
				break;



			case 1:		// Set ranges by procedure for this module section

				if(selectedmodule)
				{
					// Analyze module if it hasn't already been marked as such
					if(!selectedmodule->codedec)
					{
						Analysecode(selectedmodule);
					}
					
					// Use base address given in t_memory structure in order
					// to view procedures for the *selected* section.
					// i.e. note differences between ole32!text and ole32!orpc sections

					UpdateRangeTable(memorydescriptor->base);

					// Display Range Table window
					Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");

				} else {

					MessageBox(hwmain,
						"There is no analyzable module associated with this address.\n",
		  				"Conditional Branch Logger",
		  				MB_OK|MB_ICONINFORMATION);
				}

				break;

				
			
			case 2:		// Analyze module

				if(selectedmodule)
				{
					Analysecode(selectedmodule);

				} else {

					MessageBox(hwmain,
						"There is no analyzable module associated with this address.\n",
		  				"Conditional Branch Logger",
		  				MB_OK|MB_ICONINFORMATION);
				}
				
				break;


			default:
				break;

		};		// end switch(action)			
	}


	//====================================================================
	//====================================================================


	if (origin == PM_MODULES)		// Modules window menu items
	{

		// ODBG_Pluginaction passes a pointer to selected
		// t_module structure in *item parameter

		// Pointer to module descriptor of selected module
		selectedmodule = (t_module*)item;

		// Pointer to sorted table of loaded modules
		valmodules = (t_table *) Plugingetvalue(VAL_MODULES);
		modsorteddata = (t_sorted *)&(valmodules->data);
		
		// Get current range settings as stored in the gui dll
		returnedfromdlg = pgetcurrawsetptr();
				
		SetCursor(hCursorWait);
		

		switch(action)
		{
	
			case 0:		// Add code section as Included range for Selected module

				if(selectedmodule)
				{
					// Analyze module if it hasn't already been marked as such
					if(!selectedmodule->codedec)
					{
						Analysecode(selectedmodule);
					}

					// Add code section as Included range
					_snprintf(RangeDescriptionBuffer, SHORTLEN, selectedmodule->name);
					lstrcat(RangeDescriptionBuffer, "  .TEXT section");
					paddrange(returnedfromdlg,
						selectedmodule->codebase,
						(selectedmodule->codebase+selectedmodule->codesize),
						RangeDescriptionBuffer, FALSE, TRUE);
				}
				
				break;



			case 1:		// Add code section as Included range for Non-System modules
				
				for(i=0; i < modsorteddata->n; i++)
				{
					Selectandscroll(valmodules, i, 2);
					InvalidateRect(valmodules->hw, NULL, FALSE);
					UpdateWindow(valmodules->hw);
					selectedmodule = (t_module *)Getsortedbyselection((t_sorted *)modsorteddata->name, i);

					// Check for non-system module
					if (!(selectedmodule->issystemdll))
					{
						// Analyze module if it hasn't already been marked as such
						if(!selectedmodule->codedec)
						{
							Analysecode(selectedmodule);
						}

						// Add code section as Included range
						_snprintf(RangeDescriptionBuffer, SHORTLEN, selectedmodule->name);
						lstrcat(RangeDescriptionBuffer, "  .TEXT section");
						paddrange(returnedfromdlg,
							selectedmodule->codebase,
							(selectedmodule->codebase+selectedmodule->codesize),
							RangeDescriptionBuffer, FALSE, TRUE);
					}
				}

				break;



			case 2:		// Add code section as Included range for All modules
				
				for(i=0; i < modsorteddata->n; i++)
				{
					Selectandscroll(valmodules, i, 2);
					InvalidateRect(valmodules->hw, NULL, FALSE);
					UpdateWindow(valmodules->hw);
					selectedmodule = (t_module *)Getsortedbyselection((t_sorted *)modsorteddata->name, i);

					if (selectedmodule)
					{
						// Analyze module if it hasn't already been marked as such
						if(!selectedmodule->codedec)
						{
							Analysecode(selectedmodule);
						}
						
						// Add code section as Included range
						_snprintf(RangeDescriptionBuffer, SHORTLEN, selectedmodule->name);
						lstrcat(RangeDescriptionBuffer, "  .TEXT section");
						paddrange(returnedfromdlg,
							selectedmodule->codebase,
							(selectedmodule->codebase+selectedmodule->codesize),
							RangeDescriptionBuffer, FALSE, TRUE);
					}
				}
				
				break;


			//====================================================================


			case 3:		// Set ranges by procedure for Selected module

				if(selectedmodule)
				{
					// Analyze module if it hasn't already been marked as such
					if(!selectedmodule->codedec)
					{
						Analysecode(selectedmodule);
					}
					
					UpdateRangeTable(selectedmodule->codebase);

					// Display Range Table window
					Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");
				}

				break;



			case 4:		// Set ranges by procedure for Non-System modules

				for(i=0; i < modsorteddata->n; i++)
				{
					Selectandscroll(valmodules, i, 2);
					InvalidateRect(valmodules->hw, NULL, FALSE);
					UpdateWindow(valmodules->hw);
					selectedmodule = (t_module *)Getsortedbyselection((t_sorted *)modsorteddata->name, i);

					// Check for non-system module
					if (!(selectedmodule->issystemdll))
					{
						// Analyze module if it hasn't already been marked as such
						if(!selectedmodule->codedec)
						{
							Analysecode(selectedmodule);
						}

						UpdateRangeTable(selectedmodule->codebase);
					}
				}

				// Display Range Table window
				Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");
				
				break;



			case 5:		// Set ranges by procedure for All modules

				for(i=0; i < modsorteddata->n; i++)
				{
					Selectandscroll(valmodules, i, 2);
					InvalidateRect(valmodules->hw, NULL, FALSE);
					UpdateWindow(valmodules->hw);
					selectedmodule = (t_module *)Getsortedbyselection((t_sorted *)modsorteddata->name, i);

					if (selectedmodule)
					{
						// Analyze module if it hasn't already been marked as such
						if(!selectedmodule->codedec)
						{
							Analysecode(selectedmodule);
						}

						UpdateRangeTable(selectedmodule->codebase);
					}
				}
				
				// Display Range Table window
				Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");
				
				break;
				

			//====================================================================


			case 6:		// Analyze module

				if (selectedmodule)
				{
					Analysecode(selectedmodule);
				}

				break;


			default:

				break;

		};		// end switch(action)			
	}
}

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//						SetLoggingBreakpoints
//
// Set up the conditional logging breakpoints from the Included and Excluded
// ranges defined by the user. These may include addresses from the main exe
// as well as from system or non-system dlls.  
// For proper functioning the modules must first be analyzed by OllyDbg,
// this is taken care of automatically in most cases if the ranges have been
// defined through the plugin menu options.
//
//////////////////////////////////////////////////////////////////////////

void SetLoggingBreakpoints(t_asmmodel model[NMODELS], PCONFIG_DATA ccdata)
{
	t_table 		valref;
	PLOGGING_RANGE	temprange;
	PLOGGING_RANGE	temprange2;
	t_ref *			refdata;
	t_ref			realref;
	int 			noofent;
	PVOID			paddress;
	int				i, j, k;
	t_dump *		cpudasm;
	int				result;

	// Jump Table items
	t_jmplog		jmplog = {0};	// sorted data structure defining the jumps we are logging

	//////////////////////////////////////

	// Initialize common elements
	jmplog.size=1;
	jmplog.type=0;
			

	//////////////////////////////////////
	// Handle Included Ranges
	//////////////////////////////////////
	
	for (i = 0; i < (int)ccdata->noof_logged_ranges; i++)
	{
		temprange = *(PLOGGING_RANGE *)((PCHAR)(ccdata->logged_ranges) + (i * sizeof(DWORD)));

		if(temprange->enabled)
		{

			// Set the CPU window to display the module containing this Included range
			// so the Plugingetvalue call returns the correct t_dump structure
			Setdisasm(temprange->r_start, 0, CPU_ASMCENTER);
			
			cpudasm = (t_dump *)Plugingetvalue(VAL_CPUDASM);

			result = Findallcommands(cpudasm, model, 0, "");
			if(result <= 0)
			{
				break;	
			}
			
			valref = *(t_table *)Plugingetvalue(VAL_REFERENCES);

		
			//////////////////////////////////////
			// Handle Excluded Ranges
			//////////////////////////////////////
			for (j = 0; j < (int)ccdata->noof_excluded_ranges; j++)
			{
				temprange2 = *(PLOGGING_RANGE *)((PCHAR)(ccdata->excluded_ranges) + (j * sizeof(DWORD)));

				if(temprange2->enabled)
				{
					if( (temprange2->r_start >= cpudasm->base) &&
						(temprange2->r_end <= (cpudasm->base+cpudasm->size)))
					{
						// Remove excluded ranges from global VAL_REFERENCES data table
						Deletesorteddatarange((&valref.data), temprange2->r_start, temprange2->r_end);

						// Remove any existing entries from our custom sorted data table
						// that fall within the excluded range of addresses
						paddress = Findsorteddatarange(&jmptbl.data, temprange2->r_start, temprange2->r_end);
						
						while (paddress)
						{
							// Remove NM_PLUGCMD status for this breakpoint address
							Deletenamerange(*(ULONG*)paddress, (*(ULONG*)paddress)+1, NM_PLUGCMD);
							// Delete the associated breakpoint
							Deletebreakpoints(*(ULONG*)paddress, (*(ULONG*)paddress)+1, 1);
							// Delete the entry
							Deletesorteddata(&jmptbl.data, *(ULONG*)paddress);

							paddress = Findsorteddatarange(&jmptbl.data, temprange2->r_start, temprange2->r_end);
						}
					}
				}		
			}
					

			noofent = valref.data.n;
			refdata	= (t_ref *)valref.data.data;

		
			for(k=0; k < noofent; k++)
			{
				realref = *refdata;

				if((realref.addr > temprange->r_start) && (realref.addr < temprange->r_end))
				{
					Quickinsertname(realref.addr,NM_PLUGCMD,"!Checkjump");
					Setbreakpoint(realref.addr,TY_IMPORT | TY_KEEPCOND | TY_ACTIVE,0);

					// Jump Table: initialize items in our sorted data structure
					// add logged jump 
					jmplog.addr = realref.addr;
					// reset any previous values
					jmplog.count_jump_taken = 0;
					jmplog.count_jump_not_taken = 0;
					
					Addsorteddata(&jmptbl.data, &jmplog);
				}

				refdata++;
			}
		}		
	}

	Mergequicknames();

}

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//						Setasmmodel_JCC
//
// Fill in a t_asmmodel structure that will be used to define the type of
// breakpoints we want to set, in this case on JCC instructions.
//
// This function emulates the OllyDbg Getasmfindmodel procedure that normally
// displays a dialog box allowing user input, which in this case would be "JCC CONST"
//

/*
Type of structure that keeps assembler search model.

typedef struct t_asmmodel {   // Model to search for assembler command
  char      code[MAXCMDSIZE]; // Binary code
  char      mask[MAXCMDSIZE]; // Mask for binary code (0: bit ignored)
  int       length;           // Length of code, bytes (0: empty)
  int       jmpsize;          // Offset size if relative jump
  int       jmpoffset;        // Offset relative to IP
  int       jmppos;           // Position of jump offset in command
} t_asmmodel;

Members:

code - binary code of the command. Only bits that have 1's set in corresponding mask bits are significant;
mask - comparison mask. Search routine ignores all code bits where mask is set to 0;
length - length of code and mask, bytes. If length is 0, search model is empty or invalid;
jmpsize - if nonzero, command is a relative jump and jmpsize is a size of offset in bytes;
jmpoffset - if jmpsize is nonzero, jump offset relative to address of the following command, otherwise undefined;
jmppos - if jmpsize is nonzero, position of the first byte of the offset in code, otherwise undefined.
*/
//////////////////////////////////////////////////////////////////////////

void Setasmmodel_JCC(t_asmmodel model[NMODELS]){

	memcpy(model[0].code, &jcccode1, 1);
	memcpy(model[0].mask, &jccmask1, 1);
	model[0].length = 2;
	model[0].jmpsize = 1;
	model[0].jmpoffset = -2;
	model[0].jmppos = 1;

	memcpy(model[1].code, &jcccode2, 2);
	memcpy(model[1].mask, &jccmask2, 2);
	model[1].length = 6;
	model[1].jmpsize = 4;
	model[1].jmpoffset = -6;
	model[1].jmppos = 2;


	// Include JCXZ / JECXZ to the "JCC CONST" selection

	memcpy(model[2].code, &onebyte_opcode[JCXZ], 1);   // JCXZ
	memcpy(model[2].mask, &onebyte_mask, 1);
	model[2].length = 2;
	model[2].jmpsize = 1;
	model[2].jmpoffset = -2;
	model[2].jmppos = 1;

	memcpy(model[3].code, &twobyte_opcode[JCXZ*2], 2);   // JECXZ
	memcpy(model[3].mask, &twobyte_mask, 2);
	model[3].length = 3;
	model[3].jmpsize = 1;
	model[3].jmpoffset = -3;
	model[3].jmppos = 2;		
}

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//						Setasmmodel_Jxx
//
// Fill in a t_asmmodel structure for each individual conditional jump
// instruction as selected by the user
//
//////////////////////////////////////////////////////////////////////////

void Setasmmodel_Jxx(t_asmmodel model[NMODELS], PCONFIG_DATA ccdata)
{
	int i;

	//////////////////////////////////////

	for (i = 0,jmptype = 0; i < sizeof(BRANCHES); i++, jmptype++)
	{
		if(*(char *)((char *)(ccdata->log_spec_branch) + i))
		{
			if (jmptype == JCXZ)
			{	
				memcpy(model[0].code, &onebyte_opcode[jmptype], 1);   // JCXZ
				memcpy(model[0].mask, &onebyte_mask, 1);
				model[0].length = 2;
				model[0].jmpsize = 1;
				model[0].jmpoffset = -2;
				model[0].jmppos = 1;

				memcpy(model[1].code, &twobyte_opcode[jmptype*2], 2);   // JECXZ
				memcpy(model[1].mask, &twobyte_mask, 2);
				model[1].length = 3;
				model[1].jmpsize = 1;
				model[1].jmpoffset = -3;
				model[1].jmppos = 2;		


			} else if (jmptype == LOOPxx)
			{
				memcpy(model[0].code, &loopxx_opcode[0], 1);	// LOOPNZ / LOOPNE
				memcpy(model[0].mask, &onebyte_mask, 1);
				model[0].length = 2;
				model[0].jmpsize = 1;
				model[0].jmpoffset = -2;
				model[0].jmppos = 1;

				memcpy(model[1].code, &loopxx_opcode[1], 1);	// LOOPZ / LOOPE
				memcpy(model[1].mask, &onebyte_mask, 1);
				model[1].length = 2;
				model[1].jmpsize = 1;
				model[1].jmpoffset = -2;
				model[1].jmppos = 1;

				memcpy(model[2].code, &loopxx_opcode[2], 1);	// LOOP
				memcpy(model[2].mask, &onebyte_mask, 1);
				model[2].length = 2;
				model[2].jmpsize = 1;
				model[2].jmpoffset = -2;
				model[2].jmppos = 1;
				
			} else
			{
				memcpy(model[0].code, &onebyte_opcode[jmptype], 1);
				memcpy(model[0].mask, &onebyte_mask, 1);
				model[0].length = 2;
				model[0].jmpsize = 1;
				model[0].jmpoffset = -2;
				model[0].jmppos = 1;

				memcpy(model[1].code, &twobyte_opcode[jmptype*2], 2);
				memcpy(model[1].mask, &twobyte_mask, 2);
				model[1].length = 6;
				model[1].jmpsize = 4;
				model[1].jmpoffset = -6;
				model[1].jmppos = 2;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//				Checkjump
//
// This function is called every time the conditional logging breakpoints
// we set previously are hit. From here we update the jump count and result
// as well as add the occurence to the log file.
//
//////////////////////////////////////////////////////////////////////////

int Checkjump(char *answer, ulong parm, t_reg *reg)
{
	int	result;
	t_reg	*moreg;
	char 	buf[16]	= {0};
	ulong	ip;
	ulong	flags;
	char 	resultbuff[200] = {0};
	
	// Jump Table items
	t_jmplog *		jmplog;	// pointer to sorted data structure defining the jumps we are logging

	//////////////////////////////////////

	moreg = reg;
	ip = moreg->ip;
	flags = moreg->flags;
	Readmemory(buf,ip,1,MM_RESILENT);
	result = Checkcondition(*buf, flags);

	if(result == 1)
	{
		_snprintf(resultbuff,199, "0x%08x\tJump Taken", ip);
		pwritestrtolog(resultbuff);

		// Log jump count and result of jump to sorted data structure
		jmplog = (t_jmplog*)Findsorteddata(&jmptbl.data, ip);
		jmplog->count_jump_taken++;

		// note the space at the start of the string
		// this will improve the column sorting efficiency
		strcpy(jmplog->last_jump_result, " Jump Taken");

	} else if (result == 0) {

		_snprintf(resultbuff,199, "0x%08x\tJump Not Taken", ip);
		pwritestrtolog(resultbuff);			

		// Log jump count and result of jump to sorted data structure
		jmplog = (t_jmplog*)Findsorteddata(&jmptbl.data, ip);
		jmplog->count_jump_not_taken++;

		strcpy(jmplog->last_jump_result, "Jump Not Taken");
	}

	Go(0,0,STEP_SAME,parm,1);
	return 0;
}

//////////////////////////////////////////////////////////////////////////


extc int _export cdecl ODBG_Pluginclose(void)
{

    // Write to ollydbg.ini whether our custom window was visible when Ollydbg was shut down
    Pluginwriteinttoini(hinst,"Restore Conditional Branch Logger window", jmptbl.hw != NULL);
	return 0;
};


extc void _export cdecl ODBG_Plugindestroy(void)
{

	// Clean up Jump Table constructs
	Unregisterpluginclass(jmptblwinclass);
	Destroysorteddata(&jmptbl.data);

	// Clean up Range Table constructs
	Unregisterpluginclass(rangetblwinclass);
	Destroysorteddata(&rangetbl.data);
};


extc void _export cdecl ODBG_Pluginreset(void)
{

	// Jump Table items
	t_jmplog *		jmplog;	// pointer to sorted data structure defining the jumps we are logging
	int i;

    /////////////////////////////////////////////

	// Reinitialize Jump Table entries

	// Sort table by index before beginning
	Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

	// Get first table entry	
	jmplog = (t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
	for(i=0; i < jmptbl.data.n; i++)
	{
		jmplog->count_jump_taken = 0;
		jmplog->count_jump_not_taken = 0;
		strcpy(jmplog->last_jump_result, "");
		jmplog++;
	}

};


extc int _export cdecl ODBG_Pluginmenu(int origin,char data[4096],void *item) 
{
	switch (origin) 
	{
		case PM_MAIN:			// NULL	Main window

			strcpy(data,"0 &Configuration | 1 &Set ranges by procedure | 2 &Help");
			return 1;


		case PM_DISASM:			// (t_dump *) CPU Disassembler

			if(GUIDLLLOADED)
			{
				strcpy(data,"&Conditional Branch Logger{ \
					0 Add selection as Included range, \
					1 Add selection as Excluded range}");
				return 1;
			}


		case PM_MEMORY:			// (t_memory *)	Memory window

			if(GUIDLLLOADED)
			{
				strcpy(data,"&Conditional Branch Logger{ \
					0 Add section as Included range, \
					1 Set ranges by procedure for this module section | \
					2 Analyze module}");
				return 1;
			}


		case PM_MODULES:		// (t_module *)	Modules window

			if(GUIDLLLOADED)
			{
				strcpy(data,"&Conditional Branch Logger{ \
					0 Add code section as Included range for Selected module, \
					1 Add code section as Included range for Non-System modules, \
					2 Add code section as Included range for All modules, |\
					3 Set ranges by procedure for Selected module, \
					4 Set ranges by procedure for Non-System modules, \
					5 Set ranges by procedure for All modules | \
					6 Analyze module, \
					}");

				return 1;
			}
		
		default: break;
  	};
	return 0;
}


extc int _export cdecl ODBG_Plugincmd(int reason, t_reg *reg, char *cmd)
{
	char answer[TEXTLEN];
	t_reg *myreg;
	
	if(reg == NULL)
	{
  		Addtolist(0,1,"t_reg is null");
  		return 0;
   	}

	myreg=reg;
	if(cmd==NULL || cmd[0]!='!' || cmd[1]=='\0')
    		return 0;
  	Checkjump(answer, 0, myreg);
  	return 1;
}




//========================================================================
//
//				Custom Window procedures
//
//========================================================================


//////////////////////////////////////////////////////////////////////////
//						JmptblCreatewindow
//
// Create Jump Table window
//////////////////////////////////////////////////////////////////////////

void JmptblCreatewindow(void)
{
	// Initialize table columns as defined in t_table structure
	if (jmptbl.bar.nbar==0)
	{    
		jmptbl.bar.name[0]="Item *";		// column name
		jmptbl.bar.defdx[0]=7;				// column width
		jmptbl.bar.mode[0]=BAR_BUTTON;		// column style

		jmptbl.bar.name[1]="Address *";
		jmptbl.bar.defdx[1]=10;
		jmptbl.bar.mode[1]=BAR_BUTTON;

		jmptbl.bar.name[2]="Disassembly";
		jmptbl.bar.defdx[2]=32;
		jmptbl.bar.mode[2]=BAR_NOSORT;

		jmptbl.bar.name[3]="Taken";
		jmptbl.bar.defdx[3]=10;
		jmptbl.bar.mode[3]=BAR_NOSORT;

		jmptbl.bar.name[4]="Not Taken";
		jmptbl.bar.defdx[4]=10;
		jmptbl.bar.mode[4]=BAR_NOSORT;
		
		jmptbl.bar.name[5]="Final Jump Result *";
		jmptbl.bar.defdx[5]=32;
		jmptbl.bar.mode[5]=BAR_BUTTON;

		jmptbl.bar.nbar=6;					// number of columns

		// table style
		jmptbl.mode=TABLE_COPYMENU|TABLE_APPMENU|TABLE_SAVEPOS|TABLE_ONTOP;

		jmptbl.drawfunc=JmptblDrawFunction;	// Paint function
	};

	// Display Jump Table window
	Quicktablewindow(&jmptbl, 15, 6, jmptblwinclass, "Conditional Branch Logger");
};


//////////////////////////////////////////////////////////////////////////
//						JmptblDrawFunction
//
// Paint function for Jump Table window
//////////////////////////////////////////////////////////////////////////

int JmptblDrawFunction(char *s,char *mask,int *select,t_sortheader *ph,int column)
{
	int n = 0;
	int breakpoint_type;

	//Disasm
	ulong cmdsize;
	char cmdbuffer[MAXCMDSIZE];
	t_disasm disasm;

	// sorted data structure defining the jumps we have logged
	t_jmplog *jmplog=(t_jmplog *)ph;

	//////////////////////////////////////

	column++;							// Change to a 1-based index for clarity

	if (column==1)						// Item count
	{
		n=sprintf(s,"%i",jmplog->index);
		*select=DRAW_MASK;
		memset(mask,DRAW_GRAY,8);
		mask[8]=DRAW_NORMAL;
	}

	if (column==2)						// Address
	{
		n=sprintf(s,"%08X",jmplog->addr);

		breakpoint_type = Getbreakpointtype(jmplog->addr);

		if (breakpoint_type & TY_ACTIVE)
		{
			if(Findname(jmplog->addr, NM_PLUGCMD, "!Checkjump"))
			{
				// active logging breakpoints highlighted with
				// Ollydbg "conditional breakpoint" background
				*select=DRAW_BREAK | DRAW_SELECT;
			} else {

				// active hard breakpoints highlighted with
				// Ollydbg "breakpoint" background
				*select=DRAW_BREAK;
			}
		}

		if (breakpoint_type & TY_DISABLED)
		{
			// disabled breakpoints highlighted gray
			*select=DRAW_SELECT;			
		}
	}
	
	if (column==3)						// Disassemby
	{
		cmdsize = Readcommand(jmplog->addr, cmdbuffer);			

		if (cmdsize == 0)
		{
			*select=DRAW_GRAY; return sprintf(s, "???");

		} else {

			Disasm((UCHAR*)cmdbuffer, cmdsize, jmplog->addr, NULL, &disasm, DISASM_CODE, 0);
			strcpy(s, disasm.result);
			n=strlen(s);			
		}	
	}

	if (column==4)						// Count Jump Taken
	{
		if (jmplog->count_jump_taken)
		{
			// Output number of times "Jump Taken" was recorded for this address	
			n=sprintf(s,"%01i",jmplog->count_jump_taken);
		}
	}	

	if (column==5)						// Count Jump Not Taken
	{
		if (jmplog->count_jump_not_taken)
		{		
			// Output number of times "Jump Not Taken" was recorded for this address	
			n=sprintf(s,"%01i",jmplog->count_jump_not_taken);
		}
	}	

	if (column==6)						// Final Jump Result
	{
		// Output the last jump result recorded for this address	
		n=sprintf(s,"%s",jmplog->last_jump_result);
	}

	return n;
};


//////////////////////////////////////////////////////////////////////////
//						JmptblSortFunction
//
// Sorting function for Jump Table window
//////////////////////////////////////////////////////////////////////////

int JmptblSortFunction(t_jmplog *lParam1, t_jmplog *lParam2, int sort)
{
	int result;
	int i=0;

	//////////////////////////////////////

	if (sort==COLUMN1_SORT_UP)				 // Sort by Index
	{
		if(lParam1->index > lParam2->index)
			return 1;
		else if(lParam1->index == lParam2->index)
			i = 0;
		else
			return -1;	
	}

	if (sort==COLUMN1_SORT_DOWN)
	{
		if(lParam1->index < lParam2->index)
			return 1;
		else if(lParam1->index == lParam2->index)
			i = 0;
		else
			return -1;	
	}
	
	if (sort==COLUMN2_SORT_UP)				 // Sort by Address
	{
		if(lParam1->addr > lParam2->addr)
			return 1;
		else if(lParam1->addr == lParam2->addr)
			i = 0;
		else
			return -1;	
	}

	if (sort==COLUMN2_SORT_DOWN)
	{
		if(lParam1->addr < lParam2->addr)
			return 1;
		else if(lParam1->addr == lParam2->addr)
			i = 0;
		else
			return -1;	
	}
	
	if (sort==COLUMN6_SORT_UP)				 // Sort by Final Jump Result
	{		

		result = strncmp(lParam1->last_jump_result, lParam2->last_jump_result, 1);
		
		if (result < 0)
			return -1;
		else if (result == 0)
			i = 0;
		else
			return 1;
	}

	if (sort==COLUMN6_SORT_DOWN)
	{

		result = strncmp(lParam1->last_jump_result, lParam2->last_jump_result, 1);

		if (result > 0)
			return -1;
		else if (result == 0)
			i = 0;
		else
			return 1;
	}


	// If elements are equal or sorting is by the first column, sort by index.
	if (i==0)
	{
		if(lParam1->index > lParam2->index)
			return 1;
		else
			return -1;	
	}

	return i;	
};


//////////////////////////////////////////////////////////////////////////
//						JmptblWinproc
//
// Handler function for Jump Table window
//////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK JmptblWinproc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam)
{
	HMENU menu;
	int menuselection;
	int i;
	int column;
	int shiftkey;
	int controlkey;
	int currenttablerow;	// index (row) of selected table item

	// sorted data structure defining the jumps we have logged
	t_jmplog *jmplog;
	t_jmplog *jmplog2;
	
	//////////////////////////////////////

	switch (msg)
	{	

		////////////////////////////////////////////////////////////////
		// Handle right click context menu items
		//
		////////////////////////////////////////////////////////////////
		
		case WM_USER_MENU:

			menu = CreatePopupMenu();

			// Get selected table entry	
			jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, jmptbl.data.selected);

			if (menu != NULL && jmplog != NULL)
			{

				AppendMenu(menu,MF_STRING,DISABLE_BP,"&Disable Breakpoint\tShift-D");
				AppendMenu(menu,MF_STRING,ENABLE_AS_CONDITIONAL_LOGGED_BP,"&Enable As Conditional Logged Breakpoint\tShift-C");
				AppendMenu(menu,MF_STRING,ENABLE_AS_ACTIVE_BP,"&Enable As Active Breakpoint\tShift-A");
				AppendMenu(menu,MF_STRING,REMOVE_BP,"&Remove Breakpoint\tShift-R");
				AppendMenu(menu,MF_SEPARATOR,0,NULL);

				AppendMenu(menu,MF_STRING,DISABLE_ALL_BPS,"&Disable All Breakpoints");
				AppendMenu(menu,MF_STRING,ENABLE_ALL_BPS,"&Enable All Breakpoints");
				AppendMenu(menu,MF_STRING,REMOVE_ALL_BPS,"&Remove All Breakpoints");
				AppendMenu(menu,MF_SEPARATOR,0,NULL);

				AppendMenu(menu,MF_STRING,DELETE_SELECTED_ENTRY,"&Delete Selected Entry\tDelete");
				AppendMenu(menu,MF_STRING,DELETE_ALL_ENTRIES,"&Delete All Entries");
				AppendMenu(menu,MF_SEPARATOR,0,NULL);

				AppendMenu(menu,MF_STRING,DELETE_ENTRIES_NOT_EXECUTED,"&Delete Entries Not Executed");
				AppendMenu(menu,MF_STRING,DELETE_ENTRIES_EXECUTED,"&Delete Entries Executed");
				AppendMenu(menu,MF_STRING,RESET_JUMP_RESULTS,"&Reset Jump Results");
				AppendMenu(menu,MF_SEPARATOR,0,NULL);

				AppendMenu(menu,MF_STRING,FOLLOW_IN_DISASM,"&Follow in Disassembler\tEnter or Double Click");
				AppendMenu(menu,MF_STRING,0,"&Sort Columns\tClick * Column Header");

			}
		
			// Returns menu selection
			menuselection = Tablefunction(&jmptbl, hwnd, WM_USER_MENU, 0, (LPARAM)menu);

			if (menu!=NULL) DestroyMenu(menu);

			switch (menuselection)
			{

				case DISABLE_BP:						// Disable Breakpoint

					Setbreakpoint(jmplog->addr, TY_IMPORT | TY_DISABLED | TY_KEEPCOND, 0);
					break;
				
				case ENABLE_AS_CONDITIONAL_LOGGED_BP:	// Enable As Conditional Logged Breakpoint

					// Maintain NM_PLUGCMD status for this breakpoint address
					Insertname(jmplog->addr, NM_PLUGCMD, "!Checkjump");
					Setbreakpoint(jmplog->addr, TY_IMPORT | TY_ACTIVE | TY_KEEPCOND, 0);
					break;

				case ENABLE_AS_ACTIVE_BP:				// Enable As Active Breakpoint

					// Remove NM_PLUGCMD status for this breakpoint address
					Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
					Setbreakpoint(jmplog->addr, TY_ACTIVE, 0);
					break;

				case REMOVE_BP:							// Remove Breakpoint

					// Remove NM_PLUGCMD status for this breakpoint address
					Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
					Deletebreakpoints(jmplog->addr, (jmplog->addr+1), 1);
					break;

					
				//======================================================

				
				case DISABLE_ALL_BPS:					// Disable All Breakpoints

					SetCursor(hCursorWait);

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
					for(i=0; i < jmptbl.data.n; i++)
					{
						Setbreakpoint(jmplog->addr, TY_IMPORT | TY_DISABLED | TY_KEEPCOND, 0);
						jmplog++;
					}
					
					break;				
				

				case ENABLE_ALL_BPS:					// Enable All Breakpoints

					SetCursor(hCursorWait);

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
					for(i=0; i < jmptbl.data.n; i++)
					{
						// Maintain NM_PLUGCMD status for breakpoint address range
						Quickinsertname(jmplog->addr,NM_PLUGCMD,"!Checkjump");
						
						Setbreakpoint(jmplog->addr, TY_IMPORT | TY_ACTIVE | TY_KEEPCOND,0);
						jmplog++;
					}

					Mergequicknames();

					break;	


				case REMOVE_ALL_BPS:					// Remove All Breakpoints

					SetCursor(hCursorWait);

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
					// Get last table entry	
					jmplog2=(t_jmplog *)Getsortedbyselection(&jmptbl.data, jmptbl.data.n-1);

  
					// Remove NM_PLUGCMD status for this breakpoint address
					Deletenamerange(jmplog->addr, (jmplog2->addr+1), NM_PLUGCMD);
					Deletebreakpoints(jmplog->addr, (jmplog2->addr+1), 1);

					break;


				//======================================================
				
				case DELETE_SELECTED_ENTRY:				// Delete Selected Entry
					
					// Get table row of selected entry
					currenttablerow = Findsorteddataindex(&jmptbl.data, jmplog->addr, jmplog->addr+1);

					// Remove NM_PLUGCMD status for this breakpoint address
					Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
					// Delete the associated breakpoint
					Deletebreakpoints(jmplog->addr, (jmplog->addr+1), 1);
					// Delete the entry
					Deletesorteddata(&jmptbl.data, jmplog->addr);
					// Select the next item in table
					Selectandscroll(&jmptbl, currenttablerow, 2);

					break;


				//======================================================

				
				case DELETE_ALL_ENTRIES:		// Delete All Entries 
					
					SetCursor(hCursorWait);

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
					// Get last table entry	
					jmplog2=(t_jmplog *)Getsortedbyselection(&jmptbl.data, jmptbl.data.n-1);
  
					// Remove NM_PLUGCMD status for range
					Deletenamerange(jmplog->addr, (jmplog2->addr+1), NM_PLUGCMD);
					// Remove breakpoints for range
					Deletebreakpoints(jmplog->addr, (jmplog2->addr+1), 1);
					// Delete entries
					Deletesorteddatarange(&jmptbl.data, jmplog->addr, (jmplog2->addr+1));					
					
					break;

				
				//======================================================
				

				case DELETE_ENTRIES_NOT_EXECUTED:		// Delete Entries Not Executed

					SetCursor(hCursorWait);

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
					
					for(i=0; i < jmptbl.data.n; i++)
					{

						// Set TY_CONFIRMED flag for entries we want to keep
						if (jmplog->count_jump_taken != 0 ||
							jmplog->count_jump_not_taken != 0)
						{
							jmplog->type = (jmplog->type | TY_CONFIRMED);

						} else {

							// Remove NM_PLUGCMD status for this breakpoint address
							Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
							// Delete the associated breakpoint
							Deletebreakpoints(jmplog->addr, (jmplog->addr+1), 1);

						}
						jmplog++;
					}

					// Delete all entries without TY_CONFIRMED flag set
					Deletenonconfirmedsorteddata(&jmptbl.data);
					
					break;

				//======================================================
				

				case DELETE_ENTRIES_EXECUTED:		// Delete Entries Executed

					SetCursor(hCursorWait);

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
					for(i=0; i < jmptbl.data.n; i++)
					{
						// Set TY_CONFIRMED flag for entries we want to keep
						if (jmplog->count_jump_taken == 0 &&
							jmplog->count_jump_not_taken == 0)
						{
							jmplog->type = (jmplog->type | TY_CONFIRMED);

						} else {

							// Remove NM_PLUGCMD status for this breakpoint address
							Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
							// Delete the associated breakpoint
							Deletebreakpoints(jmplog->addr, (jmplog->addr+1), 1);

						}
						jmplog++;
					}

					// Delete all entries without TY_CONFIRMED flag set
					Deletenonconfirmedsorteddata(&jmptbl.data);
					
					break;
					

				//======================================================

					
				case RESET_JUMP_RESULTS:		// Reset Jump Results

					// Sort table by index before beginning
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					// Get first table entry	
					jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, 0);
				
					for(i=0; i < jmptbl.data.n; i++)
					{
						jmplog->count_jump_taken = 0;
						jmplog->count_jump_not_taken = 0;
						strcpy(jmplog->last_jump_result, "");
						jmplog++;
					}
				
					break;


				//======================================================

				
				case FOLLOW_IN_DISASM:		// Follow address in Disassembler

					Setcpu(0,jmplog->addr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS);

					break;

			} // end switch (menuselection)


		// Update window
		InvalidateRect(hwnd, NULL, FALSE);

		return 0;



		////////////////////////////////////////////////////////////////
		// Handle key presses
		//
		// ENTER - Follow address in Disassembler
		////////////////////////////////////////////////////////////////

		case WM_KEYDOWN:
	
			shiftkey=GetKeyState(VK_SHIFT) & 0x8000;      
			controlkey=GetKeyState(VK_CONTROL) & 0x8000;

			// Get selected table entry	
			jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, jmptbl.data.selected);
				
			if (jmplog == 0)
			{
				Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);
				break;
			}


			// Follow address in Disassembler
			if (wparam==VK_RETURN && shiftkey==0 && controlkey==0) {

				Setcpu(0,jmplog->addr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS);
			

			// Delete Entry
			} else if (wparam==VK_DELETE && shiftkey==0 && controlkey==0) {
				
				// Get table row of selected entry
				currenttablerow = Findsorteddataindex(&jmptbl.data, jmplog->addr, jmplog->addr+1);

				// Remove NM_PLUGCMD status for this breakpoint address
				Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
				// Delete the associated breakpoint
				Deletebreakpoints(jmplog->addr, (jmplog->addr+1), 1);
				// Delete the entry
				Deletesorteddata(&jmptbl.data, jmplog->addr);
				// Select the next item in table
				Selectandscroll(&jmptbl, currenttablerow, 2);
				// Redraw the window
				InvalidateRect(hwnd, NULL, FALSE);					
			
			} else {

				Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);
			}

			break;	

			
		case WM_CHAR:
	
			shiftkey=GetKeyState(VK_SHIFT) & 0x8000;      
			controlkey=GetKeyState(VK_CONTROL) & 0x8000;

			// Get selected table entry	
			jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, jmptbl.data.selected);
				
			if (jmplog == 0)
			{
				Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);
				break;
			}


			// Disable Breakpoint - Shift-D
			if (wparam==0x44 && shiftkey==0x8000 && controlkey==0) {

				Setbreakpoint(jmplog->addr, TY_IMPORT | TY_DISABLED | TY_KEEPCOND, 0);

				
			// Enable As Conditional Logged Breakpoint - Shift-C
			} else if (wparam==0x43 && shiftkey==0x8000 && controlkey==0) {

				// Maintain NM_PLUGCMD status for this breakpoint address
				Insertname(jmplog->addr, NM_PLUGCMD, "!Checkjump");
				Setbreakpoint(jmplog->addr, TY_IMPORT | TY_ACTIVE | TY_KEEPCOND, 0);
				
				
			// Enable As Active Breakpoint - Shift-A
			} else if (wparam==0x41 && shiftkey==0x8000 && controlkey==0) {

				// Remove NM_PLUGCMD status for this breakpoint address
				Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
				Setbreakpoint(jmplog->addr, TY_ACTIVE, 0);
			
			
			// Remove Breakpoint - Shift-R
			} else if (wparam==0x52 && shiftkey==0x8000 && controlkey==0) {

				// Remove NM_PLUGCMD status for this breakpoint address
				Deletenamerange(jmplog->addr, jmplog->addr+1, NM_PLUGCMD);
				Deletebreakpoints(jmplog->addr, (jmplog->addr+1), 1);
		
			
			} else {

				Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);
			}

			InvalidateRect(hwnd, NULL, FALSE);
			break;	

			
		////////////////////////////////////////////////////////////////
		// Handle double click
		//
		// Follow address in Disassembler
		////////////////////////////////////////////////////////////////

		case WM_USER_DBLCLK:

			// Get selected table entry	
			jmplog=(t_jmplog *)Getsortedbyselection(&jmptbl.data, jmptbl.data.selected);

			if (jmplog != NULL)	
			{
				Setcpu(0,jmplog->addr,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS);
			}

			return 1;


		////////////////////////////////////////////////////////////////
		// Handle table header clicks
		//
		// Sort according to column
		////////////////////////////////////////////////////////////////

		case WM_USER_BAR:	// wParam contains column

			column = wparam + 1;	// Change to 1-based index for clarity
			SetCursor(hCursorWait);	

			if (column == 1)	// column 1 clicked
			{
				if(SortFlag_Jump_Column1 == SORT_UP)
				{
					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_UP);

					SortFlag_Jump_Column1 = SORT_DOWN;						

				}else{

					Sortsorteddata(&jmptbl.data, COLUMN1_SORT_DOWN);

					SortFlag_Jump_Column1 = SORT_UP;
				}				
			}

			if (column == 2)	// column 2 clicked
			{
				if(SortFlag_Jump_Column2 == SORT_UP)
				{
					Sortsorteddata(&jmptbl.data, COLUMN2_SORT_UP);

					SortFlag_Jump_Column2 = SORT_DOWN;						

				}else{

					Sortsorteddata(&jmptbl.data, COLUMN2_SORT_DOWN);

					SortFlag_Jump_Column2 = SORT_UP;
				}				
			}

			
			if (column == 6)	// column 6 clicked
			{				
				if(SortFlag_Jump_Column6 == SORT_UP)
				{
					Sortsorteddata(&jmptbl.data, COLUMN6_SORT_UP);

					SortFlag_Jump_Column6 = SORT_DOWN;						

				}else{

					Sortsorteddata(&jmptbl.data, COLUMN6_SORT_DOWN);

					SortFlag_Jump_Column6 = SORT_UP;
				}				
			}

			InvalidateRect(hwnd, NULL, FALSE);
			break;
								

		////////////////////////////////////////////////////////////////
		// Default Ollydbg messages
		//
		////////////////////////////////////////////////////////////////
			
			
		// Standard messages. You can process them, but - unless absolutely sure -
		// always pass them to Tablefunction().
		case WM_DESTROY:
		case WM_MOUSEMOVE:
		case WM_LBUTTONDOWN:
		case WM_LBUTTONDBLCLK:
		case WM_LBUTTONUP:
		case WM_RBUTTONDOWN:
		case WM_RBUTTONDBLCLK:
		case WM_HSCROLL:
		case WM_VSCROLL:
		case WM_TIMER:
		case WM_SYSKEYDOWN:
			Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);
			break;                           // Pass message to DefMDIChildProc()

		// Custom messages responsible for scrolling and selection. User-drawn
		// windows must process them, standard OllyDbg windows without extra
		// functionality pass them to Tablefunction().
		case WM_USER_SCR:
		case WM_USER_VABS:
		case WM_USER_VREL:
		case WM_USER_VBYTE:
		case WM_USER_STS:
		case WM_USER_CNTS:
		case WM_USER_CHGS:
			return Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);


		// WM_WINDOWPOSCHANGED is responsible for always-on-top MDI windows.
		case WM_WINDOWPOSCHANGED:
			return Tablefunction(&jmptbl,hwnd,msg,wparam,lparam);
		
		
		case WM_USER_CHALL:
		case WM_USER_CHMEM:
			// Something is changed, redraw window.
			InvalidateRect(hwnd, NULL, FALSE);
			return 0;

		case WM_PAINT:
			Painttable(hwnd, &jmptbl, JmptblDrawFunction);
			return 0;

		default: break;

	} 	// end switch (msg)

	return DefMDIChildProc(hwnd,msg,wparam,lparam);
};

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//						RangetblCreatewindow
//
// Create Range Table window
//////////////////////////////////////////////////////////////////////////

void RangetblCreatewindow(void)
{
	// Initialize table columns as defined in t_table structure
	if (rangetbl.bar.nbar==0)
	{    
		rangetbl.bar.name[0]="Item *";			// column name
		rangetbl.bar.defdx[0]=7;				// column width
		rangetbl.bar.mode[0]=BAR_BUTTON;		// column style

		rangetbl.bar.name[1]="Start *";
		rangetbl.bar.defdx[1]=10;
		rangetbl.bar.mode[1]=BAR_BUTTON;

		rangetbl.bar.name[2]="End";
		rangetbl.bar.defdx[2]=10;
		rangetbl.bar.mode[2]=BAR_NOSORT;

		rangetbl.bar.name[3]="Symbolic Name";
		rangetbl.bar.defdx[3]=64;
		rangetbl.bar.mode[3]=BAR_NOSORT;

		rangetbl.bar.nbar=4;					// number of columns

		// table style
		rangetbl.mode=TABLE_COPYMENU|TABLE_APPMENU|TABLE_SAVEPOS|TABLE_ONTOP;

		rangetbl.drawfunc=RangetblDrawFunction;	// Paint function
	};

	// Display Range Table window
//	Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");

};


//////////////////////////////////////////////////////////////////////////
//						RangetblDrawFunction
//
// Paint function for Range Table window
//////////////////////////////////////////////////////////////////////////

int RangetblDrawFunction(char *s,char *mask,int *select,t_sortheader *ph,int column)
{
	int n = 0;

	// sorted data structure defining the jumps we have logged
	t_range *range = (t_range *)ph;

	//////////////////////////////////////

	column++;								// Change to a 1-based index for clarity

	if (column==1)							// Item count
	{
		n=sprintf(s,"%i",range->index);
		*select=DRAW_MASK;
		memset(mask,DRAW_GRAY,8);
		mask[8]=DRAW_NORMAL;
	}

	if (column==2)							// Start
	{
		n=sprintf(s,"%08X",range->procstart);

	}

	if (column==3)							// End
	{
		n=sprintf(s,"%08X",range->procend);

	}
	
	if (column==4)							// Symbolic Name
	{
		// Output symbolic name	
		n=sprintf(s,"%s",range->symbolicname);
	}		

	return n;
};


//////////////////////////////////////////////////////////////////////////
//						RangetblSortFunction
//
// Sorting function for Range Table window
//////////////////////////////////////////////////////////////////////////

int RangetblSortFunction(t_range *lParam1, t_range *lParam2, int sort)
{
	int i=0;

	//////////////////////////////////////

	if (sort==COLUMN1_SORT_UP)				 // Sort by Index
	{
		if(lParam1->index > lParam2->index)
			return 1;
		else if(lParam1->index == lParam2->index)
			i = 0;
		else
			return -1;	
	}

	if (sort==COLUMN1_SORT_DOWN)
	{
		if(lParam1->index < lParam2->index)
			return 1;
		else if(lParam1->index == lParam2->index)
			i = 0;
		else
			return -1;	
	}
	
	if (sort==COLUMN2_SORT_UP)				 // Sort by Start Address
	{
		if(lParam1->procstart > lParam2->procstart)
			return 1;
		else if(lParam1->procstart == lParam2->procstart)
			i = 0;
		else
			return -1;	
	}

	if (sort==COLUMN2_SORT_DOWN)
	{
		if(lParam1->procstart < lParam2->procstart)
			return 1;
		else if(lParam1->procstart == lParam2->procstart)
			i = 0;
		else
			return -1;	
	}


	// If elements are equal or sorting is by the first column, sort by index.
	if (i==0)
	{
		if(lParam1->index > lParam2->index)
			return 1;
		else
			return -1;	
	}

	return i;	
};


//////////////////////////////////////////////////////////////////////////
//						RangetblWinproc
//
// Handler function for Range Table window
//////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK RangetblWinproc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam)
{
	HMENU menu;
	int menuselection;
	int column;
	int shiftkey;
	int controlkey;

	// sorted data structure defining the proc ranges
	t_range *range;

	// Included / Excluded ranges
	PCONFIG_DATA	returnedfromdlg;
	
	//////////////////////////////////////

	switch (msg)
	{	

		////////////////////////////////////////////////////////////////
		// Handle right click context menu items
		//
		////////////////////////////////////////////////////////////////
		
		case WM_USER_MENU:

			menu = CreatePopupMenu();

			// Get latest range settings
			returnedfromdlg =  pgetcurrawsetptr();
			
			// Get selected table entry	
			range = (t_range *)Getsortedbyselection(&rangetbl.data, rangetbl.data.selected);

			if (menu != NULL && range != NULL)
			{
				
				AppendMenu(menu,MF_STRING,ADD_AS_INCLUDED,"&Add to Included Range Set\tShift-I");
				AppendMenu(menu,MF_STRING,ADD_AS_EXCLUDED,"&Add to Excluded Range Set\tShift-E");
				AppendMenu(menu,MF_SEPARATOR,0,NULL);
				AppendMenu(menu,MF_STRING,FOLLOW_IN_DISASM,"&Follow in Disassembler\tEnter or Double Click");
				AppendMenu(menu,MF_STRING,0,"&Sort Columns\tClick * Column Header");
				AppendMenu(menu,MF_SEPARATOR,0,NULL);
				AppendMenu(menu,MF_STRING,CLEAR_RANGE_TABLE,"&Clear Table");
			}
		
			// Returns menu selection
			menuselection = Tablefunction(&rangetbl, hwnd, WM_USER_MENU, 0, (LPARAM)menu);

			if (menu!=NULL) DestroyMenu(menu);

		
			switch (menuselection)
			{
		
				case ADD_AS_INCLUDED:						// Add as Included Range

					paddrange(returnedfromdlg,
						range->procstart, range->procend,
						range->symbolicname, FALSE, TRUE);
					break;
				
		
				case ADD_AS_EXCLUDED:						// Add as Excluded Range

					paddrange(returnedfromdlg,
						range->procstart, range->procend,
						range->symbolicname, TRUE, TRUE);
					break;				
				
				case CLEAR_RANGE_TABLE:						// Clear Table

					// Delete all entries without TY_CONFIRMED flag set
					Deletenonconfirmedsorteddata(&rangetbl.data);
					break;				

					
				//======================================================

					
				case FOLLOW_IN_DISASM:		// Follow address in Disassembler

					Setcpu(0,range->procstart,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS);

					break;

			} // end switch (menuselection)


		// Update window
		InvalidateRect(hwnd, NULL, FALSE);

		return 0;



		////////////////////////////////////////////////////////////////
		// Handle key presses
		//
		// ENTER - Follow address in Disassembler
		////////////////////////////////////////////////////////////////

		case WM_KEYDOWN:
	
			shiftkey=GetKeyState(VK_SHIFT) & 0x8000;      
			controlkey=GetKeyState(VK_CONTROL) & 0x8000;

			// Get selected table entry	
			range=(t_range *)Getsortedbyselection(&rangetbl.data, rangetbl.data.selected);
				
			if (range == 0)
			{
				Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);
				break;
			}


			// Follow address in Disassembler
			if (wparam==VK_RETURN && shiftkey==0 && controlkey==0) {

				Setcpu(0,range->procstart,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS);
			

			} else {

				Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);
			}

			break;	

			
		case WM_CHAR:
	
			shiftkey=GetKeyState(VK_SHIFT) & 0x8000;      
			controlkey=GetKeyState(VK_CONTROL) & 0x8000;

			// Get latest range settings
			returnedfromdlg =  pgetcurrawsetptr();
			
			// Get selected table entry	
			range=(t_range *)Getsortedbyselection(&rangetbl.data, rangetbl.data.selected);

			if (range == 0)
			{
				Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);
				break;
			}

			
			// Add as Included Range - Shift-I
			if (wparam==0x49 && shiftkey==0x8000 && controlkey==0) {

				paddrange(returnedfromdlg,
					range->procstart, range->procend,
					range->symbolicname, FALSE, TRUE);

				
			// Add as Included Range - Shift-E
			} else if (wparam==0x45 && shiftkey==0x8000 && controlkey==0) {

				paddrange(returnedfromdlg,
					range->procstart, range->procend,
					range->symbolicname, TRUE, TRUE);
		
			
			} else {

				Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);
			}

			InvalidateRect(hwnd, NULL, FALSE);
			break;	

			
		////////////////////////////////////////////////////////////////
		// Handle double click
		//
		// Follow address in Disassembler
		////////////////////////////////////////////////////////////////

		case WM_USER_DBLCLK:

			// Get selected table entry	
			range=(t_range *)Getsortedbyselection(&rangetbl.data, rangetbl.data.selected);

			if (range != NULL)	
			{
				Setcpu(0,range->procstart,0,0,CPU_ASMHIST|CPU_ASMCENTER|CPU_ASMFOCUS);
			}

			return 1;

					
		////////////////////////////////////////////////////////////////
		// Handle table header clicks
		//
		// Sort according to column
		////////////////////////////////////////////////////////////////

		case WM_USER_BAR:	// wParam contains column

			column = wparam + 1;	// Change to 1-based index for clarity

			if (column == 1)	// column 1 clicked
			{
				if(SortFlag_Range_Column1 == SORT_UP)
				{
					Sortsorteddata(&rangetbl.data, COLUMN1_SORT_UP);

					SortFlag_Range_Column1 = SORT_DOWN;						

				}else{

					Sortsorteddata(&rangetbl.data, COLUMN1_SORT_DOWN);

					SortFlag_Range_Column1 = SORT_UP;
				}				
			}

			if (column == 2)	// column 2 clicked
			{
				if(SortFlag_Range_Column2 == SORT_UP)
				{
					Sortsorteddata(&rangetbl.data, COLUMN2_SORT_UP);

					SortFlag_Range_Column2 = SORT_DOWN;						

				}else{

					Sortsorteddata(&rangetbl.data, COLUMN2_SORT_DOWN);

					SortFlag_Range_Column2 = SORT_UP;
				}				
			}
			

			InvalidateRect(hwnd, NULL, FALSE);
			break;
			

		////////////////////////////////////////////////////////////////
		// Default Ollydbg messages
		//
		////////////////////////////////////////////////////////////////
			
			
		// Standard messages. You can process them, but - unless absolutely sure -
		// always pass them to Tablefunction().
		case WM_DESTROY:
		case WM_MOUSEMOVE:
		case WM_LBUTTONDOWN:
		case WM_LBUTTONDBLCLK:
		case WM_LBUTTONUP:
		case WM_RBUTTONDOWN:
		case WM_RBUTTONDBLCLK:
		case WM_HSCROLL:
		case WM_VSCROLL:
		case WM_TIMER:
		case WM_SYSKEYDOWN:
			Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);
			break;                           // Pass message to DefMDIChildProc()

		// Custom messages responsible for scrolling and selection. User-drawn
		// windows must process them, standard OllyDbg windows without extra
		// functionality pass them to Tablefunction().
		case WM_USER_SCR:
		case WM_USER_VABS:
		case WM_USER_VREL:
		case WM_USER_VBYTE:
		case WM_USER_STS:
		case WM_USER_CNTS:
		case WM_USER_CHGS:
			return Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);


		// WM_WINDOWPOSCHANGED is responsible for always-on-top MDI windows.
		case WM_WINDOWPOSCHANGED:
			return Tablefunction(&rangetbl,hwnd,msg,wparam,lparam);
		
		
		case WM_USER_CHALL:
		case WM_USER_CHMEM:
			// Something is changed, redraw window.
			InvalidateRect(hwnd, NULL, FALSE);
			return 0;

		case WM_PAINT:
			Painttable(hwnd, &rangetbl, RangetblDrawFunction);
			return 0;

		default: break;

	} 	// end switch (msg)

	return DefMDIChildProc(hwnd,msg,wparam,lparam);
};

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//						UpdateRangeTable
//
// Add ranges which OllyDbg recognizes as discrete procedures to the Range Table.
// Note that not ALL valid functions will be included here, just those that
// have been analyzed by Olly as such.
//////////////////////////////////////////////////////////////////////////

void UpdateRangeTable(ULONG baseaddress)
{

	// Range Table items
	t_range			range = {0};	// sorted data structure defining the procedure ranges
	t_range *		prange;			// pointer to sorted data structure defining the procedure ranges
	int				i;	

	// Included / Excluded ranges
	ULONG			procstart;
	ULONG			procend;
	ULONG			nextproc;
	char 			symb[512];
	char 			comment[TEXTLEN];
	char 			decodedaddress[512];
	
	/////////////////////////////////////////////

	// Create Range Table window.
//	RangetblCreatewindow();

	// Initialize common elements
	range.size = 1;
	range.type = 0;

	// Find the first procedure
	nextproc = Findprocbegin(baseaddress);
	if(nextproc == 0)
	{
		nextproc = Findnextproc(baseaddress);
	}				

	// Find subsequent procedures
	while(nextproc)
	{
		Getproclimits(nextproc, &procstart, &procend);
			
		Decodeaddress(procstart, 0, ADC_VALID, symb, sizeof(symb), comment);
		if(strlen(symb) == 0)
		{
			_snprintf(decodedaddress, sizeof(symb), "no symbolic information");
		}
		else
		{
			_snprintf(decodedaddress, sizeof(symb), "%s", symb);
		}				
				
			// Range Table: add to sorted data structure
			range.procstart = procstart;
			range.procend = procend-1;
			strcpy(range.symbolicname, decodedaddress);
					
			Addsorteddata(&rangetbl.data, &range);
					
			nextproc = Findnextproc(procend-1);
		}

				
	// Sort table by address before displaying
	Sortsorteddata(&rangetbl.data, COLUMN2_SORT_UP);

	/////////////////////////////////////////////
			
	prange = (t_range*)rangetbl.data.data;
			
	for(i=0; i < rangetbl.data.n; i++)
	{
		prange->index = i+1;	// Initialize an Item number for each entry
		prange++;
	}  
}

//////////////////////////////////////////////////////////////////////////



//////////////////////////////////////////////////////////////////////////
//						ODBG_Pluginsaveudd
//
// Save table info and gui dll data ranges to .udd file
//
// The data will be restored from the udd file the next time the process is opened,
// if and only if the user left the window active when OllyDbg was closed. 
// 
// The other functions required for this option are
// Pluginwriteinttoini in ODBG_Pluginclose and
// Pluginreadintfromini in ODBG_Plugininit
//
//////////////////////////////////////////////////////////////////////////

#define TAG_CBLP   0x43424C50L	// tag for custom table window data     
#define TAG_GUIS   0x47554953L	// tag for gui dll data buffer size    
#define TAG_GUIB   0x47554942L	// tag for gui dll data buffer  

extc void _export cdecl ODBG_Pluginsaveudd(t_module *pmod,int ismainmodule)
{
    t_jmplog *jmplog;
    int i;

	PULONG pserialized_settings_buf;
	ULONG serialized_settings_size;

	//////////////////////////////////////

    if (ismainmodule==0)
        return; 

	if (!GUIDLLLOADED)
        return; 
	
	
	// Save table data which is displayed in the custom window.

    jmplog = (t_jmplog *)jmptbl.data.data;

    for (i=0; i<jmptbl.data.n; i++,jmplog++)
    {
		// Record data to udd file with a distinctive tag
        Pluginsaverecord(TAG_CBLP, sizeof(t_jmplog), jmplog);
    };


	// Save range data which is displayed in the gui dll
	// (included/excluded ranges, logfile name, etc.)
	//

	// Get range data
	// Function returns a pointer to a variable sized buffer containing the data,
	// as well as the size of the buffer

	pserialized_settings_buf = (PULONG)GetSerializedSettings(&serialized_settings_size);
	
	// Record both the size of the buffer and the buffer itself
	// to the udd, under two separate tags
	
	Pluginsaverecord(TAG_GUIS, sizeof(ULONG), &serialized_settings_size);
	Pluginsaverecord(TAG_GUIB, serialized_settings_size, pserialized_settings_buf);
	
};

//////////////////////////////////////////////////////////////////////////
//						ODBG_Pluginuddrecord
//
// Restore table info and gui dll data ranges from .udd file
//
// OllyDbg calls this function separately for each udd tag registered,
// in the order they were recorded.  This is fortunate because we can
// get the size of the variable sized gui dll data range buffer first,
// before we retrieve the buffer itself.
// 
//////////////////////////////////////////////////////////////////////////

extc int _export cdecl ODBG_Pluginuddrecord(t_module *pmod,int ismainmodule,
  ulong tag,ulong size,void *data)
{

    t_jmplog jmplog;
	t_jmplog *udddata=(t_jmplog *)data;

	//////////////////////////////////////

    if (ismainmodule==0)
        return 0;

	if (!GUIDLLLOADED)
        return 0; 
    
	
	if (tag == TAG_CBLP)				// custom table window data
	{
		jmplog.size=1;
		jmplog.type=0;

		SetCursor(hCursorWait);

		// Restore these table entries, the rest will be restored dynamically upon display
		jmplog.index = udddata->index;
		jmplog.addr = udddata->addr;
		jmplog.count_jump_taken = 0;
		jmplog.count_jump_not_taken = 0;    
		strcpy(jmplog.last_jump_result, "");
    
		Addsorteddata(&(jmptbl.data),&jmplog);
    
		return 1;						// Record processed

	}


    if (tag == TAG_GUIS)				// gui dll data buffer size 

	{
		g_serialized_settings_size = *(ULONG*) data;
		return 1;
	}

    if (tag == TAG_GUIB)				// gui dll data buffer

	{
		g_serialized_settings_buf = (ULONG*) data;

		// Restore data ranges to gui dll window
		SetSerializedSettings(g_serialized_settings_buf, g_serialized_settings_size);
		return 1;
	}

	return 0;

};


//////////////////////////////////////////////////////////////////////////
//						ToolbarButtonWinproc
//
// Handler function for our custom Toolbar Button
//////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK ToolbarButtonWinproc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{

	//////////////////////////////////////

	switch (uMsg)
	{
		case WM_COMMAND:

			if (lParam == (LPARAM)hToolbarButton)
			{
				// Create/restore Jump Table window.
				JmptblCreatewindow();

				// Display Range Table window
				Quicktablewindow(&rangetbl, 15, 4, rangetblwinclass, "Conditional Branch Logger - Set Ranges By Procedure");

			}

			break;

		//////////////////////////////////////		
		
		default: break;

	}

	return CallWindowProc(OldWndProc, hWnd, uMsg, wParam, lParam);

};

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//						CreateTooltip
//
// Create a tooltip for our toolbar button
//////////////////////////////////////////////////////////////////////////

void CreateTooltip(HWND hwnd)
{

    INITCOMMONCONTROLSEX	iccex; 
    HWND					hwndTT;
    TOOLINFO				ti;
    unsigned int			uid = 0;
    char					strTT[32] = "Conditional Branch Logger";
    LPTSTR					lptstr = strTT;
    RECT					rect;

	//////////////////////////////////////

    // Initialize Common Controls
    iccex.dwICC = ICC_BAR_CLASSES;
    iccex.dwSize = sizeof(INITCOMMONCONTROLSEX);
    InitCommonControlsEx(&iccex);


    // Create a tooltip window

	hwndTT = CreateWindowEx(WS_EX_TOPMOST, TOOLTIPS_CLASS, NULL,
                            WS_POPUP | TTS_NOPREFIX | TTS_ALWAYSTIP,
                            CW_USEDEFAULT, CW_USEDEFAULT,
                            CW_USEDEFAULT, CW_USEDEFAULT,
                            hwnd, NULL, hinst, NULL);

	SetWindowPos(hwndTT, HWND_TOPMOST,0, 0, 0, 0,
				SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);	

    GetClientRect (hwnd, &rect);

    // Initialize TOOLINFO structure
    ti.cbSize = sizeof(TOOLINFO);
    ti.uFlags = TTF_SUBCLASS;
    ti.hwnd = hwnd;
    ti.hinst = hinst;
    ti.uId = uid;
    ti.lpszText = lptstr;
    ti.rect.left = rect.left;    
    ti.rect.top = rect.top;
    ti.rect.right = rect.right;
    ti.rect.bottom = rect.bottom;

    SendMessage(hwndTT, TTM_ADDTOOL, 0, (LPARAM) (LPTOOLINFO) &ti);	

}

//////////////////////////////////////////////////////////////////////////


void BroadcastDllLoadFailure()
{
	Addtolist(0,1,"cbl_gui.dll failed to load correctly");
	Addtolist(0,1,"Ensure dll is in the OllyDbg plugin or main directory or system %PATH%");
	Flash("cbl_gui.dll failed to load correctly. See OllyDbg Log window");
}


void START_CODE_EXECUTION_TIMER()
{
	// Get the frequency of the hi-res timer
	QueryPerformanceFrequency(&freq);

	// Start tick counter - ignore overhead time of the API itself
	QueryPerformanceCounter(&start_time);
}


void STOP_CODE_EXECUTION_TIMER()
{

	char			szElapsedTime[64];
	char			szCBL[128];
	
	// Stop tick counter
	QueryPerformanceCounter(&stop_time);

	// Calculate time in milliseconds, omit * 1000 for time in seconds
	time_diff = ((stop_time.QuadPart - start_time.QuadPart) * 1000) / freq.QuadPart;

	if(time_diff > 1000)
	{

		sprintf(szElapsedTime, "   - %i conditional jumps processed in %.2lf s", jmptbl.data.n, time_diff/1000);

	}else{

		sprintf(szElapsedTime, "   - %i conditional jumps processed in %.1lf ms", jmptbl.data.n, time_diff);
	}
	
	strcpy(szCBL, "Conditional Branch Logger");
	lstrcat(szCBL, szElapsedTime);

	// Update table caption with processing time
	SendMessage (jmptbl.hw, WM_SETTEXT, 0, (LPARAM) szCBL);
			
}



//////////////////////////////////////////////////////////////////////////
//						HelpWindow
//
// Create an edit window to display help file
//////////////////////////////////////////////////////////////////////////

BOOL HelpWindow()
{

	WNDCLASS	wc; 
	ULONG		Wwd = 640;
	ULONG		Wht = 480;
	ULONG		Wtx;
	ULONG		Wty;
	
	//////////////////////////////////////
	
	// Register window class
	wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc   = (WNDPROC) HelpWndProc; 
    wc.cbClsExtra    = 0; 
    wc.cbWndExtra    = 0; 
    wc.hInstance     = hinst; 
    wc.hIcon         = LoadIcon(hinst, (LPCTSTR)"CBL_ICO"); 
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW); 
    wc.hbrBackground = (HBRUSH) (COLOR_APPWORKSPACE + 1); 
    wc.lpszMenuName  = NULL; 
    wc.lpszClassName = "CBL"; 
	
	RegisterClass(&wc);

	//------------------------------------------------	

	// Centre window at following size
	Wtx = (GetSystemMetrics(SM_CXSCREEN) - Wwd) / 2;
	Wty = (GetSystemMetrics(SM_CYSCREEN) - Wht) / 2;


	// Create the window
	
	hwndHelpWindow = CreateWindow((LPCSTR)"CBL", "Conditional Branch Logger",
		WS_OVERLAPPEDWINDOW,
		Wtx, Wty, Wwd, Wht,
		NULL, NULL, hinst, NULL);

	if (!hwndHelpWindow)
	{
		return FALSE;
	}

	ShowWindow(hwndHelpWindow, SW_SHOWNORMAL);
	UpdateWindow(hwndHelpWindow);

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
//						HelpWndProc
//
// Handler for the window
//////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK HelpWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{

	static HWND hwndEdit; 
	PAINTSTRUCT ps;
	RECT rt;

// "\r\n\" == "\x0D\x0A"

	//////////////////////////////////////

	CHAR Help[] = "Conditional Branch Logger (CBL) is a plugin which gives control and logging \
capabilities for conditional jumps over the full user address space of a process."
"\r\n\r\n"
  	"From the main OllyDbg plugin menu select Conditional Branch Logger -> Configuration."
"\r\n\r\n"
	"This will open a dialog for managing Included (Logged) and Excluded address ranges, \
conditional branch type selection and project settings. \
Ranges can be entered manually or selected from one of the Conditional Branch Logger context menu options \
available in other OllyDbg windows."
"\r\n\r\n"
	"Main CPU window (Alt+C): Add multiline selections as Included or Excluded ranges.\r\n"
	"Executable modules (Alt+E): Add code section as an Included range. \
Set ranges by procedure for any module. \r\n"
	"Memory map (Alt+M): Add any memory address region, including non-standard \
or memory mapped sections, as an Included range. Set ranges by procedure for any module \
section successfully analyzed."
"\r\n\r\n"
	"The Set Ranges by Procedure option, available from the main menu or one of the \
context menus, will open a window from which individual functions can be added as \
Included or Excluded ranges for any module. The module will be automatically analyzed \
via the OllyDbg code analysis function if required."
"\r\n\r\n"
	"Each time the main dialog is closed the CBL jump table window will be updated \
with the latest conditional jump breakpoint addresses. A context menu provides several \
options for further controlling the selected breakpoints both before and after analysis. \
When you are satisfied with the settings, run or single-step the target as usual. \
The results will be shown in the logfile and/or the CBL jump table window. \r\n"
	"A button on the OllyDbg toolbar can be used to show the two Conditional Branch Logger \
custom windows if they have been hidden or closed."
"\r\n\r\n"
	"A logfile can be specified from the main Configuration dialog in which all executed \
conditional branch instructions within the selected logging range and the result of whether the jump was taken or not \
are recorded.  If a logfile is not specified a default file named \"conditional_branch_logger_default.log\" \
in the OllyDbg main executable directory will be used. \
Such log files, from different runs of the same program, can then be compared using any good \
'diffing' program to find changes in the code execution path as a result of changing inputs or conditions."
"\r\n\r\n"
	"The majority of Conditional Branch Logger settings, including active breakpoints, \
are saved in the OllyDbg UDD project files and restored when the target is reopened. \
This means that you can log conditional branch instructions in system dlls such as ntdll.dll which \
occur even before the Entry Point of the target is reached."
"\r\n\r\n"
	"Any comments can be directed to the OllyDbg forums at \r\n"
	"http://www.woodmann.com"
"\r\n\r\n"	
  	"Regards,\r\n"
  	"Blabberer, dELTA and Kayaker\r\n"
; 

	//////////////////////////////////////////////////

	switch (message) 
	{

		case WM_CREATE: 

			hwndEdit = CreateWindow("EDIT", NULL,
				WS_CHILD | WS_VISIBLE | WS_VSCROLL | 
				ES_LEFT | ES_MULTILINE | ES_AUTOVSCROLL | ES_READONLY, 
				0, 0, 0, 0,
				hWnd, NULL, hinst, NULL);
 
			// Add text to the window. 
			SendMessage(hwndEdit, WM_SETTEXT, 0, (LPARAM) Help); 

			break; 
	

		case WM_SETFOCUS:
			
            SetFocus(hwndEdit); 
            break; 


        case WM_SIZE: 

            // Make the edit control the size of the window's client area. 
            MoveWindow(hwndEdit, 
                       0, 0,                  // starting x- and y-coordinates 
                       LOWORD(lParam),        // width of client area 
                       HIWORD(lParam),        // height of client area 
                       TRUE);                 // repaint window 
            break; 


        case WM_DESTROY: 

			DestroyWindow(hwndEdit);
			break; 


		case WM_PAINT:

			BeginPaint(hWnd, &ps);
			GetClientRect(hWnd, &rt);
			EndPaint(hWnd, &ps);
			break;

		case WM_CTLCOLORSTATIC:

			return (LRESULT) GetStockObject(WHITE_BRUSH); 
			// not necessary to delete stock object


		default:

			// Unhandled messages to DefWindowProc
			return DefWindowProc(hWnd, message, wParam, lParam);
   }

   return 0;
}

//////////////////////////////////////////////////////////////////////////
